C
      SUBROUTINE GMODES(G,S,NMODE,NDIM,NBLD,NS,CV)
      IMPLICIT REAL (A-H,M,O-Z)
      DIMENSION G(NMODE,NDIM,NS), S(NDIM,NS)
      DIMENSION NBLD(NS)
C-------------------------------------------------------------------------
C     Sets specified geometry perturbation modes at mode parameter 
C     values S.  S is typically either the fractional arc length 
C     from airfoil nose to TE or just x/c, although this distinction 
C     is immaterial here.  In any case, the airfoil side is indicated 
C     by the sign of S:
C
C     On airfoil top    side    0 > S >  1
C     On airfoil bottom side    0 < S < -1
C
C     N   number of side points
C     NS  number of sides
C     CV  LE curvature for defining some mode shapes
C
C                  . . .
C
C     This implementation simply reads mode number, scaling, and endpoints
C     from file "modes.xxx".  Each line in the file contains:
C
C    KDOF     mode dof number (ties it to DPOS1-n global dof)
C    IMODE    mode shape number (specifies mode shape, implemented in GFUN)
C    GWT      mode scaling factor  (- for inward displacement)
C    ST1,ST2  mode endpoint locations, using same convention as S array
C    IEL      element number to which mode pertains
C
C     IMODE can be the same on two or more lines.  This allows a geometry 
C     mode to be made up of two or more non-contiguous pieces.  A useful
C     instance of this is a pure camber or a pure thickness mode which
C     consists of two pieces opposite each other on the two airfoil sides.
C     For example,
C
C     2   5   1.0  0.2  0.7  3
C     2   5  -1.0  -.2  -.7  3
C
C     specifies a camber mode dof DMOD2 with shape #5, 
C     over interval 0.2 > |s| > 0.7 on element 3.
C     One mode can also have separate pieces on different elements.
C-------------------------------------------------------------------------
C
      PARAMETER (LMAX=125)
      DIMENSION KDOF(LMAX), IMODE(LMAX), IEL(LMAX)
      DIMENSION ST1(LMAX), ST2(LMAX), GWT(LMAX)
C
      CHARACTER*80 ARGP1, FNAME
C
C
C---- read mode info from file  modes.xxx
      CALL GETARG(1,ARGP1)
      FNAME = 'modes.' // ARGP1 
      OPEN(7,FILE=FNAME,STATUS='OLD',ERR=5)
      GO TO 9
C
C---- try to open  modes.xxx  without "_n" extension
 5    ISCORE = INDEX(ARGP1,'_')
      IF(ISCORE.EQ.0) GO TO 990
      FNAME = 'modes.' // ARGP1(1:ISCORE-1)
      OPEN(7,FILE=FNAME,STATUS='OLD',ERR=990)
C
 9    DO 10 L=1, LMAX
        READ(7,*,END=11) KDOF(L),IMODE(L),GWT(L),ST1(L),ST2(L),IEL(L)
 10   CONTINUE
      WRITE(*,*) 'GMODES: Array limit reached. Increase LMAX.'
      L = LMAX+1
 11   CONTINUE
      LL = L-1
      CLOSE(7)
C
C
C---- process file lines
      DO 20 L=1, LL
C
C------ mode dof number
        K = KDOF(L)
C
C------ element number
        N = IEL(L)
C
        IF(K.LT.1 .OR. K.GT.NMODE) THEN
         WRITE(*,*) 'GNSET: DOF index out of range. Skipping line',L
         GO TO 20
        ENDIF
C
        IF(N.LT.1 .OR. N.GT.NS/2) THEN
         WRITE(*,*) 'GNSET: Element index out of range. Skipping line',L
         GO TO 20
        ENDIF
C
C------ if mode dof number is outside passed-in range, ignore this line
        IF(K .LT. 1  .OR.  K .GT. NMODE) GO TO 20
C
C------ set curvature in same units as SNORM below
        CNORM = CV*ABS(ST2(L) - ST1(L))
C
C------ go over all surface grid nodes for specified element
        IS1 = 2*N - 1
        IS2 = 2*N
        DO 205 IS=IS1, IS2
C
          DO 2055 I=1, NBLD(N)
C
C---------- if this grid node is between specified endpoints, set the mode
            IF((S(I,IS).GE.ST1(L) .AND. S(I,IS).LE.ST2(L)) .OR.
     &         (S(I,IS).LE.ST1(L) .AND. S(I,IS).GE.ST2(L))     ) THEN
              SNORM = (S(I,IS) - ST1(L)) / (ST2(L) - ST1(L))
C
C------------ set mode shapes (customize as needed):
              IF(IMODE(L).EQ.20) THEN
C
C-------------- bump at LE ~ LE radius long
                G(K,I,IS) = GWT(L) * GFUNCV(SNORM,CNORM)
C
              ELSE IF(IMODE(L).EQ.60) THEN
C
C-------------- linear ramp
C--             (useful for hinged-flap deflection, 
C--               hinge at ST1, trailing edge at ST2)
                G(K,I,IS) = GWT(L) * SNORM
C
c              ELSE IF(IMODE(L).GT.40) THEN
cC
cC-------------- fudged sin^2 bumps
c                IMODS = IMODE(L) - 40
c                G(K,I,IS) = GWT(L) * GFUNSB(IMODS,SNORM)
C
              ELSE IF(IMODE(L).GT.20) THEN
C
C-------------- Tchebyshev polynomials
                IMODS = IMODE(L) - 20
                G(K,I,IS) = GWT(L) * GFUNTP(IMODS,SNORM)
C
              ELSE
C
C-------------- default case: sin bumps
ccc                G(K,I,IS) = GWT(L) * GFUN(IMODE(L),SNORM)

                IMODS = IABS(IMODE(L))
                G(K,I,IS) = GWT(L) * GFUNTP(IMODS,SNORM)

C
              ENDIF
            ENDIF
C
 2055     CONTINUE
C
 205    CONTINUE
C
 20   CONTINUE
C
      RETURN
C
 990  WRITE(*,1990) FNAME(1:60)
 1990 FORMAT(1X,'Open error on file ', A60)
      RETURN
C
      END ! GMODES



      FUNCTION GFUN(IFUN,S)
      IMPLICIT REAL (A-H,M,O-Z)
      DATA  PI / 3.1415926535897932384 /
C
      RF = FLOAT(IFUN)
      GFUN = SIN(RF*PI*S)/RF
      RETURN
C
      END ! GFUN



      FUNCTION GFUNTP(IFUN,S)
      IMPLICIT REAL (A-H,M,O-Z)
      DATA  PI / 3.1415926535897932384 /
C
C---- returns Tchebyshev polynomials, bunched towards s=0
C
C---- bunching factor at s=0
cc    SWT = 1.0         ! standard Tchebyshev
      SWT = 2.0
C
C
      SW = SWT*S / (1.0 + (SWT-1.0)*S)
C
      X = 1.0 - 2.0*SW
C
      X = MIN( 1.0 , X )
      X = MAX(-1.0 , X )
C
      THETA = ACOS(X)
C
      RF = FLOAT(IFUN+1)
      IF(MOD(IFUN,2).EQ.0) THEN
        GFUNTP = (  X - COS(RF*THETA))/RF
      ELSE
        GFUNTP = (1.0 - COS(RF*THETA))/RF
      ENDIF
C
      RETURN
      END ! GFUNTP



      FUNCTION GFUNSB(IFUN,S)
      IMPLICIT REAL (A-H,M,O-Z)
      DATA  PI / 3.1415926535897932384 /
C
      GO TO (1,2,3,4,5,6,7,8,9), IFUN
C
C---- center peak
cc 1    GFUNSB = SIN(PI*     S     ) ** 2
 1    GFUNSB = SIN(PI*     S     )
      RETURN
C
C---- peaks on the right
 2    GFUNSB = SIN(PI*     S **2 )
      RETURN
 3    GFUNSB = SIN(PI*     S **4 ) ** 2
      RETURN
C
C---- peaks on the left
 4    GFUNSB = SIN(PI*(1.0-S)**2 )
      RETURN
 5    GFUNSB = SIN(PI*(1.0-S)**4 )
      RETURN
 6    GFUNSB = SIN(PI*(1.0-S)**7 ) ** 2
      RETURN
 7    GFUNSB = SIN(PI*(1.0-S)**9 ) ** 2
      RETURN
 8    GFUNSB = SIN(PI*(1.0-S)**11) ** 2
      RETURN
 9    GFUNSB = SIN(PI*(1.0-S)**13) ** 2
      RETURN
C
      END ! GFUNSB


      FUNCTION GFUNCV(S,CV)
      IMPLICIT REAL (A-H,M,O-Z)
      DATA ACON / 2.0 /
C
C---- bump at LE for changing LE radius
      ARG = MIN( 15.0 , ACON*S*CV )
      GFUNCV = 16.0 * CV * S**2 * EXP(-ARG) * (1.0 - S)
C
      RETURN
      END ! GFUNCV
