C
      PROGRAM MEDP
C-------------------------------------------------------------
C     Interactive program for generating and modifying
C     specified pressure distributions for MSES.
C
C     User input can be via screen cursor or a scratch file
C     which is written out with the surface pressures or
C     isentropic Mach numbers, edited, and then read back in.
C-------------------------------------------------------------
C
      INCLUDE 'STATE.INC'
      CHARACTER*1 ANS
      CHARACTER*5 PLTDEV
      INCLUDE 'MEDP.INC'
C
      DTOR = ATAN(1.0)/45.0
C
C---- Plotting flag
      IDEV = 1   ! X11 window only
C     IDEV = 2   ! B&W PostScript output file only (no color)
C     IDEV = 3   ! both X11 and B&W PostScript file
C     IDEV = 4   ! Color PostScript output file only 
C     IDEV = 5   ! both X11 and Color PostScript file 

C---- Re-plotting flag (for hardcopy)
      IDEVRP = 2   ! B&W PostScript
C     IDEVRP = 4   ! Color PostScript
C
C---- PostScript output logical unit and file specification
      IPSLU = 0  ! output to file  plot.ps   on LU 4    (default case)
C     IPSLU = ?  ! output to file  plot?.ps  on LU 10+?
C
C---- screen fraction taken up by plot window upon opening
      SCRNFR = 0.70
C
C---- Default plot size in inches
C-    (Default plot window is 11.0 x 8.5)
      SIZE = 10.0
C
C---- character size
      CH = 0.015
C
C---- initialize plot routines and set up basic colors
      CALL PLINITIALIZE
C
C---- color indices for top & bottom pressures
      ICOL1 = 5
      ICOL2 = 7
C
C---- flag indicating whether or not a plot is active on the screen
      LPLOT = .FALSE.
C
C---- flag indicating whether or not wall pressures have been set by user
      LPSET = .FALSE.
C
C---- flag indicating whether or not  mdat.xxx  needs to be saved
      LSAVE  = .FALSE.
C
C---- flag indicating whether endpoint slopes are to be matched for Cp(s)
      LSLOP = .TRUE.
C
C---- flag indicating whether grid is to be overlaid on Cp(s) plot
      LGRID = .TRUE.
C
C---- initialize and set up specified and wall pressure buffer arrays
      CALL EDINIT
C
  500 WRITE(*,2000)
 2000 FORMAT(/'  1   Edit Cp distributions'
     &       /'  2   Re-read mdat.xxx'
     &       /'  3   Write   mdat.xxx'
     &       /'  4   Write airfoil coordinate file'
     &       /'  5   List flow conditions and forces'
     &       /'  6   Change MSES flow parameters'
     &       /'  7   Plot size change'
     &       /'  8   Write CPwall file'
     &       /'  9   Read  CPspec file'
     &       /' 10   Write Mwall file'
     &       /' 11   Read  Mspec file'
     &       /' 12   Read  new airfoil coordinate file'
     &      //'    Select option (0=quit):  ', $)
      READ(*,*,ERR=500) NOPT
C
      GO TO (5,10,20,30,40,50,60,70,80,90,100,110,120), NOPT+1
      GO TO 500
C
C=================================
    5 IF(LPSET.AND.LSAVE) THEN
       WRITE(*,*) ' Modifications not saved.  Exit?   N'
       READ(*,1000) ANS
       IF (ANS.NE.'Y' .AND. ANS.NE.'y') GO TO 500
      ENDIF
      IF(LPLOT) CALL PLCLOSE
      STOP
C
C=================================
C---- edit Cp distributions
   10 CALL EDPRES
      GO TO 500
C
C=================================
C---- re-read mdat.xxx
   20 IF(LSAVE) THEN
       WRITE(*,*) ' Modifications will be lost.  Are you sure?   Y'
       READ(*,1000) ANS
       IF (ANS.NE.'N' .AND. ANS.NE.'n') GO TO 500
      ENDIF
      REWIND(1)
      CALL EDINIT
      GO TO 500
C
C=================================
C---- write mdat.xxx
   30 CALL GETNAM
      CALL OUT
      LSAVE = .FALSE.
      GO TO 500
C
C=================================
C---- write coordinate file
   40 CALL GETNAM
      CALL SAVE
      GO TO 500
C
C=================================
C---- list forces
   50 CONTINUE
      CLI = LIFT/QU
      CDI = DRAG/QU
      CMI = MOMN/QU
      CDV = DRAGV/QU
      CDW = DRAGW/QU
      CLTOT = CLI 
      CDTOT = CDW + CDV
      CMTOT = CMI 
      ELOD = 0.0
      ADEG = ALFA / DTOR
      IF(CDTOT.NE.0.0) ELOD = CLTOT/CDTOT
C
      WRITE(*,8000) MINF,REINF*1.0E-6,ACRIT
      WRITE(*,8010) CLTOT, CDTOT, CMTOT,
     &              CDI,   CDV,   CDW,
     &              ELOD, QU
      WRITE(*,8020) CIRC, ADEG, DOUX, DOUY
      DO 1004 N = 1, NBL
        BCL  = BLIFT(N)/QU
        BCDI = BDRAG(N)/QU
        BCM  = BMOMN(N)/QU
        BCDV = BDRAGV(N)/QU
        WRITE(*,8030) N, BCL, BCM, BCDI, BCDV
        IF(REYN.NE.0.0) WRITE(*,8040) XTR(IS1(N)), XTR(IS2(N))
 1004 CONTINUE
      GO TO 500
C
C=================================
C---- change MSES flow parameters
   60 CALL PARSET
      GO TO 500
C
C=================================
   70 WRITE(*,*) 'Current plot size = ', SIZE
   71 WRITE(*,*) 'Enter new plot size:'
      READ(*,*,ERR=71) SIZE
      GO TO 500
C
 80   CALL CPDUMP
      GO TO 500
C
 90   CALL CPREAD
      GO TO 500
C
 100  CALL MDUMP
      GO TO 500
C
 110  CALL MREAD
      GO TO 500
C
 120  CALL READAI
      GO TO 500
C
C................................................................
C
 1000 FORMAT(A)
C
 8000 FORMAT(/' Ma  =',F10.4,'   Re  =',F8.3,'e6',
     &                                        '   Ncrit =',F8.2)
 8010 FORMAT(/' CL  =',F10.5,'   CD  =',F10.5,'   CM  =',F10.5
     &       /' CDp =',F10.6,'   CDv =',F10.6,'   CDw =',F10.6
     &       //' L/D =',F10.2,/' dynamic pressure =',F9.6)
 8012 FORMAT(/' CY  =',F10.5,'   CX  =',F10.5,'   CM  =',F10.5
     &       /' CXp =',F10.6,'   CXv =',F10.6,'   CXw =',F10.6
     &       //' dynamic pressure =',F9.6,'   exit pressure =',F9.5)
 8015 FORMAT(/' Sinl  =',F8.4,'    Sout  = ',F8.4)
 8020 FORMAT(/' Gamma =',F8.4,'    alpha = ',F8.4,' deg.'
     &       /' Doubx =',F8.4,'    Douby = ',F8.4)
 8030 FORMAT(/' Element :',I2
     &        /'   CL  =',F10.5,'   CM  =',F10.5
     &        /'   CDp =',F10.6,'   CDv =',F10.6)
 8040 FORMAT( ' S xtr =',F8.4,'   P xtr = ',F8.4)
C
      END ! EDP

 

      SUBROUTINE EDINIT
C-----------------------------------------
C     Initializes various variables and
C     specified and actual pressure
C     distribution buffer arrays.
C-----------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MEDP.INC'
      DIMENSION Q(IX), QS(IX), P(IX), PIDUM(IX)
C
      CALL INPUT
      CALL INDINI
C
      GM1 = GAM - 1.0
      GP1 = GAM + 1.0
C
      QU = 0.5*RHOINF*QINF**2
      REINF = REYN * (RHOINF*QINF/MUINF)
C
      CPSTAG = (PSTOUT-PINF)/QU
      CPSTAR = (PSTAR -PINF)/QU
C
      DO 100 N=1, NBL
C
        ILE = ILEB(N)
        ITE = ITEB(N)
C
        I1 = IS1(N)
        I2 = IS2(N)
        J1 = JS1(N)
        J2 = JS2(N)
C
C------ calculate streamtube quantities
        CALL PICALC(J1  ,Q,QS,P,PI(1,I1),PIDUM   )
        CALL PICALC(J2-1,Q,QS,P,PIDUM   ,PI(1,I2))
C
C------ set actual and specified normalized Cp buffer arrays
        DPINF = PINF - PSTOUT
        DO 20 IG=1, NBLD(N)
          I = ILE+IG-1
          CPW(IG,I1) = (PI(I,I1)    -DPINF)/QU
          CPW(IG,I2) = (PI(I,I2)    -DPINF)/QU
          CPS(IG,I1) = (PSPEC(IG,I1)-DPINF)/QU
          CPS(IG,I2) = (PSPEC(IG,I2)-DPINF)/QU
   20   CONTINUE
C
        IGS(1,N) = 2
        IGS(2,N) = NBLD(N)
        LPSET = .FALSE.
C
        IF(N.EQ.NMIX) THEN
         IGS(1,N) = IX0-ILE+1
         IGS(2,N) = IX1-ILE+1
         WRITE(*,1010) IX0, IX1, N
 1010    FORMAT(/' Old mixed-inverse target segment endpoints: ',
     &             2I4, '   element',I2)
         LPSET = .TRUE.
        ENDIF
C
        ISMIX = 0
C
 100  CONTINUE
C
      RETURN
      END ! EDINIT


      SUBROUTINE INDINI
C--------------------------------------
C     Initializes various grid indices
C--------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MEDP.INC'
C
      DO 20 N = 1, NBL
C
        ILE = NINL(N)
        ITE = II-NOUT(N)+1
        ILEB(N) = ILE
        ITEB(N) = ITE
C
        IS1(N) = 2*(N-1)+1
        IS2(N) = IS1(N)+1
C
        JS1(N) = JBLD(N)
        JS2(N) = JS1(N) - 1
        IF (JS2(N).LT.1) JS2(N) = JJ
C
   20 CONTINUE
C
      RETURN
      END ! INDINI


      SUBROUTINE GETNAM
      INCLUDE 'STATE.INC'
      CHARACTER*1 ANS
      INCLUDE 'MEDP.INC'
C
      WRITE(*,1000) NAME
 1000 FORMAT(/' Current airfoil name: ', A)
C
      WRITE(*,*) 'Specify new airfoil name ?   N'
      READ(*,9000) ANS
      IF(ANS.EQ.'Y' .OR. ANS.EQ.'y') THEN
       WRITE(*,*) 'Enter new name:'
       READ(*,9000) NAME
      ENDIF
C
 9000 FORMAT(A)
      RETURN
      END ! GETNAM



      SUBROUTINE EDPRES
C--------------------------------------
C     Specified-Cp editing routine.
C--------------------------------------
      INCLUDE 'STATE.INC'
      CHARACTER*1 OPT, ANS
      INCLUDE 'MEDP.INC'
C
      IF(NBL.EQ.1) NMIX = 1
C
      IF(NMIX.EQ.0) THEN
       WRITE(*,*)
 3     WRITE(*,*) 'Enter target element'
       READ (*,*,ERR=3) NMIX
       IF(NMIX.LT.1 .OR. NMIX.GT.NBL) GO TO 3
      ELSE
       WRITE(*,1200) NMIX
 1200  FORMAT(/'  Target element = ', I2)
      ENDIF
C
C
      CALL PLOTCP(.TRUE.)
C
 490  WRITE(*,1000)
 1000 FORMAT(/'   I nitialize CPspec <== CPwall for target element'
     &       /'   M odify Cp'
     &       /'   S lope-matching at input-Cp endpoints (toggle)'
     &       /'   D emark inverse segment'
     &       /'   P lot Cp vs s/smax'
     &       /'   G rid-plotting toggle'
     &       /'   V ector Cp plot'
     &       /'   B lowup'
     &       /'   R eset to original size'
     &       /'   A nnotate plot'
     &       /'   H ardcopy current plot' )
      IF(NBL.GT.1) WRITE(*,1010)
 1010 FORMAT( '   N ew target element')
C
  500 WRITE(*,1020)
 1020 FORMAT(/'     Select edit option:  ', $)
C
      READ(*,2000) OPT
C
      IF(OPT .EQ. ' ') THEN
        IF(LPLOT) CALL PLEND
        LPLOT = .FALSE.
        IF (LPSET) THEN
          N = NMIX
          I1 = IS1(N)
          I2 = IS2(N)
          J1 = JS1(N)
          J2 = JS2(N)
C
C-------- set freewall segment indices from buffer indices before returning
          IX0 = ILEB(N) - 1 + IGS(1,N)
          IX1 = ILEB(N) - 1 + IGS(2,N)
C
C-------- set 2nd pressure derivatives for pressure regularity constraints
          PXX0(I1) = PI(IX0-1,I1) - 2.0*PI(IX0,I1) + PI(IX0+1,I1)
          PXX1(I1) = PI(IX1-1,I1) - 2.0*PI(IX1,I1) + PI(IX1+1,I1)
          PXX0(I2) = PI(IX0-1,I2) - 2.0*PI(IX0,I2) + PI(IX0+1,I2)
          PXX1(I2) = PI(IX1-1,I2) - 2.0*PI(IX1,I2) + PI(IX1+1,I2)
        ENDIF
        RETURN
C
      ELSE IF(INDEX('Ii',OPT).NE.0) THEN
C
        CALL SETSP(NMIX)
        LPSET = .TRUE.
        CALL PLOTCP(.FALSE.)
        CALL FORCE(CLW,CMW,CLS,CMS,NMIX)
        WRITE(*,2100) NMIX, CLW,CMW,CLS,CMS
C
      ELSE IF(INDEX('Mm',OPT).NE.0) THEN
C
        IF(.NOT.LCPPL) CALL PLOTCP(.FALSE.)
        CALL MODIFY
        LCPPL = .FALSE.
        CALL FORCE(CLW,CMW,CLS,CMS,NMIX)
        WRITE(*,2100) NMIX,CLW,CMW,CLS,CMS
C
      ELSE IF(INDEX('Ss',OPT).NE.0) THEN
C
        LSLOP = .NOT. LSLOP
        IF(     LSLOP) WRITE(*,*) 
     &                 'Input-Cp endpoint slopes will be matched.'
        IF(.NOT.LSLOP) WRITE(*,*)
     &                 'Input-Cp endpoint slopes will not be matched.'
C
      ELSE IF(INDEX('Dd',OPT).NE.0) THEN
C
        IF(LAIPL) THEN
          CALL IGAGET
        ELSE
          IF(.NOT.LCPPL) CALL PLOTCP(.FALSE.)
          CALL IGSGET
          LCPPL = .FALSE.
        ENDIF
        IX0 = ILEB(NMIX) - 1 + IGS(1,NMIX)
        IX1 = ILEB(NMIX) - 1 + IGS(2,NMIX)
C
      ELSE IF(INDEX('Pp',OPT).NE.0) THEN
C
        CALL PLOTCP(.FALSE.)
C
      ELSE IF(INDEX('Gg',OPT).NE.0) THEN
C
        LGRID = .NOT. LGRID
        CALL PLOTCP(.FALSE.)
C
      ELSE IF(INDEX('Vv',OPT).NE.0) THEN
C
        CALL PLOTAI
C
      ELSE IF(INDEX('Bb',OPT).NE.0) THEN
C
        IF(.NOT.LCPPL) CALL PLOTCP(.FALSE.)
        XLEN = XWIND/SIZE
        YLEN = YWIND/SIZE
        CALL OFFGET(XOFF,YOFF,XSF,YSF,XLEN,YLEN,.FALSE.,.TRUE.)
        CALL PLOTCP(.FALSE.)
C
      ELSE IF(INDEX('Rr',OPT).NE.0) THEN
C
        CALL PLOTCP(.TRUE.)
C
      ELSE IF(INDEX('Aa',OPT).NE.0) THEN
C
        CALL ANNOT(CH)
C
      ELSE IF(INDEX('Hh',OPT).NE.0) THEN
C
        IF(LPLOT) CALL PLOT(0.,0.,-999)
        LPLOT = .FALSE.
        CALL REPLOT(IDEVRP)
C
      ELSE IF(INDEX('Nn',OPT).NE.0) THEN
C
        WRITE(*,*)
        WRITE(*,*) 'Current target element:', NMIX
        IF(NBL.GT.1) THEN
  151     WRITE(*,*) 'Input new element:'
          READ(*,*,ERR=151) NMIX
          IF(NMIX.LT.1 .OR. NMIX.GT.NBL) GO TO 151
        ENDIF
C
      ELSE
C
        GO TO 490
C
      ENDIF
C
      GO TO 500
C
 2000 FORMAT(A)
 2100 FORMAT (/1X, I2,'    CL    =', F9.5,5X,'CM    =',F9.5
     &        /1X, 2X,'    CLspec=', F9.5,5X,'CMspec=',F9.5)
      END ! EDPRES


      SUBROUTINE PLTINI
C---------------------------------------------
C     Initializes plot and gets window size
C---------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MEDP.INC'
C
      IF(LPLOT) CALL PLEND
C
      CALL PLOPEN(SCRNFR,IPSLU,IDEV)
C
C---- set X-window size in inches (might have been resized by user)
      CALL GETWINSIZE(XWIND,YWIND)
C
      CALL NEWFACTOR(SIZE)
      LPLOT = .TRUE.
C
      RETURN
      END


      SUBROUTINE PLOTCP(LOFINI)
C----------------------------------------------
C     Plots the actual and specified pressure
C     distributions as functions of fractional
C     arc length from LE to TE.
C     If LOFINI=t, initializes scales/offsets.
C----------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MEDP.INC'
      LOGICAL LOFINI
C
      DATA LMASK1, LMASK2, LMASK3 / -32640, -30584, -21846 /
C
      XMOD(XTMP) = XSF * (XTMP - XOFF)
      YMOD(YTMP) = YSF * (YTMP - YOFF)
C
C
C---- symbol height
      SH = 0.6*CH
C
C---- Cp and Mach annotation deltas
      DCP = 0.2
      DMA = 0.1
C
      N = NMIX
      I1 = IS1(N)
      I2 = IS2(N)
C
      CHORD = SQRT((XBTAIL(N)-XBNOSE(N))**2 + (YBTAIL(N)-YBNOSE(N))**2)
C
C---- find min CP for scaling purposes
      CPMIN = CPW(1,I1)
      DO 5 IG=1, NBLD(N)
        CPMIN = MIN( CPMIN , CPW(IG,I1) )
        CPMIN = MIN( CPMIN , CPW(IG,I2) )
    5 CONTINUE
      DCP = 0.2
      IF (CPMIN.GE.-0.6)  DCP = 0.1
      IF (CPMIN.LT.-2.5)  DCP = 0.5
C
C---- reset upper Cp limit to integer multiple of annotation delta(Cp)
      CPMIN = FLOAT( INT(CPMIN/DCP-0.001) - 1 ) * DCP
C
C---- Cp scaling factor
      PWT = 1.0/(CPSTAG-CPMIN)
C
C---- initialize plot, set current window size
      CALL PLTINI
C
C---- default offsets
      IF(LOFINI) THEN
        XADD = 0.12
        YADD = 0.10
        XSF = (XWIND/SIZE) / (1.0 + 2.0*XADD)
        YSF = (YWIND/SIZE) / (1.0 + 2.0*YADD)
        XOFF = -XADD
        YOFF = -YADD - CPSTAG*PWT
      ENDIF
C
      CALL GETCOLOR(ICOL0)
C
ccc      CALL PLOT(7.0*CH,3.0*CH,-3)
C
C======================
C      CPMIN = -2.0
C      PWT = 0.75/CPMIN
C======================
C
      CALL NEWPEN(3)
      AFAC = 0.5/CHORD
      AXOFF = XBNOSE(N) - 0.25/AFAC
      AYOFF = YBNOSE(N) - 0.20/AFAC
C
C---- draw axes
      CALL NEWPEN(1)
      CALL PLOT(XMOD(0.0),YMOD(-CPSTAG*PWT),3)
      CALL PLOT(XMOD(1.0),YMOD(-CPSTAG*PWT),2)
      CALL PLOT(XMOD(0.0),YMOD(-CPSTAG*PWT),3)
      CALL PLOT(XMOD(0.0),YMOD(-CPMIN *PWT),2)
      CALL PLOT(XMOD(1.0),YMOD(-CPSTAG*PWT),3)
      CALL PLOT(XMOD(1.0),YMOD(-CPMIN *PWT),2)
C
C---- annotate left y-axis with CPs
      NT1 = INT(  CPMIN/DCP - 0.01 )
      NT2 = INT( CPSTAG/DCP + 0.01 )
      DO 20 NT=NT1, NT2
        CPNUM = FLOAT(NT)*DCP
        YPLT = -CPNUM*PWT
        DXCH = 3.4*CH
        IF(ABS(CPNUM) .GE. 10.0) DXCH = DXCH + CH
        IF(    CPNUM  .LT.  0.0) DXCH = DXCH + CH
        CALL NEWPEN(2)
        CALL PLNUMB(XMOD(0.0)-DXCH,YMOD(YPLT)-0.5*CH,CH,CPNUM,0.0,1)
        CALL NEWPEN(1)
        CALL PLOT(XMOD(0.0)       ,YMOD(YPLT),3)
        CALL PLOT(XMOD(0.0)-0.2*CH,YMOD(YPLT),2)
   20 CONTINUE
C
      CALL NEWPEN(3)
      YPLT = (-CPMIN-1.5*DCP)*PWT
      CALL PLCHAR(XMOD(0.0)-5.7*CH,YMOD(YPLT)-0.7*CH,1.3*CH,'C',0.0,1)
      CALL PLCHAR(XMOD(0.0)-4.6*CH,YMOD(YPLT)-1.0*CH,1.0*CH,'p',0.0,1)
C
C---- draw x-axis tick marks
      CALL NEWPEN(1)
      DO 25 NT=0, 20
C
C------ set element x/c
        XTICK = 0.05*FLOAT(NT)
C
C------ absolute x/c
        XBC = XBNOSE(N) + XTICK*(XBTAIL(N)-XBNOSE(N))
C
C------ find corresponding upper and lower surface fractional arc lengths
        SUPP = SBNOSE(N) + XTICK*(SB(1     ,N)-SBNOSE(N))
        SLOW = SBNOSE(N) + XTICK*(SB(IIB(N),N)-SBNOSE(N))
        IF(NT .GT. 0) THEN
         CALL SINVRT(SUPP,XBC,XB(1,N),XPB(1,N),SB(1,N),IIB(N))
         CALL SINVRT(SLOW,XBC,XB(1,N),XPB(1,N),SB(1,N),IIB(N))
        ENDIF
C
        SFUPP = (SUPP-SBLE(N))/(SB(1     ,N)-SBLE(N))
        SFLOW = (SLOW-SBLE(N))/(SB(IIB(N),N)-SBLE(N))
C
C------ set tick mark length
        TICKL = 0.20*CH
        IF(MOD(NT, 2) .EQ. 0) TICKL = 0.40*CH
        IF(MOD(NT,10) .EQ. 0) TICKL = 0.70*CH
C
C------ draw tick marks
        YPLT = -CPSTAG*PWT
C
        IF(SUPP.LE.SBNOSE(N)) THEN
         CALL NEWCOLOR(ICOL1)
         CALL PLOT(XMOD(SFUPP),YMOD(YPLT)      ,3)
         CALL PLOT(XMOD(SFUPP),YMOD(YPLT)+TICKL,2)
        ENDIF
C
        IF(SLOW.GE.SBNOSE(N)) THEN
         CALL NEWCOLOR(ICOL2)
         CALL PLOT(XMOD(SFLOW),YMOD(YPLT)      ,3)
         CALL PLOT(XMOD(SFLOW),YMOD(YPLT)-TICKL,2)
        ENDIF
   25 CONTINUE
C
      IF(LGRID) THEN
       CALL NEWCOLOR(ICOL0)
       NXGR = 1
       NYGR = NT2 - NT1
       DXG = XSF * 1.0
       DYG = YSF * DCP*PWT
       CALL NEWPEN(1)
       CALL PLGRID(XMOD(0.0),YMOD(FLOAT(-NT2)*DCP*PWT), 
     &             NXGR,DXG, NYGR,DYG, LMASK2 )
      ENDIF
C
C---- put label on x-axis
      CALL NEWPEN(2)
      XPLT = 0.80
      YPLT = -CPSTAG*PWT
      CALL NEWCOLOR(ICOL1)
      CALL PLCHAR(XMOD(XPLT)       ,YMOD(YPLT)+1.5*CH,1.2*CH,'x'  ,0.,1)
      CALL PLCHAR(XMOD(XPLT)+1.2*CH,YMOD(YPLT)+1.1*CH,0.8*CH,'upp',0.,3)
      CALL PLCHAR(XMOD(XPLT)+3.7*CH,YMOD(YPLT)+1.5*CH,1.2*CH,'/c' ,0.,2)
      CALL NEWCOLOR(ICOL2)
      CALL PLCHAR(XMOD(XPLT)       ,YMOD(YPLT)-3.0*CH,1.2*CH,'x'  ,0.,1)
      CALL PLCHAR(XMOD(XPLT)+1.2*CH,YMOD(YPLT)-3.4*CH,0.8*CH,'low',0.,3)
      CALL PLCHAR(XMOD(XPLT)+3.7*CH,YMOD(YPLT)-3.0*CH,1.2*CH,'/c' ,0.,2)
C
C---- annotate right y-axis with isentropic Mach number
      CALL NEWCOLOR(ICOL0)
      YPLT = 0.
      DO 27 NT=1, 40
        MACH = FLOAT(NT)*DMA
        TRAT = 1.0 + 0.5*GM1*MACH**2
        PRES = PSTOUT / TRAT**(GAM/GM1)
        CP = (PRES-PINF)/QU
        YPLT = -CP*PWT
        IF(CP.LT.CPMIN) GO TO 28
        CALL NEWPEN(2)
        CALL PLNUMB(XMOD(1.0)+0.8*CH,YMOD(YPLT)-0.5*CH,CH,MACH,0.0,1)
        CALL NEWPEN(1)
        CALL PLOT(XMOD(1.0)        ,YMOD(YPLT),3)
        CALL PLOT(XMOD(1.0)+0.25*CH,YMOD(YPLT),2)
   27 CONTINUE
   28 CONTINUE
C
      IF(NT.GT.1) THEN
       CALL NEWPEN(3)
       MACH = (FLOAT(NT)-1.5)*DMA
       TRAT = 1.0 + 0.5*GM1*MACH**2
       PRES = PSTOUT / TRAT**(GAM/GM1)
       CP = (PRES-PINF)/QU
       YPLT = -CP*PWT
       CALL PLCHAR(XMOD(1.0)+3.8*CH,YMOD(YPLT)-0.7*CH,1.3*CH,'M',0.0,1)
      ENDIF
C
C---- plot sonic line
      CALL NEWPEN(1)
      IF(CPSTAR.GT.CPMIN)
     &   CALL DASH(XMOD(0.0),XMOD(1.0),YMOD(-CPSTAR*PWT))
C
C---- plot Cp = 0 line
      CALL PLOT(XMOD(0.0),YMOD(0.0),3)
      CALL PLOT(XMOD(1.0),YMOD(0.0),2)
C
C---- put nifty MSES version label on upper right corner
      CALL NEWPEN(2)
      XPLT = 1.0
      YPLT = -CPMIN*PWT
      CALL PLCHAR(XMOD(XPLT)-4.5*CH,YMOD(YPLT)-1.2*CH,0.90*CH,
     &            'MSES' ,0.0,4)
      CALL PLCHAR(XMOD(XPLT)-4.5*CH,YMOD(YPLT)-2.6*CH,0.70*CH,
     &            'v'    ,0.0,1)
      CALL PLNUMB(XMOD(XPLT)-3.0*CH,YMOD(YPLT)-2.6*CH,0.70*CH,
     &            VERSION,0.0,1)
C
      IF(NBL.GT.1) THEN
C----- element label on lower left
       RNMIX = FLOAT(NMIX)
       XPLT = 0.0
       YPLT = -CPSTAG*PWT
       CALL PLCHAR(XMOD(XPLT)       ,YMOD(YPLT)-2.8*CH,0.80*CH,
     &             'Element',0.0,7)
       CALL PLNUMB(XMOD(XPLT)+6.5*CH,YMOD(YPLT)-2.8*CH,0.80*CH,
     &             RNMIX,0.0,-1)
      ENDIF
C
      IG1 = IGS(1,N)
      IG2 = IGS(2,N)
C
      CALL NEWPEN(2)
      DO 50 IS=I1, I2
        IF(IS.EQ.I1) ICOL = ICOL1
        IF(IS.EQ.I2) ICOL = ICOL2
C
C------ plot actual pressures with symbols
        CALL NEWCOLOR(ICOL0)
        DO IG=1, NBLD(N)
          IF(IG.EQ.IG1) CALL NEWCOLOR(ICOL)
          CALL PLSYMB(XMOD(SG(IG,IS)),YMOD(-CPW(IG,IS)*PWT),SH,3,0.,0)
          IF(IG.EQ.IG2) CALL NEWCOLOR(ICOL0)
        ENDDO
C
C------ plot specified pressures with continuous line
        CALL NEWCOLOR(ICOL0)
        CALL PLOT(XMOD(SG(1,IS)),YMOD(-CPS(1,IS)*PWT),3)
        DO IG=2, NBLD(N)
          CALL PLOT(XMOD(SG(IG,IS)),YMOD(-CPS(IG,IS)*PWT),2)
          IF(IG.EQ.IG1) CALL NEWCOLOR(ICOL)
          IF(IG.EQ.IG2) CALL NEWCOLOR(ICOL0)
        ENDDO
   50 CONTINUE
C
C---- mark freewall segment endpoints with small symbols
      SH = 0.016
      CALL NEWPEN(3)
      DO 70 IS = I1, I2
        IF(IS.EQ.I1) CALL NEWCOLOR(ICOL1)
        IF(IS.EQ.I2) CALL NEWCOLOR(ICOL2)
C
        IF(IG1.NE.0) THEN
         YPLT = -CPS(IG1,IS)*PWT
         CALL PLSYMB(XMOD(SG(IG1,IS)),YMOD(YPLT),SH,5,0.0,0)
C         CALL PLOT(XMOD(SG(IG1,IS)),YMOD(YPLT)-0.02,3)
C         CALL PLOT(XMOD(SG(IG1,IS)),YMOD(YPLT)+0.02,2)
        ENDIF
        IF(IG2.NE.0) THEN
         YPLT = -CPS(IG2,IS)*PWT
         CALL PLSYMB(XMOD(SG(IG2,IS)),YMOD(YPLT),SH,5,0.0,0)
C         CALL PLOT(XMOD(SG(IG2,IS)),YMOD(YPLT)-0.02,3)
C         CALL PLOT(XMOD(SG(IG2,IS)),YMOD(YPLT)+0.02,2)
        ENDIF
   70 CONTINUE
C
      CALL PLFLUSH
      LCPPL = .TRUE.
      LAIPL = .FALSE.
C
      CALL NEWCOLOR(ICOL0)
C
      RETURN
      END  ! PLOTCP



      SUBROUTINE PLOTAI
C------------------------------------------------------------
C     Plots airfoil with surface pressure vectors (gee whiz)
C------------------------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MEDP.INC'
      XMODG(XTMP) = GSF * (XTMP - XOFG)
      YMODG(YTMP) = GSF * (YTMP - YOFG)
C
      SH = 0.012
      PAR = 1.0
C
      VFAC = 0.15
C
      N = NMIX
      I1 = IS1(N)
      I2 = IS2(N)
C
C---- set pressure vector scale VSF
      CHORD = SQRT(  (XBTAIL(N)-XBNOSE(N))**2
     &             + (YBTAIL(N)-YBNOSE(N))**2 )
      VSF = VFAC * CHORD
C
      SBSIDE(I1) = SB(1     ,N) - SBLE(N)
      SBSIDE(I2) = SB(IIB(N),N) - SBLE(N)
C
C---- find min CP
      CPMIN = 1.0E9
      DO 5 IG=1, NBLD(N)
        CPMIN = MIN( CPMIN , CPW(IG,I1) )
        CPMIN = MIN( CPMIN , CPW(IG,I2) )
    5 CONTINUE
      CPMIN = FLOAT(INT(20.0*CPMIN)-1) / 20.0
C
C---- set geometric limits, including pressure vectors
      XMIN = XB(1,N)
      XMAX = XB(1,N)
      YMIN = YB(1,N)
      YMAX = YB(1,N)
      DO 10 IS=I1, I2
        DO 101 IG=1, NBLD(N)
          SB2 = SBLE(N) + SG(IG,IS)*SBSIDE(IS)
          XB2 =  SEVAL(SB2,XB(1,N),XPB(1,N),SB(1,N),IIB(N))
          YB2 =  SEVAL(SB2,YB(1,N),YPB(1,N),SB(1,N),IIB(N))
          XN2 =  DEVAL(SB2,YB(1,N),YPB(1,N),SB(1,N),IIB(N))
          YN2 = -DEVAL(SB2,XB(1,N),XPB(1,N),SB(1,N),IIB(N))
          SLEN = ABS(CPW(IG,IS)*VSF)
          XV2 = XB2 + XN2*SLEN
          YV2 = YB2 + YN2*SLEN
          XMIN = MIN(XV2,XMIN)
          XMAX = MAX(XV2,XMAX)
          YMIN = MIN(YV2,YMIN)
          YMAX = MAX(YV2,YMAX)
 101    CONTINUE
 10   CONTINUE
C
C---- initialize plot, set window size
      CALL PLTINI
C
C---- set scale, offsets, to center airfoil+vectors in window
      XYEPS = 1.0E-9
      XRANGE = MAX(XYEPS, XMAX-XMIN)
      YRANGE = MAX(XYEPS, YMAX-YMIN)
      XADD = 0.05
      YADD = 0.05
      GSF = MIN( (XWIND/SIZE)/(XRANGE*(1.0 + 2.0*XADD)) , 
     &           (YWIND/SIZE)/(YRANGE*(1.0 + 2.0*YADD))  )
      XOFG = XMIN - 0.5*(XWIND/SIZE-GSF*XRANGE)/GSF
      YOFG = YMIN - 0.5*(YWIND/SIZE-GSF*YRANGE)/GSF
C
      CALL GETCOLOR(ICOL0)
C
      XB1 = XB(1,N)
      YB1 = YB(1,N)
      IG1 = NBLD(N)
      IG2 = 1
      IGINC = -1
      DO 20 IS=I1, I2
        IF(IS.EQ.I1) ICOL = ICOL1
        IF(IS.EQ.I2) ICOL = ICOL2
C
        CALL NEWCOLOR(ICOL0)
        DO 201 IG=IG1, IG2, IGINC
          SB2 = SBLE(N) + SG(IG,IS)*SBSIDE(IS)
          XB2 =  SEVAL(SB2,XB(1,N),XPB(1,N),SB(1,N),IIB(N))
          YB2 =  SEVAL(SB2,YB(1,N),YPB(1,N),SB(1,N),IIB(N))
          XN2 =  DEVAL(SB2,YB(1,N),YPB(1,N),SB(1,N),IIB(N))
          YN2 = -DEVAL(SB2,XB(1,N),XPB(1,N),SB(1,N),IIB(N))
C
          IF(MIN(IG,IG-IGINC) .GE. IGS(1,N) .AND.
     &       MAX(IG,IG-IGINC) .LE. IGS(2,N)       ) THEN
           CALL NEWCOLOR(ICOL)
          ELSE
           CALL NEWCOLOR(ICOL0)
          ENDIF
          CALL PLOT(XMODG(XB1),YMODG(YB1),3)
          CALL PLOT(XMODG(XB2),YMODG(YB2),2)
C
          IF(IG.GE.IGS(1,N) .AND. IG.LE.IGS(2,N)) THEN
           CALL NEWCOLOR(ICOL)
          ELSE
           CALL NEWCOLOR(ICOL0)
          ENDIF
          DX = -CPW(IG,IS)*VSF*XN2*GSF
          DY = -CPW(IG,IS)*VSF*YN2*GSF
          XPLT = XMODG(XB2)
          YPLT = YMODG(YB2)
          IF(CPW(IG,IS).LT.0.0) CALL ARROW(XPLT   ,YPLT   ,DX,DY)
          IF(CPW(IG,IS).GE.0.0) CALL ARROW(XPLT-DX,YPLT-DY,DX,DY)
C
C-------- Plot endpoints of the target segment
          IF(IG.EQ.IGS(1,N) .OR. IG.EQ.IGS(2,N)) THEN
           CALL NEWCOLOR(ICOL)
           DTICK = 0.005*CHORD
           CALL PLOT(XMODG(XB2)          ,YMODG(YB2)          ,3)
           CALL PLOT(XMODG(XB2-XN2*DTICK),YMODG(YB2-YN2*DTICK),2)
c
c           CALL NEWPEN(3)
c           CALL PLSYMB(XMODG(XB2),YMODG(YB2),SH,5,0.0,0)
c           CALL NEWPEN(1)
          ENDIF
C
          XB1 = XB2
          YB1 = YB2
  201   CONTINUE
        IG1 = 2
        IG2 = NBLD(N)
        IGINC = 1
   20 CONTINUE
C
      CALL PLFLUSH
      LAIPL = .TRUE.
      LCPPL = .FALSE.
C
      CALL NEWCOLOR(ICOL0)
      RETURN
      END  ! PLOTAI



      SUBROUTINE ARROW(X,Y,DX,DY)
      CALL PLOT(X,Y,3)
      CALL PLOT(X+DX,Y+DY,2)
      X1 = X + 0.90*DX + 0.007*DY
      Y1 = Y + 0.90*DY - 0.007*DX
      X2 = X + 0.90*DX - 0.007*DY
      Y2 = Y + 0.90*DY + 0.007*DX
      CALL PLOT(X1,Y1,2)
      CALL PLOT(X2,Y2,2)
      CALL PLOT(X+DX,Y+DY,2)
      RETURN
      END



      SUBROUTINE IGAGET
C--------------------------------------------
C     Gets freewall segment endpoints from 
C     user via cursor on airfoil plot.
C--------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MEDP.INC'
      CHARACTER*1 KCHAR
      LOGICAL LABORT
C
      XMODG(XTMP) = GSF * (XTMP - XOFG)
      YMODG(YTMP) = GSF * (YTMP - YOFG)
C
C
      SH = 0.010
C
      N = NMIX
      I1 = IS1(N)
      I2 = IS2(N)
C
      IGNEW(1) = 0
      IGNEW(2) = 0
      ISNEW(1) = 0
      ISNEW(2) = 0
C
      CALL GETCOLOR(ICOL0)
C
      CALL PABORT(1.0,0.0)
C
      WRITE(*,*)
      WRITE(*,*) 'Mark freewall segment endpoints on either surface'
      WRITE(*,*)
C
      SBSIDE(I1) = SB(1     ,N) - SBLE(N)
      SBSIDE(I2) = SB(IIB(N),N) - SBLE(N)
C
C---- go over each chosen endpoint
      DO 1 IE=1, 2
    2   CALL GETCURSORXY(XE,YE,KCHAR)
        IF(LABORT(XE,YE)) THEN
          CALL NEWCOLOR(ICOL0)
          RETURN
        ENDIF
        XE = XE/GSF + XOFG
        YE = YE/GSF + YOFG
        DMIN = 1.0E9
        DO 111 IS=I1, I2
          DO 1111 IG=1, NBLD(N)
            SS = SBLE(N) + SG(IG,IS)*SBSIDE(IS)
            XS = SEVAL(SS,XB(1,N),XPB(1,N),SB(1,N),IIB(N))
            YS = SEVAL(SS,YB(1,N),YPB(1,N),SB(1,N),IIB(N))
            DIST = (XS-XE)**2 + (YS-YE)**2
            IF(DIST.GT.DMIN) GO TO 1111
              DMIN = DIST
              IGNEW(IE) = IG
              ISNEW(IE) = IS
 1111     CONTINUE
  111   CONTINUE
C
C------ plot symbol over the endpoint node to tell user what he specified
        CALL NEWCOLORNAME('red')
        IG = IGNEW(IE)
        DO 112 IS=I1, I2
          SS = SBLE(N) + SG(IG,IS)*SBSIDE(IS)
          XS = SEVAL(SS,XB(1,N),XPB(1,N),SB(1,N),IIB(N))
          YS = SEVAL(SS,YB(1,N),YPB(1,N),SB(1,N),IIB(N))
          CALL PLSYMB(XMODG(XS),YMODG(YS),SH,5,0.0,0)
  112   CONTINUE
    1 CONTINUE
C
      IF(IGNEW(1).EQ.IGNEW(2)) THEN
       WRITE(*,*) '***  Endpoints must be distinct  ***'
       WRITE(*,*) '***  NEW SEGMENT NOT MARKED OFF  ***'
       CALL NEWCOLOR(ICOL0)
       RETURN
      ELSE IF(IGNEW(1).GT.IGNEW(2)) THEN
       IG = IGNEW(1)
       IGNEW(1) = IGNEW(2)
       IGNEW(2) = IG
      ENDIF
C
      IGS(1,N) = IGNEW(1)
      IGS(2,N) = IGNEW(2)
C
      WRITE(*,1000) (IGS(1,N)+ILEB(N)-1), (IGS(2,N)+ILEB(N)-1), N
 1000 FORMAT(/' New target segment endpoints set: ',
     &           2I4, '   element',I2)
C
      CALL NEWCOLOR(ICOL0)
      CALL PLFLUSH
      RETURN
      END ! IGAGET


      SUBROUTINE IGSGET
C-----------------------------------------------
C     Gets freewall segment endpoints from user
C     via cursor on  pressure vs. s/smax  plot.
C-----------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MEDP.INC'
      CHARACTER*1 KCHAR
      LOGICAL LABORT
C
      XMOD(XTMP) = XSF * (XTMP - XOFF)
      YMOD(YTMP) = YSF * (YTMP - YOFF)
C
C
      SH = 0.010
C
      N = NMIX
      I1 = IS1(N)
      I2 = IS2(N)
C
      IGNEW(1) = 0
      IGNEW(2) = 0
      ISNEW(1) = 0
      ISNEW(2) = 0
C
      CALL GETCOLOR(ICOL0)
C
      CALL PABORT(1.0,0.0)
C
      WRITE(*,*)
      WRITE(*,*) 'Mark freewall segment endpoints on either surface'
      WRITE(*,*)
C
      SBSIDE(I1) = SB(1     ,N) - SBLE(N)
      SBSIDE(I2) = SB(IIB(N),N) - SBLE(N)
C
C---- for each endpoint...
      DO 1 IE=1, 2
    2   CALL GETCURSORXY(XE,YE,KCHAR)
        IF(LABORT(XE,YE)) THEN
          CALL NEWCOLOR(ICOL0)
          RETURN
        ENDIF
        XE = XE/XSF + XOFF
        YE = YE/YSF + YOFF
        DMIN = 1000.
C
C------ go over airfoil side, finding the point "nearest" to cursor
        DO 111 IS = I1, I2
          DO 1111 IG=1, NBLD(N)
            XS =   SG(IG,IS)
            YS = -CPS(IG,IS)*PWT
            DIST = (XS-XE)**2 + (YS-YE)**2
            IF(DIST.GT.DMIN) GO TO 1111
              DMIN = DIST
              IGNEW(IE) = IG
              ISNEW(IE) = IS
 1111     CONTINUE
  111   CONTINUE
C
        CALL NEWCOLORNAME('red')
        IG = IGNEW(IE)
        DO 112 IS=I1, I2
          XS =   SG(IG,IS)
          YS = -CPS(IG,IS)*PWT
C
C-------- plot PLSYMB over the endpoint node to tell user what he specified
          CALL PLSYMB(XMOD(XS),YMOD(YS),SH,5,0.0,0)
CCC          CALL PLOT(XMOD(XS),YMOD(YS)-0.02,3)
CCC          CALL PLOT(XMOD(XS),YMOD(YS)+0.02,2)
 112    CONTINUE
    1 CONTINUE
C
C---- test for nonsense and proper ordering ....
      IF(IGNEW(1).EQ.IGNEW(2)) THEN
       WRITE(*,*) '***  Endpoints must be distinct  ***'
       WRITE(*,*) '***  NEW SEGMENT NOT MARKED OFF  ***'
       CALL NEWCOLOR(ICOL0)
       RETURN
      ELSE IF(IGNEW(1).GT.IGNEW(2)) THEN
       IG = IGNEW(1)
       IGNEW(1) = IGNEW(2)
       IGNEW(2) = IG
      ENDIF
C
C---- set buffer endpoint indices
      IGS(1,N) = IGNEW(1)
      IGS(2,N) = IGNEW(2)
C
      WRITE(*,1000) (IGS(1,N)+ILEB(N)-1), (IGS(2,N)+ILEB(N)-1), N
 1000 FORMAT(/' New target segment endpoints set: ',
     &           2I4, '   element',I2)
C
      CALL NEWCOLOR(ICOL0)
      CALL PLFLUSH
      RETURN
      END ! IGSGET


      SUBROUTINE MODIFY
C--------------------------------------------
C     Gets cursor-input pressures from user,
C     splines them, and puts them into the
C     specified-pressure buffer array.
C--------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MEDP.INC'
      CHARACTER*1 KCHAR
      LOGICAL LSLOP1, LSLOP2
      LOGICAL LABORT
C
      XMOD(XTMP) = XSF * (XTMP - XOFF)
      YMOD(YTMP) = YSF * (YTMP - YOFF)
C
C
      SH = 0.006
C
      N = NMIX
C
C---- zero out arc length and input pressure arrays
      DO 5 NW=1, IBX
        S(NW) = 0.
        W(NW) = 0.
        WP(NW) = 0.
    5 CONTINUE
C
      CALL GETCOLOR(ICOL0)
C
      CALL PABORT(1.0,0.0)
C
      WRITE(*,*)
      WRITE(*,*) 'Input new Cp values'
      WRITE(*,*) 'Terminate with 3 clicks on one point...'
      WRITE(*,*)
C
      CALL NEWCOLORNAME('red')
C
C---- get first point
      NW = 1
      CALL GETCURSORXY(S(NW),W(NW),KCHAR)
      IF(LABORT(S(NW),W(NW))) THEN
        CALL NEWCOLOR(ICOL0)
        RETURN
      ENDIF
C
C---- find nearest Cp point on plot to set side index ISMIX
      DSMIN = 1.0E9
      DO 7 IS=IS1(N), IS2(N)
        DO 72 IG=1, NBLD(N)
          XPNT = XMOD(  SG(IG,IS)    )
          YPNT = YMOD(-CPS(IG,IS)*PWT)
          DIST = (S(NW) - XPNT)**2 + (W(NW) - YPNT)**2
          IF(DIST.GT.DSMIN) GO TO 72
            DSMIN = DIST
            IGMIN = IG
            ISMIN = IS
 72     CONTINUE
 7    CONTINUE
C
      ISMIX = ISMIN
C
C---- transform point from plotting x,y coordinates to -Cp,s coordinates
      S(NW) = S(NW)/XSF + XOFF
      W(NW) = W(NW)/YSF + YOFF
C
C
C---- we will need splined Cp(s) for selected side
      IS = ISMIX
      IF(LSLOP) CALL SPLINE(CPS(1,IS),CPSP(1,IS),SG(1,IS),NBLD(N))
C
C---- match slopes only if inside SG limits 0..1
      LSLOP1 = LSLOP .AND. 
     &         S(NW).GE.SG(1,IS) .AND. S(NW).LE.SG(NBLD(N),IS)
C
      IF(LSLOP1) THEN
C------ reset first point from spline, draw "o" symbol there
        W(NW) = -SEVAL(S(NW),CPS(1,IS),CPSP(1,IS),SG(1,IS),NBLD(N))*PWT
        CALL PLSYMB(XMOD(S(NW)),YMOD(W(NW)),SH,1,0.0,0)
      ELSE
        CALL PLSYMB(XMOD(S(NW)),YMOD(W(NW)),SH,3,0.0,0)
      ENDIF
C
C---- get rest of points
      DO 10 NW=2, IBX
C------ fetch x-y point coordinates from user
    9   CALL GETCURSORXY(S(NW),W(NW),KCHAR)
        IF(LABORT(S(NW),W(NW))) THEN
          CALL NEWCOLOR(ICOL0)
          RETURN
        ENDIF
C
C------ transform point from plotting x,y coordinates to -Cp,s coordinates
        S(NW) = S(NW)/XSF + XOFF
        W(NW) = W(NW)/YSF + YOFF
C
C------ plot small "+" at specified point
        CALL PLSYMB(XMOD(S(NW)),YMOD(W(NW)),SH,3,0.0,0)
C
        IF(NW.LT.3) GO TO 10
C------- test for last input point (3 identical s-values)
         IF(S(NW).EQ.S(NW-1) .AND. S(NW).EQ.S(NW-2)) GO TO 11
   10 CONTINUE
   11 CONTINUE
C
C
C---- match slopes only if inside SG limits 0..1
      LSLOP2 = LSLOP .AND. 
     &         S(NW).GE.SG(1,IS) .AND. S(NW).LE.SG(NBLD(N),IS)
C
      IF(LSLOP2) THEN
C------ reset first point from spline, draw "o" symbol there
        W(NW) = -SEVAL(S(NW),CPS(1,IS),CPSP(1,IS),SG(1,IS),NBLD(N))*PWT
        CALL PLSYMB(XMOD(S(NW)),YMOD(W(NW)),SH,1,0.0,0)
      ELSE
        CALL PLSYMB(XMOD(S(NW)),YMOD(W(NW)),SH,3,0.0,0)
      ENDIF
C
C
C---- sort points in s and strip out identical s pairs
      CALL SORT(NW,S,W)
C
      IF(NW.LT.2) THEN
       WRITE(*,*)
       WRITE(*,*) '***  Need at least 2 points    ***'
       WRITE(*,*) '***     NO CHANGES MADE        ***'
       WRITE(*,*)
       CALL NEWCOLOR(ICOL0)
       RETURN
      ENDIF
C
C
C---- default natural (zero 3rd derivative) end conditions
      CPP1 = -999.0
      CPP2 = -999.0
C
C---- set spline endpoint derivatives to match current Cp(s)
      IF(LSLOP1)
     &   CPP1 = -DEVAL(S(1) ,CPS(1,IS),CPSP(1,IS),SG(1,IS),NBLD(N))*PWT
      IF(LSLOP2)
     &   CPP2 = -DEVAL(S(NW),CPS(1,IS),CPSP(1,IS),SG(1,IS),NBLD(N))*PWT
C
C---- spline input Cp points
      CALL SPLIND(W,WP,S,NW,CPP1,CPP2)
C
      CALL NEWCOLORNAME('magenta')
C
C---- go over all points on target side
      IS = ISMIX
      DO 20 IG=1, NBLD(N)
        SS = SG(IG,IS)
        IF(SS.LT.S(1)) THEN
C------- current point is outside modified interval...try next point
         CALL PLOT(XMOD(SS),YMOD(-CPS(IG,IS)*PWT),3)
         GO TO 20
        ELSE IF(SS.LE.S(NW)) THEN
C------- stuff new pressure point into Pspec array and plot it
         SW = SEVAL(SS,W,WP,S,NW)
         CPS(IG,IS) = -SW/PWT
         CALL PLOT(XMOD(SS),YMOD(SW),2)
        ELSE
C------- went past modified interval...finish up
         GO TO 21
        ENDIF
   20 CONTINUE
C
   21 CALL PLOT(XMOD(SS),YMOD(-CPS(IG,IS)*PWT),2)
      CALL PLFLUSH
C
      LSAVE = .TRUE.
C
      CALL NEWCOLOR(ICOL0)
      RETURN
      END ! MODIFY


      SUBROUTINE SORT(NW,S,W)
      IMPLICIT REAL (A-H,M,O-Z)
      DIMENSION S(NW), W(NW)
      LOGICAL DONE
C
C---- sort arrays
      DO 10 IPASS=1, 500
        DONE = .TRUE.
        DO 101 N=1, NW-1
          NP = N+1
          IF(S(NP).GE.S(N)) GO TO 101
           TEMP = S(NP)
           S(NP) = S(N)
           S(N) = TEMP
           TEMP = W(NP)
           W(NP) = W(N)
           W(N) = TEMP
           DONE = .FALSE.
  101   CONTINUE
        IF(DONE) GO TO 11
   10 CONTINUE
      STOP 'SORT failed'
C
C---- search for duplicate pairs and eliminate each one
   11 NWS = NW
      DO 20 N=1, NWS
        IF(N.GE.NW) RETURN
        IF(S(N).NE.S(N+1)) GO TO 20
C------- eliminate pair
         NW = NW-2
         DO 201 NT=N, NW
           S(NT) = S(NT+2)
           W(NT) = W(NT+2)
  201    CONTINUE
   20 CONTINUE
C
      RETURN
      END ! SORT



      SUBROUTINE SETSP(N)
C---------------------------------------------
C     Sets specified pressure buffer array
C     from actual wall pressure buffer array.
C     Also zeroes inverse DOFs.
C---------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MEDP.INC'
C
      I1 = IS1(N)
      I2 = IS2(N)
C
      DO 10 IG=1, NBLD(N)
        CPS(IG,I1) = CPW(IG,I1)
        CPS(IG,I2) = CPW(IG,I2)
 10   CONTINUE
C
ccc      PDF0 = 0.
ccc      PDF1 = 0.
ccc      PDFL = 0.
      PDX0 = 0.
      PDX1 = 0.
      PDD0 = 0.
      PDD1 = 0.
C
      RETURN
      END  ! SETSP



      SUBROUTINE OUT
C---------------------------------------------------
C     Un-normalizes and stores specified pressure
C     buffer array into STATE common PSPEC array,
C     and writes MDAT.xxx dump file.
C---------------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MEDP.INC'
C
      DO 100 N=1, NBL
C
        I1 = IS1(N)
        I2 = IS2(N)
C
        DPINF = PINF - PSTOUT
        DO 5 IG=1, NBLD(N)
          PSPEC(IG,I1) = CPS(IG,I1)*QU + DPINF
          PSPEC(IG,I2) = CPS(IG,I2)*QU + DPINF
  5     CONTINUE
C
 100  CONTINUE
C
      CALL OUTPUT
C
      RETURN
      END ! OUT


      SUBROUTINE SAVE
C-----------------------------------------------
C     Writes airfoil coordinates from MDAT.xxx
C     into a formatted file.
C-----------------------------------------------
      INCLUDE 'STATE.INC'
      CHARACTER*1 ANS
      CHARACTER*32 FNAME
      INCLUDE 'MEDP.INC'
C
      WRITE(*,*) 'Enter output filename:'
      READ(*,1300) FNAME
C
      OPEN(UNIT=9,FILE=FNAME,STATUS='OLD',ERR=30)
      WRITE(*,*)
      WRITE(*,*) '*** Output file exists.  Overwrite?  Y'
      READ (*,1300) ANS
      IF(INDEX('nN',ANS) .NE. 0) RETURN
      GO TO 35
C
 30   OPEN(UNIT=9,FILE=FNAME,STATUS='UNKNOWN',ERR=99)
 35   WRITE(9,1300) NAME
      WRITE(9,1500) XBINL, XBOUT, YBBOT, YBTOP
C
      DO 50 N=1, NBL
        DO 510 IB=1, IIB(N)
          WRITE(9,1500) XB(IB,N), YB(IB,N)
  510   CONTINUE
        IF(N.LT.NBL) WRITE(9,1550)
 50   CONTINUE
C
      CLOSE(UNIT=9)
      RETURN
C
   99 WRITE(*,*) 'File OPEN error.  Coordinates not saved'
      RETURN
C
 1300 FORMAT(A)
 1500 FORMAT(1X,4F12.6)
 1550 FORMAT(1X,'999.0    999.0')
      END ! SAVE


      SUBROUTINE PARSET
C--------------------------------------------------------
C     Allows the altering of various mdat.xxx parameters.
C--------------------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MEDP.INC'
      CHARACTER*2 OPT
C
 1000 FORMAT(A)
C
    1 WRITE(*,1010)
 1010 FORMAT(/'   1   XCENT   far field singularity x location'
     &       /'   2   YCENT   far field singularity y location'
     &       /'   3   HVIS    Sutherland''s law constant'
     &       /'   4   GAM     Cp/Cv'
     &       /'   5   NAME    Solution case name'
     &       /'   6   ICOUNT  Newton iteration counter'
     &       /'   7   INITBL  BL initialization flag'
     &       /'   8   XPEX    exponent for normal grid line spreading'
     &       /'   9   PCWT    Pcorr weighting factor'
     &       /'  10   YBOT    Bottom wall y location'
     &       /'  11   YTOP    Top wall y location'
     &      //'    Change what?  ', $)
C
    5 READ(*,1000) OPT
      IF(OPT.EQ.'  ') RETURN
C
      READ(OPT,*,ERR=5) IOPT
C
      GO TO (10, 20, 30, 40, 50, 60, 70, 80, 90, 100, 110), IOPT
      RETURN
C
   10 WRITE(*,*) 'Current XCENT = ', XCENT
   11 WRITE(*,*) 'Enter new value: '
      READ(*,*,ERR=11) XCENT
      GO TO 1
C
   20 WRITE(*,*) 'Current YCENT = ', YCENT
   21 WRITE(*,*) 'Enter new value: '
      READ(*,*,ERR=21) YCENT
      GO TO 1
C
   30 WRITE(*,*) 'Current HVIS = ', HVIS
   31 WRITE(*,*) 'Enter new value: '
      READ(*,*,ERR=31) HVIS
      GO TO 1
C
   40 WRITE(*,*) 'Current GAM = ', GAM
   41 WRITE(*,*) 'Enter new value: '
      READ(*,*,ERR=41) GAM
      GO TO 1
C
   50 WRITE(*,5010) NAME
 5010 FORMAT(' Current NAME:  ', A)
   51 WRITE(*,*) 'Enter new name: '
      READ(*,5020) NAME
 5020 FORMAT(A)
      GO TO 1
C
   60 WRITE(*,*) 'Current ICOUNT = ', ICOUNT
   61 WRITE(*,*) 'Enter new value: '
      READ(*,*,ERR=61) ICOUNT
      GO TO 1
C
   70 WRITE(*,*) 'Current INITBL = ', INITBL
   71 WRITE(*,*) 'Enter new value: '
      READ(*,*,ERR=71) INITBL
      GO TO 1
C
   80 WRITE(*,*) 'Current XPEX = ', XPEX
   81 WRITE(*,8110) 
 8110 FORMAT(/' 0 : nodes tend to uniform spacing away from surfaces'
     &       /' 1 : nodes stay the same spacing as on surfaces'
     &      //' Enter x-spacing exponent (0...1):  ',$)
      READ(*,*,ERR=81) XPEX
      GO TO 1
C
   90 WRITE(*,*) 'Current PCWT = ', PCWT
   91 WRITE(*,*) 'Enter new value: '
      READ(*,*,ERR=91) PCWT
      GO TO 1
C
  100 WRITE(*,*) 'Current YBOT = ', YBBOT
  101 WRITE(*,*) 'Enter new value: '
      READ(*,*,ERR=101) YBBOT
      GO TO 1
C
  110 WRITE(*,*) 'Current YTOP = ', YBTOP
  111 WRITE(*,*) 'Enter new value: '
      READ(*,*,ERR=111) YBTOP
      GO TO 1
C
C
      END ! PARSET



      SUBROUTINE CPDUMP
C-----------------------------------
C     Writes current wall pressures
C     and fractional arc lengths
C     to a formatted file.
C-----------------------------------
      INCLUDE 'STATE.INC'
      CHARACTER*32 FNAME
      INCLUDE 'MEDP.INC'
C
      DO 5 N=1, NBL
        SBSIDE(IS1(N)) = SB(1     ,N) - SBLE(N)
        SBSIDE(IS2(N)) = SB(IIB(N),N) - SBLE(N)
 5    CONTINUE
C
      IF(NMIX.GE.1 .AND. NMIX.LE.NBL) THEN
        N = NMIX
        IX0 = IGS(1,N) + ILEB(N) - 1
        IX1 = IGS(2,N) + ILEB(N) - 1
      ELSE
        N = 0
        IX0 = 0
        IX1 = 0
      ENDIF
C
      WRITE(*,*) 'Enter output filename:'
      READ(*,1000) FNAME
 1000 FORMAT(A32)
C
      OPEN(UNIT=9,FILE=FNAME,STATUS='UNKNOWN',ERR=99)
cc      WRITE(9,*) NMIX, IX0, IX1,   '|  Nelement,   target endpoints'
cc      WRITE(9,*) '  i        s/smax           Cp '
      WRITE(9,*) '  i       x       y       Cp      Cf     Theta'
      DO 10 N=1, NBL
        DO 110 IS=IS1(N), IS2(N)
          DO 1110 I=ILEB(N), ITEB(N)
            IG = I - ILEB(N) + 1
C
C---------- dump Cp vs. s/smax
cc            WRITE(9,1500) I, SG(IG,IS), CPW(IG,IS)
C
C---------- dump Cp and other stuff
            SBI = SBLE(N) + SBSIDE(IS)*SG(IG,IS)
            XG = SEVAL(SBI,XB(1,N),XPB(1,N),SB(1,N),IIB(N))
            YG = SEVAL(SBI,YB(1,N),YPB(1,N),SB(1,N),IIB(N))
            CF = TAU(I,IS)/QU
            WRITE(9,1500) I, XG, YG, CPW(IG,IS), CF, THET(I,IS)
 1110     CONTINUE
  110   CONTINUE
        IF(N.LT.NBL) WRITE(9,1550)
   10 CONTINUE
C
      CLOSE(UNIT=9)
      RETURN
C
   99 WRITE(*,*) 'CPDUMP:  File OPEN error.  Data not saved'
      RETURN
C
 1500 FORMAT(1X,I4, 5F11.6)
 1550 FORMAT(1X,'   0   999.0   999.0')
      END ! CPDUMP



      SUBROUTINE CPREAD
C--------------------------------------------------------
C     Reads file written by CPDUMP and probably edited
C     by user and stuffs it into the specified-pressure
C     buffer array.  Also reads in freewall segment
C     endpoint indices in the same file.
C--------------------------------------------------------
      INCLUDE 'STATE.INC'
      CHARACTER*32 FNAME, DUMMY
      INCLUDE 'MEDP.INC'
C
      WRITE(*,*) 'Enter input filename:'
      READ(*,1000) FNAME
 1000 FORMAT(A)
C
      IF(NMIX.GE.1 .AND. NMIX.LE.NBL) THEN
        N = NMIX
        IX0 = IGS(1,N) + ILEB(N) - 1
        IX1 = IGS(2,N) + ILEB(N) - 1
      ENDIF
C
      OPEN(UNIT=9,FILE=FNAME,STATUS='OLD',ERR=99)
      READ(9,*) NMIX, IX0, IX1
      READ(9,1000) DUMMY
      DO 10 N=1, NBL
        DO 110 IS=IS1(N), IS2(N)
          DO 1110 I=ILEB(N), ITEB(N)
            IG = I - ILEB(N) + 1
ccc            READ(9,*) IT, SGT, CPIN
            READ(9,*) IT, XGT, YGT, CPIN
            IF(I.NE.IT) THEN
             WRITE(*,*) 'Incompatible grid size.  Data not read in'
             RETURN
            ENDIF
            CPS(IG,IS) = CPIN
 1110     CONTINUE
  110   CONTINUE
        IF(N.LT.NBL) READ(9,1000) DUMMY
   10 CONTINUE
C
      CLOSE(UNIT=9)
C
      IF(NMIX.GE.1 .AND. NMIX.LE.NBL) THEN
        N = NMIX
        IGS(1,N) = IX0 - ILEB(N) + 1
        IGS(2,N) = IX1 - ILEB(N) + 1
        LPSET = .TRUE.
      ENDIF
C
      RETURN
C
   99 WRITE(*,*) 'CPREAD:  File OPEN error.  Data not read in'
      RETURN
      END ! CPREAD



      SUBROUTINE MDUMP
C---------------------------------------------------
C     Same as CPDUMP, but writes out isentropic
C     Mach numbers calculated from wall pressures.
C---------------------------------------------------
      INCLUDE 'STATE.INC'
      CHARACTER*32 FNAME
      INCLUDE 'MEDP.INC'
C
      DO 5 N=1, NBL
        SBSIDE(IS1(N)) = SB(1     ,N) - SBLE(N)
        SBSIDE(IS2(N)) = SB(IIB(N),N) - SBLE(N)
 5    CONTINUE
C
      IF(NMIX.GE.1 .AND. NMIX.LE.NBL) THEN
        N = NMIX
        IX0 = IGS(1,N) + ILEB(N) - 1
        IX1 = IGS(2,N) + ILEB(N) - 1
      ELSE
        N = 0
        IX0 = 0
        IX1 = 0
      ENDIF
C
      WRITE(*,*) 'Enter output filename:'
      READ(*,1000) FNAME
 1000 FORMAT(A32)
C
      OPEN(UNIT=9,FILE=FNAME,STATUS='UNKNOWN',ERR=99)
      WRITE(9,*) NMIX, IX0, IX1,   '|  Nelement,   target endpoints'
      WRITE(9,*) '  i        s/smax            Mach '
ccc   WRITE(9,*) '  i          x               Mach '
      DO 10 N=1, NBL
        DO 110 IS=IS1(N), IS2(N)
          DO 1110 I=ILEB(N), ITEB(N)
            IG = I - ILEB(N) + 1
            PRAT = ( CPW(IG,IS)*QU + PINF ) / PSTOUT
            MSQ = (PRAT**(-GM1/GAM) - 1.0) * 2.0/GM1
            MACH = SQRT(ABS(MSQ))
            MACH = SIGN( MACH , MSQ )
C
C---------- dump Mach vs. s/smax
            WRITE(9,1500) I, SG(IG,IS), MACH
C
C---------- dump Mach vs. x
cc            SBI = SBLE(N) + SBSIDE(IS)*SG(IG,IS)
cc            XG = SEVAL(SBI,XB(1,N),XPB(1,N),SB(1,N),IIB(N))
cc            WRITE(9,1500) I, XG, MACH
C
 1110     CONTINUE
  110   CONTINUE
        IF(N.LT.NBL) WRITE(9,1550)
   10 CONTINUE
C
      CLOSE(UNIT=9)
      RETURN
C
   99 WRITE(*,*) 'MDUMP:  File OPEN error.  Data not saved'
      RETURN
C
 1500 FORMAT(1X,I4, 2F11.6)
 1550 FORMAT(1X,'   0   999.0   999.0')
      END ! MDUMP



      SUBROUTINE MREAD
C---------------------------------------------------
C     Same as CPREAD, but reads in  Mach numbers
C     and converts them to specified surface
C     pressures assuming inlet stagnation pressure.
C---------------------------------------------------
      INCLUDE 'STATE.INC'
      CHARACTER*32 FNAME, DUMMY
      INCLUDE 'MEDP.INC'
C
      WRITE(*,*) 'Enter input filename:'
      READ(*,1000) FNAME
 1000 FORMAT(A)
C
      OPEN(UNIT=9,FILE=FNAME,STATUS='OLD',ERR=99)
      READ(9,*) NMIX, IX0, IX1
      READ(9,1000) DUMMY
      DO 10 N=1, NBL
        DO 10 IS=IS1(N), IS2(N)
          J = 1 + (IS-1)*(JJ-1)
          DO 110 I=ILEB(N), ITEB(N)
            IG = I - ILEB(N) + 1
            READ(9,*) IT, SGT, MACH
            IF(I.NE.IT) THEN
             WRITE(*,*) 'Incompatible grid size. Data not read in'
             RETURN
            ENDIF
            SGN = SIGN(1.0,MACH)
            PRES = PSTOUT*(1.0 + 0.5*GM1*SGN*MACH**2)**(-GAM/GM1)
            CPS(IG,IS) = (PRES-PINF)/QU
 1110     CONTINUE
  110   CONTINUE
        IF(N.LT.NBL) READ(9,1000) DUMMY
   10 CONTINUE
C
      CLOSE(UNIT=9)
C
      IF(NMIX.GE.1 .AND. NMIX.LE.NBL) THEN
        N = NMIX
        IGS(1,N) = IX0 - ILEB(N) + 1
        IGS(2,N) = IX1 - ILEB(N) + 1
        LPSET = .TRUE.
      ENDIF
C
      RETURN
C
   99 WRITE(*,*) 'MREAD:  File OPEN error.  Data not read in'
      RETURN
      END ! MREAD


      SUBROUTINE FORCE(CLW,CMW,CLS,CMS,N)
C-------------------------------------------
C     Integrates airfoil surface pressures
C-------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MEDP.INC'
C
C---- coordinates for moment reference point
      XMOMNT = 0.25
      YMOMNT = 0.0
C
      COSA = COS(ALFA)
      SINA = SIN(ALFA)
C
      CLS = 0.
      CLW = 0.
      CMS = 0.
      CMW = 0.
C
      I1 = IS1(N)
      I2 = IS2(N)
C
C---- TE pressures which are subtracted off everywhere
      CPSTE = 0.5*(CPS(NBLD(N),I1) + CPS(NBLD(N),I2))
      CPWTE = 0.5*(CPW(NBLD(N),I1) + CPW(NBLD(N),I2))
C
C---- LE point coordinates
      XMS = SEVAL(SBLE(N),XB(1,N),XPB(1,N),SB(1,N),IIB(N))
      YMS = SEVAL(SBLE(N),YB(1,N),YPB(1,N),SB(1,N),IIB(N))
      XMP = XMS
      YMP = YMS
C
C---- go from LE to TE, adding on top & bottom contributions
      DO 10 IGO=2, NBLD(N)
        IGM = IGO-1
C
C------ top & bottom (suction,pressure) side arc lengths
        SS = SBLE(N) + (SB(1     ,N)-SBLE(N))*SG(IGO,I1)
        SP = SBLE(N) + (SB(IIB(N),N)-SBLE(N))*SG(IGO,I2)
C
C------ coordinates
        XOS = SEVAL(SS,XB(1,N),XPB(1,N),SB(1,N),IIB(N))
        YOS = SEVAL(SS,YB(1,N),YPB(1,N),SB(1,N),IIB(N))
        XOP = SEVAL(SP,XB(1,N),XPB(1,N),SB(1,N),IIB(N))
        YOP = SEVAL(SP,YB(1,N),YPB(1,N),SB(1,N),IIB(N))
C
C------ dx , dy  increments
        BXS = XOS - XMS
        BYS = YOS - YMS
        BXP = XOP - XMP
        BYP = YOP - YMP
C    
C------ set local Cp's
        CPSIS = 0.5*(CPS(IGM,I1)+CPS(IGO,I1)) - CPSTE
        CPSIP = 0.5*(CPS(IGM,I2)+CPS(IGO,I2)) - CPSTE
        CPWIS = 0.5*(CPW(IGM,I1)+CPW(IGO,I1)) - CPWTE
        CPWIP = 0.5*(CPW(IGM,I2)+CPW(IGO,I2)) - CPWTE
C
C------ moment arms for Cm contribution
        XBARS = 0.5*(XOS+XMS) - XMOMNT
        YBARS = 0.5*(YOS+YMS) - YMOMNT
        XBARP = 0.5*(XOP+XMP) - XMOMNT
        YBARP = 0.5*(YOP+YMP) - YMOMNT
C
C------ force increment components
        DFSX = BYS*CPSIS - BYP*CPSIP
        DFSY = BXP*CPSIP - BXS*CPSIS
        DFWX = BYS*CPWIS - BYP*CPWIP
        DFWY = BXP*CPWIP - BXS*CPWIS
C
C------ add on increments
        CLS = CLS + DFSY*COSA - DFSX*SINA
        CLW = CLW + DFWY*COSA - DFWX*SINA
        CMS = CMS + (BXS*CPSIS*XBARS - BXP*CPSIP*XBARP)
     &            + (BYS*CPSIS*YBARS - BYP*CPSIP*YBARP)
        CMW = CMW + (BXS*CPWIS*XBARS - BXP*CPWIP*XBARP)
     &            + (BYS*CPWIS*YBARS - BYP*CPWIP*YBARP)
C
C------ set "-1" coordinates for next loop pass
        XMS = XOS
        YMS = YOS
        XMP = XOP
        YMP = YOP
   10 CONTINUE      
C
      RETURN
      END  ! FORCE


 
      SUBROUTINE READAI
      INCLUDE 'STATE.INC'
      DIMENSION XBNEW(IBX), YBNEW(IBX)
      LOGICAL LCLOCK, ERROR, PLAIN
      CHARACTER*80 ARGP1, FNAME
C
      WRITE(*,*) 'Enter filename:'
      READ (*,1000) FNAME
C
      NAME = '                                '
CCC           12345678901234567890123456789012
C
C---- read blade.xxx coordinate file
      CALL READBL(FNAME,IBX,NBX,XB,YB,
     &            IIB,NBL,
     &            NAME,XBINL,XBOUT,YBBOT,YBTOP)
C
C---- quit if read was unsuccessful
      IF(NBL.EQ.0) RETURN
C
      PLAIN = ABS(XBINL-XBOUT) .LT. 1.0E-8
C
C---- get name and domain info if this was a plain coordinate file
      IF(PLAIN) THEN
        WRITE(*,*)
        WRITE(*,*) 'Enter case name:'
        READ(*,1000) NAME
 1000   FORMAT(A)
C
 50     WRITE(*,1005)
 1005   FORMAT(/' Enter  Xinl, Xout, Ybot, Ytop :')
        READ(*,*,ERR=50) XBINL,XBOUT,YBBOT,YBTOP
      ENDIF
C
C---- process each element
      DO 100 N=1, NBL
C
C------ spline coordinates
        CALL SCALC(XB(1,N),YB(1,N),SB(1,N),IIB(N))
        CALL SEGSPL(XB(1,N),XPB(1,N),SB(1,N),IIB(N))
        CALL SEGSPL(YB(1,N),YPB(1,N),SB(1,N),IIB(N))
C
C------ get nose s location
        CALL LEFIND(SBNOSE(N),XB(1,N),XPB(1,N),
     &                        YB(1,N),YPB(1,N),SB(1,N),IIB(N))
C
C------ set LE and TE coordinates
        XBNOSE(N) = SEVAL(SBNOSE(N),XB(1,N),XPB(1,N),SB(1,N),IIB(N))
        YBNOSE(N) = SEVAL(SBNOSE(N),YB(1,N),YPB(1,N),SB(1,N),IIB(N))
C
        XBTAIL(N) = 0.5*(XB(1,N) + XB(IIB(N),N))
        YBTAIL(N) = 0.5*(YB(1,N) + YB(IIB(N),N))
C
C------ set arc length of max-curvature point
        SBCMAX(N) = SBNOSE(N)
        CALL NSFIND(SBCMAX(N),XB(1,N),XPB(1,N),
     &                        YB(1,N),YPB(1,N),SB(1,N),IIB(N))
C
        SBNORM = SBCMAX(N)/(SB(IIB(N),N) - SB(1,N))
        IF(SBNORM .LT. 0.45 .OR. SBNORM .GT. 0.55) THEN
          WRITE(*,1050) N, SBNORM
 1050     FORMAT(
     &     /' READAI: Questionable max-curvature location on element',I3
     &     /'         Scmax/Stotal  =', F8.5
     &     /'         setting Scmax = Snose.' )
          SBCMAX(N) = SBNOSE(N)
        ENDIF
C
  100 CONTINUE
C
      RETURN
      END ! READAI
