C***********************************************************************
C  POLAR PLOTTING FACILITY FOR MSES AND XFOIL
C
C    INPUT:
C     * Polar file(s) generated by MSES or XFOIL
C     * Reference data files in the format:
C
C         CD(1)  CL(1)
C         CD(2)  CL(2)
C          .      .
C          .      .
C         999.0  999.0
C         alpha(1)  CL(1)
C         alpha(2)  CL(2)
C           .       .
C           .       .
C         999.0   999.0
C         alpha(1)  Cm(1)
C         alpha(2)  Cm(2)
C           .       .
C           .       .
C         999.0   999.0
C         Xtr/c(1)  CL(1)
C         Xtr/c(2)  CL(2)
C           .        .
C           .        .
C         999.0   999.0
C         
C         The number of points in each set (CD-CL, alpha-CL, etc.) 
C         is arbitrary, and can be zero.
C
C     * pplot.def  plot parameter file (optional)
C
C***********************************************************************
C
      PROGRAM PPLOT
      INCLUDE 'PINDEX.INC'
      INCLUDE 'PPLOT.INC'
C
      DIMENSION ISORT(NAX)
      DIMENSION CSORT(NAX)
C
      CHARACTER*16 CCLENP
C
      LPLOT = .FALSE.
C
      CALL PLINITIALIZE
C
C...Get default settings
      CALL GETDEF
C
C
    1 WRITE(*,1000)
      IF(NPOL.GT.0) WRITE(*,1010)
      WRITE(*,1050)
C
 1000 FORMAT(/'  1  Read polars          (-1 to append)'
     &       /'  2  Read reference data  (-2 to append)'
     &       /'  3  Plot'
     &       /'  4  Hardcopy current plot'
     &       /'  5  Change settings for plot'
     &       /'  6  Zoom'
     &       /'  7  Unzoom'
     &       /'  8  Annotation menu')
 1010 FORMAT( ' 11  Re-read current polars'
     &       /' 12  Re-read current reference data')
 1050 FORMAT(/'   Select option (0=quit): ', $)
C
      READ(*,*,ERR=1) IOPTS
      IOPT = ABS(IOPTS)
C
      GO TO (900, 10, 20, 30, 40, 50, 60, 70, 80,900,900,
     &            10, 20                                   ), IOPT+1
      GO TO 1
C
C=============================================
C---- read polars and assign colors
 10   IREAD = -1
      IF(IOPTS.LT.0 ) IREAD = +1
      IF(IOPTS.EQ.11) IREAD = 0
      CALL GETPOL(IREAD, FNPOL,
     &            NAX,NPX,NPOL,NA,CPOL, NAME,
     &            ISX,NBL,XTR,
     &            TITLE,CODE,VERSION )
      DO IP=1, NPOL
        CALL GETTYP(NAX,NA(IP),CPOL(1,1,IP),IMATYP(IP),IRETYP(IP))
        CALL STRIP(NAME(IP),NNAME)
      ENDDO
      DO IP=1, NPOL
        ICOL(IP) = 2 + IP
      ENDDO
CCC   CALL MINMAX(NAX,NPOL,NA,CPOL,CPOLPLF)
C
C---- are these dimensional polars?
      DO IP=1, NPOL
        CALL GETCLEN(NAME(IP),CCLEN,NCLEN)
        IF(NCLEN.GT.0) THEN
         LCLEN = .TRUE.
         GO TO 1
        ENDIF
      ENDDO
C
      GO TO 1
C
C=============================================
C---- read reference data and assign colors
 20   IREAD = -1
      IF(IOPTS.LT.0 ) IREAD = +1
      IF(IOPTS.EQ.12) IREAD = 0
      CALL GETREF(IREAD, FNREF,
     &            NFX,NDX,NDAT,NF,XYREF,LABREF )
      DO ID=1, NDAT
        CALL STRIP(LABREF(ID),NLAB)
      ENDDO
      DO ID=1, NDAT
ccc     IFCOL(ID) = NCOLOR - ID + 1
        IFCOL(ID) = 2 + ID
      ENDDO
      GO TO 1
C
C=============================================
C---- Make the Plot
 30   IF (NPOL.EQ.0 .AND. NDAT.EQ.0) GO TO 1
C
C---- sort each polar by increasing alpha
      DO 302 IP=1, NPOL
        CALL SORT(NA(IP),CPOL(1,IAL,IP),ISORT)
C
        DO K=1, IPTOT
          DO IA=1, NA(IP)
            CSORT(IA) = CPOL(ISORT(IA),K,IP)
          ENDDO
          DO IA=1, NA(IP)
            CPOL(IA,K,IP) = CSORT(IA)
          ENDDO
        ENDDO
C
        DO IS=1, 2*NBL(IP)
          DO IA=1, NA(IP)
            CSORT(IA) = XTR(ISORT(IA),IS,IP)
          ENDDO
          DO IA=1, NA(IP)
            XTR(IA,IS,IP) = CSORT(IA)
          ENDDO
        ENDDO
C
 302  CONTINUE
C
      IF (LPLOT) CALL PLEND
      CALL PLOPEN(SCRNFR,IPSLU,IDEV)
      LPLOT = .TRUE.
C
C---- set 0.5" left,bottom margins
      CALL PLOTABS(0.5,0.5,-3)
C
      CALL NEWFACTOR(SIZE)
C
      CALL PLOT(6.0*CH,6.0*CH,-3)
C
      CALL POLPLT(NAX,NPOL,NA,CPOL, NAME  ,ICOL,
     &            NFX,NDAT,NF,XYREF,LABREF,IFCOL,
     &            ISX,NBL,XTR, IMATYP,IRETYP,
     &            TITLE,CODE,VERSION,
     &            AR, XCD,XAL,XOC, CH,CH2, LGRID,LCDW,LLIST,
     &            CPOLPLF, CCLEN,NCLEN )
      GO TO 1
C
C=============================================
C---- hardcopy output
 40   IF(LPLOT) CALL PLEND
      LPLOT = .FALSE.
      CALL REPLOT(IDEVRP)
      GO TO 1
C
C=============================================
C---- change settings
 50   CALL GETSET
      GO TO 1
C
C=============================================
C---- zoom
 60   CALL USETZOOM(.FALSE.,.TRUE.)
      CALL REPLOT(IDEV)
      GO TO 1
C
C=============================================
C---- unzoom
 70   CALL CLRZOOM
      CALL REPLOT(IDEV)
      GO TO 1
C
C=============================================
C---- annotate plot
 80   IF(.NOT.LPLOT) THEN
       WRITE(*,*) 'No active plot to annotate'
       GO TO 1
      ENDIF
      CALL ANNOT(CH)
      GO TO 1
C=============================================
C
  900 CALL PLCLOSE
      STOP
      END ! PPLOT


      SUBROUTINE GETCLEN(NAME,CLEN,NCLEN)
      CHARACTER*(*) NAME, CLEN
C--------------------------------------------------
C     Looks for substring  "(c=01234***)"
C     in the NAME string.  If found, then
C     the "***" string is returned in CLEN.
C     If not found, then CLEN is returned blank.
C--------------------------------------------------
C
      CLEN = ' '
C
      K1 = INDEX( NAME , '(c=' )
      IF(K1.EQ.0) RETURN
C
      NNAME = LEN(NAME)
      K2 = INDEX( NAME(K1:NNAME) , ')' ) + K1 - 2
      IF(K2-K1.LT.3) RETURN
C
      DO K = K1+3, K2
        IF(INDEX( '0123456789.,)' , NAME(K:K) ) .EQ. 0) THEN
         CLEN = NAME(K:K2)
         NCLEN = K2-K+1
         RETURN
        ENDIF
      ENDDO
C
      RETURN
      END


      SUBROUTINE POLPLT(NAX,NPOL,NA,CPOL, NAME  ,ICOL,
     &                  NFX,NDAT,NF,XYREF,LABREF,IFCOL,
     &                  ISX,NBL,XTR, IMATYP,IRETYP,
     &                  TITLE,CODE,VERSION,
     &                  AR, XCD,XAL,XOC, CH,CH2, LGRID,LCDW,LLIST,
     &                  CPOLPLF, CCLEN,NCLEN )
      INCLUDE 'PINDEX.INC'
      CHARACTER*(*) NAME(NPOL), LABREF(NDAT)
      CHARACTER*(*) CODE, TITLE, CCLEN
      LOGICAL LGRID, LCDW, LLIST
C
      DIMENSION NA(NPOL), ICOL(NPOL), NBL(NPOL),
     &          NF(4,NDAT), IFCOL(NDAT), IMATYP(NPOL),IRETYP(NPOL)
      DIMENSION CPOL(NAX,IPTOT,NPOL), XYREF(NFX,2,4,NDAT),
     &          XTR(NAX,ISX,NPOL)
      DIMENSION CPOLPLF(3,*)
C
C
      LOGICAL NAMVAR,MACVAR,REYVAR,ACRVAR
C
      DIMENSION XLIN(3), YLIN(3)
C
      PARAMETER (NPX=100)
      REAL MACH
      DIMENSION MACH(NPX), REYN(NPX), ACRIT(NPX)
C
      CHARACTER*1 CC
C
      DATA LMASK1, LMASK2, LMASK3 / -32640, -30584, -21846 /
C
      IF(NPOL.GT.NPX) THEN
        WRITE(*,*) 'POLPLT: Local array overflow. Increase NPX.'
        RETURN
      ENDIF
C
      DO IP=1, NPOL
        IF    (IMATYP(IP).EQ.1) THEN
         MACH(IP) = CPOL(1,IMA,IP)
        ELSEIF(IMATYP(IP).EQ.2) THEN
         MACH(IP) = CPOL(1,IMA,IP) * SQRT(ABS(CPOL(1,ICL,IP)))
        ELSE
         MACH(IP) = CPOL(1,IMA,IP) * ABS(CPOL(1,ICL,IP))
        ENDIF
C
        IF    (IRETYP(IP).EQ.1) THEN
         REYN(IP) = CPOL(1,IRE,IP)
        ELSEIF(IRETYP(IP).EQ.2) THEN
         REYN(IP) = CPOL(1,IRE,IP) * SQRT(ABS(CPOL(1,ICL,IP)))
        ELSE
         REYN(IP) = CPOL(1,IRE,IP) * ABS(CPOL(1,ICL,IP))
        ENDIF
C
        ACRIT(IP) = CPOL(1,INC,IP)
      ENDDO
C
      CALL GETVAR(NPOL,NAME,MACH,REYN,ACRIT,
     &            NAMVAR,MACVAR,REYVAR,ACRVAR)
C
C---- unpack plot limit array
      CLMIN = CPOLPLF(1,ICL)
      CLMAX = CPOLPLF(2,ICL)
      CLDEL = CPOLPLF(3,ICL)
C                 
      CDMIN = CPOLPLF(1,ICD)
      CDMAX = CPOLPLF(2,ICD)
      CDDEL = CPOLPLF(3,ICD)
C                 
      CMMIN = CPOLPLF(1,ICM)
      CMMAX = CPOLPLF(2,ICM)
      CMDEL = CPOLPLF(3,ICM)
C                 
      ALMIN = CPOLPLF(1,IAL)
      ALMAX = CPOLPLF(2,IAL)
      ALDEL = CPOLPLF(3,IAL)
C
C
C---- Set scale factors
      CLWT = AR  / (CLMAX-CLMIN)
      CDWT = XCD / (CDMAX-CDMIN)
      ALWT = XAL / (ALMAX-ALMIN)
C
      CMWT = CLWT*2.0*CLDEL/CMMAX
C
      SH = 0.7*CH2
C
      IF(NCLEN.EQ.0) THEN
       CC = ' '
      ELSE
       CC = 'c'
      ENDIF
C
C---- number of text lines to be plotted in left upper corner of CL-CD plot
      LINBOX = NDAT
      IF(NPOL.GT.1) LINBOX = LINBOX + NPOL + 1
      DYBOX = CH2*(2.0*FLOAT(LINBOX) + 1.0)
C

C---- set default color index
      CALL GETCOLOR(ICOL0)
C
C
      CALL PLOT(-CDWT*CDMIN,-CLWT*CLMIN,-3)
C
C---- plot Polar data - Title, airfoils: name, Mach, Re, and Ncrit
      XPLT0 = CDWT*CDMIN
      YPLT0 = CLMAX*CLWT
      CALL LABEL(NPOL, NAME ,ICOL,
     &           IMATYP, IRETYP,
     &           MACH, REYN, ACRIT,
     &           TITLE,
     &           XPLT0,YPLT0, AR, CH,CH2, LLIST, CCLEN,NCLEN )
C
      CALL NEWCOLOR(ICOL0)

      IF(XCD.EQ.0.0) GO TO 100
C
C---- CL axis for CL-CD polar
      CALL NEWPEN(2)
      CALL YAXIS(CDWT*CDMIN,CLWT*CLMIN,AR,CLWT*CLDEL,CLMIN,CLDEL,CH2,1)
C
      CALL NEWPEN(3)
      IF(NCLEN.GT.0) THEN
       XPLT = CDWT* CDMIN            - 3.0*CH - FLOAT(NCLEN)*1.2*CH
       YPLT = CLWT*(CLMAX-1.5*CLDEL) - 0.5*CH
       CALL PLCHAR(XPLT,YPLT,1.2*CH,'('  ,0.0,1)
       CALL PLCHAR(999.,YPLT,1.2*CH,CCLEN,0.0,NCLEN)
       CALL PLCHAR(999.,YPLT,1.2*CH,')'  ,0.0,1)
      ENDIF
C
      XPLT = CDWT* CDMIN            - 3.2*CH
      YPLT = CLWT*(CLMAX-0.5*CLDEL) - 0.6*CH
      IF(NCLEN.GT.0) THEN
      CALL PLCHAR(XPLT-1.1*CH,YPLT       ,1.1*CH,CC ,0.0,1)
      ENDIF
      CALL PLCHAR(XPLT       ,YPLT       ,1.4*CH,'C',0.0,1)
      CALL PLCHAR(XPLT+1.2*CH,YPLT-0.4*CH,0.9*CH,'L',0.0,1)
C
C---- CD axis for CL-CD polar
      CALL NEWPEN(2)
      CALL XAXIS(CDWT*CDMIN,CLWT*CLMIN,-XCD,CDWT*CDDEL,
     &           10000.*CDMIN,10000.*CDDEL,CH2,-1)
C
      CALL NEWPEN(3)
      NXL = INT((CDMAX-CDMIN)/CDDEL + 0.5)
      XPLT = CDWT*(CDMAX - (FLOAT((NXL+1)/2) - 0.5)*CDDEL) - 4.5*CH2
      YPLT = CLWT* CLMIN - 4.8*CH2
      CALL PLCHAR(XPLT       ,YPLT       ,1.4*CH,'10'    ,0.0,5)
      CALL PLMATH(XPLT       ,YPLT       ,1.4*CH,'  4'   ,0.0,3)
      CALL PLMATH(XPLT+3.9*CH,YPLT       ,1.0*CH,   '#'  ,0.0,1)
      IF(NCLEN.GT.0) THEN
      CALL PLCHAR(XPLT+4.9*CH,YPLT       ,1.1*CH,     CC ,0.0,1)
      ENDIF
      CALL PLCHAR(XPLT+6.0*CH,YPLT       ,1.4*CH,     'C',0.0,1)
      CALL PLCHAR(XPLT+7.2*CH,YPLT-0.4*CH,0.9*CH,     'D',0.0,1)
C
      YLINE = CLWT*CLMAX - 2.0*CH2
C
      CALL NEWPEN(3)
C
      IF(NAMVAR) THEN
       XPLT = CDWT*CDMIN + 6.0*CH2
       YPLT = YLINE
       CALL PLCHAR(XPLT    ,YPLT,    CH2,'Airfoil',0.0,7)
       YLINE = YLINE - 2.25*CH2
      ENDIF
C
      IF(REYVAR) THEN
       XPLT = CDWT*CDMIN + 7.5*CH2
       YPLT = YLINE
       ITYP = IRETYP(1)
       IF(ITYP.EQ.1) THEN
        CALL PLCHAR(XPLT        ,YPLT,    CH2,'Re'  ,0.0,2)
       ELSE IF(ITYP.EQ.2) THEN
        CALL PLMATH(XPLT-1.0*CH2,YPLT,    CH2,'  R  ',0.0,5)
        CALL PLCHAR(XPLT-1.0*CH2,YPLT,    CH2,'Re C' ,0.0,4)
        CALL PLCHAR(999.        ,999.,0.7*CH2,    'L',0.0,1)
       ELSE IF(ITYP.EQ.3) THEN
        CALL PLMATH(XPLT-1.0*CH2,YPLT,    CH2,'  #  ',0.0,5)
        CALL PLCHAR(XPLT-1.0*CH2,YPLT,    CH2,'Re C' ,0.0,4)
        CALL PLCHAR(999.        ,999.,0.7*CH2,    'L',0.0,1)
       ENDIF
       YLINE = YLINE - 2.25*CH2
      ENDIF
C
      IF(ACRVAR) THEN
       XPLT = CDWT*CDMIN + 8.0*CH2
       YPLT = YLINE
       CALL PLCHAR(XPLT,YPLT,    CH2,'N'   ,0.0,1)
       CALL PLCHAR(999.,999.,0.7*CH2,'crit',0.0,4)
       YLINE = YLINE - 2.25*CH2
      ENDIF
C
      IF(MACVAR) THEN
       XPLT = CDWT*CDMIN + 7.5*CH2
       YPLT = YLINE
       ITYP = IMATYP(1)
       IF(ITYP.EQ.1) THEN
        CALL PLCHAR(XPLT        ,YPLT,    CH2,'Ma'  ,0.0,2)
       ELSE IF(ITYP.EQ.2) THEN
        CALL PLMATH(XPLT-1.0*CH2,YPLT,    CH2,'  R  ',0.0,5)
        CALL PLCHAR(XPLT-1.0*CH2,YPLT,    CH2,'Ma C' ,0.0,4)
        CALL PLCHAR(999.        ,999.,0.7*CH2,    'L',0.0,1)
       ELSE IF(ITYP.EQ.3) THEN
        CALL PLMATH(XPLT-1.0*CH2,YPLT,    CH2,'  #  ',0.0,5)
        CALL PLCHAR(XPLT-1.0*CH2,YPLT,    CH2,'Ma C' ,0.0,4)
        CALL PLCHAR(999.        ,999.,0.7*CH2,    'L',0.0,1)
       ENDIF
       YLINE = YLINE - 2.25*CH2
      ENDIF
C
C---- plot CL-CD polar(s)
      DO IP=1, NPOL
        CALL NEWCOLOR(ICOL(IP))
        CALL NEWPEN(4)
        CALL XYLINE(NA(IP),CPOL(1,ICD,IP),CPOL(1,ICL,IP),
     &              0.,CDWT,0.,CLWT,IP)
        IF(LCDW)
     &  CALL XYLINE(NA(IP),CPOL(1,ICW,IP),CPOL(1,ICL,IP),
     &              0.,CDWT,0.,CLWT,IP)
      ENDDO
C
C---- label each polar with legend
      IF(NAMVAR .OR. REYVAR .OR. ACRVAR .OR. MACVAR) THEN
       DO IP=1, NPOL
         CALL NEWCOLOR(ICOL(IP))
         XLIN(1) =     CH2
         XLIN(2) = 3.0*CH2
         XLIN(3) = 5.0*CH2
         YLIN(1) = YLINE + 0.5*CH2
         YLIN(2) = YLINE + 0.5*CH2
         YLIN(3) = YLINE + 0.5*CH2
         CALL NEWPEN(4)
         CALL XYLINE(3,XLIN,YLIN,0.0,1.0,0.0,1.0,IP)
         CALL NEWPEN(2)
         XPT = CDWT*CDMIN + 6.5*CH2
         IF(NAMVAR) CALL PLCHAR(XPT,YLINE,.8*CH2,NAME(IP)     ,0.,11)
         IF(REYVAR) CALL PLNUMB(XPT,YLINE,.8*CH2,REYN(IP)*1.E6,0.,-1)
         IF(ACRVAR) CALL PLNUMB(XPT,YLINE,.8*CH2,ACRIT(IP)    ,0., 3)
         IF(MACVAR) CALL PLNUMB(XPT,YLINE,.8*CH2,MACH(IP)     ,0., 3)
         YLINE = YLINE - 2.0*CH2
       ENDDO
       YLINE = YLINE - 0.5*CH2
C
      ENDIF
C
C
C---- plot CL-CD reference data
      DO 13 ID=1, NDAT
        IF(NF(1,ID).EQ.0) GO TO 13
C
        CALL NEWCOLOR(IFCOL(ID))
C
        CALL NEWPEN(3)
        CALL XYSYMB(NF(1,ID),XYREF(1,1,1,ID),XYREF(1,2,1,ID),
     &              0.0,CDWT,0.0,CLWT,SH,ID)
        XPLT = CDWT*CDMIN + 1.5*CH2
        YPLT = YLINE + 0.5*CH2
        CALL PLSYMB(XPLT,YPLT,SH,ID,0.0,0)
        XPLT = CDWT*CDMIN + 3.0*CH2
        CALL NEWPEN(2)
        LABLEN = LEN(LABREF(ID))
        CALL PLCHAR(XPLT,YLINE,0.8*CH2,LABREF(ID),0.0,LABLEN)
        YLINE = YLINE - 2.0*CH2
   13 CONTINUE
C
      CALL NEWCOLOR(ICOL0)
C
C----- coarse grid
       CALL NEWPEN(1)
       DXG = CDWT*CDDEL
       DYG = CLWT*CLDEL
C
       NYGBOX = INT( DYBOX/(DYG/5.0) ) + 1
       IF (LINBOX.EQ.0) NYGBOX = 0
       DYGBOX = (DYG/5.0) * FLOAT(NYGBOX)
C
       Y0 = CLWT*CLMIN
       NXG = INT( XCD/(CDWT*CDDEL)    + 0.01 )
       NYG = INT( (CLMAX-CLMIN)/CLDEL + 0.01 )
       DO K=0, NXG
         XL = CDWT*(CDMIN + CDDEL*FLOAT(K))
         CALL PLOT(XL,Y0,3)
         IF(K.GE.NXG/2) CALL PLOT(XL, Y0 + DYG*FLOAT(NYG)       , 2)
         IF(K.LT.NXG/2) CALL PLOT(XL, Y0 + DYG*FLOAT(NYG)-DYGBOX, 2)
       ENDDO
C
       X0 = CDWT*CDMIN
       DO K=0, NYG
         YL = Y0 + DYG*FLOAT(K)
         CALL PLOT(X0, YL, 3)
CCC         IF(K.NE.NYG) CALL PLOT(X0          , YL, 3)
CCC         IF(K.EQ.NYG) CALL PLOT(X0 + 2.0*DXG, YL, 3)
         CALL PLOT(CDWT*CDMAX, YL, 2)
       ENDDO
C
      IF(LGRID) THEN
C
       CALL PLFLUSH
C
C----- fine grid
       CALL NEWPEN(1)
       DXG = CDWT*CDDEL / 5.0
       DYG = CLWT*CLDEL / 5.0
C
CCC       IF(K.GE.NXG/2) CALL PLOT(XL, Y0 + DYG*FLOAT(NYG)       , 2)
CCC       IF(K.LT.NXG/2) CALL PLOT(XL, Y0 + DYG*FLOAT(NYG)-DYGBOX, 2)
       X0 = CDWT*CDMIN
       Y0 = CLWT*CLMIN
       NXGF = 5*(NXG/2)
       NYGF = 5* NYG   - NYGBOX
       CALL PLGRID(X0,Y0, NXGF,DXG, NYGF,DYG, LMASK2 )
C
       X0 = X0 + DXG*FLOAT(NXGF)
       NXGF = 5*NXG - NXGF
       NYGF = 5*NYG
       CALL PLGRID(X0,Y0, NXGF,DXG, NYGF,DYG, LMASK2 )
C
      ENDIF
C
C---- re-origin for CL-a plot
      CALL PLOT(CDWT*CDMAX + 0.05 - ALWT*ALMIN,0.0,-3)
C
 100  CONTINUE
      IF(XAL.EQ.0.0) GO TO 200
C
C---- CL axis for CL-a plot
      CALL NEWPEN(2)
      CALL YAXIS(0.0,CLWT*CLMIN,-AR,CLWT*CLDEL,CLMIN,CLDEL,-CH2,1)
C
      CALL NEWPEN(3)
      YPLT = CLWT*(CLMAX-0.5*CLDEL) - 0.6*CH
      IF(NCLEN.GT.0) THEN
      CALL PLCHAR(0.9*CH,YPLT       ,1.1*CH,CC ,0.0,1)
      ENDIF
      CALL PLCHAR(2.0*CH,YPLT       ,1.4*CH,'C',0.0,1)
      CALL PLCHAR(3.2*CH,YPLT-0.4*CH,0.9*CH,'L',0.0,1)
C
C---- a-axis for CL-a plot
      CALL NEWPEN(2)
      XALS = XAL
      IF(CLMIN.LT.0.0) XALS = -XAL
      CALL XAXIS(ALWT*ALMIN,0.0,XALS,ALWT*ALDEL,ALMIN,ALDEL,CH2,-1)
C
      CALL NEWPEN(3)
      XPLT = ALWT*(ALMAX - 1.5*ALDEL) - 0.5*CH
      YPLT = -4.5*CH
      CALL PLMATH(XPLT,YPLT,1.4*CH,'a',0.0,1)
C
C---- plot CL-a plot
      DO IP=1, NPOL
        CALL NEWCOLOR(ICOL(IP))
        CALL NEWPEN(4)
        CALL XYLINE(NA(IP),CPOL(1,IAL,IP),CPOL(1,ICL,IP),
     &              0.0,ALWT,0.0,CLWT,IP)
      ENDDO
C
C---- plot reference data
      DO 25 ID=1, NDAT
        IF(NF(2,ID).EQ.0) GO TO 25
C
        CALL NEWCOLOR(IFCOL(ID))
        CALL NEWPEN(3)
        CALL XYSYMB(NF(2,ID),XYREF(1,1,2,ID),XYREF(1,2,2,ID),
     &              0.0,ALWT,0.0,CLWT,SH,ID)
   25 CONTINUE
C
      CALL NEWCOLOR(ICOL0)
C
C---- CM axis for CM-a plot
      CALL NEWPEN(2)
      YCM = CLWT*CLDEL*AINT(0.5*CLMAX/CLDEL + 0.51)
      CALL YAXIS(0.0,CMWT*CMMIN,-YCM,CMWT*CMDEL,-CMMIN,-CMDEL,CH2,2)
C
      CALL NEWPEN(3)
      XPLT = -4.5*CH
      YPLT = CMWT*(CMMAX-0.5*CMDEL) - 0.6*CH
      IF(NCLEN.GT.0) THEN
      CALL PLCHAR(XPLT-0.8*CH,YPLT       ,1.1*CH,CC ,0.0,1)
      CALL PLMATH(XPLT+0.2*CH,YPLT       ,1.1*CH,'2',0.0,1)
      ENDIF
      CALL PLCHAR(XPLT+1.2*CH,YPLT       ,1.4*CH,'C',0.0,1)
      CALL PLCHAR(XPLT+2.4*CH,YPLT-0.4*CH,0.9*CH,'M',0.0,1)
C
C---- plot CM-a plot
      CALL NEWPEN(4)
      DO IP=1, NPOL
        CALL NEWCOLOR(ICOL(IP))
        CALL NEWPEN(4)
        CALL XYLINE(NA(IP),CPOL(1,IAL,IP),CPOL(1,ICM,IP),
     &              0.0,ALWT,0.0,-CMWT,IP)
      ENDDO
C
C---- plot reference data
      DO 35 ID=1, NDAT
        IF(NF(3,ID).EQ.0) GO TO 35
C
        CALL NEWCOLOR(IFCOL(ID))
        CALL NEWPEN(3)
        CALL XYSYMB(NF(3,ID),XYREF(1,1,3,ID),XYREF(1,2,3,ID),
     &              0.0,ALWT,0.0,-CMWT,SH,ID)
   35 CONTINUE
C
      CALL NEWCOLOR(ICOL0)
C
C---- re-origin for xtr plot
 200  CALL PLOT( ALWT*ALMAX + 0.05, 0.0, -3 )
      IF(XOC .EQ. 0.0) GO TO 300
C
      CALL NEWPEN(2)
      CALL XAXIS(0.0,CLWT*CLMIN,XOC,0.5*XOC,0.0,0.5,CH2,1)
C
      CALL NEWPEN(3)
      XPLT = 0.75*XOC   - 2.2*CH2
      YPLT = CLWT*CLMIN - 4.7*CH2
      CALL PLCHAR(XPLT,YPLT,1.3*CH2,'x  /c',0.0,5)
      CALL PLCHAR(XPLT+1.2*CH2,YPLT-0.4*CH2,0.9*CH2,'tr',0.0,2)
C
C---- plot xtr/c
      DO IP=1, NPOL
        CALL NEWCOLOR(ICOL(IP))
        CALL NEWPEN(4)
        DO IS=1, 2*NBL(IP)
          CALL XYLINE(NA(IP),XTR(1,IS,IP),CPOL(1,ICL,IP),
     &                0.0,XOC,0.0,CLWT,IP)
        ENDDO
      ENDDO
C
C---- plot reference data
      DO 55 ID=1, NDAT
        IF(NF(4,ID).EQ.0) GO TO 55
C
        CALL NEWCOLOR(IFCOL(ID))
        CALL NEWPEN(3)
        CALL XYSYMB(NF(4,ID),XYREF(1,1,4,ID),XYREF(1,2,4,ID),
     &              0.0,XOC,0.0,CLWT,SH,ID)
   55 CONTINUE
C
      CALL NEWCOLOR(ICOL0)
C
C----- coarse grid
       CALL NEWPEN(1)
       CALL PLOT(0.0    ,CLWT*CLMIN,3)
       CALL PLOT(0.0    ,CLWT*CLMAX,2)
       CALL PLOT(0.5*XOC,CLWT*CLMIN,3)
       CALL PLOT(0.5*XOC,CLWT*CLMAX,2)
       CALL PLOT(    XOC,CLWT*CLMIN,3)
       CALL PLOT(    XOC,CLWT*CLMAX,2)
C
       DYG = CLWT*CLDEL
       Y0  = CLWT*CLMIN
       NYG = INT( (CLMAX-CLMIN)/CLDEL + 0.01 )
       DO K=0, NYG
         YL = Y0 + DYG*FLOAT(K)
         CALL PLOT(0.0,YL,3)
         CALL PLOT(XOC,YL,2)
       ENDDO
C
      IF(LGRID) THEN
C
       CALL PLFLUSH
C
C----- fine grid
       CALL NEWPEN(1)
       DXG =  XOC*0.5   / 5.0
       DYG = CLWT*CLDEL / 5.0
       X0 = 0.0
       Y0 = CLWT*CLMIN
       NXG = 10
       NYG = INT( (CLMAX-CLMIN)/CLDEL + 0.01 ) * 5
       CALL PLGRID(X0,Y0, NXG,DXG, NYG,DYG, LMASK2 )
C
      ENDIF
C
 300  CONTINUE
C
C---- code and version identifier
      CHI = 0.75*CH2
      CALL NEWPEN(2)
      XPLT = XOC        - 12.0*CHI
      YPLT = CLWT*CLMAX +  0.5*CHI
      CALL PLCHAR(XPLT        ,YPLT,CHI,CODE   ,0.0,5)
      CALL PLCHAR(XPLT+6.0*CHI,YPLT,CHI,'V'    ,0.0,1)
      CALL PLNUMB(XPLT+8.0*CHI,YPLT,CHI,VERSION,0.0,2)
C
      CALL PLFLUSH
C
      RETURN
      END ! POLPLT
 


      SUBROUTINE LABEL(NPOL, NAME ,ICOL,
     &                 IMATYP, IRETYP,
     &                 MACH, REYN, ACRIT,
     &                 TITLE,
     &                 XPLT0,YPLT0, AR, CH,CH2, LLIST, CCLEN,NCLEN )
      INCLUDE 'PINDEX.INC'
C
      CHARACTER*(*) NAME(NPOL)
      CHARACTER*(*) TITLE, CCLEN
C
      DIMENSION ICOL(NPOL), IMATYP(NPOL),IRETYP(NPOL)
      REAL MACH
      DIMENSION MACH(NPOL), REYN(NPOL), ACRIT(NPOL)
      LOGICAL LLIST
C
      CH3 = 0.90*CH2
      CH4 = 1.10*CH2
C
C---- y-spacing for label lines
      YSPC = 1.9*CH4
C
C...Put up title
C
      XPLT = XPLT0 - CH2
      YPLT = YPLT0 + 0.6*CH4
      IF(LLIST) YPLT = YPLT + YSPC*(NPOL+1)
C
      CALL NEWPEN(3)
      LENT = LEN(TITLE)
      CALL PLCHAR(XPLT,YPLT,1.2*CH4,TITLE,0.0,LENT)
C
      IF(.NOT.LLIST) RETURN
C
C
C...Put up polar identification data: name, flow conditions
      NMAX = 0
      DO IP = 1, NPOL
        CALL STRIP(NAME(IP),NNAME)
        NMAX = MAX(NMAX,NNAME)
      ENDDO
C
      DO 900 IP = 1, NPOL
C
      CALL NEWCOLOR(ICOL(IP))
C
      XPLT = XPLT0
      YPLT = YPLT0 + YSPC*(NPOL-IP+1)
C
      CALL NEWPEN(3)
      CALL PLCHAR(XPLT,YPLT,CH4,NAME(IP),0.0,NMAX)
      XPLT = XPLT + CH4*FLOAT(NMAX)
C
      CALL NEWPEN(2)
C
       ITYP = IRETYP(IP)
       IF(ITYP.EQ.1) THEN
        CALL PLCHAR(XPLT,YPLT,CH3,'   Re = '   ,0.0,  8)
        XPLT = XPLT + CH3*8.0
       ELSE IF(ITYP.EQ.2) THEN
        CALL PLCHAR(XPLT,YPLT,CH3,'   Re CL = ',0.0, 11)
        CALL PLMATH(XPLT,YPLT,CH3,'     R   = ',0.0, 11)
        XPLT = XPLT + CH3*11.0
       ELSE IF(ITYP.EQ.3) THEN
        CALL PLCHAR(XPLT,YPLT,CH3,'   Re CL = ',0.0, 11)
        XPLT = XPLT + CH3*11.0
       ENDIF
       CALL PLNUMB(XPLT,YPLT,CH3,REYN(IP)*1.0E6,0.0,-1)
       IF(NCLEN.GT.0) THEN
        CALL PLCHAR(999.,YPLT,CH3,'/'  ,0.0,1)
        CALL PLCHAR(999.,YPLT,CH3,CCLEN,0.0,NCLEN)
        XPLT = XPLT + CH3*FLOAT(1+NCLEN)
       ENDIF
       XPLT = XPLT + CH3*7.0
C
       ITYP = IMATYP(IP)
       IF(ITYP.EQ.1) THEN
        CALL PLCHAR(XPLT,YPLT,CH3,'   Ma = '   ,0.0,  8)
        XPLT = XPLT + CH3*8.0
       ELSE IF(ITYP.EQ.2) THEN
        CALL PLCHAR(XPLT,YPLT,CH3,'   Ma CL = ',0.0, 11)
        CALL PLMATH(XPLT,YPLT,CH3,'     R   = ',0.0, 11)
        XPLT = XPLT + CH3*11.0
       ELSE IF(ITYP.EQ.3) THEN
        CALL PLCHAR(XPLT,YPLT,CH3,'   Ma CL = ',0.0, 11)
        XPLT = XPLT + CH3*11.0
       ENDIF
       CALL PLNUMB(XPLT,YPLT,CH3,    MACH(IP)  ,0.0,3)
       XPLT = XPLT + CH3*5.0
C
       CALL PLCHAR(XPLT,YPLT,    CH3,'   N',0.0,4)
       XPLT = XPLT + CH3*4.0
       CALL PLCHAR(XPLT,YPLT,0.8*CH3,'crit',0.0,4)
       XPLT = XPLT + CH3*3.2
       CALL PLCHAR(XPLT,YPLT,    CH3,' = ' ,0.0,3)
       XPLT = XPLT + CH3*3.0
       CALL PLNUMB(XPLT,YPLT,    CH3,ACRIT(IP) ,0.0,3)
       XPLT = XPLT + CH3*6.0
C
  900 CONTINUE
C
      RETURN
      END ! LABEL


 
      SUBROUTINE MINMAX(NAX,NPOL,NA,CPOL,CPOLPLF)
      INCLUDE 'PINDEX.INC'
      DIMENSION NA(NPOL)
      DIMENSION CPOL(NAX,IPTOT,NPOL), CPOLPLF(3,*)
C--------------------------------------------
C     Determines max and min limits of polar
C     quantities among all polars passed in.
C--------------------------------------------
C
      DO IP=1, NPOL
        DO K=1, 4
          CPOLPLF(1,K) = CPOL(1,K,IP)
          CPOLPLF(2,K) = CPOL(1,K,IP)
          DO I=2, NA(IP)
            CPOLPLF(1,K) = MIN( CPOL(I,K,IP) , CPOLPLF(1,K) )
            CPOLPLF(2,K) = MAX( CPOL(I,K,IP) , CPOLPLF(2,K) )
          ENDDO
        ENDDO
      ENDDO
C
      RETURN
      END ! MINMAX


 
      SUBROUTINE GETDEF
      INCLUDE 'PINDEX.INC'
      INCLUDE 'PPLOT.INC'
C
C---- Plotting flag
      IDEV = 1   ! X11 window only
c     IDEV = 2   ! B&W PostScript output file only (no color)
c     IDEV = 3   ! both X11 and B&W PostScript file
c     IDEV = 4   ! Color PostScript output file only 
c     IDEV = 5   ! both X11 and Color PostScript file 
C
C---- Re-plotting flag (for hardcopy)
      IDEVRP = 2   ! B&W PostScript
c     IDEVRP = 4   ! Color PostScript
C
C---- PostScript output logical unit and file specification
      IPSLU = 0  ! output to file  plot.ps   on LU 4    (default case)
c     IPSLU = ?  ! output to file  plot?.ps  on LU 10+?
C
C---- screen fraction taken up by plot window upon opening
      SCRNFR = 0.70
C
C---- Default plot size in inches
C-    (Default plot window is 11.0 x 8.5)
      SIZE = 8.0
C
C---- plot aspect ratio V/H
      AR = 0.60
C
C---- character height
      CH  = 0.016
      CH2 = 0.0135
C
C---- set default color table and get number of colors
      CALL COLORMAPDEFAULT
      CALL GETNUMCOLOR(NCOLOR)
C
      LGRID = .TRUE.
      LCDW  = .FALSE.
      LLIST = .TRUE.
      LCLEN = .FALSE.
C
      CPOLPLF(1,ICL) = 0.0   ! CLmax
      CPOLPLF(2,ICL) = 1.5   ! CLmin
      CPOLPLF(3,ICL) = 0.5   ! Axis CL increment
C         
      CPOLPLF(1,ICD) = 0.0   ! CDmax
      CPOLPLF(2,ICD) = 0.01  ! CDmin
      CPOLPLF(3,ICD) = 0.005 ! Axis CD increment
C         
      CPOLPLF(1,ICM) = 0.0   ! -CMmax
      CPOLPLF(2,ICM) = 0.3   ! -CMmin
      CPOLPLF(3,ICM) = 0.1   ! Axis -CM increment
C         
      CPOLPLF(1,IAL) = -4.0  ! ALmax
      CPOLPLF(2,IAL) = 10.0  ! ALmin
      CPOLPLF(3,IAL) =  2.0  ! Axis AL increment
C
C
      XCD = 0.45
      XAL = 0.25
      XOC = 0.20
C
C
      TITLE = '                                '
CCC            12345678901234567890123456789012
C
C...Try to read  pplot.def  file
      OPEN(UNIT=10,FILE='pplot.def',STATUS='OLD',ERR=900)
      READ(10,*) CPOLPLF(1,ICL), CPOLPLF(2,ICL), CPOLPLF(3,ICL)
      READ(10,*) CPOLPLF(1,ICD), CPOLPLF(2,ICD), CPOLPLF(3,ICD)
      READ(10,*) CPOLPLF(1,ICM), CPOLPLF(2,ICM), CPOLPLF(3,ICM)
      READ(10,*) CPOLPLF(1,IAL), CPOLPLF(2,IAL), CPOLPLF(3,IAL)
      READ(10,*) XCD, XAL, XOC
      READ(10,*) IDEV, SIZE
      READ(10,*) AR, LGRID
      READ(10,*) CH, CH2
      CLOSE(UNIT=10)
      RETURN
C
  900 WRITE(*,*)
      WRITE(*,*) 'No  pplot.def  file found'
      WRITE(*,*) 'Hard-wired defaults used'
      WRITE(*,*)
      RETURN
      END ! GETDEF


 
      SUBROUTINE WRTDEF(LU)
      INCLUDE 'PINDEX.INC'
      INCLUDE 'PPLOT.INC'
      CHARACTER*1 CGRID
C
      CGRID = 'F'
      IF(LGRID) CGRID = 'T'
C
      WRITE(LU,1010) CPOLPLF(1,ICL), CPOLPLF(2,ICL), CPOLPLF(3,ICL)
      WRITE(LU,1020) CPOLPLF(1,ICD), CPOLPLF(2,ICD), CPOLPLF(3,ICD)
      WRITE(LU,1030) CPOLPLF(1,ICM), CPOLPLF(2,ICM), CPOLPLF(3,ICM)
      WRITE(LU,1040) CPOLPLF(1,IAL), CPOLPLF(2,IAL), CPOLPLF(3,IAL)
      WRITE(LU,1050) XCD, XAL, XOC
      WRITE(LU,1060) IDEV, SIZE
      WRITE(LU,1070) AR, CGRID
      WRITE(LU,1080) CH, CH2
      RETURN
C
C...............................................
 1010 FORMAT(1X, F9.4,F9.4,F9.4,' | CLmin   CLmax    dCL')
 1020 FORMAT(1X, F9.4,F9.4,F9.4,' | CDmin   CDmax    dCD')
 1030 FORMAT(1X, F9.4,F9.4,F9.4,' | CMmin   CMmax    dCM')
 1040 FORMAT(1X, F9.4,F9.4,F9.4,' | ALmin   ALmax    dAL')
 1050 FORMAT(1X, F9.4,F9.4,F9.4,' | CL-CD   CL-alpha  CL-Xtr  (widths)')
 1060 FORMAT(1X,I3,6X,F9.4,9X  ,' | device  width(in)')
 1070 FORMAT(1X,F9.4,8X,A1,9X  ,' | height/width  grid_plot_flag')
 1080 FORMAT(1X,F9.4,F9.4 ,9X  ,' | char.height1  char.height2')
      END ! WRTDEF
 

      SUBROUTINE GETSET
      INCLUDE 'PINDEX.INC'
      INCLUDE 'PPLOT.INC'
      LOGICAL OK
      CHARACTER*2 OPTION
      CHARACTER*16 CCLEN1

C---- Change plotting parameters
C
    1 WRITE(*,1000)
 1000 FORMAT(/ '  1   Change CL scaling'
     &       / '  2   Change CD scaling'
     &       / '  3   Change CM scaling'
     &       / '  4   Change ALPHA scaling'
     &       / '  5   Plot Size'
     &       / '  6   Plot Title'
     &       / '  7   Wave-CD plot toggle'
     &       / '  8   Write settings to  pplot.def  file'
     &       / '  9   Toggle airfoil list'
     &       / ' 10   Change reference-length unit'
     &      // '    Select option:  ',$)
C
      READ(*,1005) OPTION
 1005 FORMAT(A)
C
      IF(OPTION .EQ. '  ' .OR. OPTION.EQ.'0 ') THEN
C
        RETURN
C
      ELSE IF(OPTION.EQ.'1 ') THEN
C
        WRITE(*,1100) (CPOLPLF(K,ICL), K=1, 3)
        READ (*,*   ) (CPOLPLF(K,ICL), K=1, 3)
C
      ELSE IF(OPTION.EQ.'2 ') THEN
C
        WRITE(*,1200) (CPOLPLF(K,ICD), K=1, 3)
        READ (*,*   ) (CPOLPLF(K,ICD), K=1, 3)
C
      ELSE IF(OPTION.EQ.'3 ') THEN
C
        WRITE(*,1300) (CPOLPLF(K,ICM), K=1, 3)
        READ (*,*   ) (CPOLPLF(K,ICM), K=1, 3)
C
      ELSE IF(OPTION.EQ.'4 ') THEN
C
        WRITE(*,1400) (CPOLPLF(K,IAL), K=1, 3)
        READ (*,*   ) (CPOLPLF(K,IAL), K=1, 3)
C
      ELSE IF(OPTION.EQ.'5 ') THEN
C
        WRITE(*,1500) SIZE
        READ (*,*) SIZE
C
      ELSE IF(OPTION.EQ.'6 ') THEN
C
        TITLE = '                                '
CCC              12345678901234567890123456789012
        CALL ASKS('Enter plot title (32 chars)^',TITLE)
        CALL STRIP(TITLE,NTITLE)
C
      ELSE IF(OPTION.EQ.'7 ') THEN
C
        LCDW = .NOT. LCDW
        IF(     LCDW) WRITE(*,*) 'CDwave will be plotted'
        IF(.NOT.LCDW) WRITE(*,*) 'CDwave will not be plotted'
C
      ELSE IF(OPTION.EQ.'8 ') THEN
C
        OPEN(10,FILE='pplot.def',STATUS='OLD',ERR=803)
        CALL ASKL('File  pplot.def  exists.  Overwrite ?^',OK)
        IF(OK) THEN
         REWIND 10
         GO TO 806
        ENDIF
        WRITE(*,*)
        WRITE(*,*) 'No action taken'
        CLOSE(10)
        GO TO 1
C
 803    OPEN(10,FILE='pplot.def',STATUS='UNKNOWN')
 806    CALL WRTDEF(10)
        WRITE(*,*)
        WRITE(*,*) 'File pplot.def written'
        CLOSE(10)
C
      ELSE IF(OPTION.EQ.'9 ') THEN
C
        LLIST = .NOT. LLIST
        IF(     LLIST) WRITE(*,*) 'List of polars will be plotted'
        IF(.NOT.LLIST) WRITE(*,*) 'List of polars will not be plotted'
C
      ELSE IF(OPTION.EQ.'10') THEN
C
        WRITE(*,*)
        WRITE(*,*) 'Current reference length unit: ', CCLEN
        CALL ASKS(
     &   'Enter new reference length unit (<return> if none)^',CCLEN)
        CALL STRIP(CCLEN,NCLEN)
C
      ENDIF
      GO TO 1
C
 1100 FORMAT(/' Current   CLmin, CLmax, dCL = ',3F10.5
     &       /' Enter new CLmin, CLmax, dCL:  ',$)
 1200 FORMAT(/' Current   CDmin, CDmax, dCD = ',3F10.5
     &       /' Enter new CDmin, CDmax, dCD:  ',$)
 1300 FORMAT(/' Current   CMmin, CMmax, dCM = ',3F10.5
     &       /' Enter new CMmin, CMmax, dCM:  ',$)
 1400 FORMAT(/' Current   ALmin, ALmax, dAL = ',3F10.5
     &       /' Enter new ALmin, ALmax, dAL:  ',$)
 1500 FORMAT(/' Current   plot size = ', F10.5
     &       /' Enter new plot size:  ',$)
      END ! GETDEF



      SUBROUTINE GETPOL(IREAD,FNPOL,
     &            NAX,NPX,NPOL,NA,CPOL, NAME,
     &            ISX,NBL,XTR,
     &            TITLE,CODE,VERSION )
      INCLUDE 'PINDEX.INC'
      CHARACTER*(*) FNPOL(NPX), NAME(NPX)
      CHARACTER*(*) CODE, TITLE
C
      DIMENSION NA(NPX), NBL(NPX)
      DIMENSION CPOL(NAX,IPTOT,NPX),
     &          XTR(NAX,ISX,NPX)
C
      REAL MACH
C
      CHARACTER*80 LINE
      CHARACTER*1 DUMMY
C
      IF    (IREAD.LE.-1) THEN
C------ read new polars
        IP1 = 1
        IPN = NPX
      ELSEIF(IREAD.EQ.0 ) THEN
C------ re-read old polars
        IP1 = 1
        IPN = NPOL
      ELSE
C------ read additional polars
        IP1 = NPOL+1
        IPN = NPX
      ENDIF
C
C
      DO 900 IP=IP1, IPN
C
      IF(IREAD.NE.0) THEN
        CALL ASKS('Enter polar data filename or <return>^',FNPOL(IP))
      ENDIF
C
      OPEN(UNIT=9,FILE=FNPOL(IP),STATUS='OLD',ERR=901)
C
      READ(9,1000,END=901,ERR=901) DUMMY
      IF(IP.EQ.1) THEN
       READ(9,1005,END=901) CODE, VERSION
       CALL STRIP(CODE,NCODE)
      ELSE
       READ(9,1000,END=901) DUMMY
      ENDIF
      READ(9,1000,END=901) DUMMY
      READ(9,1000,END=901) LINE
C
      READ(LINE,1010,END=901) NAME(IP)
      K = INDEX(LINE,'element')
      NBL(IP) = 1
      IF(K.NE.0) READ(LINE(K-3:K-1),*,ERR=901) NBL(IP)
C
      READ(9,1000,END=901) DUMMY
      READ(9,1015,END=901,ERR=29) IRETYP, IMATYP
      GO TO 30
C
C----- no CL-dependence flags (V 3.2 ISES polar).  Assume Re, Ma are fixed.
   29  IRETYP = 1
       IMATYP = 1
C
   30 IF(IRETYP .EQ. 0) IRETYP = 1
      IF(IMATYP .EQ. 0) IMATYP = 1
      READ(9,1000,END=901) DUMMY
      READ(9,1000,END=901) DUMMY
      XTRS1 = 1.0
      XTRP1 = 1.0
      READ(9,1030,END=901,ERR=901) MACH, REYN, ACRIT
      READ(9,1000,END=901) DUMMY
      READ(9,1000,END=901) DUMMY
      READ(9,1000,END=901) DUMMY
      DO 40 IA=1, NAX
        READ(9,*,ERR=901,END=42)
     &     CPOL(IA,IAL,IP),
     &     CPOL(IA,ICL,IP),
     &     CPOL(IA,ICD,IP),
     &     CPOL(IA,ICW,IP),
     &     CPOL(IA,ICM,IP),
     &     XTR(IA,1,IP),
     &     XTR(IA,2,IP)
        CPOL(IA,IMA,IP) = MACH
        CPOL(IA,IRE,IP) = REYN
        CPOL(IA,INC,IP) = ACRIT
        CPOL(IA,IXT,IP) = XTRS1
        CPOL(IA,IXB,IP) = XTRP1
C
        ACL = MAX( ABS( CPOL(IA,ICL,IP) ) , 0.0001 )
        IF(IRETYP.EQ.2) CPOL(IA,IRE,IP) = REYN/SQRT(ACL)
        IF(IMATYP.EQ.2) CPOL(IA,IMA,IP) = MACH/SQRT(ACL)
        IF(IRETYP.EQ.3) CPOL(IA,IRE,IP) = REYN/ACL
        IF(IMATYP.EQ.3) CPOL(IA,IMA,IP) = MACH/ACL
C
   40 CONTINUE
   42 NA(IP) = IA - 1
C
      CLOSE(UNIT=9)
C
      WRITE(*,8000) NAME(IP)
      IF(IMATYP.EQ.1) WRITE(*,8011) MACH 
      IF(IMATYP.EQ.2) WRITE(*,8012) MACH
      IF(IMATYP.EQ.3) WRITE(*,8013) MACH
      IF(IRETYP.EQ.1) WRITE(*,8021) REYN 
      IF(IRETYP.EQ.2) WRITE(*,8022) REYN
      IF(IRETYP.EQ.3) WRITE(*,8023) REYN
      WRITE(*,8030) ACRIT
C
  900 CONTINUE
  901 NPOL = IP-1
C
      RETURN
C
C..........................................
 1000 FORMAT(A)
 1005 FORMAT(A16,12X,F10.0)
 1010 FORMAT(22X,A32)
 1015 FORMAT(2I2)
 1030 FORMAT( 8X,F7.3,10X,F9.3,17X,F7.3)
 8000 FORMAT(1X,A32)
 8011 FORMAT('             Ma =', F7.3,       $)
 8012 FORMAT('    sqrt(CL)*Ma =', F7.3,       $)
 8013 FORMAT('          CL*Ma =', F7.3,       $)
 8021 FORMAT('             Re =', F7.3,' e 6',$)
 8022 FORMAT('    sqrt(CL)*Re =', F7.3,' e 6',$)
 8023 FORMAT('          CL*Re =', F7.3,' e 6',$)
 8030 FORMAT('          Ncrit =', F6.2         )
      END ! GETPOL



      SUBROUTINE GETREF(IREAD, FNREF,
     &            NFX,NDX,NDAT,NF,XYREF,LABREF )
      INCLUDE 'PINDEX.INC'
      CHARACTER*(*) FNREF(NDX),LABREF(NDX)
      DIMENSION NF(4,NDX)
      DIMENSION XYREF(NFX,2,4,NDX)
C
      CHARACTER*80 LINE
C
      IF    (IREAD.LE.-1) THEN
C------ read new data sets
        ID1 = 1
        IDN = NDX
      ELSEIF(IREAD.EQ.0 ) THEN
C------ re-read old data sets
        ID1 = 1
        IDN = NDAT
      ELSE
C------ read additional data sets
        ID1 = NDAT+1
        IDN = NDX
      ENDIF
C
C
      DO 900 ID=ID1, IDN
C
      IF(IREAD.NE.0) THEN
       CALL ASKS('Enter reference data filename or <return>^',FNREF(ID))
      ENDIF
C
      OPEN(UNIT=9,FILE=FNREF(ID),STATUS='OLD',ERR=901)
C
C---- try to read data label
      READ(9,1000,END=901) LINE
 1000 FORMAT(A)
C
      IF(LINE(1:1).EQ.'#') THEN
C------ set data label
        LABREF(ID) = LINE(2:80)
      ELSE
C------ ask for data label
        IF(IREAD.NE.0) THEN
          CALL ASKS('Enter label for reference data^',LABREF(ID))
        ENDIF
C------ go read data
        REWIND(9)
      ENDIF
C
      DO 100 K=1, 4
C
        DO 10 I=1, NFX
          READ(9,*,END=11,ERR=901) XYREF(I,1,K,ID), XYREF(I,2,K,ID)
          IF(XYREF(I,1,K,ID).EQ.999.0) GO TO 11
   10   CONTINUE
   11   NF(K,ID) = I-1
C
  100 CONTINUE
      CLOSE(UNIT=9)
C
  900 CONTINUE
C
  901 NDAT = ID-1
      WRITE(*,*)
      WRITE(*,*) NDAT, ' reference data files read in'
C
      RETURN
      END ! GETREF


 
      SUBROUTINE SORT(N,A,ISORT)
      DIMENSION A(N), ISORT(N)
C
      LOGICAL DONE
C
      DO I=1, N
        ISORT(I) = I
      ENDDO
C
      DO IPASS=1, 12345
        DONE = .TRUE.
C
        DO 20 I=2, N
C
          IO = ISORT(I  )
          IM = ISORT(I-1)
C
          IF(A(IO) .GE. A(IM)) GO TO 20
C
           DONE = .FALSE.
C
           ISORT(I  ) = IM
           ISORT(I-1) = IO
C
 20     CONTINUE
C
        IF(DONE) RETURN
C
      ENDDO
C
      END ! SORT
 
 


      SUBROUTINE GETTYP(NAX,NA,CPOL, IMATYP,IRETYP )
C
C---- Determines type of Ma(CL) and Re(CL) dependence
C
      INCLUDE 'PINDEX.INC'
C
      DIMENSION CPOL(NAX,IPTOT)
C
      IF(CPOL(NA,ICL)*CPOL(1,ICL) .LE. 0.0) THEN
        IMATYP = 1
        IRETYP = 1
        RETURN
      ENDIF
C
      IF(CPOL(NA,IMA)*CPOL(1,IMA) .LE. 0.0) THEN
        IMATYP = 1
      ELSE
        EX = LOG( CPOL(NA,IMA)/CPOL(1,IMA) )
     &     / LOG( CPOL(NA,ICL)/CPOL(1,ICL) )
        IF     (ABS(EX) .LT. 0.25) THEN
          IMATYP = 1
        ELSEIF (ABS(EX) .LT. 0.75) THEN
          IMATYP = 2
        ELSE
          IMATYP = 3
        ENDIF
      ENDIF
C
      IF(CPOL(NA,IRE)*CPOL(1,IRE) .LE. 0.0) THEN
        IRETYP = 1
      ELSE
        EX = LOG( CPOL(NA,IRE)/CPOL(1,IRE) )
     &     / LOG( CPOL(NA,ICL)/CPOL(1,ICL) )
        IF     (ABS(EX) .LT. 0.25) THEN
          IRETYP = 1
        ELSEIF (ABS(EX) .LT. 0.75) THEN
          IRETYP = 2
        ELSE
          IRETYP = 3
        ENDIF
      ENDIF
C
      RETURN
      END ! GETTYP



      SUBROUTINE GETVAR(NPOL,NAME,MACH,REYN,ACRIT,
     &                  NAMVAR,MACVAR,REYVAR,ACRVAR)
      CHARACTER*(*) NAME
      REAL MACH
      LOGICAL NAMVAR,MACVAR,REYVAR,ACRVAR
C
      DIMENSION NAME(NPOL),MACH(NPOL),REYN(NPOL),ACRIT(NPOL)
C
      NAMVAR = .FALSE.
      MACVAR = .FALSE.
      REYVAR = .FALSE.
      ACRVAR = .FALSE.
C
      DO 10 IP=1, NPOL-1
        IF(NAME(IP) .NE. NAME(IP+1)) THEN
         NAMVAR = .TRUE.
         RETURN
        ENDIF
   10 CONTINUE      
C
      DO 20 IP=1, NPOL-1
        IF(MACH(IP) .NE. MACH(IP+1)) THEN
         MACVAR = .TRUE.
         RETURN
        ENDIF
   20 CONTINUE      
C
      DO 30 IP=1, NPOL-1
        IF(REYN(IP) .NE. REYN(IP+1)) THEN
         REYVAR = .TRUE.
         RETURN
        ENDIF
   30 CONTINUE      
C
      DO 40 IP=1, NPOL-1
        IF(ACRIT(IP) .NE. ACRIT(IP+1)) THEN
         ACRVAR = .TRUE.
         RETURN
        ENDIF
   40 CONTINUE      
C
      RETURN
      END ! GETVAR
