
      SUBROUTINE SETEXP(S,DS1,SMAX,NN)
C==========================================================
C     Sets geometrically stretched array S:
C
C       S(i+1) - S(i)  =  r * [S(i) - S(i-1)]
C
C       S     (output)  array to be set  
C       DS1   (input)   first S increment:  S(2) - S(1)
C       SMAX  (input)   final S value:      S(NN)
C       NN    (input)   number of points
C==========================================================
      IMPLICIT REAL (A-H,M,O-Z)
      DIMENSION S(NN)
C
      SIGMA = SMAX/DS1
      NEX = NN-1
      RNEX = FLOAT(NEX)
      RNI = 1.0/RNEX
C
C---- solve quadratic for initial geometric ratio guess
      AAA = RNEX*(RNEX-1.0)*(RNEX-2.0) / 6.0
      BBB = RNEX*(RNEX-1.0) / 2.0
      CCC = RNEX - SIGMA
C
      DISC = BBB**2 - 4.0*AAA*CCC
      DISC = MAX( 0.0 , DISC )
C
      IF(NEX.LE.0) THEN
       STOP 'SETEXP: Cannot fill array.  N<0 too small.'
      ELSE IF(NEX.EQ.1) THEN
       WRITE(*,*) 'SETEXP: N=2 passed in.  Specified DS1 is not set.'
       S(1) = 0.0
       S(2) = SMAX
       RETURN
      ELSE IF(NEX.EQ.2) THEN
       RATIO = -CCC/BBB  +  1.0
      ELSE
       RATIO = (-BBB + SQRT(DISC))/(2.0*AAA)  +  1.0
      ENDIF
C
      IF(RATIO.EQ.1.0) GO TO 11
C
C---- Newton iteration for actual geometric ratio
      DO 1 ITER=1, 100
        SIGMAN = (RATIO**NEX - 1.0) / (RATIO - 1.0)
        RES = SIGMAN**RNI - SIGMA**RNI
        DRESDR = RNI*SIGMAN**RNI
     &         * (RNEX*RATIO**(NEX-1) - SIGMAN) / (RATIO**NEX - 1.0)
C
        DRATIO = -RES/DRESDR
        RATIO = RATIO + DRATIO
C
        IF(ABS(DRATIO) .LT. 1.0E-5) GO TO 11
C
    1 CONTINUE
      WRITE(*,*) 'SETEXP: Convergence failed.  dRatio =', DRATIO
C
C---- set up stretched array using converged geometric ratio
   11 S(1) = 0.0
      DS = DS1
      DO 2 N=2, NN
        S(N) = S(N-1) + DS
        DS = DS*RATIO
    2 CONTINUE
C
      RETURN
      END ! SETEXP
 

      SUBROUTINE SETEX2(S,DS1,DSN,SMAX,N)
C==========================================================
C     Sets array S stretched so that a prescribed spacing is 
C     obtained at each end.  The interior spacing is a blend 
C     of two geometric stretchings "shot" from each end.
C
C       S     (output)  array to be set  
C       DS1   (input)   approximate first S increment:  S(2) - S(1)
C       DSN   (input)   approximate last  S increment:  S(N) - S(N-1)
C       SMAX  (input)   final S value:      S(N)
C       N     (input)   number of points
C==========================================================
C
      IMPLICIT REAL (A-H,M,O-Z)
      DIMENSION S(N)
C
      PARAMETER (NDIM=500)
      DIMENSION S1(NDIM), SN(NDIM)
C
C---- with bigger FEND, the actual end increments will get closer 
C-    to DS1 & DSN, but the interior spacing might get screwy.
      DATA FEND / 2.0 /
C
      IF(N.GT.NDIM) STOP 'SETEX2:  Array overflow.'
C
C---- calculate spacing arrays each having the prescribed end increment
      CALL SETEXP(S1,DS1,SMAX,N)
      CALL SETEXP(SN,DSN,SMAX,N)
C
cc      S(1) = 0.
cc      S(N) = SMAX
C
C---- blend spacing arrays with power-function weights
      DO 10 I=1, N
        IN = N-I+1
        SS1 = S1(I)
        SSN = SMAX - SN(IN)
C
C------ power function of integer index
        WT1 = FLOAT(N-I)**FEND
        WTN = FLOAT(I-1)**FEND
C
C------ power function of coordinate
CCC     WT1 = (1.0 - SSN/SMAX)**FEND
CCC     WTN = (      SS1/SMAX)**FEND
C
        S(I) = (SS1*WT1 + SSN*WTN) / (WT1 + WTN)
   10 CONTINUE
C
C---- check for monotonicity
      SGN = SIGN( 1.0 , S(N)-S(1) )
      DO 20 I=2, N
        IF(SGN*S(I) .LE. SGN*S(I-1)) THEN
         WRITE(*,*) 'SETEX2: Warning. Returned array not monotonic.'
         RETURN
        ENDIF
   20 CONTINUE
C
      RETURN
      END ! SETEX2

