


      SUBROUTINE SGINIT
C---------------------------------------------------------------
C     Sets initial point distributions on airfoils based on 
C     curvature, and point distributions on inlet and outlet 
C     streamlines by geometric stretching. These distributions,
C     which will in general be incompatible between adjacent
C     elements, are then reconciled in SGFUDG.
C---------------------------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MSET.INC'
      LOGICAL PLOTP
C
C---- initialize number of airfoil surface points for each element
C
      CHMAX = CHRD(1)
      DO N=1, NBL
        CHMAX = MAX(CHMAX,CHRD(N))
      ENDDO
C
      DO N=1, NBL
        NBLD0(N) = INT( FLOAT(NBLD1) * (CHRD(N)/CHMAX)**FNBLD1 )
C------ use odd number of points to permit refinement
        IF(MOD(NBLD0(N),2) .EQ. 0) NBLD0(N) = NBLD0(N) + 1
      ENDDO
C
C
C---- set stagnation point arc length(s)
      CALL STFIND(SBLE)
C
C---- set LE and TE block indices for each streamline
      DO IS=1, 2*NBL+2
        ILLE(IS) = 0
        ILTE(IS) = 0
        DO IL=1, NLIN(IS)
          IF(ISOR(IL,IS).EQ.IS .AND.      LEOR(IL,IS)) ILLE(IS) = IL
          IF(ISOR(IL,IS).EQ.IS .AND. .NOT.LEOR(IL,IS)) ILTE(IS) = IL
        ENDDO
      ENDDO
C
C
      WRITE(*,*) 'Generating unresolved point spacings ...'
C
C---- Generate surface spacing for elements(s) based on curvature
      DO 10 N=1, NBL
        IS = 2*(N-1) + 1
C
        IF(.NOT.DSINIT) THEN
C------- set default LE and TE grid spacing ratios
         FSLE(N) = 0.4
         FSTE(N) = 0.8
         CVLEWT(N) = 1.0
        ENDIF
C
C------ parameters for spacing plot display in SGCURV
        PLOTP = .TRUE.
        XSIZ = 1.0
        YSIZ = AR
C
        CALL SGCURV(PLOTP,XSIZ,YSIZ,
     &       IIB(N),XB(1,N),XPB(1,N),YB(1,N),YPB(1,N),SB(1,N),
     &       SBLE(N),IBLE(N),SG0(1,IS),IX,NBLD0(N),
     &       CVLEWT(N),FSLE(N),FSTE(N),
     &       XSREF1(N),XSREF2(N),XPREF1(N),XPREF2(N),
     &       CRRATS(N),CRRATP(N))
   10 CONTINUE
C
      DSINIT = .TRUE.
C
C---- set potential increments over first and last streamline segments
      IS = 1
      DPINL = PHIST(ISTLE(IS),IS) - PHIINL     + 1000.0
      DPOUT = PHIEND(IS) - PHIST(ISTTE(IS),IS) + 1000.0
CCC   DO 15 IS=1, 2*NBL
      DO 15 IS=1, 2*NBL, 2     !!!  27 Jan 92
        DPINL = MIN( PHIST(ISTLE(IS),IS) - PHIINL     , DPINL )
        DPOUT = MIN( PHIEND(IS) - PHIST(ISTTE(IS),IS) , DPOUT )
   15 CONTINUE
C
C---- a reasonable estimate of the number of streamwise points needed
      IITRY = IX - (NBX-NBL)*25
C
C---- Generate free streamline spacing(s) with exponential stretching
      DO 20 N=1, NBL
        IS = 2*N - 1
C
C------ set streamline potential increments upstream and downstream
        DPLEFT = PHIST(ISTLE(IS),IS) - PHIINL     - DPINL
        DPRITE = PHIEND(IS) - PHIST(ISTTE(IS),IS) - DPOUT
C
C------ leftmost LE has NINL1 points upstream, and rightmost TE has NOUT1 points
C-      downstream, ... apportion remaining available points to other elements
        IF(NBL.EQ.1 .OR. (DPLEFT.EQ.0.0 .AND. DPRITE.EQ.0.0)) THEN
         NINL0(N) = NINL1
         NOUT0(N) = NOUT1
        ELSE
         NREM = IITRY - NBLD0(N) - NINL1 - NOUT1
         NINL0(N) = NINL1 + NREM*DPLEFT/(DPLEFT + DPRITE)
         NOUT0(N) = NOUT1 + NREM*DPRITE/(DPLEFT + DPRITE)
        ENDIF
C
C------ set surface arc lengths
ccc        SSURFO = SBLE(N) - SB(1     ,N)
ccc        SSURFP = SB(IIB(N),N) - SBLE(N)
        SSURFO = SST(ISTTE(IS  ),IS  ) - SST(ISTLE(IS  ),IS  )
        SSURFP = SST(ISTTE(IS+1),IS+1) - SST(ISTLE(IS+1),IS+1)
C
C------ generate spacing on streamline attached to LE
        STINL(N) = SST(ISTLE(IS),IS) - SSTL(1,IS)
        DSB = (  SSURFO*(SG0(2,IS  )-SG0(1,IS  ))
     &         + SSURFP*(SG0(2,IS+1)-SG0(1,IS+1)) )*0.5
        CALL SETEXP(SGINL0(1,N),DSB/STINL(N),1.0,NINL0(N))
C
C------ flip array, since it's inconveniently indexed from LE to upstream
        DO 210 IG=1, NINL0(N)/2
          IGBACK = NINL0(N) - IG + 1
          TEMP = SGINL0(IG,N)
          SGINL0(IG,N)     = SGINL0(IGBACK,N)
          SGINL0(IGBACK,N) = TEMP
  210   CONTINUE
        DO 220 IG=1, NINL0(N)
          SGINL0(IG,N) = 1.0 - SGINL0(IG,N)
  220   CONTINUE
C
C------ set endpoints exactly
        SGINL0(       1,N) = 0.0
        SGINL0(NINL0(N),N) = 1.0
C
C
C------ generate spacing on streamline attached to TE
        STOUT(N) = SSTL(NLIN(IS),IS) - SST(ISTTE(IS),IS)
        IG = NBLD0(N)
        DSB = (  SSURFO*(SG0(IG,IS  )-SG0(IG-1,IS  ))
     &         + SSURFP*(SG0(IG,IS+1)-SG0(IG-1,IS+1)) )*0.5
        CALL SETEXP(SGOUT0(1,N),DSB/STOUT(N),1.0,NOUT0(N))
C
C------ set endpoints exactly
        SGOUT0(       1,N) = 0.0
        SGOUT0(NOUT0(N),N) = 1.0
C
   20 CONTINUE
C
C
C**** Set number of points between normal lines
C
C---- go over all streamlines
      DO 30 IS=1, 2*NBL
        N = (IS+1)/2
C
C------ set LE and TE arc lengths for this streamline
        STLE = SST(ISTLE(IS),IS)
        STTE = SST(ISTTE(IS),IS)
        STEX = SSTL(NLIN(IS),IS)
C
ccc        IF(MOD(IS,2).EQ.1) SSURF = SBLE(N) - SB(1     ,N)
ccc        IF(MOD(IS,2).EQ.0) SSURF = SB(IIB(N),N) - SBLE(N)
        SSURF = STTE - STLE
C
C------ go over streamline segments
        DO 310 ILO=1, NLIN(IS)-1
          ILP = ILO + 1
C
C******** find number of points intercepted by each streamline segment
C
          IF(SSTL(ILP,IS) .LE. STLE) THEN
C--------- inlet streamline
C
C--------- locate end points
           DO 3110 IG=1, NINL0(N)-1
             STG = SSTL(1,IS) + SGINL0(IG,N)*STINL(N)
             IF(STG .LE. SSTL(ILO,IS)) IGO = IG
             IF(STG .LT. SSTL(ILP,IS)) IGP = IG
 3110      CONTINUE
C
C--------- last array point is checked separately to circumvent roundoff
           IG = NINL0(N)
           STG = STLE
           IF(STG .LE. SSTL(ILO,IS)) IGO = IG
           IF(STG .LT. SSTL(ILP,IS)) IGP = IG
C
C--------- set number of intercepted points
           IF((IGP-IGO+1).GT.IGLX) THEN
            WRITE(*,*) 'SGINIT: Array overflow.',
     &                 '   Increase IGLX to', (IGP-IGO+1)
            STOP
           ENDIF
C
           NGL0(ILO,IS) = IGP - IGO + 1
C
           IF(ILP.EQ.ILLE(IS)) NGL0(ILO,IS) = NGL0(ILO,IS) + 1
C
          ELSE IF(SSTL(ILP,IS) .LE. STTE) THEN
C--------- airfoil surface
C
           DO 3120 IG=1, NBLD0(N)-1
             STG = STLE + SG0(IG,IS)*SSURF
             IF(STG .LE. SSTL(ILO,IS)) IGO = IG
             IF(STG .LT. SSTL(ILP,IS)) IGP = IG
 3120      CONTINUE
C
C--------- last array point is checked separately to circumvent roundoff
           IG = NBLD0(N)
           STG = STTE
           IF(STG .LE. SSTL(ILO,IS)) IGO = IG
           IF(STG .LT. SSTL(ILP,IS)) IGP = IG
C
           IF((IGP-IGO+1).GT.IGLX) THEN
            WRITE(*,*) 'SGINIT: Array overflow.',
     &                 '   Increase IGLX to', (IGP-IGO+1)
            STOP
           ENDIF
C
           NGL0(ILO,IS) = IGP - IGO + 1
C
           IF(ILP.EQ.ILTE(IS)) NGL0(ILO,IS) = NGL0(ILO,IS) + 1
C
          ELSE
C--------- exit streamline
C
           DO 3130 IG=1, NOUT0(N)-1
             STG = STTE + SGOUT0(IG,N)*STOUT(N)
             IF(STG .LE. SSTL(ILO,IS)) IGO = IG
             IF(STG .LT. SSTL(ILP,IS)) IGP = IG
 3130      CONTINUE
C
C--------- last array point is checked separately to circumvent roundoff
           IG = NOUT0(N)
           STG = STEX
           IF(STG .LE. SSTL(ILO,IS)) IGO = IG
           IF(STG .LT. SSTL(ILP,IS)) IGP = IG
C
           IF((IGP-IGO+1).GT.IGLX) THEN
            WRITE(*,*) 'SGINIT: Array overflow.',
     &                 '   Increase IGLX to', (IGP-IGO+1)
            STOP
           ENDIF
C
           NGL0(ILO,IS) = IGP - IGO + 1
C
           IF(ILP.EQ.NLIN(IS)) NGL0(ILO,IS) = NGL0(ILO,IS) + 1
C
          ENDIF
C
  310   CONTINUE
   30 CONTINUE
C
C
C
C**** Each segment needs at least two points --- shift points if needed
C
C---- go over all streamlines
      DO 50 IS=1, 2*NBL
        N = (IS+1)/2
C
C------ go over inlet streamline segments
        DO 510 ILO=1, ILLE(IS)-1
          ILM = ILO - 1
          ILP = ILO + 1
C
          IF(NGL0(ILO,IS) .GE. 2) GO TO 510
C
C-------- number of points to be shifted to present segment
          NADD = 2 - NGL0(ILO,IS)
C
C-------- shift points from upstream or downstream segment
          IF(ILO.EQ.1) THEN
           NGL0(ILO,IS) = NGL0(ILO,IS) + NADD
           NGL0(ILP,IS) = NGL0(ILP,IS) - NADD
          ELSE IF(ILP.EQ.ILLE(IS)) THEN
           NGL0(ILO,IS) = NGL0(ILO,IS) + NADD
           NGL0(ILM,IS) = NGL0(ILM,IS) - NADD
          ELSE
           NGL0(ILO,IS) = NGL0(ILO,IS) + NADD
           IF(NGL0(ILM,IS) .GT. NGL0(ILP,IS)) THEN
            NGL0(ILM,IS) = NGL0(ILM,IS) - NADD
           ELSE
            NGL0(ILP,IS) = NGL0(ILP,IS) - NADD
           ENDIF
          ENDIF
C
  510   CONTINUE
C
C
C------ go over element surface streamline segments
        DO 520 ILO=ILLE(IS), ILTE(IS)-1
          ILM = ILO - 1
          ILP = ILO + 1
C
          IF(NGL0(ILO,IS) .GE. 2) GO TO 520
C
C-------- number of points to be shifted to present segment
          NADD = 2 - NGL0(ILO,IS)
C
C-------- shift points from upstream or downstream segment
          IF(ILO.EQ.ILLE(IS)) THEN
           NGL0(ILO,IS) = NGL0(ILO,IS) + NADD
           NGL0(ILP,IS) = NGL0(ILP,IS) - NADD
          ELSE IF(ILP.EQ.ILTE(IS)) THEN
           NGL0(ILO,IS) = NGL0(ILO,IS) + NADD
           NGL0(ILM,IS) = NGL0(ILM,IS) - NADD
          ELSE
           NGL0(ILO,IS) = NGL0(ILO,IS) + NADD
           IF(NGL0(ILM,IS) .GT. NGL0(ILP,IS)) THEN
            NGL0(ILM,IS) = NGL0(ILM,IS) - NADD
           ELSE
            NGL0(ILP,IS) = NGL0(ILP,IS) - NADD
           ENDIF
          ENDIF
C
  520   CONTINUE
C
C
C------ go over exit streamline segments
        DO 530 ILO=ILTE(IS), NLIN(IS)-1
          ILM = ILO - 1
          ILP = ILO + 1
C
          IF(NGL0(ILO,IS) .GE. 2) GO TO 530
C
C-------- number of points to be shifted to present segment
          NADD = 2 - NGL0(ILO,IS)
C
C-------- shift points from upstream or downstream segment
          IF(ILO.EQ.ILTE(IS)) THEN
           NGL0(ILO,IS) = NGL0(ILO,IS) + NADD
           NGL0(ILP,IS) = NGL0(ILP,IS) - NADD
          ELSE IF(ILP.EQ.NLIN(IS)) THEN
           NGL0(ILO,IS) = NGL0(ILO,IS) + NADD
           NGL0(ILM,IS) = NGL0(ILM,IS) - NADD
          ELSE
           NGL0(ILO,IS) = NGL0(ILO,IS) + NADD
           IF(NGL0(ILM,IS) .GT. NGL0(ILP,IS)) THEN
            NGL0(ILM,IS) = NGL0(ILM,IS) - NADD
           ELSE
            NGL0(ILP,IS) = NGL0(ILP,IS) - NADD
           ENDIF
          ENDIF
C
  530   CONTINUE
C
   50 CONTINUE
C
C
C
C**** Set spacings over segments
C
C---- go over all streamlines
      DO 60 IS=1, 2*NBL
        N = (IS+1)/2
C
C------ set LE and TE arc lengths for this streamline
        STLE = SST(ISTLE(IS),IS)
        STTE = SST(ISTTE(IS),IS)
C
ccc        IF(MOD(IS,2).EQ.1) SSURF = SBLE(N) - SB(1     ,N)
ccc        IF(MOD(IS,2).EQ.0) SSURF = SB(IIB(N),N) - SBLE(N)
        SSURF = STTE - STLE
C
C
C------ go over inlet streamline segments
        IGO = 1
        DO 610 ILO=1, ILLE(IS)-1
          IGP = IGO + NGL0(ILO,IS) - 1
C
C-------- set normalized spacing array
          DO 6115 JG=1, NGL0(ILO,IS)
            IG = IGO + JG - 1
            SGL0(JG,ILO,IS) = (SGINL0(IG ,N)-SGINL0(IGO,N))
     &                      / (SGINL0(IGP,N)-SGINL0(IGO,N))
 6115     CONTINUE
C
          IGO = IGP
  610   CONTINUE
C
C
C------ go over surface streamline segments
        IGO = 1
        DO 620 ILO=ILLE(IS), ILTE(IS)-1
          IGP = IGO + NGL0(ILO,IS) - 1
C
          DO 6225 JG=1, NGL0(ILO,IS)
            IG = IGO + JG - 1
            SGL0(JG,ILO,IS) = (SG0(IG ,IS)-SG0(IGO,IS))
     &                      / (SG0(IGP,IS)-SG0(IGO,IS))
 6225     CONTINUE
C
          IGO = IGP
  620   CONTINUE
C
C
C------ go over exit streamline segments
        IGO = 1
        DO 630 ILO=ILTE(IS), NLIN(IS)-1
          IGP = IGO + NGL0(ILO,IS) - 1
C
          DO 6335 JG=1, NGL0(ILO,IS)
            IG = IGO + JG - 1
            SGL0(JG,ILO,IS) = (SGOUT0(IG ,N)-SGOUT0(IGO,N))
     &                      / (SGOUT0(IGP,N)-SGOUT0(IGO,N))
 6335     CONTINUE
C
          IGO = IGP
  630   CONTINUE
C
   60 CONTINUE
C
C
C
C---- As a debugging aid, check to see if opposite airfoil sides have 
C-    the same number of points (if not, general havoc will result)
      DO 70 N=1, NBL
        IS1 = 2*N-1
        IS2 = 2*N
C
        ISUM1 = 0
        DO 710 IL=1, NLIN(IS1)-1
          IF(SSTL(IL,IS1) .LT. SST(ISTLE(IS1),IS1)) GO TO 710
          IF(SSTL(IL,IS1) .GE. SST(ISTTE(IS1),IS1)) GO TO 711
          ISUM1 = ISUM1 + NGL0(IL,IS1) - 1
  710   CONTINUE
  711   CONTINUE
C
        ISUM2 = 0
        DO 720 IL=1, NLIN(IS2)-1
          IF(SSTL(IL,IS2) .LT. SST(ISTLE(IS2),IS2)) GO TO 720
          IF(SSTL(IL,IS2) .GE. SST(ISTTE(IS2),IS2)) GO TO 721
          ISUM2 = ISUM2 + NGL0(IL,IS2) - 1
  720   CONTINUE
  721   CONTINUE
C
        IF(ISUM1 .EQ. ISUM2) GO TO 70
C
        WRITE(*,7000) N, ISUM1, ISUM2
 7000   FORMAT(/' *** SGINIT: Internal error ***'
     &         /' Element',I2,' sides have unequal points: ', 2I5)
c
   70 CONTINUE
C
      RETURN
      END ! SGINIT



      SUBROUTINE SGFUDG
C------------------------------------------------------------------------
C     Reconciles conflicting element point spacings generated by SGINIT.
C     Each element tries to impose its point number on its neighbors.
C     Repeated passes are performed with everybody slugging it out
C     according to a few averaging rules until no further changes are made.
C------------------------------------------------------------------------
      INCLUDE 'STATE.INC'
      INCLUDE 'MSET.INC'
      LOGICAL DONE, SGLMOD
      LOGICAL LESTR, LSURF, LWAKE
C
C---- set initial point numbers
      DO 20 IS=1, 2*NBL
        DO 210 IL=1, NLIN(IS)
           NGL(IL,IS) = NGL0(IL,IS)
           DO 2110 IG=1, NGL(IL,IS)
             SGL(IG,IL,IS) = SGL0(IG,IL,IS)
 2110      CONTINUE
  210   CONTINUE
   20 CONTINUE
C
C
C---- perform repeated passes, trying to resolve conflicting point numbers
      DO 8000 IPASS=1, 4*NBL
C
      DONE = .TRUE.
C
C---- go over each element, trying to impose its point number on others
      DO 1000 N=1, NBL
C
cc         write(*,*)
cc         write(*,*) '***** Element', N,'   *****'
C
C------ go to the two stagnation streamlines above and below this element
        DO 700 IDIR=-1, 1, 2
C
C------ set index of streamline above or below
        IF(IDIR.EQ.-1) THEN
         IS = 2*N-1
         ISEND = 1
        ELSE
         IS = 2*N
         ISEND = 2*NBL
        ENDIF
C
cc         write(*,*)
cc         write(*,*) '   ======= IS', IS,'   ====='
c
C------ go over streamline segments on surfaces of the element
        DO 40 ILO=ILLE(IS), ILTE(IS)-1
          ILP = ILO + 1
C
C-------- initialize matching segment index for streamlines above current one
          JLO = ILO
          JLP = ILP
C
cc         write(*,*)
cc         write(*,*) '      ++++ IL', ILO,'   +++++'
c
cc$$$
c          IF(MOD(NGL(ILO,IS),2).EQ.0) THEN 
c           CALL NEWNUM(SGL(1,ILO,IS),NGL(ILO,IS),NGL(ILO,IS)+1)
c           NGL(ILO,IS) = NGL(ILO,IS) + 1
c          ENDIF
C
C-------- set flag indicating if SGL array on side IS has been modified
          SGLMOD = .FALSE.
C
C-------- initialize potential jump accumulator
          POFF = 0.0
C
C-------- go over matching segment on each streamline above or below
          DO 410 JS=IS+IDIR, ISEND, IDIR
C
cc         write(*,*)
cc         write(*,*) '         - JS', JS,'   -----'

C---------- is this an inlet streamline, or wake ?
            LESTR = JLO .LT. ILLE(JS)
            LWAKE = JLP .GT. ILTE(JS)
C
            IF(LESTR) THEN
C
C----------- set new number of points, but preserve end spacing
             IF(JLP .EQ. ILLE(JS) .AND. .NOT.SGLMOD) THEN
C------------ inlet streamline has LE at right point ... 
C-            "average" the spacing arrays to preserve LE spacings
              CALL NEWNUM(SGL(1,JLO,JS),NGL(JLO,JS),NGL(ILO,IS))
              CALL SGAVG (SGL(1,JLO,JS),SGL(1,ILO,IS),NGL(ILO,IS),1.0)
              CALL SGAVG2(SGL(1,JLO,JS),SGL(1,ILO,IS),NGL(ILO,IS))
              SGLMOD = .TRUE.
             ELSE
              CALL SGCOPY(SGL(1,JLO,JS),SGL(1,ILO,IS),NGL(ILO,IS))
             ENDIF
C
             NGL(JLO,JS) = NGL(ILO,IS)
             GO TO 410
C
            ELSE IF(LWAKE) THEN
C
C----------- set new number of points, but preserve point bunching distribution
             IF(JLO .EQ. ILTE(JS) .AND. .NOT.SGLMOD) THEN
C------------ wake streamline has TE at left point ... 
C-            "average" the spacing arrays to preserve normal Phi lines
              DSGLJ = SGL(2,JLO,JS) - SGL(1,JLO,JS)
              DSGLI = SGL(2,ILO,IS) - SGL(1,ILO,IS)
C%%%
ccc              DSGL = DSGLJ
              DSGL = MIN(DSGLI,DSGLJ)
C%%%
              CALL NEWNUM(SGL(1,JLO,JS),NGL(JLO,JS),NGL(ILO,IS))
              CALL SETEXP(SGL(1,JLO,JS),DSGL ,1.0,NGL(ILO,IS))
              CALL SGAVG (SGL(1,ILO,IS),SGL(1,JLO,JS),NGL(ILO,IS),4.0)
              CALL SGCOPF(SGL(1,JLO,JS),SGL(1,ILO,IS),NGL(ILO,IS),
     &                    SSTL(JLO,JS),(SSTL(JLP,JS)-SSTL(JLO,JS)),
     &                    SSTL(ILO,IS),(SSTL(ILP,IS)-SSTL(ILO,IS)),POFF,
     &                    PHIST(1,JS),QST(1,JS),SST(1,JS),NST(JS),
     &                    PHIST(1,IS),QST(1,IS),SST(1,IS),NST(IS) )
              SGLMOD = .TRUE.
             ELSE
              CALL SGCOPY(SGL(1,JLO,JS),SGL(1,ILO,IS),NGL(ILO,IS))
              CALL SGCOPF(SGL(1,JLO,JS),SGL(1,ILO,IS),NGL(ILO,IS),
     &                    SSTL(JLO,JS),(SSTL(JLP,JS)-SSTL(JLO,JS)),
     &                    SSTL(ILO,IS),(SSTL(ILP,IS)-SSTL(ILO,IS)),POFF,
     &                    PHIST(1,JS),QST(1,JS),SST(1,JS),NST(JS),
     &                    PHIST(1,IS),QST(1,IS),SST(1,IS),NST(IS) )
             ENDIF
C
             NGL(JLO,JS) = NGL(ILO,IS)
             IF(JS.EQ.ISEND) GO TO 410
C
C----------- reset matching segment index since number of segments may not match
             JLO = JLO + NLIN(JS+IDIR) - NLIN(JS)
             JLP = JLO + 1
C
C----------- add on to potential offset if we're about to cross a wake
             IF(IDIR.EQ.1 .AND. MOD(JS,2).EQ.1) THEN
              JEL = (JS+1)/2
              POFF = POFF
     &             - (PHIST(ISTTE(JS),JS) - PHIST(ISTTE(JS+1),JS+1))
             ENDIF
C
             IF(IDIR.EQ.-1 .AND. MOD(JS,2).EQ.0) THEN
              JEL = JS/2
              POFF = POFF
     &             +(PHIST(ISTTE(JS-1),JS-1) - PHIST(ISTTE(JS),JS))
             ENDIF
C
             GO TO 410
C
            ELSE
C
C----------- We ran into another element -- must add points.
             NADD = NGL(ILO,IS) - NGL(JLO,JS)
C
C----------- jump out and go to next block segment on this streamline
C-           if no points are to be added here
             IF(NADD.LE.0) GO TO 40
C
             DONE = .FALSE.
C
C----------- assume that points will only be added, not "stolen"
             NADDM = 0
             NADDP = 0
C
cC----------- see if these points can be "stolen" from neighboring segments
c             IF(JLP .EQ. ILTE(JS)) THEN
cC------------ take from upstream segment
c              NADDM = NADD
c             ELSE IF(JLO .EQ. ILLE(JS)) THEN
cC------------ take from downstream segment
c              NADDP = NADD
c             ELSE
cC------------ take from both segments, proportional to their point number
c              FRACM = FLOAT( NGL(JLO-1,JS) - 1 )
c     &              / FLOAT( NGL(JLO-1,JS) + NGL(JLO+1,JS) - 2 )
c              NADDM = INT( FRACM * FLOAT(NADD) )
c              NADDP = NADD - NADDM
c             ENDIF
C
C----------- limit "stolen" points to 1/3 available amount
             NADDM = MIN0( NADDM , NGL(JLO-1,JS)/3 )
             NADDP = MIN0( NADDP , NGL(JLO+1,JS)/3 )
C
c             write(*,*)
c             write(*,*) 'side', js, -nADDm, nADD, -nADDp
C
C----------- add points to segment in question
             CALL NEWNUM(SGL(1,JLO,JS),NGL(JLO,JS),(NGL(JLO,JS)+NADD))
             NGL(JLO,JS) = NGL(JLO,JS) + NADD
C
             IF(NADDM .NE. 0) THEN
C------------ remove points from upstream segment
              CALL NEWNUM(SGL(1,JLO-1,JS),NGL(JLO-1,JS),
     &                                   (NGL(JLO-1,JS)-NADDM))
              NGL(JLO-1,JS) = NGL(JLO-1,JS) - NADDM
             ENDIF
C
             IF(NADDP .NE. 0) THEN
C------------ remove points from downstream segment
              CALL NEWNUM(SGL(1,JLO+1,JS),NGL(JLO+1,JS),
     &                                   (NGL(JLO+1,JS)-NADDP))
              NGL(JLO+1,JS) = NGL(JLO+1,JS) - NADDP
             ENDIF
C
C
C----------- if more points were added than removed, we still have to
C-           add the difference to opposite element side.
             NDEL = NADD - (NADDM + NADDP)
C
C----------- go to next segment station if no new points need to be added
             IF(NDEL .EQ. 0) GO TO 40
C
C----------- side index of side opposite JS on the same element
             KS = JS+IDIR
C
C----------- first set total number of surface points on side KS
             NTOT = 0
             DO 414 KL=ILLE(KS), ILTE(KS)-1
               NTOT = NTOT + NGL(KL,KS) - 1
 414         CONTINUE
C
             WRITE(*,9400) (JS+1)/2, NTOT, NTOT+NDEL
C
C----------- go over all segments on side KS
             NREM = NTOT
             DO 415 KL=ILLE(KS), ILTE(KS)-1
C
C------------- this segment gets a fraction of new points equal to the
C-             fraction of points it has relative to the whole element
               NDSEG = NDEL * FLOAT(NGL(KL,KS)-1)/FLOAT(NREM)
               CALL NEWNUM(SGL(1,KL,KS),NGL(KL,KS),(NGL(KL,KS)+NDSEG))
ccc               write(*,*) 'new num:', ks, ngl(kl,ks)-1, nrem, ndseg
C
C------------- set number of points in remaining segments
               NREM = NREM - (NGL(KL,KS)-1)
C
C------------- set new number of remaining points to be still added
               NDEL = NDEL - NDSEG
C
               NGL(KL,KS) = NGL(KL,KS) + NDSEG
C
 415         CONTINUE
C
C----------- since we ran into another element surface,
C-           jump out and go to next block segment on this streamline
             GO TO 40
C
            ENDIF
C
  410     CONTINUE ! with next streamline at this segment position
C
   40   CONTINUE   ! with next streamwise segment position
C
  700   CONTINUE   ! with next streamline for this element
C
 1000 CONTINUE     ! with next element
C
C
C---- jump out if no change was made on this pass
      IF(DONE) GO TO 8010
C
 8000 CONTINUE ! with next point resolution pass
      WRITE(*,*) 'SGFUDG: Streamwise spacing may not be resolved.'
 8010 CONTINUE
C
C
C---- Find any interior inlet streamline and wake segments which do not 
C-    overlap with any element surface.
      NLE = 0
      NTE = 0
C
      IDIR = 1
      IS = 1
      ISEND = 2*NBL
C
C---- go over segments of streamline IS
      DO 50 ILO=3, NLIN(IS)-3
        ILP = ILO+1
C
C------ if this normal lines is attached to an LE or _not_ a TE, it is covered
        IF(LEOR(ILO,IS) .OR. .NOT.LEOR(ILP,IS)) GO TO 50
C
        JLO = ILO
        JLP = ILP
C
C------ check streamlines below and including IS to see if we hit an element
        DO 510 JS=IS, ISEND, IDIR
C
C-------- is this an inlet streamline, or wake ?
          LESTR = JLO .LT. ILLE(JS)
          LWAKE = JLP .GT. ILTE(JS)
C
C-------- if we hit an element surface, move on to next streamwise segment
          IF(.NOT.LESTR .AND. .NOT.LWAKE) GO TO 50
C
C-------- save element number if segment endpoint is LE or TE
          IF(JLP.EQ.ILLE(JS)) NLE = (JS+1)/2
          IF(JLO.EQ.ILTE(JS)) NTE = (JS+1)/2
C
          IF(LWAKE) THEN
C--------- reset segment index  (since it may not match across wake)
           JLO = JLO + NLIN(JS+IDIR) - NLIN(JS)
           JLP = JLO + 1
          ENDIF
C
  510   CONTINUE
C
C------ if we came out here, this is an uncovered segment
ccc        WRITE(*,*) 'Uncovered interior segment:  nLE nTE =', NLE, NTE
C
C------ set LE and TE segment indices
        JSLE = 2*NLE
        JSTE = 2*NTE
C
        JLOLE = ILLE(JSLE)-1
        JLOTE = ILTE(JSTE)
C
C------ new number of points on all streamlines over uncovered segment
        NGLNEW = (NGL(JLOLE,JSLE) + NGL(JLOTE,JSTE))/3
        NGLNEW = MAX0( NGLNEW , 3 )
cc$$$
c        IF(MOD(NGLNEW,2).EQ.0) NGLNEW = NGLNEW + 1
C
cc        WRITE(*,*) 'Point numbers:', NGL(JLOLE,JSLE),NGL(JLOTE,JSTE),
cc     &             '  =>', NGLNEW
C
C------ set LE spacing array for new point number
        NGLOLD = NGL(JLOLE,JSLE)
        DSGLE = SGL(NGLOLD,JLOLE,JSLE) - SGL(NGLOLD-1,JLOLE,JSLE)
        CALL SETEXP(SGL(1,JLOLE,JSLE),DSGLE,1.0,NGLNEW)
        DO 512 IGL=1, (NGLNEW+1)/2
          IGLOPP = NGLNEW - IGL + 1
          SGTMP = SGL(IGL,JLOLE,JSLE)
          SGL(IGL   ,JLOLE,JSLE) = 1.0 - SGL(IGLOPP,JLOLE,JSLE)
          SGL(IGLOPP,JLOLE,JSLE) = 1.0 - SGTMP
  512   CONTINUE
C
C
C------ set TE spacing array for new point number
        DSGTE = SGL(2,JLOTE,JSTE) - SGL(1,JLOTE,JSTE)
        CALL SETEXP(SGL(1,JLOTE,JSTE),DSGTE,1.0,NGLNEW)
C
CCC        CALL NEWNUM(SGL(1,JLOLE,JSLE),NGL(JLOLE,JSLE),NGLNEW)
CCC        CALL NEWNUM(SGL(1,JLOTE,JSTE),NGL(JLOTE,JSTE),NGLNEW)
C
C
C------ set "averaged" spacing arrays to preserve LE and TE spacings
        CALL SGAVG(SGL(1,JLOLE,JSLE),SGL(1,JLOTE,JSTE),NGLNEW,1.0)
C
C
C------ set spacing arrays on all other streamlines
        JLO = ILO
        DO 520 JS=1, 2*NBL
          CALL SGCOPY(SGL(1,JLO,JS),SGL(1,JLOLE,JSLE),NGLNEW)
          NGL(JLO,JS) = NGLNEW
          IF(JLO.GE.ILTE(JS)) JLO = JLO + NLIN(JS+1) - NLIN(JS)
  520   CONTINUE
C
   50 CONTINUE
C
C
C---- find element that has the leftmost LE (its inlet streamline then sets
C-    the spacing on all other inlet streamline first segments)
      ILO = 1
      ILP = 2
      DO 70 IS=1, 2*NBL
        IF(ILLE(IS) .EQ. ILP) GO TO 71
   70 CONTINUE
   71 CONTINUE
C
ccc      write(*,*) 'Leftmost-LE  side =', is
c
C---- set point number on first streamline segments
      DO 75 JS=1, 2*NBL
        JLO = ILO
        CALL SGCOPY(SGL(1,JLO,JS),SGL(1,ILO,IS),NGL(ILO,IS))
        NGL(JLO,JS) = NGL(ILO,IS)
   75 CONTINUE
C
C
C---- find which element has the rightmost TE (its wake the sets
C-     the spacing on all other wake streamline last segments)
      DO 80 IS=1, 2*NBL
        ILO = NLIN(IS) - 1
        ILP = NLIN(IS)
        IF(ILTE(IS) .EQ. ILO) GO TO 81
   80 CONTINUE
   81 CONTINUE
C
ccc      write(*,*) 'Rightmost-TE side =', is
c
C---- initialize potential offset so that POFF=0 when IS=JS
      POFF = 0.0
      DO 82 JS=IS, 2, -1
        IF(MOD(JS,2).EQ.0) THEN
         JEL = JS/2
         POFF = POFF
     &        + (PHIST(ISTTE(JS-1),JS-1) - PHIST(ISTTE(JS),JS))
        ENDIF
 82   CONTINUE
C
C---- set point number on last streamline segments, forcing orthogonality
      DO 85 JS=1, 2*NBL
        JLO = NLIN(JS) - 1
        JLP = NLIN(JS)
C
        CALL SGCOPY(SGL(1,JLO,JS),SGL(1,ILO,IS),NGL(ILO,IS))
        CALL SGCOPF(SGL(1,JLO,JS),SGL(1,ILO,IS),NGL(ILO,IS),
     &              SSTL(JLO,JS),(SSTL(JLP,JS)-SSTL(JLO,JS)),
     &              SSTL(ILO,IS),(SSTL(ILP,IS)-SSTL(ILO,IS)),POFF,
     &              PHIST(1,JS),QST(1,JS),SST(1,JS),NST(JS),
     &              PHIST(1,IS),QST(1,IS),SST(1,IS),NST(IS) )
        NGL(JLO,JS) = NGL(ILO,IS)
C
C------ add on to potential offset if we're about to cross a wake
        IF(MOD(JS,2).EQ.1) THEN
         JEL = (JS+1)/2
         POFF = POFF
     &        - (PHIST(ISTTE(JS),JS) - PHIST(ISTTE(JS+1),JS+1))
        ENDIF
   85 CONTINUE
C
C
C---- set point number on topmost streamline
      IS = 1
      JS = 2*NBL+2
      DO 90 ILO=1, NLIN(IS)-1
        CALL SGCOPY(SGL(1,ILO,JS),SGL(1,ILO,IS),NGL(ILO,IS))
        NGL(ILO,JS) = NGL(ILO,IS)
   90 CONTINUE
C
C---- set point number on bottommost streamline
      IS = 2*NBL
      JS = 2*NBL+1
      DO 95 ILO=1, NLIN(IS)-1
        CALL SGCOPY(SGL(1,ILO,JS),SGL(1,ILO,IS),NGL(ILO,IS))
        NGL(ILO,JS) = NGL(ILO,IS)
   95 CONTINUE
C
C
      RETURN
C
 9400 FORMAT(' SGFUDG: Increasing element',I3,
     &       '  point number from', I4,'  to',I5)
      END ! SGFUDG





      SUBROUTINE SETCRN
      INCLUDE 'STATE.INC'
      INCLUDE 'MSET.INC'
      LOGICAL LTOP,LBOT,LCORN
C---------------------------------------------------------
C     Finds doubled points in airfoil geometry arrays, 
C     and tags them as "corner points".  The spacing arrays
C     are then tweaked so that a grid node falls exactly
C     on the corner point.  Only one corner point per 
C     airfoil side is currently permitted.
C
C     The panel solver allows the initial displacement 
C     streamline to peel away from the airfoil at the
C     corner point, giving a good initial guess for a
C     cove recirculation zone and the like.  Each point
C     on the initial displacement streamline which is 
C     "free" from the surface is tagged as such with 
C     LSFREE=t. 
C---------------------------------------------------------
C
C---- go over all airfoil sides
      DO 4000 IS=1, 2*NBL
        N = (IS+1)/2
C
C------ first assume no corners or free streamlines
        IGCORN(IS) = 0
        DO 10 IG=1, NBLD(N)
          LSFREE(IG,IS) = .FALSE.
   10   CONTINUE
C
C------ test if this is a top or bottom surface
        LBOT = MOD(IS,2) .EQ. 0
        LTOP = .NOT. LBOT
C
C------ find any doubled point in SB array on current surface
        LCORN = .FALSE.
        DO 20 IB=1, IIB(N)-1
          IF(LTOP .AND. SB(IB,N).GE.SBLE(N)) GO TO 20
          IF(LBOT .AND. SB(IB,N).LE.SBLE(N)) GO TO 20
          IF(SB(IB,N) .EQ. SB(IB+1,N)) THEN
           LCORN = .TRUE.
           SBCORN = SB(IB,N)
          ENDIF
   20   CONTINUE
C
        IF(LTOP) DSB = (SB(1     ,N) - SBLE(N))
        IF(LBOT) DSB = (SB(IIB(N),N) - SBLE(N))
C
C------ treat corner if found
        IF(LCORN)THEN
C
C-------- corner SG value
          SGO = (SBCORN - SBLE(N))/DSB
C
C-------- SG values at LE and TE (should be 0 and 1, actually)
          SGM = SG(      1,IS)
          SGP = SG(NBLD(N),IS)
C
C-------- find current node interval which contains corner
          DO 30 IG=2, NBLD(N)-2
            IF(SG(IG,IS).LT.SGO .AND. SG(IG+1,IS).GT.SGO) THEN
C----------- find spacing array SG point closest to corner
             IF(ABS(SG(IG,IS)-SGO) .LT. ABS(SG(IG+1,IS)-SGO)) THEN
              IGO = IG
             ELSE
              IGO = IG+1
             ENDIF
             GO TO 31
            ENDIF
   30     CONTINUE
          WRITE(*,*) '*** SBCORN: Internal error ***'
          WRITE(*,*) 'Known corner not located in spacing array SG.'
   31     CONTINUE
C
C-------- fudge spacings to force spacing array point to corner location
          FUDGM = (SGO-SGM) / (SG(IGO,IS)-SGM)
          FUDGP = (SGO-SGP) / (SG(IGO,IS)-SGP)
          DO 62 IG=2, IGO-1
            SG(IG,IS) = SGM + (SG(IG,IS)-SGM)*FUDGM
   62     CONTINUE
          DO 64 IG=IGO+1, NBLD(N)-1
            SG(IG,IS) = SGP + (SG(IG,IS)-SGP)*FUDGP
   64     CONTINUE
C
C-------- set corner node
          SG(IGO,IS) = SGO
          IGCORN(IS) = IGO
C
C-------- set free streamline flag for nodes past corner to one before TE
          DO 70 IG=IGO+1, NBLD(N)-1
            LSFREE(IG,IS) = .TRUE.
   70     CONTINUE
C
        ENDIF
C

 4000 CONTINUE
C
      RETURN
      END ! SETCRN



      SUBROUTINE SETSGF
      INCLUDE 'STATE.INC'
      INCLUDE 'MSET.INC'
C
C**** sets indices of "pinned" points on element surfaces
C
C---- first assume that all pinned points are at TE of each element
      DO 50 N=1, NBL
        IGFIX(2*N-1) = NBLD(N)
        IGFIX(2*N  ) = NBLD(N)
   50 CONTINUE
C
C---- go over stagnation streamline TE points (except for top and bottommost)
      DO 60 IS=2, 2*NBL-1
        IL = ILTE(IS)
C
C------ go to the adjacent stagnation streamline
        IF(MOD(IS,2).EQ.1) THEN
         ISA = IS-1
        ELSE
         ISA = IS+1
        ENDIF
C
C------ go on if the normal line from this TE doesn't run into an element
        IF(IL .LE. ILLE(ISA) .OR. IL .GE. ILTE(ISA)) GO TO 60
C
C------ find IG index of the point where the normal line falls
        IG = 1
        DO 605 JL=ILLE(ISA), IL-1
          IG = IG + NGL(JL,ISA) - 1
  605   CONTINUE
C
C------ set index of this pinned point
        IGFIX(ISA) = IG
C
   60 CONTINUE
C
C---- go over stagnation streamlines again, pinning any corners
      DO 70 IS=1, 2*NBL
        IF(IGCORN(IS) .NE. 0) IGFIX(IS) = MIN0(IGFIX(IS),IGCORN(IS))
   70 CONTINUE
C
      RETURN
      END ! SETSGF


      SUBROUTINE SGIBO
      INCLUDE 'STATE.INC'
      INCLUDE 'MSET.INC'
C
C---- set final inlet, outlet, and surface spacing arrays
      DO 10 N=1, NBL
        IS = 2*N-1
C
        NINL(N) = 0
        DO 110 IL=1, ILLE(IS)-1
          DSST = SSTL(IL+1,IS) - SSTL(IL,IS)
          DO 1110 IGL=1, NGL(IL,IS)-1
            IG = NINL(N) + IGL
            SGINL(IG,N) = SSTL(IL,IS) + DSST*SGL(IGL,IL,IS)
 1110     CONTINUE
          NINL(N) = NINL(N) + NGL(IL,IS) - 1
  110   CONTINUE
        NINL(N) = NINL(N) + 1
        SGINL(NINL(N),N) = SSTL(ILLE(IS),IS)
C
        DSG = SGINL(NINL(N),N) - SGINL(1,N)
        DO 120 IG=NINL(N), 1, -1
          SGINL(IG,N) = (SGINL(IG,N) - SGINL(1,N)) / DSG
  120   CONTINUE
C
C------ smooth inlet spacing array
        CALL FILTER(SGINL(1,N),3.0,NINL(N))
C
C
        NOUT(N) = 0
        DO 130 IL=ILTE(IS), NLIN(IS)-1
          DSST = SSTL(IL+1,IS) - SSTL(IL,IS)
          DO 1310 IGL=1, NGL(IL,IS)-1
            IG = NOUT(N) + IGL
            SGOUT(IG,N) = SSTL(IL,IS) + DSST*SGL(IGL,IL,IS)
 1310     CONTINUE
          NOUT(N) = NOUT(N) + NGL(IL,IS) - 1
  130   CONTINUE
        NOUT(N) = NOUT(N) + 1
        SGOUT(NOUT(N),N) = SSTL(NLIN(IS),IS)
C
        DSG = SGOUT(NOUT(N),N) - SGOUT(1,N)
        DO 140 IG=NOUT(N), 1, -1
          SGOUT(IG,N) = (SGOUT(IG,N) - SGOUT(1,N)) / DSG
  140   CONTINUE
C
C------ smooth outlet spacing array
        CALL FILTER(SGOUT(1,N),2.0,NOUT(N))
C
   10 CONTINUE
C
C
      DO 40 IS=1, 2*NBL
        N = (IS+1)/2
C
        NBLD(N) = 0
        DO 410 IL=ILLE(IS), ILTE(IS)-1
          DSST = SSTL(IL+1,IS) - SSTL(IL,IS)
          DO 4110 IGL=1, NGL(IL,IS)-1
            IG = NBLD(N) + IGL
            SG(IG,IS) = SSTL(IL,IS) + DSST*SGL(IGL,IL,IS)
 4110     CONTINUE
          NBLD(N) = NBLD(N) + NGL(IL,IS) - 1
  410   CONTINUE
        NBLD(N) = NBLD(N) + 1
        SG(NBLD(N),IS) = SSTL(ILTE(IS),IS)
C
C------ smooth spacing array
        CALL FILTER(SG(1,IS),1.0,NBLD(N))
C
        DSG = SG(NBLD(N),IS) - SG(1,IS)
        DO 470 IG=NBLD(N), 1, -1
          SG(IG,IS) = (SG(IG,IS) - SG(1,IS)) / DSG
  470   CONTINUE
C
   40 CONTINUE
C
      RETURN
      END ! SGIBO
