/*                                                                          
        Source: check_http.c
        Author: Adam G. Bowen (agbowen@bealenet.com)
        $Revision: 1.2 $
        $Date: 1999/05/17 14:35:26 $
 
 * Program: HTTP socket plugin for NetSaint
 * License: GPL
 * Copyright (c) 1999 Adam G. Bowen (agbowen@bealenet.com)
 *
 * Description:
 *
 * This plugin will establish a socket on the http port of the host specified by
 * ipaddress and receives a responce from the host.  If any error occures while
 * creating the socket, receiving data from the socket or closing the socket,
 * this error will result in an alert.  The incorrect responce or responce time
 * above the threshold vaules will also cause an alert. 
 *
 * Other errors result in a STATE_UNKNOWN return.
 *
 * Command line:
 *
 * check_http ipaddress sys_type
 * check_http ipaddress sys_type <warn_ms> <crit_ms>
 *
 * Required input:
 *
 * ipaddress = The ipaddress of the remote system to run the check on.
 * sys_type = The remote system type.
 *
 * Optional input:
 *
 * <warn_ms> = Number of miliseconds necessary to result in a WARNING state.
 * <crit_ms> = Number of miliseconds necessary to result in a CRITICAL state.
 *
 * Notes:
 *
 * If <warn_ms> and <crit_ms> are not passed on the command line, they will be 
 * set to the default values in the check_fs config file.
 *
 * sys_type is used to determine which config file to use to generate the
 * remote command.
 *
 * The configuration file /usr/local/netsaint/config/check_http/<sys_type>
 * contains the following values:
 *
 * HTTP_PORT|<http port on system sys_type>|
 * HTTP_PROTO|<protocol to use for socket>|
 * HTTP_EXPECT|<string to receive from remote system>|
 * WARN_MS|<default number of miliseconds needed to generate a warning>|
 * CRIT_MS|<default number of miliseconds needed to generate a critical>|
 *
 * $Log: check_http.c,v $
 * Revision 1.2  1999/05/17 14:35:26  netsaint
 * Changed the plugin to use a seperate config directory.  This directory is the
 * CONFIG_DIR/command_name directory.
 *
 * Revision 1.1  1999/05/14 02:49:59  netsaint
 * Initial revision
 *
 *
*/

#include "/usr/local/src/netsaint/include/plugins.h"

int main(int argc, char *argv[])
{
  char command_name[MAX_CHARS];
  char config_file_net[MAX_CHARS];
  char expected[MAX_CHARS];
  char ip_address[MAX_CHARS];
  char ms_crit[MAX_CHARS];
  char ms_warn[MAX_CHARS];
  char port_http[MAX_CHARS];
  char protocol[MAX_CHARS];
  char send_buffer[MAX_CHARS];
  char system_name[MAX_CHARS];

  int crit_ms;
  int end_time;
  int http_port;
  int get_defaults;
  int micro_seconds;
  int result;
  int return_value;
  int seconds;
  int socket_name;
  int start_time;
  int total_time;
  int warn_ms;

  struct timeval end_buff;
  struct timeval start_buff;
  struct timezone end_tz;
  struct timezone start_tz;

  /* Initialize alarm signal handling */

  signal(SIGALRM,alarm_signal);

  strcpy(command_name,get_command_name(argv[0]));
  if(!((argc==3) || (argc==5)))
  {
    printf("\n");
    printf(" Incorrect number of arguments supplied\n");
    printf("\n");
    printf(" HTTP socket plugin for NetSaint\n");
    printf(" Copyright (c) 1999 Adam G. Bowen (agbowen@bealenet.com)\n");
    printf(" $Revision: 1.2 $\n");
    printf(" Last Modified $Date: 1999/05/17 14:35:26 $\n");
    printf(" License: GPL\n");
    printf("\n");
    printf(" Description:\n");
    printf("\n");
    printf(" This plugin will establish a socket on the http port of the host specified by\n");
    printf(" ipaddress and receives a responce from the host.  If any error occures while\n");
    printf(" creating the socket, receiving data from the socket or closing the socket, this\n");
    printf(" error will result in an alert.  The incorrect responce or responce time above\n");
    printf(" the threshold vaules will also cause an alert. \n");
    printf("\n");
    printf(" Usage: %s ipaddress sys_type\n",command_name);
    printf(" Usage: %s ipaddress sys_type <warn_ms> <crit_ms>\n",command_name);
    printf("\n");
    printf(" Required input:\n");
    printf("\n");
    printf(" ipaddress = The ipaddress of the remote system to run the check on.\n");
    printf(" sys_type = The remote system type.\n");
    printf("\n");
    printf(" Optional input:\n");
    printf("\n");
    printf(" <warn_ms> = Number of miliseconds necessary to result in a WARNING state.\n");
    printf(" <crit_ms> = Number of miliseconds necessary to result in a CRITICAL state.\n");
    printf("\n");
    printf(" If <warn_ms> and <crit_ms> are not passed on the command line, they will be\n");
    printf(" set to the default values in the %s config file.\n", command_name);
    printf("\n");
    printf(" sys_type is used to determine which config file to use to generate the\n");
    printf(" remote command.\n");
    printf("\n");
    return_value = STATE_UNKNOWN;
  }
  else
  {
    /* Set up config files and get the command line information */

    strcpy(ip_address,argv[1]);
    strcpy(system_name,argv[2]);

    strcpy(config_file_net,CONFIG_DIR);
    strcat(config_file_net,command_name);
    strcat(config_file_net,"/");
    strcat(config_file_net,system_name);

    if(argc == 3)
    {
      get_defaults = TRUE;
    }
    else
    {
      get_defaults = FALSE;
      strcpy(ms_warn,argv[3]);
      strcpy(ms_crit,argv[4]);
    }

    /* Check if config files exist */

    if (access(config_file_net, EXISTS) != 0 )
    {
      printf("Config file %s does not exist!\n",config_file_net);
      return_value = STATE_UNKNOWN;
    }
    else
    {

      /* Network config file variables */

      if((get_defaults == TRUE) && ((return_value=get_var("WARN_MS", config_file_net, ms_warn)) != STATE_OK))
      {
        printf("WARN_MS entry not found in config file %s!\n",config_file_net);
      }
      else if((get_defaults == TRUE) && ((return_value=get_var("CRIT_MS", config_file_net, ms_crit)) != STATE_OK))
      {
        printf("CRIT_MS entry not found in config file %s!\n",config_file_net);
      }
      else if((return_value=get_var("HTTP_PORT", config_file_net, port_http)) != STATE_OK)
      {
        printf("HTTP_PORT entry not found in config file %s!\n",config_file_net);
      }
      else if((return_value=get_var("HTTP_PROTO", config_file_net, protocol)) != STATE_OK)
      {
        printf("HTTP_PROTO entry not found in config file %s!\n",config_file_net);
      }
      else if((return_value=get_var("HTTP_EXPECT", config_file_net, expected)) != STATE_OK)
      {
        printf("HTTP_EXPECT entry not found in config file %s!\n",config_file_net);
      }
      else
      {

        /* Check alert level consistency */

        warn_ms=atoi(ms_warn);
        crit_ms=atoi(ms_crit);
        return_value = check_consistency(warn_ms, crit_ms);
        if(return_value == STATE_OK)
        {

          /* Check the network */

          http_port=atoi(port_http);
          gettimeofday(&start_buff, &start_tz);

          /* Set alarm */

          alarm(TIME_OUT); 
          if((result=open_socket(&socket_name, ip_address, http_port, protocol)) != STATE_OK)
          {
            return_value=exit_error(result,ip_address,protocol,http_port);
          } 
          else
          {
            sprintf(send_buffer,"GET / HTTP/1.0\n");
            if((result=send_socket(&socket_name, send_buffer)) != STATE_OK)
            {
              return_value=exit_error(result,ip_address,protocol,http_port);
            }
            else
            {
              sprintf(send_buffer,"\n");
              if((result=send_socket(&socket_name, send_buffer)) != STATE_OK)
              {
                return_value=exit_error(result,ip_address,protocol,http_port);
              }
              else if((result=recv_socket(&socket_name, expected)) != STATE_OK)
              {
                return_value=exit_error(result,ip_address,protocol,http_port);
              }
              else if((result=close_socket(&socket_name)) != STATE_OK)
              {
                return_value=exit_error(result,ip_address,protocol,http_port);
              }
              else
              {

                /* Nothing hungup reset alarm */

                alarm(0); 
                gettimeofday(&end_buff, &end_tz);
                seconds=(start_buff.tv_sec%1000);
                micro_seconds=start_buff.tv_usec;
                start_time=((seconds * 1000) + (micro_seconds/1000));
                seconds=(end_buff.tv_sec%1000);
                micro_seconds=end_buff.tv_usec;
                end_time=((seconds * 1000) + (micro_seconds/1000));
                total_time = end_time - start_time;
                if(total_time > crit_ms)
                {
                  return_value = STATE_CRITICAL;
                }
                else if(total_time > warn_ms)
                {
                  return_value = STATE_WARNING;
                }
                else
                {
                  return_value = STATE_OK;
                }
                if(return_value == STATE_OK)
                {
                  printf("HTTP ok - %d ms responce time.\n",total_time);
                }
                else
                {
                  printf("HTTP error slow responce time - %d ms.\n",total_time);
                }
              }
            }
          }
        }
      }
    }
  }
  return return_value;
}
