/**************************************************************************
 *
 * CMD.C -  NetSaint Command CGI
 *
 * Copyright (c) 1999-2000 Ethan Galstad (netsaint@netsaint.org)
 * Last Modified: 09-23-2000
 *
 * License:
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *************************************************************************/

#include "../common/config.h"
#include "../common/locations.h"
#include "../common/common.h"
#include "../common/objects.h"
#include "../common/comments.h"

#include "cgiutils.h"
#include "getcgi.h"
#include "auth.h"

extern char main_config_file[MAX_FILENAME_LENGTH];
extern char command_file[MAX_FILENAME_LENGTH];
extern char comment_file[MAX_FILENAME_LENGTH];

extern char url_stylesheets_path[MAX_FILENAME_LENGTH];

extern int  netsaint_process_state;

extern int  check_external_commands;

extern int  use_authentication;

extern comment *comment_list;



/* don't commit any commands for processing while in demo mode... */
/*#define DEMO_MODE*/

#define MAX_AUTHOR_LENGTH	64
#define MAX_COMMENT_LENGTH	1024


char host_name[MAX_HOSTNAME_LENGTH]="";
char service_desc[MAX_SERVICEDESC_LENGTH]="";
char comment_author[MAX_AUTHOR_LENGTH]="";
char comment_data[MAX_COMMENT_LENGTH]="";
int comment_id=0;
int notification_delay=0;
int schedule_delay=0;
int persistent_comment=FALSE;

int command_type=CMD_NONE;
int command_mode=CMDMODE_REQUEST;

authdata current_authdata;

void show_command_help(int);
void request_command_data(int);
void commit_command_data(int);
int commit_command(int);
int write_command_to_file(char *);
void clean_comment_data(char *);

void document_header(int);
void document_footer(void);
int process_cgivars(void);


int main(void){
	int result=OK;
	
	/* get the arguments passed in the URL */
	process_cgivars();

	/* reset internal variables */
	reset_cgi_vars();

	/* read the CGI configuration file */
	result=read_cgi_config_file(DEFAULT_CGI_CONFIG_FILE);
	if(result==ERROR){
		document_header(FALSE);
		printf("<P><DIV class='errorMessage'>Error: Could not open CGI configuration file '%s' for reading!</DIV></P>\n",DEFAULT_CGI_CONFIG_FILE);
		document_footer();
		return ERROR;
	        }

	document_header(TRUE);

	/* read the main configuration file */
	result=read_main_config_file(main_config_file);
	if(result==ERROR){
		printf("<P><DIV class='errorMessage'>Error: Could not open main configuration file '%s' for reading!</DIV></P>\n",main_config_file);
		document_footer();
		return ERROR;
	        }

	/* read all object configuration data */
	result=read_all_object_configuration_data(main_config_file,READ_HOSTGROUPS|READ_CONTACTGROUPS|READ_HOSTS|READ_SERVICES);
	if(result==ERROR){
		printf("<P><DIV class='errorMessage'>Error: Could not read some or all object configuration data!</DIV></P>\n");
		document_footer();
		return ERROR;
                }

	/* get authentication information */
	get_authentication_information(&current_authdata);

	/* begin top table */
	printf("<table border=0 width=100%%>\n");
	printf("<tr>\n");

	/* left column of the first row */
	printf("<td align=left valign=top width=33%%>\n");
	display_info_table("External Command Interface",FALSE,&current_authdata);
	printf("</td>\n");

	/* end of top table */
	printf("</tr>\n");
	printf("</table>\n");

	/* if no command was specified... */
	if(command_type==CMD_NONE){
		printf("<P><DIV CLASS='errorMessage'>Error: No command was specified</DIV></P>\n");
	        }

	/* if this is the first request for a command, present option */
	else if(command_mode==CMDMODE_REQUEST)
		request_command_data(command_type);

	/* the user wants to commit the command */
	else if(command_mode==CMDMODE_COMMIT)
		commit_command_data(command_type);

	document_footer();

	/* free allocated memory */
	free_memory();
	free_object_data();

	return OK;
        }



void document_header(int use_stylesheet){

	printf("Content-type: text/html\r\n\r\n");

	printf("<html>\n");
	printf("<head>\n");
	printf("<title>\n");
	printf("External Command Interface\n");
	printf("</title>\n");

	if(use_stylesheet==TRUE)
		printf("<LINK REL='stylesheet' TYPE='text/css' HREF='%s%s'>\n",url_stylesheets_path,COMMAND_CSS);

	printf("</head>\n");

	printf("<body CLASS='cmd'>\n");

	return;
        }


void document_footer(void){

	printf("</body>\n");
	printf("</html>\n");

	return;
        }


int process_cgivars(void){
	char **variables;
	int error=FALSE;
	int x;

	variables=getcgivars();

	for(x=0;variables[x]!=NULL;x++){

		/* do some basic length checking on the variable identifier to prevent buffer overflows */
		if(strlen(variables[x])>=MAX_INPUT_BUFFER-1){
			x++;
			continue;
		        }

		/* we found the command type */
		else if(!strcmp(variables[x],"cmd_typ")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			command_type=atoi(variables[x]);
		        }

		/* we found the command mode */
		else if(!strcmp(variables[x],"cmd_mod")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			command_mode=atoi(variables[x]);
		        }

		/* we found the comment id */
		else if(!strcmp(variables[x],"com_id")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			comment_id=atoi(variables[x]);
		        }

		/* we found the notification delay */
		else if(!strcmp(variables[x],"not_dly")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			notification_delay=atoi(variables[x]);
		        }

		/* we found the schedule delay */
		else if(!strcmp(variables[x],"sched_dly")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			schedule_delay=atoi(variables[x]);
		        }

		/* we found the comment author */
		else if(!strcmp(variables[x],"com_author")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			/* protect against buffer overflows */
			if(strlen(variables[x])>=MAX_AUTHOR_LENGTH-1){
				error=TRUE;
				break;
			        }
			else
				strcpy(comment_author,variables[x]);
			}

		/* we found the comment data */
		else if(!strcmp(variables[x],"com_data")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			/* protect against buffer overflows */
			if(strlen(variables[x])>=MAX_COMMENT_LENGTH-1){
				error=TRUE;
				break;
			        }
			else
				strcpy(comment_data,variables[x]);
			}

		/* we found the host name */
		else if(!strcmp(variables[x],"host")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			/* protect against buffer overflows */
			if(strlen(variables[x])>=MAX_HOSTNAME_LENGTH-1){
				error=TRUE;
				break;
			        }
			else
				strcpy(host_name,variables[x]);
			}

		/* we found the service name */
		else if(!strcmp(variables[x],"service")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			/* protect against buffer overflows */
			if(strlen(variables[x])>=MAX_SERVICEDESC_LENGTH-1){
				error=TRUE;
				break;
			        }
			else
				strcpy(service_desc,variables[x]);
			}

		/* we got the persistence option for a comment */
		else if(!strcmp(variables[x],"persistent"))
			persistent_comment=TRUE;

	        }

	return error;
        }



void request_command_data(int cmd){

	printf("<DIV ALIGN=CENTER CLASS='cmdType'>You are requesting to ");

	switch(cmd){

	case CMD_ADD_HOST_COMMENT:
		printf("add a host comment");
		break;
		
	case CMD_ADD_SVC_COMMENT:
		printf("add a service comment");
		break;

	case CMD_DEL_HOST_COMMENT:
		printf("delete a host comment");
		break;
		
	case CMD_DEL_SVC_COMMENT:
		printf("delete a service comment");
		break;
		
	case CMD_DELAY_HOST_NOTIFICATION:
		printf("delay a host notification");
		break;

	case CMD_DELAY_SVC_NOTIFICATION:
		printf("delay a service notification");
		break;

	case CMD_DELAY_SVC_CHECK:
		printf("delay a service check");
		break;

	case CMD_IMMEDIATE_SVC_CHECK:
		printf("schedule an immediate service check");
		break;

	case CMD_ENABLE_SVC_CHECK:
		printf("enable a service check");
		break;
		
	case CMD_DISABLE_SVC_CHECK:
		printf("disable a service check");
		break;
		
	case CMD_ENTER_STANDBY_MODE:
		printf("enter STANDBY mode");
		break;
		
	case CMD_ENTER_ACTIVE_MODE:
		printf("enter ACTIVE mode");
		break;
		
	case CMD_SHUTDOWN_PROCESS:
		printf("shutdown the NetSaint process");
		break;

	case CMD_RESTART_PROCESS:
		printf("restart the NetSaint process");
		break;

	case CMD_ENABLE_HOST_SVC_CHECKS:
		printf("enable all service checks for a host");
		break;
		
	case CMD_DISABLE_HOST_SVC_CHECKS:
		printf("disable all service checks for a host");
		break;

	case CMD_DELAY_HOST_SVC_CHECKS:
		printf("delay all service checks for a host");
		break;

	case CMD_IMMEDIATE_HOST_SVC_CHECKS:
		printf("schedule an immediate check of all services for a host");
		break;

	case CMD_DEL_ALL_HOST_COMMENTS:
		printf("delete all comments for a host");
		break;

	case CMD_DEL_ALL_SVC_COMMENTS:
		printf("delete all comments for a service");
		break;

	case CMD_ENABLE_SVC_NOTIFICATIONS:
		printf("enable notifications for a service");
		break;

	case CMD_DISABLE_SVC_NOTIFICATIONS:
		printf("temporarily disable notifications for a service");
		break;

	case CMD_ENABLE_HOST_NOTIFICATIONS:
		printf("enable notifications for a host");
		break;

	case CMD_DISABLE_HOST_NOTIFICATIONS:
		printf("temporarily disable notifications for a host");
		break;

	case CMD_ENABLE_ALL_NOTIFICATIONS_BEYOND_HOST:
		printf("enable notifications for all hosts and services beyond a host");
		break;

	case CMD_DISABLE_ALL_NOTIFICATIONS_BEYOND_HOST:
		printf("temporarily disable notifications for all hosts and services beyond a host");
		break;
		
	case CMD_ENABLE_HOST_SVC_NOTIFICATIONS:
		printf("enable notifications for all services on a host");
		break;

	case CMD_DISABLE_HOST_SVC_NOTIFICATIONS:
		printf("temporarily disable notifications for all services on a host");
		break;

	case CMD_ACKNOWLEDGE_HOST_PROBLEM:
		printf("acknowledge a host problem");
		break;

	case CMD_ACKNOWLEDGE_SVC_PROBLEM:
		printf("acknowledge a service problem");
		break;

	case CMD_START_EXECUTING_SVC_CHECKS:
		printf("start executing service checks");
		break;

	case CMD_STOP_EXECUTING_SVC_CHECKS:
		printf("stop executing service checks");
		break;

	case CMD_START_ACCEPTING_PASSIVE_SVC_CHECKS:
		printf("start accepting passive service checks");
		break;

	case CMD_STOP_ACCEPTING_PASSIVE_SVC_CHECKS:
		printf("stop accepting passive service checks");
		break;

	case CMD_ENABLE_PASSIVE_SVC_CHECKS:
		printf("start accepting passive service checks for a particular service");
		break;

	case CMD_DISABLE_PASSIVE_SVC_CHECKS:
		printf("stop accepting passive service checks for a particular service");
		break;

	case CMD_ENABLE_EVENT_HANDLERS:
		printf("enable event handlers");
		break;

	case CMD_DISABLE_EVENT_HANDLERS:
		printf("disable event handlers");
		break;

	case CMD_ENABLE_HOST_EVENT_HANDLER:
		printf("enable the event handler for a particular host");
		break;

	case CMD_DISABLE_HOST_EVENT_HANDLER:
		printf("disable the event handler for a particular host");
		break;

	case CMD_ENABLE_SVC_EVENT_HANDLER:
		printf("enable the event handler for a particular service");
		break;

	case CMD_DISABLE_SVC_EVENT_HANDLER:
		printf("disable the event handler for a particular service");
		break;

	case CMD_ENABLE_HOST_CHECK:
		printf("enable checks of a particular host");
		break;

	case CMD_DISABLE_HOST_CHECK:
		printf("disable checks of a particular host");
		break;

	case CMD_STOP_OBSESSING_OVER_SVC_CHECKS:
		printf("stop obsessing over service checks");
		break;

	case CMD_START_OBSESSING_OVER_SVC_CHECKS:
		printf("start obsessing over service checks");
		break;

	case CMD_REMOVE_HOST_ACKNOWLEDGEMENT:
		printf("remove a host acknowledgement");
		break;

	case CMD_REMOVE_SVC_ACKNOWLEDGEMENT:
		printf("remove a service acknowledgement");
		break;

	default:
		printf("execute an unknown command.  Shame on you!</DIV>");
		return;
	        }

	printf("</DIV>\n");

	printf("<p>\n");
	printf("<div align='center'>\n");

	printf("<table border=0 width=70%%>\n");
	printf("<tr>\n");
	printf("<td align=center valign=top>\n");

	printf("<DIV ALIGN=CENTER CLASS='optBoxTitle'>Command Options</DIV>\n");

	printf("<TABLE CELLSPACING=0 CELLPADDING=0 BORDER=1 CLASS='optBox'>\n");
	printf("<TR><TD CLASS='optBoxItem'>\n");
	printf("<TABLE CELLSPACING=0 CELLPADDING=0 CLASS='optBox'>\n");

	printf("<tr><td CLASS='optBoxItem'><form method='post' action='%s'></td><td><INPUT TYPE='HIDDEN' NAME='cmd_typ' VALUE='%d'><INPUT TYPE='HIDDEN' NAME='cmd_mod' VALUE='%d'></td></tr>\n",COMMAND_CGI,cmd,CMDMODE_COMMIT);

	switch(cmd){

	case CMD_ADD_HOST_COMMENT:
	case CMD_ACKNOWLEDGE_HOST_PROBLEM:
		printf("<tr><td CLASS='optBoxRequiredItem'>Host Name:</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='host' VALUE='%s'>",host_name);
		printf("</b></td></tr>\n");
		printf("<tr><td CLASS='optBoxItem'>Persistent:</td><td><b>");
		printf("<INPUT TYPE='checkbox' NAME='persistent'>");
		printf("<tr><td CLASS='optBoxRequiredItem'>Author (Your Name):</td><td><b>");
		printf("<INPUT TYPE'TEXT' NAME='com_author' VALUE='%s'>",comment_author);
		printf("<tr><td CLASS='optBoxRequiredItem'>Comment:</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='com_data' VALUE='%s' SIZE=40>",comment_data);
		printf("</b></td></tr>\n");
		break;
		
	case CMD_ADD_SVC_COMMENT:
	case CMD_ACKNOWLEDGE_SVC_PROBLEM:
		printf("<tr><td CLASS='optBoxRequiredItem'>Host Name:</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='host' VALUE='%s'>",host_name);
		printf("</b></td></tr>\n");
		printf("<tr><td CLASS='optBoxRequiredItem'>Service:</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='service' VALUE='%s'>",service_desc);
		printf("<tr><td CLASS='optBoxItem'>Persistent:</td><td><b>");
		printf("<INPUT TYPE='checkbox' NAME='persistent'>");
		printf("<tr><td CLASS='optBoxRequiredItem'>Author (Your Name):</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='com_author' VALUE='%s'>",comment_author);
		printf("<tr><td CLASS='optBoxRequiredItem'>Comment:</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='com_data' VALUE='%s' SIZE=40>",comment_data);
		printf("</b></td></tr>\n");
		break;

	case CMD_DEL_HOST_COMMENT:
	case CMD_DEL_SVC_COMMENT:
		printf("<tr><td CLASS='optBoxRequiredItem'>Comment ID:</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='com_id' VALUE='%d'>",comment_id);
		printf("</b></td></tr>\n");
		break;
		
	case CMD_DELAY_HOST_NOTIFICATION:
		printf("<tr><td CLASS='optBoxRequiredItem'>Host Name:</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='host' VALUE='%s'>",host_name);
		printf("</b></td></tr>\n");
		printf("<tr><td CLASS='optBoxRequiredItem'>Notification Delay (minutes from now):</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='not_dly' VALUE='%d'>",notification_delay);
		printf("</b></td></tr>\n");
		break;

	case CMD_DELAY_SVC_NOTIFICATION:
		printf("<tr><td CLASS='optBoxRequiredItem'>Host Name:</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='host' VALUE='%s'>",host_name);
		printf("</b></td></tr>\n");
		printf("<tr><td CLASS='optBoxRequiredItem'>Service:</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='service' VALUE='%s'>",service_desc);
		printf("<tr><td CLASS='optBoxRequiredItem'>Notification Delay (minutes from now):</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='not_dly' VALUE='%d'>",notification_delay);
		printf("</b></td></tr>\n");
		break;

	case CMD_DELAY_SVC_CHECK:
		printf("<tr><td CLASS='optBoxRequiredItem'>Host Name:</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='host' VALUE='%s'>",host_name);
		printf("</b></td></tr>\n");
		printf("<tr><td CLASS='optBoxRequiredItem'>Service:</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='service' VALUE='%s'>",service_desc);
		printf("<tr><td CLASS='optBoxRequiredItem'>Check Delay (minutes from now):</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='sched_dly' VALUE='%d'>",schedule_delay);
		printf("</b></td></tr>\n");
		break;

	case CMD_DELAY_HOST_SVC_CHECKS:
		printf("<tr><td CLASS='optBoxRequiredItem'>Host Name:</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='host' VALUE='%s'>",host_name);
		printf("</b></td></tr>\n");
		printf("<tr><td CLASS='optBoxRequiredItem'>Check Delay (minutes from now):</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='sched_dly' VALUE='%d'>",schedule_delay);
		printf("</b></td></tr>\n");
		break;

	case CMD_IMMEDIATE_SVC_CHECK:
	case CMD_ENABLE_SVC_CHECK:
	case CMD_DISABLE_SVC_CHECK:
	case CMD_DEL_ALL_SVC_COMMENTS:
	case CMD_ENABLE_SVC_NOTIFICATIONS:
	case CMD_DISABLE_SVC_NOTIFICATIONS:
	case CMD_ENABLE_PASSIVE_SVC_CHECKS:
	case CMD_DISABLE_PASSIVE_SVC_CHECKS:
	case CMD_ENABLE_SVC_EVENT_HANDLER:
	case CMD_DISABLE_SVC_EVENT_HANDLER:
	case CMD_REMOVE_SVC_ACKNOWLEDGEMENT:
		printf("<tr><td CLASS='optBoxRequiredItem'>Host Name:</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='host' VALUE='%s'>",host_name);
		printf("</b></td></tr>\n");
		printf("<tr><td CLASS='optBoxRequiredItem'>Service:</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='service' VALUE='%s'>",service_desc);
		printf("</b></td></tr>\n");
		break;
		
	case CMD_ENABLE_HOST_SVC_CHECKS:
	case CMD_DISABLE_HOST_SVC_CHECKS:
	case CMD_DEL_ALL_HOST_COMMENTS:
	case CMD_IMMEDIATE_HOST_SVC_CHECKS:
	case CMD_ENABLE_HOST_NOTIFICATIONS:
	case CMD_DISABLE_HOST_NOTIFICATIONS:
	case CMD_ENABLE_ALL_NOTIFICATIONS_BEYOND_HOST:
	case CMD_DISABLE_ALL_NOTIFICATIONS_BEYOND_HOST:
	case CMD_ENABLE_HOST_SVC_NOTIFICATIONS:
	case CMD_DISABLE_HOST_SVC_NOTIFICATIONS:
	case CMD_ENABLE_HOST_EVENT_HANDLER:
	case CMD_DISABLE_HOST_EVENT_HANDLER:
	case CMD_ENABLE_HOST_CHECK:
	case CMD_DISABLE_HOST_CHECK:
	case CMD_REMOVE_HOST_ACKNOWLEDGEMENT:
		printf("<tr><td CLASS='optBoxRequiredItem'>Host Name:</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='host' VALUE='%s'>",host_name);
		printf("</b></td></tr>\n");
		break;
		
	case CMD_ENTER_STANDBY_MODE:
	case CMD_ENTER_ACTIVE_MODE:
	case CMD_SHUTDOWN_PROCESS:
	case CMD_RESTART_PROCESS:
		printf("<tr><td CLASS='optBoxRequiredItem'>Execution Delay (minutes from now):</td><td><b>");
		printf("<INPUT TYPE='TEXT' NAME='sched_dly' VALUE='%d'>",schedule_delay);
		printf("</b></td></tr>\n");
		break;

	case CMD_START_EXECUTING_SVC_CHECKS:
	case CMD_STOP_EXECUTING_SVC_CHECKS:
	case CMD_START_ACCEPTING_PASSIVE_SVC_CHECKS:
	case CMD_STOP_ACCEPTING_PASSIVE_SVC_CHECKS:
	case CMD_ENABLE_EVENT_HANDLERS:
	case CMD_DISABLE_EVENT_HANDLERS:
	case CMD_START_OBSESSING_OVER_SVC_CHECKS:
	case CMD_STOP_OBSESSING_OVER_SVC_CHECKS:
		printf("<tr><td CLASS='optBoxItem' colspan=2>There are no options for this command.<br>Click the 'Commit' button to submit the command.</td></tr>");
		break;
		
	default:
		printf("<tr><td CLASS='optBoxItem'>This should not be happening... :-(</td><td></td></tr>\n");
	        }


	printf("<tr><td CLASS='optBoxItem' COLSPAN=2></td></tr>\n");
	printf("<tr><td CLASS='optBoxItem'></td><td CLASS='optBoxItem'><INPUT TYPE='submit' NAME='btnSubmit' VALUE='Commit'><INPUT TYPE='reset' VALUE='Reset'></FORM></td></tr>\n");

	printf("</table>\n");

	printf("</td>\n");
	printf("</tr>\n");
	printf("</table>\n");

	printf("</td>\n");
	printf("<td align=center valign=top width=50%%>\n");

	/* show information about the command... */
	show_command_help(cmd);
	
	printf("</td>\n");
	printf("</tr>\n");
	printf("</table>\n");

	printf("</div>\n");
	printf("</p>\n");


	printf("<P><DIV CLASS='infoMessage'>Please enter any and all information that may be required before committing the command.<br>Failure to do so will result in an error.</DIV></P>");

	return;
        }


void commit_command_data(int cmd){
	int error=TRUE;
	int result=OK;
	int authorized=FALSE;
	service *temp_service;
	host *temp_host;
	comment *temp_comment;

#ifdef DEMO_MODE
	/* demo protection */
	printf("<P><DIV CLASS='errorMessage'>Sorry, but commands cannot be committed in demo mode!</DIV></P>\n");
	return;
#endif

	/* get authentication information */
	get_authentication_information(&current_authdata);

	switch(cmd){
	case CMD_ADD_HOST_COMMENT:
	case CMD_ACKNOWLEDGE_HOST_PROBLEM:

		/* make sure we have some host name, author name, and comment data... */
		if(strcmp(host_name,"") && strcmp(comment_author,"") && strcmp(comment_data,""))
			error=FALSE;
		else
			error=TRUE;

		/* clean up the comment data */
		clean_comment_data(comment_author);
		clean_comment_data(comment_data);

		/* see if the user is authorized to issue a command... */
		temp_host=find_host(host_name,NULL);
		if(is_authorized_for_host_commands(temp_host,&current_authdata)==TRUE)
			authorized=TRUE;
		break;
		
	case CMD_ADD_SVC_COMMENT:
	case CMD_ACKNOWLEDGE_SVC_PROBLEM:

		/* make sure we have some host name and service description, author name, and comment data... */
		if(strcmp(host_name,"") && strcmp(service_desc,"") && strcmp(comment_author,"") && strcmp(comment_data,""))
			error=FALSE;
		else
			error=TRUE;

		/* clean up the comment data */
		clean_comment_data(comment_author);
		clean_comment_data(comment_data);

		/* see if the user is authorized to issue a command... */
		temp_service=find_service(host_name,service_desc,NULL);
		if(is_authorized_for_service_commands(temp_service,&current_authdata)==TRUE)
			authorized=TRUE;
		break;

	case CMD_DEL_HOST_COMMENT:
		if(comment_id>0)
			error=FALSE;
		else
			error=TRUE;

		/* read comments */
		read_comment_data(main_config_file);

		/* find the comment */
		temp_comment=find_comment(comment_id);

		/* see if the user is authorized to issue a command... */
		if(temp_comment!=NULL){
			temp_host=find_host(temp_comment->host_name,NULL);
			if(is_authorized_for_host_commands(temp_host,&current_authdata)==TRUE)
			authorized=TRUE;
		        }

		/* free comment data */
		free_comment_data();

		break;
		
	case CMD_DEL_SVC_COMMENT:
		if(comment_id>0)
			error=FALSE;
		else
			error=TRUE;

		/* read comments */
		read_comment_data(main_config_file);

		/* find the comment */
		temp_comment=find_comment(comment_id);

		/* see if the user is authorized to issue a command... */
		if(temp_comment!=NULL){
			temp_service=find_service(temp_comment->host_name,temp_comment->service_description,NULL);
			if(is_authorized_for_service_commands(temp_service,&current_authdata)==TRUE)
				authorized=TRUE;
		        }

		/* free comment memory */
		free_comment_data();

		break;
		
	case CMD_DELAY_HOST_NOTIFICATION:
		/* make sure we have some host name as well as delay time... */
		if(strcmp(host_name,"") && notification_delay>0)
			error=FALSE;
		else
			error=TRUE;

		/* see if the user is authorized to issue a command... */
		temp_host=find_host(host_name,NULL);
		if(is_authorized_for_host_commands(temp_host,&current_authdata)==TRUE)
			authorized=TRUE;
		break;

	case CMD_DELAY_SVC_NOTIFICATION:
		/* make sure we have some host name and service description as well as notification delay... */
		if(strcmp(host_name,"") && strcmp(service_desc,"") && notification_delay>0)
			error=FALSE;
		else
			error=TRUE;

		/* see if the user is authorized to issue a command... */
		temp_service=find_service(host_name,service_desc,NULL);
		if(is_authorized_for_service_commands(temp_service,&current_authdata)==TRUE)
			authorized=TRUE;
		break;

	case CMD_DELAY_SVC_CHECK:
	case CMD_IMMEDIATE_SVC_CHECK:
	case CMD_ENABLE_SVC_CHECK:
	case CMD_DISABLE_SVC_CHECK:
	case CMD_DEL_ALL_SVC_COMMENTS:
	case CMD_ENABLE_SVC_NOTIFICATIONS:
	case CMD_DISABLE_SVC_NOTIFICATIONS:
	case CMD_ENABLE_PASSIVE_SVC_CHECKS:
	case CMD_DISABLE_PASSIVE_SVC_CHECKS:
	case CMD_ENABLE_SVC_EVENT_HANDLER:
	case CMD_DISABLE_SVC_EVENT_HANDLER:
	case CMD_REMOVE_SVC_ACKNOWLEDGEMENT:

		/* make sure we have some host name and service description... */
		if(strcmp(host_name,"") && strcmp(service_desc,""))
			error=FALSE;
		else
			error=TRUE;

		/* see if the user is authorized to issue a command... */
		temp_service=find_service(host_name,service_desc,NULL);
		if(is_authorized_for_service_commands(temp_service,&current_authdata)==TRUE)
			authorized=TRUE;
		break;
		
	case CMD_ENTER_STANDBY_MODE:
	case CMD_ENTER_ACTIVE_MODE:
	case CMD_SHUTDOWN_PROCESS:
	case CMD_RESTART_PROCESS:
	case CMD_START_EXECUTING_SVC_CHECKS:
	case CMD_STOP_EXECUTING_SVC_CHECKS:
	case CMD_START_ACCEPTING_PASSIVE_SVC_CHECKS:
	case CMD_STOP_ACCEPTING_PASSIVE_SVC_CHECKS:
	case CMD_ENABLE_EVENT_HANDLERS:
	case CMD_DISABLE_EVENT_HANDLERS:
	case CMD_START_OBSESSING_OVER_SVC_CHECKS:
	case CMD_STOP_OBSESSING_OVER_SVC_CHECKS:
		error=FALSE;

		/* see if the user is authorized to issue a command... */
		if(is_authorized_for_system_commands(&current_authdata)==TRUE)
			authorized=TRUE;
		break;
		
	case CMD_ENABLE_HOST_SVC_CHECKS:
	case CMD_DISABLE_HOST_SVC_CHECKS:
	case CMD_DEL_ALL_HOST_COMMENTS:
	case CMD_IMMEDIATE_HOST_SVC_CHECKS:
	case CMD_DELAY_HOST_SVC_CHECKS:
	case CMD_ENABLE_HOST_NOTIFICATIONS:
	case CMD_DISABLE_HOST_NOTIFICATIONS:
	case CMD_ENABLE_ALL_NOTIFICATIONS_BEYOND_HOST:
	case CMD_DISABLE_ALL_NOTIFICATIONS_BEYOND_HOST:
	case CMD_ENABLE_HOST_SVC_NOTIFICATIONS:
	case CMD_DISABLE_HOST_SVC_NOTIFICATIONS:
	case CMD_ENABLE_HOST_EVENT_HANDLER:
	case CMD_DISABLE_HOST_EVENT_HANDLER:
	case CMD_ENABLE_HOST_CHECK:
	case CMD_DISABLE_HOST_CHECK:
	case CMD_REMOVE_HOST_ACKNOWLEDGEMENT:

		/* make sure we have some host name... */
		if(strcmp(host_name,""))
			error=FALSE;
		else
			error=TRUE;

		/* see if the user is authorized to issue a command... */
		temp_host=find_host(host_name,NULL);
		if(is_authorized_for_host_commands(temp_host,&current_authdata)==TRUE)
			authorized=TRUE;
		break;

	default:
		error=TRUE;
	        }


	/* to be safe, we are going to REQUIRE that the authentication functionality is enabled... */
	if(use_authentication==FALSE){
		printf("<P>\n");
		printf("<DIV CLASS='errorMessage'>Sorry Dave, I can't let you do that...</DIV><br>");
		printf("<DIV CLASS='errorDescription'>");
		printf("It seems that you have chosen to not use the authentication functionality of the CGIs.<br><br>");
		printf("I don't want to be personally responsible for what may happen as a result of allowing unauthorized users to issue commands to NetSaint,");
		printf("so you'll have to disable this safeguard if you are really stubborn and want to invite trouble...<br><br>");
		printf("<strong>Read the section on CGI authentication in the HTML documentation to learn how you can enable authentication and why you should want to.</strong>\n");
		printf("</DIV>\n");
		printf("</P>\n");
	        }

	/* the user is not authorized to issue the given command */
	else if(authorized==FALSE){
		printf("<P><DIV CLASS='errorMessage'>Sorry, but you are not authorized to commit the specified command...</DIV></P>\n");
		printf("<P><DIV CLASS='errorDescription'>Read the section of the docs that deals with authentication and authorization in the CGIs for more information...</DIV></P>\n");
		printf("<P><A HREF='javascript:window.history.go(-2)'>Return from whence you came...</A></P>\n");
	        }

	/* some error occurred (data was probably missing) */
	else if(error==TRUE){
		printf("<P><DIV CLASS='errorMessage'>An error occurred while processing your command.</DIV></P>\n");
		printf("<P><DIV CLASS='errorDescription'>Go back and verify that you entered all required information correctly...</DIV></P>\n");
		printf("<P><A HREF='javascript:window.history.go(-2)'>Return from whence you came...</A></P>\n");
	        }

	/* if NetSaint isn't checking external commands, don't do anything... */
	else if(!check_external_commands){
		printf("<P><DIV CLASS='errorMessage'>Sorry, but NetSaint is currently not checking for external commands, so your command will not be committed!</DIV></P>\n");
		printf("<P><DIV CLASS='errorDescription'>Read the documentation for information on how to enable external commands...</DIV></P>\n");
		printf("<P><A HREF='javascript:window.history.go(-2)'>Return from whence you came...</A></P>\n");
	        }
	
	/* everything looks okay, so let's go ahead and commit the command... */
	else{

		/* commit the command */
		result=commit_command(cmd);

		if(result==OK){
			printf("<P><DIV CLASS='infoMessage'>Your command request was successfully submitted to NetSaint for processing.<BR><BR>\n");
			printf("Note: It may take a few minutes before the command is actually processed...<BR><BR>\n");
			printf("<A HREF='javascript:window.history.go(-2)'>Done</A></DIV></P>");
		        }
		else{
			printf("<P><DIV CLASS='errorMessage'>An error occurred while attempting to commit your command to NetSaint for processing...<BR><BR>\n");
			printf("<A HREF='javascript:window.history.go(-2)'>Return from whence you came...</A></DIV></P>\n");
		        }
	        }

	return;
        }


int commit_command(int cmd){
	char command_buffer[MAX_INPUT_BUFFER];
	time_t current_time;
	time_t scheduled_time;
	time_t notification_time;
	int result;

	/* get the current time */
	time(&current_time);

	/* get the scheduled time */
	scheduled_time=current_time+(schedule_delay*60);

	/* get the notification time */
	notification_time=current_time+(notification_delay*60);

	/* decide how to form the command line... */
	switch(cmd){

	case CMD_ADD_HOST_COMMENT:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] ADD_HOST_COMMENT;%s;%d;%s;%s\n",current_time,host_name,(persistent_comment==TRUE)?1:0,comment_author,comment_data);
		break;
		
	case CMD_ADD_SVC_COMMENT:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] ADD_SVC_COMMENT;%s;%s;%d;%s;%s\n",current_time,host_name,service_desc,(persistent_comment==TRUE)?1:0,comment_author,comment_data);
		break;

	case CMD_DEL_HOST_COMMENT:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] DEL_HOST_COMMENT;%d\n",current_time,comment_id);
		break;
		
	case CMD_DEL_SVC_COMMENT:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] DEL_SVC_COMMENT;%d\n",current_time,comment_id);
		break;
		
	case CMD_DELAY_HOST_NOTIFICATION:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] DELAY_HOST_NOTIFICATION;%s;%lu\n",current_time,host_name,notification_time);
		break;

	case CMD_DELAY_SVC_NOTIFICATION:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] DELAY_SVC_NOTIFICATION;%s;%s;%lu\n",current_time,host_name,service_desc,notification_time);
		break;

	case CMD_DELAY_SVC_CHECK:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] SCHEDULE_SVC_CHECK;%s;%s;%lu\n",current_time,host_name,service_desc,scheduled_time);
		break;

	case CMD_IMMEDIATE_SVC_CHECK:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] SCHEDULE_SVC_CHECK;%s;%s;%lu\n",current_time,host_name,service_desc,current_time);
		break;

	case CMD_ENABLE_SVC_CHECK:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] ENABLE_SVC_CHECK;%s;%s\n",current_time,host_name,service_desc);
		break;
		
	case CMD_DISABLE_SVC_CHECK:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] DISABLE_SVC_CHECK;%s;%s\n",current_time,host_name,service_desc);
		break;
		
	case CMD_ENTER_STANDBY_MODE:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] ENTER_STANDBY_MODE;%lu\n",current_time,scheduled_time);
		break;
		
	case CMD_ENTER_ACTIVE_MODE:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] ENTER_ACTIVE_MODE;%lu\n",current_time,scheduled_time);
		break;
		
	case CMD_SHUTDOWN_PROCESS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] SHUTDOWN_PROGRAM;%lu\n",current_time,scheduled_time);
		break;

	case CMD_RESTART_PROCESS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] RESTART_PROGRAM;%lu\n",current_time,scheduled_time);
		break;

	case CMD_ENABLE_HOST_SVC_CHECKS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] ENABLE_HOST_SVC_CHECKS;%s\n",current_time,host_name);
		break;
		
	case CMD_DISABLE_HOST_SVC_CHECKS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] DISABLE_HOST_SVC_CHECKS;%s\n",current_time,host_name);
		break;
		
	case CMD_DELAY_HOST_SVC_CHECKS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] SCHEDULE_HOST_SVC_CHECKS;%s;%lu\n",current_time,host_name,scheduled_time);
		break;

	case CMD_IMMEDIATE_HOST_SVC_CHECKS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] SCHEDULE_HOST_SVC_CHECKS;%s;%lu\n",current_time,host_name,current_time);
		break;

	case CMD_DEL_ALL_HOST_COMMENTS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] DEL_ALL_HOST_COMMENTS;%s\n",current_time,host_name);
		break;
		
	case CMD_DEL_ALL_SVC_COMMENTS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] DEL_ALL_SVC_COMMENTS;%s;%s\n",current_time,host_name,service_desc);
		break;

	case CMD_ENABLE_SVC_NOTIFICATIONS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] ENABLE_SVC_NOTIFICATIONS;%s;%s\n",current_time,host_name,service_desc);
		break;
		
	case CMD_DISABLE_SVC_NOTIFICATIONS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] DISABLE_SVC_NOTIFICATIONS;%s;%s\n",current_time,host_name,service_desc);
		break;
		
	case CMD_ENABLE_HOST_NOTIFICATIONS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] ENABLE_HOST_NOTIFICATIONS;%s\n",current_time,host_name);
		break;
		
	case CMD_DISABLE_HOST_NOTIFICATIONS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] DISABLE_HOST_NOTIFICATIONS;%s\n",current_time,host_name);
		break;
		
	case CMD_ENABLE_ALL_NOTIFICATIONS_BEYOND_HOST:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] ENABLE_ALL_NOTIFICATIONS_BEYOND_HOST;%s\n",current_time,host_name);
		break;
		
	case CMD_DISABLE_ALL_NOTIFICATIONS_BEYOND_HOST:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] DISABLE_ALL_NOTIFICATIONS_BEYOND_HOST;%s\n",current_time,host_name);
		break;
		
	case CMD_ENABLE_HOST_SVC_NOTIFICATIONS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] ENABLE_HOST_SVC_NOTIFICATIONS;%s\n",current_time,host_name);
		break;
		
	case CMD_DISABLE_HOST_SVC_NOTIFICATIONS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] DISABLE_HOST_SVC_NOTIFICATIONS;%s\n",current_time,host_name);
		break;
		
	case CMD_ACKNOWLEDGE_HOST_PROBLEM:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] ADD_HOST_COMMENT;%s;%d;%s;ACKNOWLEDGEMENT: %s\n[%lu] ACKNOWLEDGE_HOST_PROBLEM;%s;%s\n",current_time,host_name,(persistent_comment==TRUE)?1:0,comment_author,comment_data,current_time,host_name,comment_data);
		break;
		
	case CMD_ACKNOWLEDGE_SVC_PROBLEM:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] ADD_SVC_COMMENT;%s;%s;%d;%s;ACKNOWLEDGEMENT: %s\n[%lu] ACKNOWLEDGE_SVC_PROBLEM;%s;%s;%s\n",current_time,host_name,service_desc,(persistent_comment==TRUE)?1:0,comment_author,comment_data,current_time,host_name,service_desc,comment_data);
		break;

	case CMD_START_EXECUTING_SVC_CHECKS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] START_EXECUTING_SVC_CHECKS;\n",current_time);
		break;

	case CMD_STOP_EXECUTING_SVC_CHECKS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] STOP_EXECUTING_SVC_CHECKS;\n",current_time);
		break;

	case CMD_START_ACCEPTING_PASSIVE_SVC_CHECKS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] START_ACCEPTING_PASSIVE_SVC_CHECKS;\n",current_time);
		break;

	case CMD_STOP_ACCEPTING_PASSIVE_SVC_CHECKS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] STOP_ACCEPTING_PASSIVE_SVC_CHECKS;\n",current_time);
		break;

	case CMD_ENABLE_PASSIVE_SVC_CHECKS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] ENABLE_PASSIVE_SVC_CHECKS;%s;%s\n",current_time,host_name,service_desc);
		break;
		
	case CMD_DISABLE_PASSIVE_SVC_CHECKS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] DISABLE_PASSIVE_SVC_CHECKS;%s;%s\n",current_time,host_name,service_desc);
		break;
		
	case CMD_ENABLE_EVENT_HANDLERS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] ENABLE_EVENT_HANDLERS;\n",current_time);
		break;

	case CMD_DISABLE_EVENT_HANDLERS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] DISABLE_EVENT_HANDLERS;\n",current_time);
		break;

	case CMD_ENABLE_SVC_EVENT_HANDLER:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] ENABLE_SVC_EVENT_HANDLER;%s;%s\n",current_time,host_name,service_desc);
		break;
		
	case CMD_DISABLE_SVC_EVENT_HANDLER:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] DISABLE_SVC_EVENT_HANDLER;%s;%s\n",current_time,host_name,service_desc);
		break;
		
	case CMD_ENABLE_HOST_EVENT_HANDLER:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] ENABLE_HOST_EVENT_HANDLER;%s\n",current_time,host_name);
		break;
		
	case CMD_DISABLE_HOST_EVENT_HANDLER:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] DISABLE_HOST_EVENT_HANDLER;%s\n",current_time,host_name);
		break;
		
	case CMD_ENABLE_HOST_CHECK:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] ENABLE_HOST_CHECK;%s\n",current_time,host_name);
		break;
		
	case CMD_DISABLE_HOST_CHECK:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] DISABLE_HOST_CHECK;%s\n",current_time,host_name);
		break;
		
	case CMD_START_OBSESSING_OVER_SVC_CHECKS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] START_OBSESSING_OVER_SVC_CHECKS;\n",current_time);
		break;
		
	case CMD_STOP_OBSESSING_OVER_SVC_CHECKS:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] STOP_OBSESSING_OVER_SVC_CHECKS;\n",current_time);
		break;
		
	case CMD_REMOVE_HOST_ACKNOWLEDGEMENT:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] REMOVE_HOST_ACKNOWLEDGEMENT;%s\n",current_time,host_name);
		break;
		
	case CMD_REMOVE_SVC_ACKNOWLEDGEMENT:
		snprintf(command_buffer,sizeof(command_buffer)-1,"[%lu] REMOVE_SVC_ACKNOWLEDGEMENT;%s;%s\n",current_time,host_name,service_desc);
		break;
		
	default:
		return ERROR;
	        }

	/* make sure command buffer is terminated */
	command_buffer[sizeof(command_buffer)-1]='\x0';

	/* write the command to the command file */
	result=write_command_to_file(command_buffer);

	return result;
        }


/* write a command entry to the command file */
int write_command_to_file(char *cmd){
	FILE *fp;
	struct flock lock;

 	/* open the command for appending */
	fp=fopen(command_file,"a");
	if(fp==NULL){

		printf("<P><DIV CLASS='errorMessage'>Error: Could not open command file '%s' for update!</DIV></P>\n",command_file);

		printf("<P><DIV CLASS='errorDescription'>This may be due to one of two problems:</DIV></P>\n");
		printf("<P><DIV CLASS='errorDescription'><OL>\n");
		printf("<LI>The permissions on the external command file and/or directory may be incorrect.  Read the FAQs on how to setup proper permissions.\n");
		printf("<LI>Another process may have a lock on the command file which is preventing me from modifying the file.\n");
		printf("If you believe this is the case, try hitting the RELOAD button on your browser to resubmit the command.\n");
		printf("</OL></DIV></P>\n");

		return ERROR;
	        }

	/* place a write lock on the command file... */
	lock.l_type=F_WRLCK;
	lock.l_start=0;
	lock.l_whence=SEEK_SET;
	lock.l_len=0;
	
	/* don't block while trying to get a lock... */
	if(fcntl(fileno(fp),F_SETLK,&lock)<0){

		printf("<P><DIV CLASS='errorMessage'>Error: Could not lock command file '%s' for writing!</DIV></P>\n",command_file);

		printf("<P><DIV CLASS='errorDescription'>Try clicking the RELOAD button on your browser to resubmit the command.  ");
		printf("If you continue to have problems, check the permission on the external command file/directory.</DIV></P>\n");

		fclose(fp);

		return ERROR;
                }

	/* write the command to file */
	fputs(cmd,fp);

	/* clear write lock on the command file... */
	lock.l_type=F_UNLCK;
	lock.l_start=0;
	lock.l_whence=SEEK_SET;
	lock.l_len=0;
		
	if(fcntl(fileno(fp),F_SETLK,&lock)<0)
		printf("<P><DIV CLASS='errorMessage'>Warning: I couldn't clear the write lock on command file '%s'!</DIV></P>\n",command_file);

	fclose(fp);

	return OK;
        }


/* strips out semicolons from comment data */
void clean_comment_data(char *buffer){
	int x;
	int y;

	y=(int)strlen(buffer);
	
	for(x=0;x<y;x++){
		if(buffer[x]==';')
			buffer[x]=' ';
	        }

	return;
        }


/* display information about a command */
void show_command_help(cmd){

	printf("<DIV ALIGN=CENTER CLASS='descriptionTitle'>Command Description</DIV>\n");
	printf("<TABLE BORDER=1 CELLSPACING=0 CELLPADDING=0 CLASS='commandDescription'>\n");
	printf("<TR><TD CLASS='commandDescription'>\n");

	/* decide what information to print out... */
	switch(cmd){

	case CMD_ADD_HOST_COMMENT:
		printf("This command is used to add a comment for the specified host.  If you work with other administrators, you may find it useful to share information about a host\n");
		printf("that is having problems if more than one of you may be working on it.  If you do not check the 'persistent' option, the comment will be automatically be deleted\n");
		printf("the next time NetSaint is restarted.\n");
		break;
		
	case CMD_ADD_SVC_COMMENT:
		printf("This command is used to add a comment for the specified service.  If you work with other administrators, you may find it useful to share information about a host\n");
		printf("or service that is having problems if more than one of you may be working on it.  If you do not check the 'persistent' option, the comment will automatically be\n");
		printf("deleted the next time NetSaint is restarted.\n");
		break;

	case CMD_DEL_HOST_COMMENT:
		printf("This command is used to delete a specific host comment.\n");
		break;
		
	case CMD_DEL_SVC_COMMENT:
		printf("This command is used to delete a specific service comment.\n");
		break;
		
	case CMD_DELAY_HOST_NOTIFICATION:
		printf("This command is used to delay the next problem notification that is sent out for the specified host.  The notification delay will be disregarded if\n");
		printf("the host changes state before the next notification is scheduled to be sent out.  This command has no effect if the host is currently UP.\n");
		break;

	case CMD_DELAY_SVC_NOTIFICATION:
		printf("This command is used to delay the next problem notification that is sent out for the specified service.  The notification delay will be disregarded if\n");
		printf("the service changes state before the next notification is scheduled to be sent out.  This command has no effect if the service is currently in an OK state.\n");
		break;

	case CMD_DELAY_SVC_CHECK:
		printf("This command is used to delay the next scheduled check of the specified service.  NetSaint will re-queue the service and will not not check it again until the number\n");
		printf("of minutes you specified with the <b><i>check delay</i></b> option has elapsed from the time the command is committed.  Specifying a value of 0 for the delay value\n");
		printf("is equivalent to scheduling an immediate service check.\n");
		break;

	case CMD_IMMEDIATE_SVC_CHECK:
		printf("This command will schedule an immediate check of the specified service.  Note that the check is <i>scheduled</i> immediately, not necessary executed immediately.  If NetSaint\n");
		printf("has fallen behind in its scheduling queue, it will check services that were queued prior to this one.\n");
		break;

	case CMD_ENABLE_SVC_CHECK:
		printf("This command is used to re-enable a service check that has been disabled.  Once a service check is enabled, NetSaint will begin to monitor the service as usual.  If the service\n");
		printf("has recovered from a problem that was detected before the check was disabled, contacts might be notified of the recovery.\n");
		break;
		
	case CMD_DISABLE_SVC_CHECK:
		printf("This command is used to disable a service check.  When a service is disabled NetSaint will not monitor the service.  Doing this will prevent any notifications being sent out for\n");
		printf("the specified service while it is disabled.  In order to have NetSaint check the service in the future you will have to re-enable the service.\n");
		printf("Note that disabling service checks may not necessarily prevent notifications from being sent out about the host which those services are associated with.\n");
		break;
		
	case CMD_ENTER_STANDBY_MODE:
		printf("This command is used to put NetSaint in standby mode.  By supplying a non-zero value for the <b><i>execution delay</i></b> option you can schedule a delayed mode change.\n");
		printf("Specifying a value of 0 for the delay will cause NetSaint to change modes immediately upon processing the request.  In standby mode NetSaint will continue to monitor all hosts\n");
		printf("and services that you have defined, but it will prevent any notifications from being sent out.  Contrast this with active mode, where NetSaint sends out notifications when necessary.\n");
		break;
		
	case CMD_ENTER_ACTIVE_MODE:
		printf("This command is used to put NetSaint in active mode.  By supplying a non-zero value for the <b><i>execution delay</i></b> option you can schedule a delayed mode change.\n");
		printf("Specifying a value of 0 for the delay will cause NetSaint to change modes immediately upon processing the request.  In active mode NetSaint will send out notifications to contacts\n");
		printf("if necessary.  Contrast this with standby mode, where NetSaint will prevent any notifications from being sent to contacts.\n");
		break;
		
	case CMD_SHUTDOWN_PROCESS:
		printf("This command is used to shutdown the NetSaint process.  By supplying a non-zero value for the <b><i>execution delay</i></b> option you can schedule a delayed shutdown.\n");
		printf("Specifying a value of 0 for the delay will cause NetSaint to shut down immediately upon processing the request.  Note: Once the NetSaint has been shutdown, it\n");
		printf("cannot be restarted via the web interface!\n");
		break;

	case CMD_RESTART_PROCESS:
		printf("This command is used to restart the NetSaint process.  By supplying a non-zero value for the <b><i>execution delay</i></b> variable you can schedule a delayed restart.\n");
		printf("Specifying a value of 0 for the delay will cause NetSaint to restart immediately upon processing the request.  Executing a restart command is equivalent to sending\n");
		printf("the process a HUP signal.  All information will be flushed from memory, the configuration files will be re-read, and NetSaint will start monitoring with the new configuration information.\n");
		break;

	case CMD_ENABLE_HOST_SVC_CHECKS:
		printf("This command is used to enable all service checks associated with the specified host.\n");
		break;
		
	case CMD_DISABLE_HOST_SVC_CHECKS:
		printf("This command is used to disable all service checks associated with the specified host.  When a service is disabled NetSaint will not monitor the service.  Doing this will prevent any notifications being sent out for\n");
		printf("the specified service while it is disabled.  In order to have NetSaint check the service in the future you will have to re-enable the service.\n");
		printf("Note that disabling service checks may not necessarily prevent notifications from being sent out about the host which those services are associated with.\n");
		break;
		
	case CMD_DELAY_HOST_SVC_CHECKS:
		printf("This command is used to delay the next scheduled check of all services on the specified host.  NetSaint will re-queue the services and will not not check them again until the number\n");
		printf("of minutes you specified with the <b><i>check delay</i></b> option has elapsed from the time the command is committed.  Specifying a value of 0 for the delay value\n");
		printf("is equivalent to scheduling an immediate check of all the services.\n");
		break;

	case CMD_IMMEDIATE_HOST_SVC_CHECKS:
		printf("This command will schedule an immediate check of all service on the specified host.  Note that the checks are <i>scheduled</i> immediately, not necessary executed immediately.  If NetSaint\n");
		printf("has fallen behind in its scheduling queue, it will check services that were queued prior to these services.\n");
		break;

	case CMD_DEL_ALL_HOST_COMMENTS:
		printf("This command is used to delete all comments associated with the specified host.\n");
		break;
		
	case CMD_DEL_ALL_SVC_COMMENTS:
		printf("This command is used to delete all comments associated with the specified service.\n");
		break;

	case CMD_ENABLE_SVC_NOTIFICATIONS:
		printf("This command is used to enable notifications for the specified service.  Notifications will only be sent out for the\n");
		printf("service state types you defined in your service definition.\n");
		break;

	case CMD_DISABLE_SVC_NOTIFICATIONS:
		printf("This command is used to prevent notifications from being sent out for the specified service.  You will have to re-enable notifications\n");
		printf("for this service before any alerts can be sent out in the future.\n");
		break;

	case CMD_ENABLE_HOST_NOTIFICATIONS:
		printf("This command is used to enable notifications for the specified host.  Notifications will only be sent out for the\n");
		printf("host state types you defined in your host definition.  Note that this command <i>does not</i> enable notifications\n");
		printf("for services associated with this host.\n");
		break;

	case CMD_DISABLE_HOST_NOTIFICATIONS:
		printf("This command is used to prevent notifications from being sent out for the specified host.  You will have to re-enable notifications for this host\n");
		printf("before any alerts can be sent out in the future.  Note that this command <i>does not</i> disable notifications for services associated with this host.\n");
		break;

	case CMD_ENABLE_ALL_NOTIFICATIONS_BEYOND_HOST:
		printf("This command is used to enable notifications for all hosts and services that lie \"beyond\" the specified host\n");
		printf("(from the view of NetSaint).\n");
		break;

	case CMD_DISABLE_ALL_NOTIFICATIONS_BEYOND_HOST:
		printf("This command is used to temporarily prevent notifications from being sent out for all hosts and services that lie\n");
		printf("\"beyone\" the specified host (from the view of NetSaint).\n");
		break;
		
	case CMD_ENABLE_HOST_SVC_NOTIFICATIONS:
		printf("This command is used to enable notifications for all services on the specified host.  Notifications will only be sent out for the\n");
		printf("service state types you defined in your service definition.\n");
		break;

	case CMD_DISABLE_HOST_SVC_NOTIFICATIONS:
		printf("This command is used to prevent notifications from being sent out for all services on the specified host.  You will have to re-enable notifications for\n");
		printf("all services associated with this host before any alerts can be sent out in the future.  This <i>does not</i> prevent notifications from being sent out about the host.\n");
		break;

	case CMD_ACKNOWLEDGE_HOST_PROBLEM:
		printf("This command is used to acknowledge a host problem.  When a host problem is acknowledged, future notifications about problems are temporarily disabled until the host changes state (i.e. recovers).\n");
		printf("Contacts for this host will receive a notification about the acknowledgement, so they are aware that someone is working on the problem.  Additionally, a comment will also be added to the host.\n");
		printf("Make sure to enter your name and fill in a brief description of what you are doing in the comment field.  If you would like the host comment to be retained between restarts of NetSaint, check\n");
		printf("the 'Persistent' checkbox.\n");
		break;

	case CMD_ACKNOWLEDGE_SVC_PROBLEM:
		printf("This command is used to acknowledge a service problem.  When a service problem is acknowledged, future notifications about problems are temporarily disabled until the service changes state (i.e. recovers).\n");
		printf("Contacts for this service will receive a notification about the acknowledgement, so they are aware that someone is working on the problem.  Additionally, a comment will also be added to the service.\n");
		printf("Make sure to enter your name and fill in a brief description of what you are doing in the comment field.  If you would like the service comment to be retained between restarts of NetSaint, check\n");
		printf("the 'Persistent' checkbox.\n");
		break;

	case CMD_START_EXECUTING_SVC_CHECKS:
		printf("This command is used to resume execution of service checks on a program-wide basis.  Individual services which are disabled will still not be checked.\n");
		break;

	case CMD_STOP_EXECUTING_SVC_CHECKS:
		printf("This command is used to temporarily stop NetSaint from executing any service checks.  This will have the side effect of preventing any notifications from being sent out (for any and all services and hosts).\n");
		printf("Service checks will not be executed again until you issue a command to resume service check execution.\n");
		break;

	case CMD_START_ACCEPTING_PASSIVE_SVC_CHECKS:
		printf("This command is used to make NetSaint start accepting passive service check results that it finds in the external command file\n");
		break;

	case CMD_STOP_ACCEPTING_PASSIVE_SVC_CHECKS:
		printf("This command is use to make NetSaint stop accepting passive service check results that it finds in the external command file.  All passive check results that are found will be ignored.\n");
		break;

	case CMD_ENABLE_PASSIVE_SVC_CHECKS:
		printf("This command is used to allow NetSaint to accept passive service check results that it finds in the external command file for this particular service.\n");
		break;

	case CMD_DISABLE_PASSIVE_SVC_CHECKS:
		printf("This command is used to make NetSaint stop accepting passive service check results that it finds in the external command file for this particular service.  All passive check results that are found for this service will be ignored.\n");
		break;

	case CMD_ENABLE_EVENT_HANDLERS:
		printf("This command is used to allow NetSaint to run host and service event handlers.\n");
		break;

	case CMD_DISABLE_EVENT_HANDLERS:
		printf("This command is used to temporarily prevent NetSaint from running any host or service event handlers.\n");
		break;

	case CMD_ENABLE_SVC_EVENT_HANDLER:
		printf("This command is used to allow NetSaint to run the service event handler for a particular service when necessary (if one is defined).\n");
		break;

	case CMD_DISABLE_SVC_EVENT_HANDLER:
		printf("This command is used to temporarily prevent NetSaint from running the service event handler for a particular service.\n");
		break;

	case CMD_ENABLE_HOST_EVENT_HANDLER:
		printf("This command is used to allow NetSaint to run the host event handler for a particular service when necessary (if one is defined).\n");
		break;

	case CMD_DISABLE_HOST_EVENT_HANDLER:
		printf("This command is used to temporarily prevent NetSaint from running the host event handler for a particular host.\n");
		break;

	case CMD_ENABLE_HOST_CHECK:
		printf("This command is used to enable checks of this host.\n");
		break;

	case CMD_DISABLE_HOST_CHECK:
		printf("This command is used to temporarily prevent NetSaint from checking the status of a particular host.  If NetSaint needs to check the status of this host, it will assume that it is in the same state that it was in before checks were disabled.\n");
		break;

	case CMD_START_OBSESSING_OVER_SVC_CHECKS:
		printf("This command is used to have NetSaint start obsessing over service checks.  Read the documentation on distributed monitoring for more information on this.\n");
		break;

	case CMD_STOP_OBSESSING_OVER_SVC_CHECKS:
		printf("This command is used to have NetSaint stop obsessing over service checks.  Read the documentation on distributed monitoring for more information on this.\n");
		break;

	case CMD_REMOVE_HOST_ACKNOWLEDGEMENT:
		printf("This commands is used to remove an acknowledgement for a particular host problem.  Once the acknowledgement is removed, notifications may start being\n");
		printf("sent out about the host problem.  Note: Removing the acknowledgement does <i>not</i> remove the host comment that was originally associated\n");
		printf("with the acknowledgement.  You'll have to remove that as well if that's what you want.\n");
		break;

	case CMD_REMOVE_SVC_ACKNOWLEDGEMENT:
		printf("This commands is used to remove an acknowledgement for a particular service problem.  Once the acknowledgement is removed, notifications may start being\n");
		printf("sent out about the service problem.  Note: Removing the acknowledgement does <i>not</i> remove the service comment that was originally associated\n");
		printf("with the acknowledgement.  You'll have to remove that as well if that's what you want.\n");
		break;

	default:
		printf("Sorry, but no information is available for this command.");
	        }

	printf("</TD></TR>\n");
	printf("</TABLE>\n");

	return;
        }
