/*****************************************************************************
 *
 * STATUSMAP.C - NetSaint Network Status Map CGI
 *
 * Copyright (c) 1999-2000 Ethan Galstad (netsaint@netsaint.org)
 * Last Modified:   07-07-2000
 *
 * Description:
 *
 * This CGI will dynamically create a map of all hosts that are being
 * monitored on your network.
 *
 * License:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *****************************************************************************/

#include "../common/config.h"
#include "../common/locations.h"
#include "../common/common.h"
#include "../common/objects.h"
#include "../common/statusdata.h"

#include "cgiutils.h"
#include "getcgi.h"
#include "auth.h"
#include "edata.h"

#include <gd.h>			/* Boutell's GD library function */
#include <gdfonts.h>		/* GD library small font definition */

extern int             refresh_rate;

#define IMAGE_OUTPUT_FILENAME	"statusmap.png"
#define NETSAINT_ICON_FILENAME	"logo.gd2"
#define UNKNOWN_ICON_FILENAME	"unknown.gd2"

extern char main_config_file[MAX_FILENAME_LENGTH];
extern char physical_images_path[MAX_FILENAME_LENGTH];
extern char url_images_path[MAX_FILENAME_LENGTH];
extern char url_logo_images_path[MAX_FILENAME_LENGTH];
extern char url_stylesheets_path[MAX_FILENAME_LENGTH];
extern char log_file[MAX_FILENAME_LENGTH];

extern hostgroup *hostgroup_list;
extern hoststatus *hoststatus_list;
extern host *host_list;
extern servicestatus *servicestatus_list;

#define DEFAULT_NODE_WIDTH		40
#define DEFAULT_NODE_HEIGHT		65
#define DEFAULT_HORIZONTAL_SPACING	40
#define DEFAULT_VERTICAL_SPACING	75

#define CREATE_HTML	0
#define CREATE_IMAGE	1


void draw_secondary_child_links(void);
int get_host_draw_data(host *);
void draw_host_and_child_links(host *,int,int);
void draw_host(host *,int,int);
void draw_child_link(host *,int,int,int,int,int);
void draw_line(int,int,int,int,int);
void draw_dashed_line(int,int,int,int,int);
int draw_status_map(host *);
void display_map(void);

void write_popup_code(void);
void write_popup_text(host *);

void document_header(int);
void document_footer(void);
int process_cgivars(void);

char image_input_file[MAX_FILENAME_LENGTH];
char physical_logo_images_path[MAX_FILENAME_LENGTH];

authdata current_authdata;

int top_number_of_immediate_children=0;
int top_max_child_drawing_width=0;
int top_max_child_depth;

int vertical_spacing=DEFAULT_VERTICAL_SPACING;
int use_vertical_spacing=FALSE;
int horizontal_spacing=DEFAULT_HORIZONTAL_SPACING;
int use_horizontal_spacing=FALSE;
int node_width=DEFAULT_NODE_WIDTH;
int use_node_width=FALSE;
int node_height=DEFAULT_NODE_HEIGHT;
int use_node_height=FALSE;

int image_width=0;
int image_height=0;
int max_image_width=0;
int use_max_width=FALSE;
int max_image_height=0;
int use_max_height=FALSE;
int scale_image=FALSE;
double scaling_factor=1.0L;

int show_secondary_links=TRUE;

char *host_name="all";
int show_all_hosts=TRUE;
int create_type=CREATE_HTML;

gdImagePtr unknown_logo_image=0;
gdImagePtr status_image=0;
int color_white=0;
int color_black=0;
int color_red=0;
int color_green=0;
int color_blue=0;
int color_yellow=0;
int color_orange=0;
int color_grey=0;

FILE *unknown_logo_file;

int embedded=FALSE;
int display_header=TRUE;





int main(int argc, char **argv){
	int result;
	char temp_buffer[MAX_INPUT_BUFFER];


	/* get the arguments passed in the URL */
	process_cgivars();

	/* reset internal variables */
	reset_cgi_vars();

	/* read the CGI configuration file */
	result=read_cgi_config_file(DEFAULT_CGI_CONFIG_FILE);
	if(result==ERROR){
		document_header(FALSE);
		if(create_type==CREATE_HTML)
			printf("<P><DIV class='errorMessage'>Error: Could not open CGI configuration file '%s' for reading!</DIV></P>\n",DEFAULT_CGI_CONFIG_FILE);
		document_footer();
		return ERROR;
	        }

	document_header(TRUE);

	/* read the main configuration file */
	result=read_main_config_file(main_config_file);
	if(result==ERROR){
		if(create_type==CREATE_HTML)
			printf("<P><DIV class='errorMessage'>Error: Could not open main configuration file '%s' for reading!</DIV></P>\n",main_config_file);
		document_footer();
		return ERROR;
	        }

	/* read all object configuration data */
	result=read_all_object_configuration_data(main_config_file,READ_HOSTGROUPS|READ_CONTACTGROUPS|READ_HOSTS|READ_SERVICES);
	if(result==ERROR){
		if(create_type==CREATE_HTML)
			printf("<P><DIV class='errorMessage'>Error: Could not read some or all object configuration data!</DIV></P>\n");
		document_footer();
		return ERROR;
                }

	/* read all status data */
	result=read_all_status_data(main_config_file,READ_PROGRAM_STATUS|READ_HOST_STATUS|READ_SERVICE_STATUS);
	if(result==ERROR){
		printf("<P><DIV class='errorMessage'>Error: Could not read host and service status information!</DIV></P>\n");
		document_footer();
		free_memory();
		return ERROR;
                }


	/* get authentication information */
	get_authentication_information(&current_authdata);

	/* read in extended host information */
	read_extended_object_config_data(DEFAULT_CGI_CONFIG_FILE);


	if(create_type==CREATE_HTML && display_header==TRUE){

		/* begin top table */
		printf("<table border=0 width=100%% cellspacing=0 cellpadding=0>\n");
		printf("<tr>\n");

		/* left column of the first row */
		printf("<td align=left valign=top width=50%%>\n");

		if(show_all_hosts==TRUE)
			snprintf(temp_buffer,sizeof(temp_buffer)-1,"Network Map For All Hosts");
		else
			snprintf(temp_buffer,sizeof(temp_buffer)-1,"Network Map For Host <I>%s</I>",host_name);
		temp_buffer[sizeof(temp_buffer)-1]='\x0';
		display_info_table(temp_buffer,TRUE,&current_authdata);

		printf("<TABLE BORDER=1 CELLPADDING=0 CELLSPACING=0 CLASS='linkBox'>\n");
		printf("<TR><TD CLASS='linkBox'>\n");

		if(show_all_hosts==FALSE)
			printf("<a href='%s?host=%s'>View Status Detail For This Host</a><BR>\n",STATUS_CGI,url_encode(host_name));
		printf("<a href='%s?host=all'>View Status Detail For All Hosts</a><BR>\n",STATUS_CGI);
		printf("<a href='%s?hostgroup=all'>View Status Overview For All Hosts</a>\n",STATUS_CGI);

		printf("</TD></TR>\n");
		printf("</TABLE>\n");

		printf("</td>\n");

		/* right hand column of top row */
		printf("<td align=right valign=top width=50%%>\n");

		printf("<table border=0 CLASS='optBox'>\n");
		printf("<tr><td valign=top>\n");
		printf("<form method=\"GET\" action=\"%s\">\n",STATUSMAP_CGI);
		printf("<input type='hidden' name='host' value='%s'>\n",host_name);

		printf("<table border=0>\n");
		printf("<tr><td CLASS='optBoxItem'>\n");
		printf("Horizontal node spacing:<br>\n");
		printf("<select name='hspacing'>\n");
		printf("<option value=%d %s>%d\n",DEFAULT_HORIZONTAL_SPACING,(horizontal_spacing==DEFAULT_HORIZONTAL_SPACING)?"selected":"",DEFAULT_HORIZONTAL_SPACING);
		printf("<option value=10 %s>10\n",(horizontal_spacing==10)?"selected":"");
		printf("<option value=20 %s>20\n",(horizontal_spacing==20)?"selected":"");
		printf("<option value=60 %s>60\n",(horizontal_spacing==60)?"selected":"");
		printf("</select>\n");
		printf("</td></tr>\n");
		printf("<tr><td CLASS='optBoxItem'>\n");
		printf("Vertical node spacing:<br>\n");
		printf("<select name='vspacing'>\n");
		printf("<option value=%d %s>%d\n",DEFAULT_VERTICAL_SPACING,(vertical_spacing==DEFAULT_VERTICAL_SPACING)?"selected":"",DEFAULT_VERTICAL_SPACING);
		printf("<option value=25 %s>25\n",(vertical_spacing==25)?"selected":"");
		printf("<option value=50 %s>50\n",(vertical_spacing==50)?"selected":"");
		printf("<option value=100 %s>100\n",(vertical_spacing==100)?"selected":"");
		printf("</select>\n");
		printf("</td></tr>\n");
		printf("</table>\n");

		printf("</td><td valign=top>\n");

		printf("<table border=0>\n");
		printf("<tr><td CLASS='optBoxItem'>\n");
		printf("Max image width:<br>\n");
		printf("<select name='maxwidth'>\n");
		printf("<option value=0 %s>Unlimited\n",(max_image_width==0)?"selected":"");
		printf("<option value=600 %s>600 pixels\n",(max_image_width==600)?"selected":"");
		printf("<option value=800 %s>800 pixels\n",(max_image_width==800)?"selected":"");
		printf("<option value=1000 %s>1000 pixels\n",(max_image_width==1000)?"selected":"");
		printf("<option value=1200 %s>1200 pixels\n",(max_image_width==1200)?"selected":"");
		printf("<option value=1400 %s>1400 pixels\n",(max_image_width==1400)?"selected":"");
		printf("</select>\n");
		printf("</td></tr>\n");
		printf("<tr><td CLASS='optBoxItem'>\n");
		printf("Max image height:<br>\n");
		printf("<select name='maxheight'>\n");
		printf("<option value=0 %s>Unlimited\n",(max_image_height==0)?"selected":"");
		printf("<option value=300 %s>300 pixels\n",(max_image_height==300)?"selected":"");
		printf("<option value=400 %s>400 pixels\n",(max_image_height==400)?"selected":"");
		printf("<option value=600 %s>600 pixels\n",(max_image_height==600)?"selected":"");
		printf("<option value=800 %s>800 pixels\n",(max_image_height==800)?"selected":"");
		printf("<option value=1000 %s>1000 pixels\n",(max_image_height==1000)?"selected":"");
		printf("</select>\n");
		printf("<input type='submit' value='Update'>\n");
		printf("</td></tr>\n");
		printf("</table>\n");

		printf("</form>\n");
		printf("</td></tr>\n");
		printf("</table>\n");


		printf("</td>\n");
	
		/* end of top table */
		printf("</tr>\n");
		printf("</table>\n");
	        }


	/* display the network map... */
	display_map();

	document_footer();

	/* free all allocated memory */
	free_memory();
	free_extended_data();

	return OK;
        }



void document_header(int use_stylesheet){
	char date_time[48];
	time_t current_time;
	time_t expire_time;

	if(create_type==CREATE_HTML){
		printf("Cache-Control: no-store\n");
		printf("Pragma: no-cache\n");
		printf("Refresh: %d\n",refresh_rate);

		time(&current_time);
		get_expire_time_string(&current_time,date_time,sizeof(date_time));
		printf("Last-Modified: %s\n",date_time);

		expire_time=0L;
		get_expire_time_string(&expire_time,date_time,sizeof(date_time));
		printf("Expires: %s\n",date_time);

		printf("Content-Type: text/html\n\n");

		if(embedded==TRUE)
			return;

		printf("<html>\n");
		printf("<head>\n");
		printf("<title>\n");
		printf("Network Map\n");
		printf("</title>\n");

		if(use_stylesheet==TRUE)
			printf("<LINK REL='stylesheet' TYPE='text/css' HREF='%s%s'>\n",url_stylesheets_path,STATUSMAP_CSS);

		/* write JavaScript code for popup window */
		write_popup_code();

		printf("</head>\n");
		
		printf("<body CLASS='statusmap'>\n");

		printf("<div id=\"popup\" style=\"position:absolute; z-index:1; visibility: hidden\"></div>\n");
	        }

	else{
		printf("Cache-Control: no-store\n");
		printf("Pragma: no-cache\n");

		time(&current_time);
		get_expire_time_string(&current_time,date_time,sizeof(date_time));
		printf("Last-Modified: %s\n",date_time);

		expire_time=(time_t)0L;
		get_expire_time_string(&expire_time,date_time,sizeof(date_time));
		printf("Expires: %s\n",date_time);

		printf("Content-Type: image/png\n\n");
	        }

	return;
        }


void document_footer(void){

	if(embedded==TRUE)
		return;

	if(create_type==CREATE_HTML){
		printf("</body>\n");
		printf("</html>\n");
	        }

	return;
        }



int process_cgivars(void){
	char **variables;
	int error=FALSE;
	int x;

	variables=getcgivars();

	for(x=0;variables[x]!=NULL;x++){

		/* do some basic length checking on the variable identifier to prevent buffer overflows */
		if(strlen(variables[x])>=MAX_INPUT_BUFFER-1){
			x++;
			continue;
		        }


		/* we found the host argument */
		else if(!strcmp(variables[x],"host")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			host_name=(char *)malloc(strlen(variables[x])+1);
			if(host_name==NULL)
				host_name="all";
			else
				strcpy(host_name,variables[x]);

			if(!strcmp(host_name,"all"))
				show_all_hosts=TRUE;
			else
				show_all_hosts=FALSE;
		        }

		/* we found the node width argument */
		else if(!strcmp(variables[x],"nodewidth")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			node_width=atoi(variables[x]);
			if(node_width<=0)
				node_width=DEFAULT_NODE_WIDTH;
			use_node_width=TRUE;
		        }

		/* we found the node height argument */
		else if(!strcmp(variables[x],"nodeheight")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			node_height=atoi(variables[x]);
			if(node_height<=0)
				node_height=DEFAULT_NODE_HEIGHT;
			use_node_height=TRUE;
		        }

		/* we found the horizontal spacing argument */
		else if(!strcmp(variables[x],"hspacing")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			horizontal_spacing=atoi(variables[x]);
			if(horizontal_spacing<=0)
				horizontal_spacing=DEFAULT_HORIZONTAL_SPACING;
			use_horizontal_spacing=TRUE;
		        }

		/* we found the vertical spacing argument */
		else if(!strcmp(variables[x],"vspacing")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			vertical_spacing=atoi(variables[x]);
			if(vertical_spacing<=0)
				vertical_spacing=DEFAULT_VERTICAL_SPACING;
			use_vertical_spacing=TRUE;
		        }

		/* we found the max image width argument */
		else if(!strcmp(variables[x],"maxwidth")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			max_image_width=atoi(variables[x]);
			if(max_image_width<=0){
				max_image_width=1;
				use_max_width=FALSE;
			        }
			else{
				scale_image=TRUE;
				use_max_width=TRUE;
			        }
		        }

		/* we found the max image height argument */
		else if(!strcmp(variables[x],"maxheight")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			max_image_height=atoi(variables[x]);
			if(max_image_height<=0){
				max_image_height=1;
				use_max_height=FALSE;
			        }
			else{
				scale_image=TRUE;
				use_max_height=TRUE;
			        }
		        }

		/* we found the image creation option */
		else if(!strcmp(variables[x],"createimage")){
			create_type=CREATE_IMAGE;
		        }

		/* we found the option for no secondary links */
		else if(!strcmp(variables[x],"nosecondarylinks")){
			show_secondary_links=FALSE;
		        }

		/* we found the embed option */
		else if(!strcmp(variables[x],"embedded"))
			embedded=TRUE;

		/* we found the noheader option */
		else if(!strcmp(variables[x],"noheader"))
			display_header=FALSE;
	        }

	return error;
        }



/* top-level map generation... */
void display_map(void){
	host *temp_host=NULL;


	if(create_type==CREATE_IMAGE){

		/* get the path where we will be reading logo images from (GD2 format)... */
		snprintf(physical_logo_images_path,sizeof(physical_logo_images_path),"%slogos/",physical_images_path);
		physical_logo_images_path[sizeof(physical_logo_images_path)-1]='\x0';

		/* load the unknown icon to use for hosts that don't have pretty images associated with them... */
		snprintf(image_input_file,sizeof(image_input_file),"%s%s",physical_logo_images_path,UNKNOWN_ICON_FILENAME);
		image_input_file[sizeof(image_input_file)-1]='\x0';

		unknown_logo_file=fopen(image_input_file,"rb");
		if(unknown_logo_file!=NULL){
			unknown_logo_image=gdImageCreateFromGd2(unknown_logo_file);
			fclose(unknown_logo_file);
	                }
	        }

	if(create_type==CREATE_HTML)
		printf("<map name='statusmap'>\n");

	/* create the network image and print out the imagemap coordinates... */
	if(show_all_hosts==TRUE){
		temp_host=NULL;
		draw_status_map(temp_host);
	        }
	else{
		temp_host=find_host(host_name,NULL);
		if(temp_host==NULL){
			show_all_hosts=TRUE;
			draw_status_map(NULL);
		        }
		else
			draw_status_map(temp_host);
                }

	if(create_type==CREATE_IMAGE){

		/* free memory allocated to the unknown image */
		if(unknown_logo_image)
			gdImageDestroy(unknown_logo_image);
	        }

	if(create_type==CREATE_HTML){

		printf("</map>\n");

		/* display raw and scaled image sizes - more for debugging information than anything... */
		printf("<P><DIV ALIGN=CENTER CLASS='imageInfo'>\n");
		printf("( Raw Image Size: %dx%d pixels, ",image_width,image_height);
		printf("Scaled Image Size: %dx%d pixels )",(int)((double)image_width*scaling_factor),(int)((double)image_height*scaling_factor));
		printf("</DIV></P>\n");

		if(show_all_hosts==FALSE){
			
			/* display a link to go "up" the dependency tree (using the "primary" parent)... */
			printf("<P>\n");
			printf("<A HREF='%s?host=%s",STATUSMAP_CGI,(temp_host->parent_hosts==NULL)?"all":url_encode(temp_host->parent_hosts->host_name));

			if(use_node_width==TRUE)
				printf("&nodewidth=%d",node_width);
			if(use_node_height==TRUE)
				printf("&nodeheight=%d",node_height);
			if(use_horizontal_spacing==TRUE)
				printf("&hspacing=%d",horizontal_spacing);
			if(use_vertical_spacing==TRUE)
				printf("&vspacing=%d",vertical_spacing);
			if(use_max_width==TRUE)
				printf("&maxwidth=%d",max_image_width);
			if(use_max_height==TRUE)
				printf("&maxheight=%d",max_image_height);

			printf("'><IMG SRC='%s%s' BORDER=0 ALT='Move up one host dependency level'></A>\n",url_images_path,PARENT_TRAVERSAL_ICON);
			printf("</P>\n");
		        }

		/* write the URL location for the image we just generated - the web browser will come and get it... */
		printf("<P><DIV ALIGN=center>\n");
		printf("<img src='%s?host=%s&createimage",STATUSMAP_CGI,url_encode(host_name));

		if(use_node_width==TRUE)
			printf("&nodewidth=%d",node_width);
		if(use_node_height==TRUE)
			printf("&nodeheight=%d",node_height);
		if(use_horizontal_spacing==TRUE)
			printf("&hspacing=%d",horizontal_spacing);
		if(use_vertical_spacing==TRUE)
			printf("&vspacing=%d",vertical_spacing);
		if(use_max_width==TRUE)
			printf("&maxwidth=%d",max_image_width);
		if(use_max_height==TRUE)
			printf("&maxheight=%d",max_image_height);

		printf("' width=%d height=%d border=0 name='statusimage' useMap='#statusmap'>\n",(int)((double)image_width*scaling_factor),(int)((double)image_height*scaling_factor));
		printf("</DIV></P>\n");
	        }

	return;
        }



int draw_status_map(host *hst){
	FILE *image_file=NULL;
	int max_child_drawing_width;
	double min_scaling_factor=1.0L;
	double temp_scaling_factor=1.0L;
	int max_child_depth;

	if(create_type==CREATE_IMAGE){

		/* use STDOUT for writing the image data... */
		image_file=stdout;
	        }

	/* calculate drawing info */
	get_host_draw_data(hst);

	/* get the top level maximum child drawing width */
	if(hst==NULL)
		max_child_drawing_width=top_max_child_drawing_width;
	else
		max_child_drawing_width=hst->max_child_drawing_width;

	/* get the max child depth to use */
	if(hst==NULL)
		max_child_depth=top_max_child_depth;
	else
		max_child_depth=hst->max_child_depth;

	/* calculate the width of the image we have to create */
	image_width=(max_child_drawing_width*node_width)+((max_child_drawing_width-1)*horizontal_spacing)+horizontal_spacing+2;

	/* calculate the height of the image we have to create */
	image_height=(max_child_depth*node_height)+((max_child_depth-1)*vertical_spacing)+3;

	/* decide if we need to scale the image */
	if(scale_image==TRUE){

		/* if the user specified a max width and we're over it, compute the scaling factor */
		if(use_max_width && image_width>max_image_width){
			temp_scaling_factor=(double)((double)max_image_width/(double)image_width);
			if(temp_scaling_factor<min_scaling_factor)
				min_scaling_factor=temp_scaling_factor;
		        }

		/* if the user specified a max height and we're over it, compute the scaling factor */
		if(use_max_height==TRUE && image_height>max_image_height){
			temp_scaling_factor=(double)((double)max_image_height/(double)image_height);
			if(temp_scaling_factor<min_scaling_factor)
				min_scaling_factor=temp_scaling_factor;
		        }

		/* check our scaling factor sanity */
		if(min_scaling_factor>=1.0L){
			scale_image=FALSE;
			scaling_factor=1.0L;
		        }
		else
			scaling_factor=min_scaling_factor;
	        }


	if(create_type==CREATE_IMAGE){

		/* allocate buffer for storing image */
		status_image=gdImageCreate(image_width,image_height);
		if(status_image==NULL){
			printf("Error: Could not allocate memory for image\n");
			return ERROR;
	                }

		/* allocate colors used for drawing */
		color_white=gdImageColorAllocate(status_image,255,255,255);
		color_black=gdImageColorAllocate(status_image,0,0,0);
		color_grey=gdImageColorAllocate(status_image,128,128,128);
		color_red=gdImageColorAllocate(status_image,255,0,0);
		color_green=gdImageColorAllocate(status_image,0,175,0);
		color_blue=gdImageColorAllocate(status_image,0,0,255);
		color_yellow=gdImageColorAllocate(status_image,255,255,0);
		color_orange=gdImageColorAllocate(status_image,255,100,25);

		/* set transparency index */
		gdImageColorTransparent(status_image,color_white);

		/* make sure the graphic is interlaced */
		gdImageInterlace(status_image,1);
	        }

	/* start drawing all hosts below and including the "top" host */
	draw_host_and_child_links(hst,image_width/2,1);

	if(create_type==CREATE_IMAGE){

		/* draw secondary parent/child links */
		if(show_secondary_links==TRUE)
			draw_secondary_child_links();

		/* write the image to file */
		gdImagePng(status_image,image_file);

		/* free memory allocated to image */
		gdImageDestroy(status_image);
	        }

	return OK;
        }



/* gets data necessary for drawing the host and its child hosts */
int get_host_draw_data(host *hst){
	int max_width=0;
	int temp_width=0;
	int children=0;
	host *temp_host;
	int max_depth=0;

	/* get the max depth of child hosts for this host */
	max_depth=get_max_host_child_depth(hst);
	if(hst==NULL)
		top_max_child_depth=max_depth;
	else
		hst->max_child_depth=max_depth;

	/* get the number of immediate children to this host */
	children=number_of_immediate_child_hosts(hst);
	if(hst==NULL)
		top_number_of_immediate_children=children;
	else
		hst->number_of_immediate_children=children;

	/* if host has no children, max drawing width is one unit */
	if(children==0)
		max_width=1;

	/* else calculate max width of all children hosts */
	else for(temp_host=host_list;temp_host!=NULL;temp_host=temp_host->next){

		if(is_host_primary_immediate_child_of_host(hst,temp_host)){

			temp_width=get_host_draw_data(temp_host);
			max_width+=temp_width;
			}
		}

	if(max_width==0)
		max_width=1;

	if(hst==NULL)
		top_max_child_drawing_width=max_width;
	else
		hst->max_child_drawing_width=max_width;

	return max_width;
	}



/* draws a host and any child links */
void draw_host_and_child_links(host *hst,int x,int y){
	host *temp_host;
	int has_children=0;
	int current_child;
	int number_of_immediate_children;
	int max_child_drawing_width;
	int this_draw_width;
	int x1, xx;
	int y0, y1;
	int link_color;
	hoststatus *temp_status;

	/* draw the host (logo, name, and status) */
	draw_host(hst,x,y);

	/* does the host have any children? */
	if(hst==NULL){
		if(top_number_of_immediate_children>0)
			has_children=1;
		else
			has_children=0;
	        }
	else if(hst->number_of_immediate_children>0)
		has_children=1;
	else
		has_children=0;

	/* if the host has children, draw the liks to the child hosts.. */
	if(has_children){

		/* the default child link color is black */
		link_color=color_black;

		/* change the child link color if this host is not up... */
		if(hst!=NULL){

			/* find the status entry for this host */
			temp_status=find_hoststatus(hst->name);

			/* the link to the children should be red if this parent host is either down or unreachable */
			if(temp_status!=NULL){
				if(temp_status->status==HOST_UNREACHABLE || temp_status->status==HOST_DOWN)
					link_color=color_red;
		                 }
		        }

		/* get the number of immediate children and the max child drawing width for this host */
		if(hst==NULL){
			number_of_immediate_children=top_number_of_immediate_children;
			max_child_drawing_width=top_max_child_drawing_width;
		        }
		else{
			number_of_immediate_children=hst->number_of_immediate_children;
			max_child_drawing_width=hst->max_child_drawing_width;
		        }


		/* calculate starting points for drawing... */
		y0=y+node_height;
		y1=y0+vertical_spacing;
		x1=x-(((max_child_drawing_width*node_width)+((max_child_drawing_width-1)*horizontal_spacing))/2);
		xx=x1;

		/* the first potential child is the first host in the host list... */
		temp_host=host_list;

		/* draw the links to all child hosts... */
		for(current_child=1;current_child<=number_of_immediate_children;current_child++){

			/* find the next child host - never go back to the head of the list or we'll have duplicates... */
			for(;temp_host!=NULL;temp_host=temp_host->next){
				if(is_host_primary_immediate_child_of_host(hst,temp_host))
					break;
				}
			
			/* if for some reason we're at the end of the host list, exit draw loop */
			if(temp_host==NULL)
				break;

			/* calculate the drawing width for this host... */
			this_draw_width=((temp_host->max_child_drawing_width*node_width)+((temp_host->max_child_drawing_width-1)*horizontal_spacing))/2;
			xx+=this_draw_width;

			if(create_type==CREATE_IMAGE){

				/* draw the link to the child host */
				draw_child_link(temp_host,x,y0,xx,y1,link_color);
			        }

			/* draw the child host and its child links */
			draw_host_and_child_links(temp_host,xx,y1);

			/* update the drawing coordinates... */
			xx+=this_draw_width;
			xx+=horizontal_spacing;

			/* go to the next host in the host list - its a potential child... */
			temp_host=temp_host->next;
			}
		}

	return;
	}



/* draws a host */
void draw_host(host *hst,int x,int y){
	int x1=0;
	int x2=0;
	int y2=0;
	int centerx=0;
	int centery=0;
	int color=color_black;
	FILE *logo_file=NULL;
	gdImagePtr logo_image;
	int string_width=0;
	int string_height=0;
	char temp_buffer[MAX_INPUT_BUFFER];
	int status_color;
	int has_image=0;
	hostextinfo *temp_hostextinfo=NULL;
	hoststatus *temp_status=NULL;
	int authorized_for_host=FALSE;

	/* treat the top level "host" differently and just place the NetSaint logo there */
	if(hst==NULL){

		if(create_type==CREATE_HTML){

			/* get upper left hand corner of bounding box */
			x1=x-(node_width/2);
			x2=x1+node_width;
			y2=y+node_height;

			/* print out coordinates for the imagemap in the HTML code... */
			if(scale_image==TRUE)
				printf("<AREA shape='rect' coords='%d,%d,%d,%d' ",(int)((double)x1*scaling_factor),(int)((double)y*scaling_factor),(int)((double)x2*scaling_factor),(int)((double)y2*scaling_factor));
			else
				printf("<AREA shape='rect' coords='%d,%d,%d,%d' ",x1,y,x2,y2);

			/* Javascript popup */
			printf("onMouseOver='showPopup(\"<STRONG>NetSaint Process</STRONG>\",event)' onMouseOut='hidePopup()'>\n");
		        }


		if(create_type==CREATE_IMAGE){

			/* get the name of the image file to open for the logo */
			snprintf(image_input_file,sizeof(image_input_file),"%s%s",physical_logo_images_path,NETSAINT_ICON_FILENAME);
			image_input_file[sizeof(image_input_file)-1]='\x0';

			/* read in the logo... */
			logo_file=fopen(image_input_file,"rb");
			if(logo_file!=NULL){

				logo_image=gdImageCreateFromGd2(logo_file);
				fclose(logo_file);

				/* if we were able to load the image... */
				if(logo_image!=NULL){

					/* get upper left hand corner of bounding box (image dimensions may be different than expected... ) */
					x1=x-(logo_image->sx/2);
					x2=x1+logo_image->sx;
					y2=y+logo_image->sy;

				        /* copy the logo image to the main image we are creating... */
					gdImageCopy(status_image,logo_image,x1,y,0,0,logo_image->sx,logo_image->sy);
					gdImageDestroy(logo_image);
				        }

				/* if not... */
				else{

					/* get upper left hand corner of bounding box */
					x1=x-(node_width/2);
					x2=x1+node_width;
					y2=y+node_height;

					/* last ditch effort - draw a host bounding box */
					draw_line(x1,y,x2,y,color);
					draw_line(x1,y,x1,y2,color);
					draw_line(x1,y2,x2,y2,color);
					draw_line(x2,y2,x2,y,color);
				        }
		                }
		        }
	        }

	/* display a pretty icon and some status information for the host... */
	else{

		/* see if user is authorized to view this host - if they're not, we'll just display an unknown icon... */
		authorized_for_host=is_authorized_for_host(hst,&current_authdata);

		/* get corners of host bounding box */
		x1=x-(node_width/2);
		x2=x1+node_width;
		y2=y+node_height;


		if(create_type==CREATE_IMAGE){

			/* get the center of the host bounding box */
			centerx=((x2-x1)/2)+x1;
			centery=((y2-y)/2)+y;

			/* save the host drawing coordinates */
			hst->has_been_drawn=TRUE;
			hst->x_coordinate=(float)centerx;
			hst->y_coordinate=(float)y;

			/* see if this host has extended data associated with it (i.e. an icon) */
			temp_hostextinfo=find_hostextinfo(hst->name);
			if(temp_hostextinfo!=NULL && temp_hostextinfo->gd2_icon_image!=NULL)
				has_image=TRUE;

			/* if the user isn't authorized to view this host, just place the unknown image here... */
			if(authorized_for_host==FALSE)
				has_image=FALSE;

			/* if the host doesn't have an image associated with it (or the user doesn't have rights to see this host), use the unknown image */
			if(has_image==FALSE){
				if(unknown_logo_image!=NULL)
					gdImageCopy(status_image,unknown_logo_image,centerx-(unknown_logo_image->sx/2),y,0,0,unknown_logo_image->sx,unknown_logo_image->sy);
		                }

			/* else load the logo associated with this host */
			else{

				/* get the name of the image file to open for the logo */
				snprintf(image_input_file,sizeof(image_input_file),"%s%s",physical_logo_images_path,temp_hostextinfo->gd2_icon_image);
				image_input_file[sizeof(image_input_file)-1]='\x0';

				/* read in the image... */
				logo_file=fopen(image_input_file,"r");
				if(logo_file!=NULL){

					logo_image=gdImageCreateFromGd2(logo_file);
					fclose(logo_file);

				        /* copy the logo to the image being generated... */
					if(logo_image!=NULL){
						gdImageCopy(status_image,logo_image,centerx-(logo_image->sx/2),y,0,0,logo_image->sx,logo_image->sy);
						gdImageDestroy(logo_image);
			                        }
			                }

				/* try to use the unknown icon if we couldn't load the image... */
				else if(unknown_logo_image!=NULL)
					gdImageCopy(status_image,unknown_logo_image,centerx-(unknown_logo_image->sx/2),y,0,0,unknown_logo_image->sx,unknown_logo_image->sy);
				else{

					/* last ditch effort - draw a host bounding box */
					draw_line(x1,y,x2,y,color);
					draw_line(x1,y,x1,y2,color);
					draw_line(x1,y2,x2,y2,color);
					draw_line(x2,y2,x2,y,color);
				        }
		                }

			/* draw the name of the host using the small font... */
			snprintf(temp_buffer,sizeof(temp_buffer),"%s",(authorized_for_host==FALSE)?"UNKNOWN":hst->name);
			temp_buffer[sizeof(temp_buffer)-1]='\x0';
			string_height=gdFontSmall->h;
			string_width=gdFontSmall->w*strlen(temp_buffer);
			gdImageString(status_image,gdFontSmall,centerx-(string_width/2),y2-(string_height*2)-2,(unsigned char *)temp_buffer,color_black);
		        }


		/* if the user isn't authorized to view information on this host, don't print any more information... */
		if(authorized_for_host==FALSE)
			return;


		if(create_type==CREATE_IMAGE){

			/* find the status entry for this host */
			temp_status=find_hoststatus(hst->name);

			/* get the status of the host (pending, up, down, or unreachable) */
			if(temp_status!=NULL){

				/* draw the status string */
				if(temp_status->status==HOST_DOWN){
					strncpy(temp_buffer,"Down",sizeof(temp_buffer));
					status_color=color_red;
		                        }
				else if(temp_status->status==HOST_UNREACHABLE){
					strncpy(temp_buffer,"Unreachable",sizeof(temp_buffer));
					status_color=color_red;
		                        }
				else if(temp_status->status==HOST_UP){
					strncpy(temp_buffer,"Up",sizeof(temp_buffer));
					status_color=color_green;
		                        }
				else if(temp_status->status==HOST_PENDING){
					strncpy(temp_buffer,"Pending",sizeof(temp_buffer));
					status_color=color_grey;
		                        }
				else{
					strncpy(temp_buffer,"Unknown",sizeof(temp_buffer));
					status_color=color_orange;
			                }

				temp_buffer[sizeof(temp_buffer)-1]='\x0';

				/* write the host status string to the generated imate... */
				string_width=gdFontSmall->w*strlen(temp_buffer);
				gdImageString(status_image,gdFontSmall,centerx-(string_width/2),y2-string_height-2,(unsigned char *)temp_buffer,status_color);
		                }
		        }

		if(create_type==CREATE_HTML){

			/* start the imagemap area... */
			printf("<AREA shape='rect'");

			/* write out coordinates for the image are associated with this host (scale if necessary) */
			if(scale_image==TRUE)
				printf(" coords='%d,%d,%d,%d'",(int)((double)x1*scaling_factor),(int)((double)y*scaling_factor),(int)((double)x2*scaling_factor),(int)((double)y2*scaling_factor));
			else
				printf(" coords='%d,%d,%d,%d'",x1,y,x2,y2);

			printf(" href='%s?host=%s",STATUSMAP_CGI,hst->name);

			/* print CGI options... */
			if(use_node_width==TRUE)
				printf("&nodewidth=%d",node_width);
			if(use_node_height==TRUE)
				printf("&nodeheight=%d",node_height);
			if(use_horizontal_spacing==TRUE)
				printf("&hspacing=%d",horizontal_spacing);
			if(use_vertical_spacing==TRUE)
				printf("&vspacing=%d",vertical_spacing);
			if(use_max_width==TRUE)
				printf("&maxwidth=%d",max_image_width);
			if(use_max_height==TRUE)
				printf("&maxheight=%d",max_image_height);

			/* end of CGI options and HREF... */
			printf("'");

			/* Javascript popup */
			printf(" onMouseOver='showPopup(\"");
			write_popup_text(hst);
			printf("\",event)' onMouseOut='hidePopup()'>\n");
	                }
	        }

	return;
	}



/* writes popup text for a specific host */
void write_popup_text(host *hst){
	hostextinfo *temp_hostextinfo;
	hoststatus *temp_status;
	int service_totals;


	/* display nothing if user is not authorized to view this host */
	if(is_authorized_for_host(hst,&current_authdata)==FALSE)
		return;

	printf("<table border=0 cellpadding=0 cellspacing=2>");

	temp_hostextinfo=find_hostextinfo(hst->name);
	if(temp_hostextinfo!=NULL){
		printf("<tr><td>");
		printf("<tr><td><img src=%s%s border=0 width=80 height=80></td>",url_logo_images_path,(temp_hostextinfo->icon_image==NULL)?"":temp_hostextinfo->icon_image);
		printf("<td><i>%s</i></td></tr>",(temp_hostextinfo->icon_image_alt==NULL)?"":temp_hostextinfo->icon_image_alt);
	        }

	printf("<tr><td>Name:</td><td><b>%s</b></td></tr>",hst->name);
	printf("<tr><td>Alias:</td><td><b>%s</b></td></tr>",hst->alias);
	printf("<tr><td>Address:</td><td><b>%s</b></td></tr>",hst->address);
	printf("<tr><td>State:</td><td><b>");

	/* find the status entry for this host */
	temp_status=find_hoststatus(hst->name);

	/* get the status of the host (pending, up, down, or unreachable) */
	if(temp_status!=NULL){

		/* draw the status string */
		if(temp_status->status==HOST_DOWN)
			printf("Down");

		else if(temp_status->status==HOST_UNREACHABLE)
			printf("Unreachable");

		else if(temp_status->status==HOST_UP)
			printf("Up");

		else if(temp_status->status==HOST_PENDING)
			printf("Pending");
	        }

	printf("</b></td></tr>");
	printf("</table>");

	printf("<br><b><u>Services:</u></b><br>");

	service_totals=get_servicestatus_count(hst->name,SERVICE_OK)+get_servicestatus_count(hst->name,SERVICE_RECOVERY);
	if(service_totals>0)
		printf("- %d ok<br>",service_totals);
	service_totals=get_servicestatus_count(hst->name,SERVICE_CRITICAL)+get_servicestatus_count(hst->name,SERVICE_UNREACHABLE)+get_servicestatus_count(hst->name,SERVICE_HOST_DOWN);
	if(service_totals>0)
		printf("- %d critical<br>",service_totals);
	service_totals=get_servicestatus_count(hst->name,SERVICE_WARNING);
	if(service_totals>0)
		printf("- %d warning<br>",service_totals);
	service_totals=get_servicestatus_count(hst->name,SERVICE_UNKNOWN);
	if(service_totals>0)
		printf("- %d unknown<br>",service_totals);
	service_totals=get_servicestatus_count(hst->name,SERVICE_PENDING);
	if(service_totals>0)
		printf("- %d pending<br>",service_totals);

	return;
        }



/* write JavaScript code an layer for popup window */
void write_popup_code(void){
	char *border_color="#000000";
	char *background_color="#ffffcc";
	int border=1;
	int padding=3;
	int x_offset=3;
	int y_offset=3;
	char *font="face='Verdana, Arial, Helvetica, sans-serif' size=2";

	printf("<SCRIPT LANGUAGE='JavaScript'>\n");
	printf("<!--\n");
	printf("// JavaScript popup based on code originally found at http://www.helpmaster.com/htmlhelp/javascript/popjbpopup.htm\n");
	printf("function showPopup(text, eventObj){\n");
	printf("ieLayer = 'document.all[\\'popup\\']';\n");
	printf("nnLayer = 'document.layers[\\'popup\\']';\n");

	printf("if(!(document.all||document.layers)) return;\n");

	printf("if(document.all) document.popup=eval(ieLayer);\n");
	printf("else document.popup=eval(nnLayer);\n");

	printf("var table = \"\";\n");

	printf("if (document.all){\n");
	printf("table += \"<table bgcolor='%s' border=%d cellpadding=%d cellspacing=0>\";\n",background_color,border,padding);
	printf("table += \"<tr><td>\";\n");
	printf("table += \"<table cellspacing=0 cellpadding=%d>\";\n",padding);
	printf("table += \"<tr><td bgcolor='%s'><font %s>\" + text + \"</font></td></tr>\";\n",background_color,font);
	printf("table += \"</table></td></tr></table>\"\n");
	printf("document.popup.innerHTML = table;\n");
	printf("document.popup.style.left = eventObj.x + %d;\n",x_offset);
	printf("document.popup.style.top  = eventObj.y + %d;\n",y_offset);
	printf("document.popup.style.visibility = \"visible\";\n");
	printf("} \n");
 

	printf("else{\n");
	printf("table += \"<table cellpadding=%d border=%d cellspacing=0 bordercolor='%s'>\";\n",padding,border,border_color);
	printf("table += \"<tr><td bgcolor='%s'><font %s>\" + text + \"</font></td></tr></table>\";\n",background_color,font);
	printf("document.popup.document.open();\n");
	printf("document.popup.document.write(table);\n");
	printf("document.popup.document.close();\n");

	/* set x coordinate */
	printf("document.popup.left = eventObj.layerX + %d;\n",x_offset);
	
	/* make sure we don't overlap the right side of the screen */
	printf("if(document.popup.left + document.popup.document.width + %d > window.innerWidth) document.popup.left = window.innerWidth - document.popup.document.width - %d - 16;\n",x_offset,x_offset);
		
	/* set y coordinate */
	printf("document.popup.top  = eventObj.layerY + %d;\n",y_offset);
	
	/* make sure we don't overlap the bottom edge of the screen */
	printf("if(document.popup.top + document.popup.document.height + %d > window.innerHeight) document.popup.top = window.innerHeight - document.popup.document.height - %d - 16;\n",y_offset,y_offset);
		
	/* make the popup visible */
	printf("document.popup.visibility = \"visible\";\n");
	printf("}\n");
	printf("}\n");

	printf("function hidePopup(){ \n");
	printf("if (!(document.all || document.layers)) return;\n");
	printf("if (document.popup == null){ }\n");
	printf("else if (document.all) document.popup.style.visibility = \"hidden\";\n");
	printf("else document.popup.visibility = \"hidden\";\n");
	printf("document.popup = null;\n");
	printf("}\n");
	printf("//-->\n");

	printf("</SCRIPT>\n");

	return;
        }



/* draws a link to a child host */
void draw_child_link(host *hst,int x0,int y0,int x1,int y1,int color){

	if(create_type==CREATE_HTML)
		return;

	/* draw the link */
	draw_line(x0,y0,x1,y1,color);	

	return;
	}



/* draws a line */
void draw_line(int x1,int y1,int x2,int y2,int color){

	if(create_type==CREATE_HTML)
		return;

	/* draws a line */
	gdImageLine(status_image,x1,y1,x2,y2,color);

	return;
	}


/* draws a dashed line */
void draw_dashed_line(int x1,int y1,int x2,int y2,int color){
	int styleDashed[12];

	if(create_type==CREATE_HTML)
		return;

	styleDashed[0]=color;
	styleDashed[1]=color;
	styleDashed[2]=color;
	styleDashed[3]=color;
	styleDashed[4]=color;
	styleDashed[5]=color;
	styleDashed[6]=gdTransparent;
	styleDashed[7]=gdTransparent;
	styleDashed[8]=gdTransparent;
	styleDashed[9]=gdTransparent;
	styleDashed[10]=gdTransparent;
	styleDashed[11]=gdTransparent;

	/* sets current style to a dashed line */
	gdImageSetStyle(status_image,styleDashed,12);

	/* draws a line (dashed) */
	gdImageLine(status_image,x1,y1,x2,y2,gdStyled);

	return;
	}


/* draws secondary parent/child links */
void draw_secondary_child_links(void){
	host *temp_host;
	host *temp_host2;
	hostsmember *temp_hostsmember;
	hoststatus *temp_hoststatus;
	int link_color;

	for(temp_host=host_list;temp_host!=NULL;temp_host=temp_host->next){

		if(temp_host->has_been_drawn==FALSE)
			continue;

		if(temp_host->parent_hosts==NULL)
			continue;

		for(temp_hostsmember=temp_host->parent_hosts;temp_hostsmember!=NULL;temp_hostsmember=temp_hostsmember->next){

			temp_host2=find_host(temp_hostsmember->host_name,NULL);

			if(temp_host2==NULL)
				continue;

			if(temp_host2->has_been_drawn==FALSE)
				continue;

			if(is_host_primary_immediate_child_of_host(temp_host2,temp_host)==TRUE)
				continue;

			temp_hoststatus=find_hoststatus(temp_host2->name);
			if(temp_hoststatus==NULL)
				link_color=color_black;
			else if(temp_hoststatus->status==HOST_DOWN || temp_hoststatus->status==HOST_UNREACHABLE)
				link_color=color_red;
			else
				link_color=color_black;

			draw_dashed_line(temp_host->x_coordinate,temp_host->y_coordinate,temp_host2->x_coordinate,temp_host2->y_coordinate+node_height,link_color);
		        }
	        }

	return;
        }
