/*****************************************************************************
 *
 * STATUSWRL.C - NetSaint 3-D Network Status View
 *
 * Copyright (c) 1999-2000 Ethan Galstad (netsaint@netsaint.org)
 * Last Modified:   07-07-2000
 *
 * Description:
 *
 * This CGI will dynamically create a 3-D VRML model of all hosts that are
 * being monitored on your network.
 *
 * License:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *****************************************************************************/

#include "../common/config.h"
#include "../common/locations.h"
#include "../common/common.h"
#include "../common/objects.h"
#include "../common/statusdata.h"

#include "cgiutils.h"
#include "getcgi.h"
#include "auth.h"
#include "edata.h"

extern char main_config_file[MAX_FILENAME_LENGTH];
extern char url_images_path[MAX_FILENAME_LENGTH];

extern int             netsaint_process_state;

extern hoststatus *hoststatus_list;
extern host *host_list;
extern servicestatus *servicestatus_list;

#define DEFAULT_NODE_WIDTH		0.5
#define DEFAULT_HORIZONTAL_SPACING	1.0
#define DEFAULT_VERTICAL_SPACING	1.0

#ifdef OLDSTYLE
void draw_line(int,int,int,int,int);
#endif

void display_world(void);
void write_global_vrml_data(void);
int get_host_draw_data(host *);
void draw_host_and_child_links(host *,float,float);
void draw_host(host *,float,float);
void draw_child_link(host *,float,float,float,float);
void draw_secondary_child_links(void);

void document_header(void);
void document_footer(void);
int process_cgivars(void);

char url_logo_images_path[MAX_FILENAME_LENGTH];

authdata current_authdata;

int top_number_of_immediate_children=0;
int top_max_child_drawing_width=0;
/*int max_host_depth=0;*/
int max_child_depth=0;
int max_child_drawing_width=0;

float link_radius=0.016;

float floor_width=0.0;
float floor_depth=0.0;

float vertical_spacing=DEFAULT_VERTICAL_SPACING;
float horizontal_spacing=DEFAULT_HORIZONTAL_SPACING;
float node_width=DEFAULT_NODE_WIDTH;
float node_height=DEFAULT_NODE_WIDTH;	/* should be the same as the node width */

char *host_name="all";
int show_all_hosts=TRUE;

int use_textures=TRUE;
int use_text=TRUE;





int main(int argc, char **argv){
	int result;

	/* get the arguments passed in the URL */
	process_cgivars();

	/* reset internal variables */
	reset_cgi_vars();

	/* read the CGI configuration file */
	result=read_cgi_config_file(DEFAULT_CGI_CONFIG_FILE);
	if(result==ERROR){
		document_header();
		printf("<P><DIV class='errorMessage'>Error: Could not open CGI configuration file '%s' for reading!</DIV></P>\n",DEFAULT_CGI_CONFIG_FILE);
		document_footer();
		return ERROR;
	        }

	document_header();

	/* read the main configuration file */
	result=read_main_config_file(main_config_file);
	if(result==ERROR){
		printf("<P><DIV class='errorMessage'>Error: Could not open main configuration file '%s' for reading!</DIV></P>\n",main_config_file);
		document_footer();
		return ERROR;
	        }

	/* read all object configuration data */
	result=read_all_object_configuration_data(main_config_file,READ_HOSTGROUPS|READ_CONTACTGROUPS|READ_HOSTS|READ_SERVICES);
	if(result==ERROR){
		printf("<P><DIV class='errorMessage'>Error: Could not read some or all object configuration data!</DIV></P>\n");
		document_footer();
		return ERROR;
                }

	/* read all status data */
	result=read_all_status_data(main_config_file,READ_PROGRAM_STATUS|READ_HOST_STATUS|READ_SERVICE_STATUS);
	if(result==ERROR){
		printf("<P><DIV class='errorMessage'>Error: Could not read host and service status information!</DIV></P>\n");
		document_footer();
		free_memory();
		return ERROR;
                }

	/* get authentication information */
	get_authentication_information(&current_authdata);

	/* read in extended host information */
	read_extended_object_config_data(DEFAULT_CGI_CONFIG_FILE);

	/* display the 3-D VRML world... */
	display_world();

	document_footer();

	/* free all allocated memory */
	free_memory();
	free_extended_data();

	return OK;
        }



void document_header(void){
	char date_time[48];
	time_t current_time;
	time_t expire_time;


	printf("Cache-Control: no-store\n");
	printf("Pragma: no-cache\n");

	time(&current_time);
	get_expire_time_string(&current_time,date_time,sizeof(date_time));
	printf("Last-Modified: %s\n",date_time);

	expire_time=0L;
	get_expire_time_string(&expire_time,date_time,sizeof(date_time));
	printf("Expires: %s\n",date_time);

	printf("Content-Type: x-world/x-vrml\n\n");

	return;
        }


void document_footer(void){

	return;
        }



int process_cgivars(void){
	char **variables;
	int error=FALSE;
	int x;

	variables=getcgivars();

	for(x=0;variables[x]!=NULL;x++){

		/* do some basic length checking on the variable identifier to prevent buffer overflows */
		if(strlen(variables[x])>=MAX_INPUT_BUFFER-1){
			x++;
			continue;
		        }


		/* we found the host argument */
		else if(!strcmp(variables[x],"host")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			host_name=(char *)malloc(strlen(variables[x])+1);
			if(host_name==NULL)
				host_name="all";
			else
				strcpy(host_name,variables[x]);

			if(!strcmp(host_name,"all"))
				show_all_hosts=TRUE;
			else
				show_all_hosts=FALSE;
		        }

		/* we found the node width argument */
		else if(!strcmp(variables[x],"nodewidth")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			node_width=atoi(variables[x]);
			if(node_width<=0)
				node_width=DEFAULT_NODE_WIDTH;

			node_height=node_width;
		        }

		/* we found the horizontal spacing argument */
		else if(!strcmp(variables[x],"hspacing")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			horizontal_spacing=atoi(variables[x]);
			if(horizontal_spacing<=0)
				horizontal_spacing=DEFAULT_HORIZONTAL_SPACING;
		        }

		/* we found the vertical spacing argument */
		else if(!strcmp(variables[x],"vspacing")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			vertical_spacing=atoi(variables[x]);
			if(vertical_spacing<=0)
				vertical_spacing=DEFAULT_VERTICAL_SPACING;
		        }

		/* we found the no textures argument*/
		else if(!strcmp(variables[x],"notextures"))
			use_textures=FALSE;

		/* we found the no text argument*/
		else if(!strcmp(variables[x],"notext"))
			use_text=FALSE;
		}

	return error;
        }



/* top-level VRML world generation... */
void display_world(void){
	host *temp_host=NULL;

	/* get the url we will use to grab the logo images... */
	snprintf(url_logo_images_path,sizeof(url_logo_images_path),"%slogos/",url_images_path);
	url_logo_images_path[sizeof(url_logo_images_path)-1]='\x0';

	/* find the host we should be starting with */
	if(show_all_hosts==TRUE)
		temp_host=NULL;
	else
		temp_host=find_host(host_name,NULL);

	/* calculate drawing info */
	get_host_draw_data(temp_host);

	/* get the top level maximum child drawing width */
	if(temp_host==NULL)
		max_child_drawing_width=top_max_child_drawing_width;
	else
		max_child_drawing_width=temp_host->max_child_drawing_width;

	/* get the max depth of child hosts beneath the top host */
	max_child_depth=get_max_host_child_depth(temp_host);

	/* get the floor dimensions */
	floor_width=(float)((max_child_drawing_width-1)*(node_width+horizontal_spacing))+(node_width*2);
	floor_depth=(float)((max_child_depth-1)*(node_height+vertical_spacing))+(node_height*2);

	/* write global VRML data */
	write_global_vrml_data();

	/* create the VRML map */
	draw_host_and_child_links(temp_host,0.0,0.0);

	/* draw secondary child links */
	draw_secondary_child_links();

	return;
        }



/* write global VRML data */
void write_global_vrml_data(void){

	/* write VRML code header */
	printf("#VRML V2.0 utf8\n");

	/* write world information */
	printf("\n");
	printf("WorldInfo{\n");
	printf("title \"NetSaint 3-D Network Status View\"\n");
	printf("info [\"Copyright (c) 1999 Ethan Galstad\"\n");
        printf("\"netsaint@netsaint.org\"]\n");
	printf("}\n");

	/* write background color */
	printf("\n");
	printf("Background{\n");
	printf("skyColor 0.2 0.2 0.2\n");
	printf("}\n");

	/* write fog information */
	printf("\n");
	printf("Fog{\n");
	printf("color 0.2 0.2 0.2\n");
	printf("fogType \"EXPONENTIAL\"\n");
	printf("visibilityRange %2.2f\n",(floor_depth>floor_width)?(float)(floor_depth*2.0)+20.0:(float)(floor_width*2.0)+20.0);
	printf("}\n");

	/* initial viewpoint */
	printf("\n");
	printf("Viewpoint{\n");
	printf("position %2.2f %2.2f %2.2f\n",(float)node_width*2.0,(float)node_width*2.0,(float)node_width*10.0);
	printf("fieldOfView 0.78\n");
	printf("description \"Entry Point\"\n");
	printf("}\n");

	/* ariel viewpoint */
	printf("\n");
	printf("Viewpoint{\n");
	printf("position 0 %2.2f 0\n",(floor_width>floor_depth)?(float)(floor_width*0.65):(float)(floor_width*0.65)+((float)node_width*5.0));
	printf("orientation 1 0 0 4.71\n");
	printf("fieldOfView 0.78\n");
	printf("description \"Ariel View\"\n");
	printf("}\n");

	/* host node prototype */
	printf("\n");
	printf("PROTO HostNode [\n");
	printf("field SFColor emissive_color 0.2 0.9 0.2\n");
	printf("field SFColor diffuse_color 0.2 0.9 0.2\n");
	if(use_textures==TRUE)
		printf("field MFString texture_image \"\"\n");
	printf("]\n");
	printf("{\n");
	printf("Shape{\n");
	printf("appearance Appearance{\n");
	printf("material Material{\n");
	printf("emissiveColor IS emissive_color\n");
	printf("diffuseColor IS diffuse_color\n");
	printf("transparency 0.2\n");
	printf("}\n");
	if(use_textures==TRUE){
		printf("texture ImageTexture{\n");
		printf("url IS texture_image\n");
		printf("}\n");
		}
	printf("}\n");
	printf("geometry Box{\n");
	printf("size %2.2f %2.2f %2.2f\n",node_width,node_width,node_width);
	printf("}\n");
	printf("}\n");
	printf("}\n");

	/* host link prototype */
	printf("PROTO HostLink [\n");
	printf("field SFColor emissive_color 0.6 0.6 0.6\n");
	printf("field SFColor diffuse_color 0.6 0.6 0.6\n");
	printf("field SFFloat link_length %2.2f\n",vertical_spacing);
	printf("]\n");
	printf("{\n");
	printf("Shape{\n");
	printf("appearance Appearance{\n");
	printf("material Material{\n");
	printf("emissiveColor IS emissive_color\n");
	printf("diffuseColor IS diffuse_color\n");
	printf("transparency 0.1\n");
	printf("}\n");
	printf("}\n");
	printf("geometry Cylinder{\n");
	printf("radius %2.4f\n",link_radius);
	printf("height IS link_length\n");
	printf("top FALSE\n");
	printf("}\n");
	printf("}\n");
	printf("}\n");

	/* host text prototype */
	printf("PROTO HostText[\n");
	printf("field MFString the_text [\"\"]\n");
	printf("field SFColor font_color 0.6 0.6 0.6");
	printf("]\n");
	printf("{\n");
	printf("Billboard{\n");
	printf("children[\n");
	printf("Shape{\n");
	printf("appearance Appearance {\n");
	printf("material Material {\n");
	printf("diffuseColor IS font_color\n");
	printf("}\n");
	printf("}\n");
	printf("geometry Text {\n");
	printf("string IS the_text\n");
	printf("fontStyle FontStyle {\n");
	printf("size 0.1\n");
	printf("justify \"MIDDLE\"\n");
	printf("}\n");
	printf("}\n");
	printf("}\n");
	printf("]\n");
	printf("}\n");
	printf("}\n");

	return;
	}


/* gets data necessary for drawing the host and its child hosts */
int get_host_draw_data(host *hst){
	int max_width=0;
	int temp_width=0;
	int children=0;
	host *temp_host;
	int max_depth=0;

	/* get the max depth of child hosts for this host */
	max_depth=get_max_host_child_depth(hst);
	if(hst==NULL)
		max_child_depth=max_depth;
	else
		hst->max_child_depth=max_depth;

	/* get the number of immediate children to this host */
	children=number_of_immediate_child_hosts(hst);
	if(hst==NULL)
		top_number_of_immediate_children=children;
	else
		hst->number_of_immediate_children=children;

	/* if host has no children, max drawing width is one unit */
	if(children==0)
		max_width=1;

	/* else calculate max width of all children hosts */
	else for(temp_host=host_list;temp_host!=NULL;temp_host=temp_host->next){

		if(is_host_primary_immediate_child_of_host(hst,temp_host)){

			temp_width=get_host_draw_data(temp_host);
			max_width+=temp_width;
			}
		}

	if(max_width==0)
		max_width=1;

	if(hst==NULL)
		top_max_child_drawing_width=max_width;
	else
		hst->max_child_drawing_width=max_width;

	return max_width;
	}


/* draws a host and any child links */
void draw_host_and_child_links(host *hst,float x,float y){
	host *temp_host;
	int has_children=0;
	int current_child;
	int number_of_immediate_children;
	int max_child_drawing_width;
	float this_draw_width;
	float x1, xx;
	float y0, y1;

	/* draw the host */
	draw_host(hst,x,y);

	/* does the host have any children? */
	if(hst==NULL){
		if(top_number_of_immediate_children>0)
			has_children=1;
		else
			has_children=0;
	        }
	else if(hst->number_of_immediate_children>0){
		has_children=1;
	        }
	else{
		has_children=0;
	        }

	/* if the host has children, draw the liks to the child hosts.. */
	if(has_children){

		/* get the number of immediate children and the max child drawing width for this host */
		if(hst==NULL){
			number_of_immediate_children=top_number_of_immediate_children;
			max_child_drawing_width=top_max_child_drawing_width;
		        }
		else{
			number_of_immediate_children=hst->number_of_immediate_children;
			max_child_drawing_width=hst->max_child_drawing_width;
		        }


		/* calculate starting points for drawing... */
		y0=y;
		y1=y0+vertical_spacing+node_height;
		x1=x-(((max_child_drawing_width*node_width)+((max_child_drawing_width-1)*horizontal_spacing))/2);
		xx=x1;

		/* the first potential child is the first host in the host list... */
		temp_host=host_list;

		/* draw the links to all child hosts... */
		for(current_child=1;current_child<=number_of_immediate_children;current_child++){

			/* find the next child host - never go back to the head of the list or we'll have duplicates... */
			for(;temp_host!=NULL;temp_host=temp_host->next){
				if(is_host_primary_immediate_child_of_host(hst,temp_host))
					break;
				}
			
			/* if for some reason we're at the end of the host list, exit draw loop */
			if(temp_host==NULL)
				break;

			/* calculate the drawing width for this host... */
			this_draw_width=((temp_host->max_child_drawing_width*node_width)+((temp_host->max_child_drawing_width-1)*horizontal_spacing))/2.0;
			xx+=this_draw_width;

			printf("#link [%2.2f,%2.2f] to [%2.2f,%2.2f]\n",x,y0+(node_width)/2.0,xx,y1-(node_width/2.0));

			/* draw the link to the child host */
			draw_child_link(hst,x,y0+(node_width/2.0),xx,y1-(node_width/2.0));

			/* draw the child host and its child links */
			draw_host_and_child_links(temp_host,xx,y1);

			/* update the drawing coordinates... */
			xx+=this_draw_width;
			xx+=horizontal_spacing;

			/* go to the next host in the host list - its a potential child... */
			temp_host=temp_host->next;
			}
		}

	return;
	}



/* draws a host */
void draw_host(host *hst,float x,float y){
	hoststatus *temp_hoststatus=NULL;
	hostextinfo *temp_hostextinfo=NULL;
	char state_string[16]="";
	int authorized_for_host=FALSE;


	/* see if user is authorized to view this host - if they're not, we'll just display an grey box... */
	if(hst!=NULL)
		authorized_for_host=is_authorized_for_host(hst,&current_authdata);


	printf("\n");

	if(hst==NULL || authorized_for_host==TRUE){
		printf("Anchor{\n");
		printf("children[\n");
	        }

	printf("Transform {\n");
	printf("translation %2.2f %2.2f -%2.2f\n",x,(float)(node_width/2.0),y);
	printf("children [\n");

	/* save the host drawing coordinates */
	if(hst!=NULL){
		hst->has_been_drawn=TRUE;
		hst->x_coordinate=x;
		hst->y_coordinate=y;
	        }

	/* this is a normal host... */
	if(hst!=NULL){
		temp_hoststatus=find_hoststatus(hst->name);
		temp_hostextinfo=find_hostextinfo(hst->name);
		if(temp_hoststatus==NULL || authorized_for_host==FALSE)
			printf("HostNode{\nemissive_color 0.2 0.2 0.2\ndiffuse_color 0.2 0.2 0.2\n");
		else if(temp_hoststatus->status==HOST_UP)
			printf("HostNode{\n");
		else
			printf("HostNode{\nemissive_color 0.9 0.6 0.6\ndiffuse_color 0.9 0.6 0.6\n");
		if(temp_hostextinfo!=NULL && temp_hostextinfo->vrml_image!=NULL && use_textures==TRUE && authorized_for_host==TRUE)
			printf("texture_image \"%s%s\"\n",url_logo_images_path,temp_hostextinfo->vrml_image);
		printf("}\n");
		}

	/* this is the NetSaint process */
	else{
		printf("HostNode{\n");
		get_netsaint_process_info();
		if(netsaint_process_state!=STATE_OK)
			printf("emissive_color 0.9 0.6 0.6\ndiffuse_color 0.9 0.6 0.6\n");
		if(use_textures==TRUE)
			printf("texture_image \"%ssblogo.gif\"\n",url_images_path);
		printf("}\n");
		}

	printf("]\n");
	printf("}\n");

	if(hst!=NULL){
		if(authorized_for_host==TRUE){
			printf("]\n");
			printf("description \"View status details for host '%s' (%s)\"\n",hst->name,hst->alias);
			printf("url \"%s?host=%s\"\n",STATUS_CGI,hst->name);
			printf("}\n");
		        }
		}
	else{
		printf("]\n");
		printf("description \"View status details for all hosts\"\n");
		printf("url \"%s?host=all\"\n",STATUS_CGI);
		printf("}\n");
		}

	/* draw status text */
	if(use_text==TRUE){

		printf("\n");
		printf("Transform{\n");
		printf("translation %2.3f %2.3f -%2.3f\n",x,(float)((node_width/2.0)+0.5),y);
		printf("children[\n");
		printf("HostText{\n");

		if(hst!=NULL){

			/* the user is authorized to view information for this host */
			if(authorized_for_host==TRUE){

				if(temp_hoststatus!=NULL){
					if(temp_hoststatus->status==HOST_UP)
						printf("font_color 0 0.8 0\n");
					else if(temp_hoststatus->status==HOST_DOWN || temp_hoststatus->status==HOST_UNREACHABLE)
						printf("font_color 1 0 0\n");
			                }
				printf("the_text [\"%s\", \"%s\", ",hst->name,hst->alias);
				if(temp_hoststatus==NULL)
					strcpy(state_string,"UNKNOWN");
				else{
					if(temp_hoststatus->status==HOST_DOWN)
						strcpy(state_string,"DOWN");
					else if(temp_hoststatus->status==HOST_UNREACHABLE)
						strcpy(state_string,"UNREACHABLE");
					else if(temp_hoststatus->status==HOST_PENDING)
						strcpy(state_string,"PENDING");
					else
						strcpy(state_string,"UP");
			                }
				printf("\"%s\"]\n",state_string);
			        }

			/* the user is not authorized to view this host... */
			else{
				printf("font_color 0.4 0.4 0.4\n");
				printf("the_text [\"unknown\"]\n");
			        }
		        }

		else{

			if(netsaint_process_state==STATE_OK){
					printf("font_color 0 0.8 0\n");
					printf("the_text [\"NetSaint Process\", \"RUNNING\"]\n");
			        }
			else{
					printf("font_color 1 0 0\n");
					printf("the_text [\"NetSaint Process\", \"NOT RUNNING\"]\n");
			        }
		        }

		printf("}\n");
		printf("]\n");
		printf("}\n");
		}

	return;
	}




/* draws a link to a child host */
void draw_child_link(host *hst,float x0,float y0,float x1,float y1){
	hoststatus *temp_hoststatus=NULL;
	float link_length=1.0;
	float y2=0.0;
	float y3=0.0;
	float x2=0.0;

	printf("\n");

	if(hst!=NULL)
		printf("# Host '%s' LINK\n",hst->name);

	printf("Transform{\n");
	printf("translation %2.3f %2.3f -%2.3f\n",((x0+x1)/2.0),(node_width/2.0),((y0+y1)/2.0));

	/* calculate link length */
	link_length=(float)sqrt((double)(((x0-x1)*(x0-x1))+((y0-y1)*(y0-y1))));

	/* child host is directly below parent host */
	if(x0==x1)
		printf("rotation 1 0 0 4.71\n");
	else{

		/* calculate rotation vector */
		x2=y1-y0;
		y2=x1-x0;
		y3=y2;

		printf("rotation %2.3f 0 %2.3f 1.57\n",x2,y3);
		}

	printf("children [\n");

	if(hst!=NULL){
		temp_hoststatus=find_hoststatus(hst->name);
		if(temp_hoststatus==NULL)
			printf("HostLink{\nlink_length %2.2f}\n",link_length);
		else if(temp_hoststatus->status==HOST_UP)
			printf("HostLink{\nlink_length %2.2f\n}\n",link_length);
		else
			printf("HostLink{\nlink_length %2.2f\nemissive_color 0.9 0.6 0.6\ndiffuse_color 0.9 0.6 0.6\n}\n",link_length);
		}
	else
		printf("HostLink{\nlink_length %2.2f}\n",link_length);

	printf("]\n");
	printf("}\n");

	return;
	}



/* draws secondary parent/child links */
void draw_secondary_child_links(void){
	host *temp_host;
	host *temp_host2;
	hostsmember *temp_hostsmember;

	for(temp_host=host_list;temp_host!=NULL;temp_host=temp_host->next){

		if(temp_host->has_been_drawn==FALSE)
			continue;

		if(temp_host->parent_hosts==NULL)
			continue;

		for(temp_hostsmember=temp_host->parent_hosts;temp_hostsmember!=NULL;temp_hostsmember=temp_hostsmember->next){

			temp_host2=find_host(temp_hostsmember->host_name,NULL);

			if(temp_host2==NULL)
				continue;

			if(temp_host2->has_been_drawn==FALSE)
				continue;

			if(is_host_primary_immediate_child_of_host(temp_host2,temp_host)==TRUE)
				continue;

			draw_child_link(temp_host2,temp_host2->x_coordinate,temp_host2->y_coordinate+(node_width/2.0),temp_host->x_coordinate,temp_host->y_coordinate-(node_width/2.0));
		        }
	        }

	return;
        }

