/*****************************************************************************
 *
 * XSDDEFAULT.C - Default external status data input routines for NetSaint
 *
 * Copyright (c) 2000 Ethan Galstad (netsaint@netsaint.org)
 * Last Modified:   06-11-2000
 *
 * License:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *****************************************************************************/


/*********** COMMON HEADER FILES ***********/

#include "../common/config.h"
#include "../common/common.h"
#include "../common/locations.h"
#include "../common/statusdata.h"

#ifdef NSCORE
#include "../base/netsaint.h"
#endif

#ifdef NSCGI
#include "../cgi/cgiutils.h"
#endif


/**** IMPLEMENTATION SPECIFIC HEADER FILES ****/
#include "xsddefault.h"


char xsddefault_status_log[MAX_FILENAME_LENGTH]="";
char xsddefault_temp_file[MAX_FILENAME_LENGTH]="";



/******************************************************************/
/***************** COMMON CONFIG INITIALIZATION  ******************/
/******************************************************************/

/* grab configuration information */
int xsddefault_grab_config_info(char *main_config_file){
	char input_buffer[MAX_INPUT_BUFFER];
	char *temp_buffer;
	FILE *fp;

	/* initialize the location of the status log */
	strncpy(xsddefault_status_log,DEFAULT_STATUS_FILE,sizeof(xsddefault_status_log)-1);
	strncpy(xsddefault_temp_file,DEFAULT_TEMP_FILE,sizeof(xsddefault_temp_file)-1);
	xsddefault_status_log[sizeof(xsddefault_status_log)-1]='\x0';
	xsddefault_temp_file[sizeof(xsddefault_temp_file)-1]='\x0';

	/* open the main config file for reading */
	fp=fopen(main_config_file,"r");
	if(fp==NULL)
		return ERROR;

	/* read in all lines from the main config file */
	for(fgets(input_buffer,sizeof(input_buffer)-1,fp);!feof(fp);fgets(input_buffer,sizeof(input_buffer)-1,fp)){

		/* skip blank lines and comments */
		if(input_buffer[0]=='#' || input_buffer[0]=='\x0' || input_buffer[0]=='\n' || input_buffer[0]=='\r')
			continue;

		strip(input_buffer);

		/* skip lines that don't contain the status log definition */
		if((strstr(input_buffer,"status_file")==input_buffer) || (strstr(input_buffer,"xsddefault_status_log")==input_buffer)){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\n");
			if(temp_buffer==NULL)
				continue;
			strncpy(xsddefault_status_log,temp_buffer,sizeof(xsddefault_status_log)-1);
			xsddefault_status_log[sizeof(xsddefault_status_log)-1]='\x0';
		        }


		/* skip lines that don't contain the status log definition */
		if((strstr(input_buffer,"temp_file")==input_buffer) || (strstr(input_buffer,"xsddefault_temp_file")==input_buffer)){

			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\n");
			if(temp_buffer==NULL)
				continue;
			strncpy(xsddefault_temp_file,temp_buffer,sizeof(xsddefault_temp_file)-1);
			xsddefault_temp_file[sizeof(xsddefault_temp_file)-1]='\x0';
		        }
	        }

	fclose(fp);

	/* we didn't find the status log name */
	if(!strcmp(xsddefault_status_log,""))
		return ERROR;

	/* we didn't find the temp file */
	if(!strcmp(xsddefault_temp_file,""))
		return ERROR;

	return OK;
        }



#ifdef NSCORE

/******************************************************************/
/***************** DEFAULT DATA OUTPUT FUNCTIONS ******************/
/******************************************************************/


/* initialize status data */
int xsddefault_initialize_status_data(char *main_config_file){
	FILE *fp;
	int result;

	/* grab configuration data */
	result=xsddefault_grab_config_info(main_config_file);
	if(result==ERROR)
		return ERROR;

	/* delete the old status log (it might not exist) */
	unlink(xsddefault_status_log);

	/* create an empty status log */
	fp=fopen(xsddefault_status_log,"w");
	if(fp==NULL)
		return ERROR;
	fclose(fp);

	return OK;
        }


/* cleanup status data before terminating */
int xsddefault_cleanup_status_data(char *main_config_file, int delete_status_data){

	/* delete the status log */
	if(delete_status_data==TRUE){
		if(unlink(xsddefault_status_log))
			return ERROR;
	        }

	return OK;
        }


/* updates program status data */
int xsddefault_update_program_status(time_t _program_start,int _daemon_mode,time_t _program_mode,time_t _last_mode_change,time_t _last_command_check, time_t _last_log_rotation, int _execute_service_checks, int _accept_passive_service_checks, int _event_handlers_enabled, int _obsess_over_services, int initial_data){
	FILE *fp,*fp2;
	char input_buffer[MAX_INPUT_BUFFER];
	char program_string[MAX_INPUT_BUFFER];
	char new_program_string[MAX_INPUT_BUFFER];
	char mode_string[8];
	time_t current_time;
	int found_entry=FALSE;

	time(&current_time);

	/* get the program_mode string */
	if(_program_mode==STANDBY_MODE)
		strcpy(mode_string,"STANDBY");
	else
		strcpy(mode_string,"ACTIVE");

	/* create a string to identify previous entries in the log file for the program info */
	snprintf(program_string,sizeof(program_string),"] PROGRAM;");
	program_string[sizeof(program_string)-1]='\x0';

	snprintf(new_program_string,sizeof(program_string)-1,"[%lu] PROGRAM;%lu;%d;%s;%lu;%lu;%lu;%d;%d;%d;%d\n",(unsigned long)current_time,(unsigned long)_program_start,_daemon_mode,mode_string,(unsigned long)_last_mode_change,(unsigned long)_last_command_check,(unsigned long)_last_log_rotation,_execute_service_checks,_accept_passive_service_checks,_event_handlers_enabled,_obsess_over_services);
	new_program_string[sizeof(new_program_string)-1]='\x0';

	/* we're creating the initial data */
	if(initial_data==TRUE){
		fp=fopen(xsddefault_status_log,"a");
		if(fp==NULL)
			return ERROR;
		fputs(new_program_string,fp);
		fclose(fp);
		return OK;
	        }

	/* create a temp file for writing the new status log */
	fp2=fopen(xsddefault_temp_file,"w");
	if(fp2==NULL)
		return ERROR;

	/* open the existing status log */
	fp=fopen(xsddefault_status_log,"r");
	if(fp==NULL){
		fclose(fp2);
		return ERROR;
	        }

	/* read all lines from the original status log */
	while(fgets(input_buffer,MAX_INPUT_BUFFER-1,fp)){

		/* if we haven't found the program entry yet, search for a match... */
		if(found_entry==FALSE){

			/* if this is the old program string, write the new program info instead */
			if(strstr(input_buffer,program_string)){
				fprintf(fp2,"%s",new_program_string);
				found_entry=TRUE;
			        }

			/* if this is not a duplicate entry, write it to the new log file */
			else
				fputs(input_buffer,fp2);
		        }

		/* we already found the program entry, so just write this line to the new log file */
		else
			fputs(input_buffer,fp2);
	        }

	/* we didn't find the program entry, so write one */
	if(found_entry==FALSE)
		fprintf(fp2,"%s",new_program_string);

	fclose(fp);
	fclose(fp2);

	/* move the temp file to the status log (overwrite the old status log) */
	if(rename(xsddefault_temp_file,xsddefault_status_log))
		return ERROR;

	return OK;
        }


/* updates host status data */
int xsddefault_update_host_status(char *host_name, char *status, time_t last_update, time_t last_check, time_t last_state_change, int problem_has_been_acknowledged, unsigned long time_up, unsigned long time_down, unsigned long time_unreachable, time_t last_notification, int current_notification_number, int notifications_enabled, int event_handler_enabled, int checks_enabled, char *plugin_output, int initial_data){
	FILE *fp,*fp2;
	char input_buffer[MAX_INPUT_BUFFER];
	char host_string[MAX_INPUT_BUFFER];
	char new_host_string[MAX_INPUT_BUFFER];
	int found_entry=FALSE;


	/* create a host string to identify previous entries in the log file for this host */
	snprintf(host_string,sizeof(host_string),"] HOST;%s;",host_name);
	host_string[sizeof(host_string)-1]='\x0';

	if(last_check==(time_t)0)
		snprintf(new_host_string,sizeof(new_host_string)-1,"[%lu] HOST;%s;PENDING;0;0;0;0;0;0;0;0;%d;%d;%d;(Not enough data to determine host status yet)\n",(unsigned long)last_update,host_name,notifications_enabled,event_handler_enabled,checks_enabled);
	else
		snprintf(new_host_string,sizeof(new_host_string)-1,"[%lu] HOST;%s;%s;%lu;%lu;%d;%lu;%lu;%lu;%lu;%d;%d;%d;%d;%s\n",(unsigned long)last_update,host_name,status,(unsigned long)last_check,(unsigned long)last_state_change,problem_has_been_acknowledged,time_up,time_down,time_unreachable,(unsigned long)last_notification,current_notification_number,notifications_enabled,event_handler_enabled,checks_enabled,plugin_output);
	new_host_string[sizeof(new_host_string)-1]='\x0';

	/* we're creating the initial data */
	if(initial_data==TRUE){
		fp=fopen(xsddefault_status_log,"a");
		if(fp==NULL)
			return ERROR;
		fputs(new_host_string,fp);
		fclose(fp);
		return OK;
	        }

	/* create a temp file for writing the new status log */
	fp2=fopen(xsddefault_temp_file,"w");
	if(fp2==NULL)
		return ERROR;

	/* open the existing status log */
	fp=fopen(xsddefault_status_log,"r");
	if(fp==NULL){
		fclose(fp2);
		return ERROR;
	        }

	/* read all lines from the original status log */
	while(fgets(input_buffer,MAX_INPUT_BUFFER-1,fp)){

		/* if we haven't found the host entry yet, scan for it */
		if(found_entry==FALSE){

			/* if this is the old host string, write the new host info instead */
			if(strstr(input_buffer,host_string)){
				fprintf(fp2,"%s",new_host_string);
				found_entry=TRUE;
			        }

			/* this was not a match, so just write it to the new log file */
			else
				fputs(input_buffer,fp2);
		        }

		/* a matching host entry was already found, so just write this to the new log file */
		else
			fputs(input_buffer,fp2);
	        }

	/* we didn't find the host entry, so write one */
	if(found_entry==FALSE)
		fprintf(fp2,"%s",new_host_string);

	fclose(fp);
	fclose(fp2);

	/* move the temp file to the status log (overwrite the old status log) */
	if(rename(xsddefault_temp_file,xsddefault_status_log))
		return ERROR;

	return OK;
        }


/* updates service status data */
int xsddefault_update_service_status(char *host_name, char *description, char *status, time_t last_update, int current_attempt, int max_attempts, int state_type, time_t last_check, time_t next_check, int should_be_scheduled, int check_type, int checks_enabled, int accept_passive_service_checks, int event_handler_enabled, time_t last_state_change, int problem_has_been_acknowledged, char *last_hard_state, unsigned long time_ok, unsigned long time_warning, unsigned long time_unknown, unsigned long time_critical, time_t last_notification, int current_notification_number, int notifications_enabled, unsigned long latency, unsigned long execution_time, char *plugin_output, int initial_data){
	FILE *fp,*fp2;
	char input_buffer[MAX_INPUT_BUFFER];
	char service_string[MAX_INPUT_BUFFER];
	char new_service_string[MAX_INPUT_BUFFER];
	int found_entry=FALSE;


	/* create a service string to identify previous entries in the log for this service/host pair */
	snprintf(service_string,sizeof(service_string),"] SERVICE;%s;%s;",host_name,description);
	service_string[sizeof(service_string)-1]='\x0';

	/* create the new service status string */
	if(last_check==(time_t)0)
		snprintf(new_service_string,sizeof(new_service_string)-1,"[%lu] SERVICE;%s;%s;PENDING;0/%d;HARD;0;%lu;ACTIVE;%d;%d;%d;0;0;OK;0;0;0;0;0;0;%d;%lu;%lu;%s%s",(unsigned long)last_update,host_name,description,max_attempts,(should_be_scheduled==TRUE)?(unsigned long)next_check:0L,checks_enabled,accept_passive_service_checks,event_handler_enabled,notifications_enabled,latency,execution_time,(should_be_scheduled==TRUE)?"Service check scheduled for ":"Service check is not scheduled for execution...\n",(should_be_scheduled==TRUE)?ctime(&next_check):"");

	else
		snprintf(new_service_string,sizeof(new_service_string)-1,"[%lu] SERVICE;%s;%s;%s;%d/%d;%s;%lu;%lu;%s;%d;%d;%d;%lu;%d;%s;%lu;%lu;%lu;%lu;%lu;%d;%d;%lu;%lu;%s\n",(unsigned long)last_update,host_name,description,status,(state_type==SOFT_STATE)?current_attempt-1:current_attempt,max_attempts,(state_type==SOFT_STATE)?"SOFT":"HARD",(unsigned long)last_check,(should_be_scheduled==TRUE)?(unsigned long)next_check:0L,(check_type==SERVICE_CHECK_ACTIVE)?"ACTIVE":"PASSIVE",checks_enabled,accept_passive_service_checks,event_handler_enabled,(unsigned long)last_state_change,problem_has_been_acknowledged,last_hard_state,time_ok,time_unknown,time_warning,time_critical,(unsigned long)last_notification,current_notification_number,notifications_enabled,latency,execution_time,plugin_output);

	new_service_string[sizeof(new_service_string)-1]='\x0';

	/* we're creating the initial data */
	if(initial_data==TRUE){
		fp=fopen(xsddefault_status_log,"a");
		if(fp==NULL)
			return ERROR;
		fputs(new_service_string,fp);
		fclose(fp);
		return OK;
	        }

	/* create a temp file for writing the new status log */
	fp2=fopen(xsddefault_temp_file,"w");
	if(fp2==NULL)
		return ERROR;

	/* open the existing status log */
	fp=fopen(xsddefault_status_log,"r");
	if(fp==NULL){
		fclose(fp2);
		return ERROR;
	        }

	/* read all lines from the original status log */
	while(fgets(input_buffer,MAX_INPUT_BUFFER-1,fp)){

		/* if we haven't found the entry yet, search for it */
		if(found_entry==FALSE){

			/* if this is the old service string, write the new service info instead */
			if(strstr(input_buffer,service_string)){
				fprintf(fp2,"%s",new_service_string);
				found_entry=TRUE;
			        }

			/* this is not a matching entry, so just write it to the new log file */
			else
				fputs(input_buffer,fp2);
		        }

		/* the service entry has already been found, so just write this to the new log file */
		else
			fputs(input_buffer,fp2);
	        }

	/* we didn't find the service entry, so write one */
	if(found_entry==FALSE)
		fprintf(fp2,"%s",new_service_string);

	fclose(fp);
	fclose(fp2);

	/* move the temp file to the status log (overwrite the old status log) */
	if(rename(xsddefault_temp_file,xsddefault_status_log))
		return ERROR;


	return OK;
        }

#endif



#ifdef NSCGI

/******************************************************************/
/****************** DEFAULT DATA INPUT FUNCTIONS ******************/
/******************************************************************/

/* read all program, host, and service status information */
int xsddefault_read_status_data(char *main_config_file,int options){
	char input_buffer[MAX_INPUT_BUFFER];
	FILE *fp;
	int result;

	/* grab configuration data */
	result=xsddefault_grab_config_info(main_config_file);
	if(result==ERROR)
		return ERROR;

	/* opent the status log for reading */
	fp=fopen(xsddefault_status_log,"r");
	if(fp==NULL)
		return ERROR;

	/* read in all lines from the status log */
	for(fgets(input_buffer,sizeof(input_buffer)-1,fp);!feof(fp);fgets(input_buffer,sizeof(input_buffer)-1,fp)){

		/* skip blank lines and comments */
		if(input_buffer[0]=='#' || input_buffer[0]=='\x0' || input_buffer[0]=='\n' || input_buffer[0]=='\r')
			continue;

		strip(input_buffer);

		if(strstr(input_buffer,"] PROGRAM;") && (options & READ_PROGRAM_STATUS)){
			result=xsddefault_add_program_status(input_buffer);
			if(result!=OK){
				fclose(fp);
				return ERROR;
			        }
		        }
		else if(strstr(input_buffer,"] HOST;") && (options & READ_HOST_STATUS)){
			result=xsddefault_add_host_status(input_buffer);
		        }
		else if(strstr(input_buffer,"] SERVICE;") && (options & READ_SERVICE_STATUS)){
			result=xsddefault_add_service_status(input_buffer);
		        }
	        }

	fclose(fp);

	return OK;
        }



/* adds program status information */
int xsddefault_add_program_status(char *input_buffer){
	char *temp_buffer;
	time_t _program_start;
	int _daemon_mode;
	int _program_mode;
	time_t _last_mode_change;
	time_t _last_command_check;
	time_t _last_log_rotation;
	int _execute_service_checks;
	int _accept_passive_service_checks;
	int _event_handlers_enabled;
	int _obsess_over_services;
	int result;


	if(input_buffer==NULL)
		return ERROR;

	/* skip the identifier string */
	temp_buffer=my_strtok(input_buffer,";");
	if(temp_buffer==NULL)
		return ERROR;

	/* program start time */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL)
		return ERROR;
	_program_start=(time_t)strtoul(temp_buffer,NULL,10);

	/* daemon mode flag */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL)
		return ERROR;
	_daemon_mode=(temp_buffer==NULL)?0:atoi(temp_buffer);

	/* program mode */
	temp_buffer=my_strtok(NULL,";");
	if(!strcmp(temp_buffer,"STANDBY"))
		_program_mode=STANDBY_MODE;
	else
		_program_mode=ACTIVE_MODE;

	/* last mode change */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL)
		return ERROR;
	_last_mode_change=(time_t)strtoul(temp_buffer,NULL,10);

	/* last command check */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL)
		return ERROR;
	_last_command_check=(time_t)strtoul(temp_buffer,NULL,10);

	/* last log rotation */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL)
		return ERROR;
	_last_log_rotation=(time_t)strtoul(temp_buffer,NULL,10);

	/* execute service checks option */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL)
		return ERROR;
	_execute_service_checks=atoi(temp_buffer);

	/* accept passive service checks option */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL)
		return ERROR;
	_accept_passive_service_checks=atoi(temp_buffer);

	/* event handlers enabled option */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL)
		return ERROR;
	_event_handlers_enabled=atoi(temp_buffer);

	/* obsess over services option */
	temp_buffer=my_strtok(NULL,"\n");
	if(temp_buffer==NULL)
		return ERROR;
	_obsess_over_services=atoi(temp_buffer);

	/* add program status information */
	result=add_program_status(_program_start,_daemon_mode,_program_mode,_last_mode_change,_last_command_check,_last_log_rotation,_execute_service_checks,_accept_passive_service_checks,_event_handlers_enabled,_obsess_over_services);
	if(result!=OK)
		return ERROR;


	return OK;
        }



/* adds host status information */
int xsddefault_add_host_status(char *input_buffer){
	char *temp_buffer;
	int result;
	time_t last_update;
	char *host_name="";
	char status[16];
	time_t last_check;
	time_t last_state_change;
	int problem_has_been_acknowledged;
	unsigned long time_up;
	unsigned long time_down;
	unsigned long time_unreachable;
	time_t last_notification;
	int current_notification_number;
	int notifications_enabled;
	int event_handler_enabled;
	int checks_enabled;


	/* get the last host update time */
	temp_buffer=my_strtok(input_buffer,"]");
	if(temp_buffer==NULL)
		return ERROR;
	last_update=(time_t)strtoul(temp_buffer+1,NULL,10);

	/* get the host name */
	temp_buffer=my_strtok(NULL,";");
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL)
		return ERROR;
	host_name=(char *)malloc(strlen(temp_buffer)+1);
	if(host_name==NULL)
		return ERROR;
	strcpy(host_name,temp_buffer);

	/* get the host status */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		return ERROR;
	        }
	strncpy(status,temp_buffer,sizeof(status)-1);
	status[sizeof(status)-1]='\x0';

	/* get the last check time */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		return ERROR;
	        }
	last_check=(time_t)strtoul(temp_buffer,NULL,10);

	/* get the last state change time */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		return ERROR;
	        }
	last_state_change=(time_t)strtoul(temp_buffer,NULL,10);

	/* get the acknowledgement option */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		return ERROR;
	        }
	problem_has_been_acknowledged=atoi(temp_buffer);

	/* get the time up */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		return ERROR;
	        }
	time_up=strtoul(temp_buffer,NULL,10);

	/* get the time down */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		return ERROR;
	        }
	time_down=strtoul(temp_buffer,NULL,10);

	/* get the time unreachable */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		return ERROR;
	        }
	time_unreachable=strtoul(temp_buffer,NULL,10);

	/* get the last notification time */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		return ERROR;
	        }
	last_notification=(time_t)strtoul(temp_buffer,NULL,10);

	/* get the current notification number */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		return ERROR;
	        }
	current_notification_number=atoi(temp_buffer);

	/* get the notifications enabled option */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		return ERROR;
	        }
	notifications_enabled=atoi(temp_buffer);

	/* get the event handler enabled option */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		return ERROR;
	        }
	event_handler_enabled=atoi(temp_buffer);

	/* get the check enabled option */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		return ERROR;
	        }
	checks_enabled=atoi(temp_buffer);

	/* get the plugin output */
	temp_buffer=my_strtok(NULL,"\n");
	if(temp_buffer==NULL){
		free(host_name);
		return ERROR;
	        }


	/* add this host status */
	result=add_host_status(host_name,status,last_update,last_check,last_state_change,problem_has_been_acknowledged,time_up,time_down,time_unreachable,last_notification,current_notification_number,notifications_enabled,event_handler_enabled,checks_enabled,temp_buffer);
	if(result!=OK)
		return ERROR;

	return OK;
        }



/* adds service status information */
int xsddefault_add_service_status(char *input_buffer){
	char *temp_buffer;
	int result;
	time_t last_update;
	char *host_name="";
	char *description="";
	char status[16];
	int current_attempt;
	int max_attempts;
	int state_type;
	time_t last_check;
	time_t next_check;
	int check_type;
	int checks_enabled;
	int accept_passive_service_checks;
	int event_handler_enabled;
	time_t last_state_change;
	int problem_has_been_acknowledged;
	char last_hard_state[16];
	unsigned long time_ok;
	unsigned long time_warning;
	unsigned long time_unknown;
	unsigned long time_critical;
	time_t last_notification;
	int current_notification_number;
	int notifications_enabled;
	unsigned long latency;
	unsigned long execution_time;


	/* get the last update time */
	temp_buffer=my_strtok(input_buffer,"]");
	if(temp_buffer==NULL)
		return ERROR;
	last_update=(time_t)strtoul(temp_buffer+1,NULL,10);

	/* get the host name */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL)
		return ERROR;
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL)
		return ERROR;
	host_name=(char *)malloc(strlen(temp_buffer)+1);
	if(host_name==NULL)
		return ERROR;
	strcpy(host_name,temp_buffer);

	/* get the service description */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		return ERROR;
	        }
	description=(char *)malloc(strlen(temp_buffer)+1);
	if(description==NULL){
		free(host_name);
		return ERROR;
	        }
	strcpy(description,temp_buffer);

	/* get the service status */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	strncpy(status,temp_buffer,sizeof(status)-1);
	status[sizeof(status)-1]='\x0';

	/* get the current attempt */
	temp_buffer=my_strtok(NULL,"/");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	current_attempt=atoi(temp_buffer);

	/* get the max attempts */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	max_attempts=atoi(temp_buffer);

	/* get the state type */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	state_type=(!strcmp(temp_buffer,"SOFT"))?SOFT_STATE:HARD_STATE;

	/* get the last check time */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	last_check=(time_t)strtoul(temp_buffer,NULL,10);

	/* get the next check time */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	next_check=(time_t)strtoul(temp_buffer,NULL,10);

	/* get the check type */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	check_type=(!strcmp(temp_buffer,"ACTIVE"))?SERVICE_CHECK_ACTIVE:SERVICE_CHECK_PASSIVE;


	/* get the check enabled argument */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	checks_enabled=atoi(temp_buffer);

	/* get the passive check argument */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	accept_passive_service_checks=atoi(temp_buffer);

	/* get the event handler enabled argument */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	event_handler_enabled=atoi(temp_buffer);

	/* get the last state change time */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	last_state_change=(time_t)strtoul(temp_buffer,NULL,10);

	/* get the acknowledgement option */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	problem_has_been_acknowledged=atoi(temp_buffer);

	/* get the last hard state */
	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	strncpy(last_hard_state,temp_buffer,sizeof(last_hard_state)-1);
	last_hard_state[sizeof(last_hard_state)-1]='\x0';

	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	time_ok=strtoul(temp_buffer,NULL,10);

	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	time_unknown=strtoul(temp_buffer,NULL,10);

	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	time_warning=strtoul(temp_buffer,NULL,10);

	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	time_critical=strtoul(temp_buffer,NULL,10);

	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	last_notification=(time_t)strtoul(temp_buffer,NULL,10);

	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	current_notification_number=atoi(temp_buffer);

	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	notifications_enabled=atoi(temp_buffer);

	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	latency=strtoul(temp_buffer,NULL,10);

	temp_buffer=my_strtok(NULL,";");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }
	execution_time=strtoul(temp_buffer,NULL,10);

	/* get the plugin output */
	temp_buffer=my_strtok(NULL,"\n");
	if(temp_buffer==NULL){
		free(host_name);
		free(description);
		return ERROR;
	        }

	/* add this service status */
	result=add_service_status(host_name,description,status,last_update,current_attempt,max_attempts,state_type,last_check,next_check,check_type,checks_enabled,accept_passive_service_checks,event_handler_enabled,last_state_change,problem_has_been_acknowledged,last_hard_state,time_ok,time_warning,time_unknown,time_critical,last_notification,current_notification_number,notifications_enabled,latency,execution_time,temp_buffer);
	if(result!=OK)
		return ERROR;

	return OK;
        }


#endif

