/***********************************************************************
 *
 * CGIUTILS.C - Common utilities for NetSaint CGIs
 * 
 * Copyright (c) 1999-2000 Ethan Galstad (netsaint@netsaint.org)
 * Last Modified: 09-06-2000
 *
 * License:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 ***********************************************************************/

#include "../common/config.h"
#include "../common/common.h"
#include "../common/locations.h"
#include "../common/objects.h"
#include "../common/statusdata.h"

#include "cgiutils.h"
#include "popen.h"
#include "edata.h"
#include "lifo.h"

char            main_config_file[MAX_FILENAME_LENGTH];
char            log_file[MAX_FILENAME_LENGTH];
char            log_archive_path[MAX_FILENAME_LENGTH];
char            command_file[MAX_FILENAME_LENGTH];

char            physical_html_path[MAX_FILENAME_LENGTH];
char            physical_images_path[MAX_FILENAME_LENGTH];
char            url_html_path[MAX_FILENAME_LENGTH];
char            url_docs_path[MAX_FILENAME_LENGTH];
char            url_images_path[MAX_FILENAME_LENGTH];
char            url_logo_images_path[MAX_FILENAME_LENGTH];
char            url_stylesheets_path[MAX_FILENAME_LENGTH];
char            url_media_path[MAX_FILENAME_LENGTH];

char            *service_critical_sound=NULL;
char            *service_warning_sound=NULL;
char            *service_unknown_sound=NULL;
char            *host_down_sound=NULL;
char            *host_unreachable_sound=NULL;
char            *normal_sound=NULL;
char            *statusmap_background_image=NULL;

char            netsaint_check_command[MAX_INPUT_BUFFER];

char            netsaint_process_info[MAX_INPUT_BUFFER];
int             netsaint_process_state=STATE_OK;

time_t          program_start=0L;
int             program_mode=ACTIVE_MODE;
int             daemon_mode=FALSE;
time_t          last_mode_change=0L;

int             execute_service_checks=TRUE;
int             accept_passive_service_checks=TRUE;
int             event_handlers_enabled=TRUE;
int             obsess_over_services=FALSE;

time_t          last_command_check=0L;

int             check_external_commands=0;

int             log_rotation_method=LOG_ROTATION_NONE;
time_t          last_log_rotation=0L;

time_t          this_scheduled_log_rotation=0L;
time_t          last_scheduled_log_rotation=0L;
time_t          next_scheduled_log_rotation=0L;

int             use_authentication=TRUE;

int             suppress_alert_window=FALSE;

int             interval_length=60;

unsigned long   max_lifo_size=DEFAULT_MAX_LIFO_SIZE;

int             hosts_have_been_read=FALSE;
int             hostgroups_have_been_read=FALSE;
int             contacts_have_been_read=FALSE;
int             contactgroups_have_been_read=FALSE;
int             services_have_been_read=FALSE;
int             timeperiods_have_been_read=FALSE;
int             commands_have_been_read=FALSE;
int             serviceescalations_have_been_read=FALSE;
int             hostgroupescalations_have_been_read=FALSE;

int             host_status_has_been_read=FALSE;
int             service_status_has_been_read=FALSE;
int             program_status_has_been_read=FALSE;

int             refresh_rate=DEFAULT_REFRESH_RATE;


extern hostgroup       *hostgroup_list;
extern host            *host_list;
extern service         *service_list;
extern contactgroup    *contactgroup_list;
extern command         *command_list;
extern timeperiod      *timeperiod_list;
extern contact         *contact_list;
extern serviceescalation *serviceescalation_list;
extern hostgroupescalation *hostgroupescalation_list;

extern hoststatus      *hoststatus_list;
extern servicestatus   *servicestatus_list;

char            *my_strtok_buffer=NULL;
char            *original_my_strtok_buffer=NULL;

char encoded_url_string[MAX_INPUT_BUFFER];

#ifdef HAVE_TZNAME
extern char     *tzname[2];
#endif




/**********************************************************
 ***************** CLEANUP FUNCTIONS **********************
 **********************************************************/

/* reset all variables used by the CGIs */
void reset_cgi_vars(void){

	strcpy(main_config_file,"");

	strcpy(url_html_path,"");
	strcpy(url_docs_path,"");
	strcpy(url_stylesheets_path,"");
	strcpy(url_media_path,"");
	strcpy(url_images_path,"");

	strcpy(log_file,"");
	strcpy(log_archive_path,DEFAULT_LOG_ARCHIVE_PATH);
	if(log_archive_path[strlen(log_archive_path)-1]!='/' && strlen(log_archive_path)<sizeof(log_archive_path)-2)
		strcat(log_archive_path,"/");
	strcpy(command_file,DEFAULT_COMMAND_FILE);

	strcpy(netsaint_check_command,"");
	strcpy(netsaint_process_info,"");
	netsaint_process_state=STATE_OK;

	log_rotation_method=LOG_ROTATION_NONE;

	use_authentication=TRUE;

	suppress_alert_window=FALSE;

	interval_length=60;

	refresh_rate=DEFAULT_REFRESH_RATE;
	
	max_lifo_size=DEFAULT_MAX_LIFO_SIZE;

	return;
        }



/* free all memory for object definitions */
void free_memory(void){

	/* free memory for common object definitions */
	free_object_data();

	/* free memory for status data */
	free_status_data();

	return;
        }




/**********************************************************
 *************** CONFIG FILE FUNCTIONS ********************
 **********************************************************/


/* read the CGI configuration file */
int read_cgi_config_file(char *filename){
	char input_buffer[MAX_INPUT_BUFFER];
	char *temp_buffer;
	FILE *fp;

	fp=fopen(filename,"r");
	if(fp==NULL)
		return ERROR;

	while(read_line(input_buffer,MAX_INPUT_BUFFER,fp)){

	        if(feof(fp))
		        break;

		if(input_buffer==NULL)
		        continue;

		if(strstr(input_buffer,"main_config_file=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\x0");
			if(temp_buffer!=NULL){
				strncpy(main_config_file,temp_buffer,sizeof(main_config_file));
				main_config_file[sizeof(main_config_file)-1]='\x0';
				strip(main_config_file);
			        }
		        }

		else if((strstr(input_buffer,"use_authentication=")==input_buffer)){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\n");
			if(temp_buffer==NULL)
				use_authentication=TRUE;
			else if(atoi(temp_buffer)==0)
				use_authentication=FALSE;
			else
				use_authentication=TRUE;
		        }

		else if(strstr(input_buffer,"netsaint_check_command=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\x0");
			if(temp_buffer!=NULL){
				strncpy(netsaint_check_command,temp_buffer,sizeof(netsaint_check_command));
				netsaint_check_command[sizeof(netsaint_check_command)-1]='\x0';
				strip(netsaint_check_command);
			        }
		        }

		else if(strstr(input_buffer,"suppress_alert_window=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\x0");
			suppress_alert_window=(atoi((temp_buffer==NULL)?"0":temp_buffer)==0)?FALSE:TRUE;
		        }

		else if(strstr(input_buffer,"refresh_rate=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\x0");
			refresh_rate=atoi((temp_buffer==NULL)?"60":temp_buffer);
		        }

		else if(strstr(input_buffer,"max_lifo_size=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\x0");
			max_lifo_size=strtoul((temp_buffer==NULL)?"524288":temp_buffer,NULL,10);
		        }

		else if(strstr(input_buffer,"physical_html_path=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\n");
			strncpy(physical_html_path,(temp_buffer==NULL)?"":temp_buffer,sizeof(physical_html_path));
			physical_html_path[sizeof(physical_html_path)-1]='\x0';
			strip(physical_html_path);
			if(physical_html_path[strlen(physical_html_path)-1]!='/' && (strlen(physical_html_path) < sizeof(physical_html_path)-1))
				strcat(physical_html_path,"/");
			snprintf(physical_images_path,sizeof(physical_images_path),"%simages/",physical_html_path);
			physical_images_path[sizeof(physical_images_path)-1]='\x0';
		        }

		else if(strstr(input_buffer,"url_html_path=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\n");

			strncpy(url_html_path,(temp_buffer==NULL)?"":temp_buffer,sizeof(url_html_path));
			url_html_path[sizeof(url_html_path)-1]='\x0';

			strip(url_html_path);
			if(url_html_path[strlen(url_html_path)-1]!='/' && (strlen(url_html_path) < sizeof(url_html_path)-1))
				strcat(url_html_path,"/");

			snprintf(url_docs_path,sizeof(url_docs_path),"%sdocs/",url_html_path);
			url_docs_path[sizeof(url_docs_path)-1]='\x0';

			snprintf(url_images_path,sizeof(url_images_path),"%simages/",url_html_path);
			url_images_path[sizeof(url_images_path)-1]='\x0';

			snprintf(url_logo_images_path,sizeof(url_logo_images_path),"%slogos/",url_images_path);
			url_logo_images_path[sizeof(url_logo_images_path)-1]='\x0';

			snprintf(url_stylesheets_path,sizeof(url_stylesheets_path),"%sstylesheets/",url_html_path);
			url_stylesheets_path[sizeof(url_stylesheets_path)-1]='\x0';

			snprintf(url_media_path,sizeof(url_media_path),"%smedia/",url_html_path);
			url_media_path[sizeof(url_media_path)-1]='\x0';
		        }

		else if(strstr(input_buffer,"service_critical_sound=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\n");
			if(temp_buffer==NULL)
				continue;
			service_critical_sound=(char *)malloc(strlen(temp_buffer)+1);
			if(service_critical_sound!=NULL)
				strcpy(service_critical_sound,temp_buffer);
		        }

		else if(strstr(input_buffer,"service_warning_sound=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\n");
			if(temp_buffer==NULL)
				continue;
			service_warning_sound=(char *)malloc(strlen(temp_buffer)+1);
			if(service_warning_sound!=NULL)
				strcpy(service_warning_sound,temp_buffer);
		        }

		else if(strstr(input_buffer,"service_unknown_sound=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\n");
			if(temp_buffer==NULL)
				continue;
			service_unknown_sound=(char *)malloc(strlen(temp_buffer)+1);
			if(service_unknown_sound!=NULL)
				strcpy(service_unknown_sound,temp_buffer);
		        }

		else if(strstr(input_buffer,"host_down_sound=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\n");
			if(temp_buffer==NULL)
				continue;
			host_down_sound=(char *)malloc(strlen(temp_buffer)+1);
			if(host_down_sound!=NULL)
				strcpy(host_down_sound,temp_buffer);
		        }

		else if(strstr(input_buffer,"host_unreachable_sound=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\n");
			if(temp_buffer==NULL)
				continue;
			host_unreachable_sound=(char *)malloc(strlen(temp_buffer)+1);
			if(host_unreachable_sound!=NULL)
				strcpy(host_unreachable_sound,temp_buffer);
		        }

		else if(strstr(input_buffer,"normal_sound=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\n");
			if(temp_buffer==NULL)
				continue;
			normal_sound=(char *)malloc(strlen(temp_buffer)+1);
			if(normal_sound!=NULL)
				strcpy(normal_sound,temp_buffer);
		        }

		else if(strstr(input_buffer,"statusmap_background_image=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\n");
			if(temp_buffer==NULL)
				continue;
			statusmap_background_image=(char *)malloc(strlen(temp_buffer)+1);
			if(statusmap_background_image!=NULL)
				strcpy(statusmap_background_image,temp_buffer);
		        }
 	        }

	fclose(fp);

	if(!strcmp(main_config_file,""))
		return ERROR;
	else
		return OK;
        }



/* read the main configuration file */
int read_main_config_file(char *filename){
	char input_buffer[MAX_INPUT_BUFFER];
	char *temp_buffer;
	FILE *fp;

	
	fp=fopen(filename,"r");
	if(fp==NULL)
		return ERROR;

	while(read_line(input_buffer,MAX_INPUT_BUFFER,fp)){

	        if(feof(fp))
		        break;

		if(input_buffer==NULL)
		        continue;

		if(strstr(input_buffer,"interval_length=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\x0");
			interval_length=(temp_buffer==NULL)?60:atoi(temp_buffer);
		        }

		else if(strstr(input_buffer,"log_file=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\x0");
			strncpy(log_file,(temp_buffer==NULL)?"":temp_buffer,sizeof(log_file));
			log_file[sizeof(log_file)-1]='\x0';
			strip(log_file);
		        }

		else if(strstr(input_buffer,"log_archive_path=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\n");
			strncpy(log_archive_path,(temp_buffer==NULL)?"":temp_buffer,sizeof(log_archive_path));
			log_archive_path[sizeof(log_archive_path)-1]='\x0';
			strip(physical_html_path);
			if(log_archive_path[strlen(log_archive_path)-1]!='/' && (strlen(log_archive_path) < sizeof(log_archive_path)-1))
				strcat(log_archive_path,"/");
		        }

		else if(strstr(input_buffer,"log_rotation_method=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\x0");
			if(temp_buffer==NULL)
				log_rotation_method=LOG_ROTATION_NONE;
			else if(!strcmp(temp_buffer,"h"))
				log_rotation_method=LOG_ROTATION_HOURLY;
			else if(!strcmp(temp_buffer,"d"))
				log_rotation_method=LOG_ROTATION_DAILY;
			else if(!strcmp(temp_buffer,"w"))
				log_rotation_method=LOG_ROTATION_WEEKLY;
			else if(!strcmp(temp_buffer,"m"))
				log_rotation_method=LOG_ROTATION_MONTHLY;
		        }

		else if(strstr(input_buffer,"command_file=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\x0");
			strncpy(command_file,(temp_buffer==NULL)?"":temp_buffer,sizeof(command_file));
			command_file[sizeof(command_file)-1]='\x0';
			strip(command_file);
		        }

		else if(strstr(input_buffer,"check_external_commands=")==input_buffer){
			temp_buffer=strtok(input_buffer,"=");
			temp_buffer=strtok(NULL,"\x0");
			check_external_commands=(temp_buffer==NULL)?0:atoi(temp_buffer);
		        }
               }

	fclose(fp);

	return OK;
        }



/* read all object definitions */
int read_all_object_configuration_data(char *config_file,int options){

	/* read in all external config data of the desired type(s) */
	read_object_config_data(config_file,options);

	/* mark what items we've read in... */
	if(options & READ_HOSTS)
		hosts_have_been_read=TRUE;
	if(options & READ_HOSTGROUPS)
		hostgroups_have_been_read=TRUE;
	if(options & READ_CONTACTS)
		contacts_have_been_read=TRUE;
	if(options & READ_CONTACTGROUPS)
		contactgroups_have_been_read=TRUE;
	if(options & READ_SERVICES)
		services_have_been_read=TRUE;
	if(options & READ_TIMEPERIODS)
		timeperiods_have_been_read=TRUE;
	if(options & READ_COMMANDS)
		commands_have_been_read=TRUE;
	if(options & READ_SERVICEESCALATIONS)
		serviceescalations_have_been_read=TRUE;
	if(options & READ_HOSTGROUPESCALATIONS)
		hostgroupescalations_have_been_read=TRUE;

	return OK;
        }


/* read all status data */
int read_all_status_data(char *config_file,int options){
	int result=OK;

	/* don't duplicate things we've already read in */
	if(program_status_has_been_read==TRUE && (options & READ_PROGRAM_STATUS))
		options-=READ_PROGRAM_STATUS;
	if(host_status_has_been_read==TRUE && (options & READ_HOST_STATUS))
		options-=READ_HOST_STATUS;
	if(service_status_has_been_read==TRUE && (options & READ_SERVICE_STATUS))
		options-=READ_SERVICE_STATUS;

	/* read in all external status data */
	result=read_status_data(config_file,options);

	/* mark what items we've read in... */
	if(options & READ_PROGRAM_STATUS)
		program_status_has_been_read=TRUE;
	if(options & READ_HOST_STATUS)
		host_status_has_been_read=TRUE;
	if(options & READ_SERVICE_STATUS)
		service_status_has_been_read=TRUE;

	return result;
        }



/**********************************************************
 *************** MISC UTILITY FUNCTIONS *******************
 **********************************************************/


/* recursively strips newline characters, spaces, and carriage returns from the end of a string */
void strip(char *buffer){
	int x;

	if(buffer==NULL)
		return;

	x=(int)strlen(buffer);

	if(!x)
		return;
	else
		x--;

	if(buffer[x]==' ' || buffer[x]=='\n' || buffer[x]=='\r' || buffer[x]=='\t' || buffer[x]==13)
		buffer[x]='\x0';

	else
		return;

	strip(buffer);

	return;
        }


/* reads a line of text from a file and strips it clean */
char * read_raw_line(char *buffer, int maxlength, FILE *fp){
	char *result;

	/* clear the buffer */
	strcpy(buffer,"");

	/* read in one line of text */
	result=fgets(buffer,maxlength-1,fp);

	/* if we were able to read something, strip extra characters from the end */
	if(result){
		buffer[maxlength-1]='\x0';
		strip(buffer);
	        }

	/* return the number of characters read */
	return result;
        }


/* read a line of text from a file, skipping comments and empty lines */
char * read_line(char *buffer, int maxlength, FILE *fp){
	char *result;
	int keep_line=TRUE;

	while((result=read_raw_line(buffer,maxlength,fp))){

		if(buffer[0]=='#')
			keep_line=FALSE;
		else if(buffer[0]=='\r')
			keep_line=FALSE;
		else if(buffer[0]=='\n')
			keep_line=FALSE;
		else if(buffer[0]=='\x0')
			keep_line=FALSE;
		else
			keep_line=TRUE;

		if(keep_line==TRUE)
			break;
	        }

	return result;
        }



/* get date/time string */
void get_time_string(time_t *raw_time,char *buffer,int buffer_length){
	time_t t;
	struct tm *tm_ptr;
	int day;
	int hour;
	int minute;
	int second;
	int year;
	char *weekdays[7]={"Sun","Mon","Tue","Wed","Thu","Fri","Sat"};
	char *months[12]={"Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sept","Oct","Nov","Dec"};
	char *tzone="";

	if(raw_time==NULL)
		time(&t);
	else 
		t=*raw_time;

	tm_ptr=localtime(&t);

	hour=tm_ptr->tm_hour;
	minute=tm_ptr->tm_min;
	second=tm_ptr->tm_sec;
	day=tm_ptr->tm_mday;
	year=tm_ptr->tm_year+1900;

#ifdef HAVE_TM_ZONE
	tzone=(char *)tm_ptr->tm_zone;
#else
	tzone=(char *)tzname[1];
#endif

	snprintf(buffer,buffer_length,"%s %s %d %02d:%02d:%02d %s %d",weekdays[tm_ptr->tm_wday],months[tm_ptr->tm_mon],day,hour,minute,second,tzone,year);
	buffer[buffer_length-1]='\x0';

	return;
        }


/* get time string for an interval of time */
void get_interval_time_string(int time_units,char *buffer,int buffer_length){
	unsigned long total_seconds;
	int hours=0;
	int minutes=0;
	int seconds=0;

	total_seconds=(unsigned long)(time_units*interval_length);
	hours=(int)total_seconds/3600;
	total_seconds%=3600;
	minutes=(int)total_seconds/60;
	total_seconds%=60;
	seconds=(int)total_seconds;
	snprintf(buffer,buffer_length,"%dh %dm %ds",hours,minutes,seconds);
	buffer[buffer_length-1]='\x0';

	return;
        }


/* get date/time string used in META tags for page expiration */
void get_expire_time_string(time_t *raw_time,char *buffer,int buffer_length){
	time_t t;
	struct tm *tm_ptr;
	int day;
	int hour;
	int minute;
	int second;
	int year;
	char *weekdays[7]={"Sun","Mon","Tue","Wed","Thu","Fri","Sat"};
	char *months[12]={"Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sept","Oct","Nov","Dec"};

	if(raw_time==NULL)
		time(&t);
	else 
		t=*raw_time;

	tm_ptr=gmtime(&t);

	hour=tm_ptr->tm_hour;
	minute=tm_ptr->tm_min;
	second=tm_ptr->tm_sec;
	day=tm_ptr->tm_mday;
	year=tm_ptr->tm_year+1900;

	snprintf(buffer,buffer_length,"%s, %02d %s %d %02d:%02d:%02d GMT",weekdays[tm_ptr->tm_wday],day,months[tm_ptr->tm_mon],year,hour,minute,second);
	buffer[buffer_length-1]='\x0';

	return;
        }



/* fix the problem with strtok() skipping empty options between tokens */	
char *my_strtok(char *buffer,char *tokens){
	char *token_position;
	char *sequence_head;

	if(buffer!=NULL){
		if(original_my_strtok_buffer!=NULL)
			free(original_my_strtok_buffer);
		my_strtok_buffer=malloc(strlen(buffer)+1);
		if(my_strtok_buffer==NULL)
			return NULL;
		original_my_strtok_buffer=my_strtok_buffer;
		strcpy(my_strtok_buffer,buffer);
	        }
	
	sequence_head=my_strtok_buffer;

	if(sequence_head[0]=='\x0')
		return NULL;
	
	token_position=index(my_strtok_buffer,tokens[0]);

	if(token_position==NULL){
		my_strtok_buffer=index(my_strtok_buffer,'\x0');
		return sequence_head;
	        }

	token_position[0]='\x0';
	my_strtok_buffer=token_position+1;

	return sequence_head;

        }


/* fixes compiler problems under Solaris, since strsep() isn't included */
/* this code is taken from the glibc source */
char *my_strsep (char **stringp, const char *delim){
	char *begin, *end;

	begin = *stringp;
	if (begin == NULL)
		return NULL;

	/* A frequent case is when the delimiter string contains only one
	   character.  Here we don't need to call the expensive `strpbrk'
	   function and instead work using `strchr'.  */
	if(delim[0]=='\0' || delim[1]=='\0'){
		char ch = delim[0];

		if(ch=='\0')
			end=NULL;
		else{
			if(*begin==ch)
				end=begin;
			else
				end=strchr(begin+1,ch);
			}
		}

	else
		/* Find the end of the token.  */
		end = strpbrk (begin, delim);

	if(end){

		/* Terminate the token and set *STRINGP past NUL character.  */
		*end++='\0';
		*stringp=end;
		}
	else
		/* No more delimiters; this is the last token.  */
		*stringp=NULL;

	return begin;
	}


/* get days, hours, minutes, and seconds from a raw time_t format or total seconds */
void get_time_breakdown(unsigned long raw_time,int *days,int *hours,int *minutes,int *seconds){
	unsigned long temp_time;
	int temp_days;
	int temp_hours;
	int temp_minutes;
	int temp_seconds;

	temp_time=raw_time;

	temp_days=temp_time/86400;
	temp_time-=(temp_days * 86400);
	temp_hours=temp_time/3600;
	temp_time-=(temp_hours * 3600);
	temp_minutes=temp_time/60;
	temp_time-=(temp_minutes * 60);
	temp_seconds=(int)temp_time;

	*days=temp_days;
	*hours=temp_hours;
	*minutes=temp_minutes;
	*seconds=temp_seconds;

	return;
	}



/* encodes a string in proper URL format */
char * url_encode(char *input){
	int len,output_len;
	int x,y;
	char temp_expansion[4];

	len=(int)strlen(input);
	output_len=(int)sizeof(encoded_url_string);

	encoded_url_string[0]='\x0';

	for(x=0,y=0;x<=len && y<output_len-1;x++){

		if((char)input[x]==(char)'\x0'){
			encoded_url_string[y]='\x0';
			break;
		        }
		else if((char)input[x]<=(char)' '){
			encoded_url_string[y]='+';
			y++;
		        }
		else if((char)input[x]>='0' && (char)input[x]<='9'){
			encoded_url_string[y]=input[x];
			y++;
		        }
		else if((char)input[x]>='A' && (char)input[x]<='Z'){
			encoded_url_string[y]=input[x];
			y++;
		        }
		else if((char)input[x]>=(char)'a' && (char)input[x]<=(char)'z'){
			encoded_url_string[y]=input[x];
			y++;
		        }
		else{
			encoded_url_string[y]='\x0';
			if((int)strlen(encoded_url_string)<(output_len-3)){
				sprintf(temp_expansion,"%%%02X",(unsigned int)input[x]);
				strcat(encoded_url_string,temp_expansion);
				y+=3;
			        }
		        }
	        }

	encoded_url_string[sizeof(encoded_url_string)-1]='\x0';

	return &encoded_url_string[0];
        }



/* determines the log file we should use */
void get_log_archive_to_use(int archive,char *buffer,int buffer_length){
	time_t current_time;
	struct tm *t;

	time(&current_time);

	determine_log_rotation_times(&current_time,archive);

	/* if we're not rotating the logs or if we want the current log, use the main one... */
	if(log_rotation_method==LOG_ROTATION_NONE || archive<=0){
		strncpy(buffer,log_file,buffer_length);
		buffer[buffer_length-1]='\x0';
		return;
	        }

	t=localtime(&this_scheduled_log_rotation);

	snprintf(buffer,buffer_length,"%snetsaint-%02d-%02d-%d-%02d.log",log_archive_path,t->tm_mon+1,t->tm_mday,t->tm_year+1900,t->tm_hour);
	buffer[buffer_length-1]='\x0';

	return;
        }


/* determines log archive to use, given a specific time */
int determine_archive_to_use_from_time(time_t target_time){
	time_t current_time;
	int current_archive=0;

	if(log_rotation_method==LOG_ROTATION_NONE)
		return 0;

	current_time=time(NULL);
	if(target_time>=current_time)
		return 0;

	for(current_archive=0;;current_archive++){
		determine_log_rotation_times(&current_time,current_archive);

		if(target_time>=this_scheduled_log_rotation && target_time<=next_scheduled_log_rotation)
			return current_archive;
	        }
        }


/* determines the log rotation times - past, present, future */
void determine_log_rotation_times(time_t *start_time,int archive){
	time_t past_hour;
	time_t past_day;
	time_t past_week;
	time_t past_month;
	struct tm *t;

	if(archive<0)
		return;

	t=localtime(start_time);

	/* calculate the start of the hour */
	t->tm_sec=0;
	t->tm_min=0;
	past_hour=mktime(t);

	/* calculate the start of the day (midnight, 00:00 hours) */
	t->tm_hour=0;
	past_day=mktime(t);

	/* calculate the start of the day (midnight, 00:00 hours) for this past weekend */
	past_week=(time_t)(past_day-(60*60*24*t->tm_wday));

	/* calculate the start of the day (midnight, 00:00 hours) for the first day this month */
	t->tm_mday=1;
	past_month=mktime(t);

	switch(log_rotation_method){
	case LOG_ROTATION_HOURLY:
		this_scheduled_log_rotation=past_hour;
		last_scheduled_log_rotation=(time_t)(past_hour-(60*60));
		next_scheduled_log_rotation=(time_t)(past_hour+(60*60));
		break;
	case LOG_ROTATION_DAILY:
		this_scheduled_log_rotation=past_day;
		last_scheduled_log_rotation=(time_t)(past_day-(60*60*24));
		next_scheduled_log_rotation=(time_t)(past_day+(60*60*24));
		break;
	case LOG_ROTATION_WEEKLY:
		this_scheduled_log_rotation=past_week;
		last_scheduled_log_rotation=(time_t)(past_week-(60*60*24*7));
		next_scheduled_log_rotation=(time_t)(past_week+(60*60*24*7));
		break;
	case LOG_ROTATION_MONTHLY:
		this_scheduled_log_rotation=(time_t)past_month;
		if(t->tm_mon==0){
			t->tm_mon=11;
			t->tm_year--;
		        }
		else
			t->tm_mon--;
		last_scheduled_log_rotation=mktime(t);
		t=localtime(&past_month);
		if(t->tm_mon==11){
			t->tm_mon=0;
			t->tm_year++;
		        }
		else
			t->tm_mon++;
		next_scheduled_log_rotation=mktime(t);
		break;
	default:
		break;
	        }

	/* recursively work our way backwards in time... */
	if(archive>1)
		determine_log_rotation_times(&last_scheduled_log_rotation,archive-1);

	return;
        }


/* get the status of the netsaint process */
int get_netsaint_process_info(void){
	char input_buffer[MAX_INPUT_BUFFER];
	int process_state=STATE_OK;
	int result=STATE_OK;
	FILE *fp;

	/* if there is no check command defined, we don't know what's going on... */
	if(!strcmp(netsaint_check_command,"")){
		process_state=STATE_UNKNOWN;
		strcpy(netsaint_process_info,"No process check command defined");
	        }

	/* else run the check... */
	else{

		/* clear out the old check results */
		strcpy(netsaint_process_info,"");
		strcpy(input_buffer,"");

		/* run the process check command */
		fp=spopen(netsaint_check_command);
		if(fp==NULL){
			process_state=STATE_UNKNOWN;
			strncpy(netsaint_process_info,"Could not open pipe for process check command",sizeof(netsaint_process_info));
			netsaint_process_info[sizeof(netsaint_process_info)-1]='\x0';
	                }
		else{
			/* grab command output */
			fgets(input_buffer,MAX_INPUT_BUFFER-1,fp);
			result=spclose(fp);

			/* get the program return code */
			process_state=WEXITSTATUS(result);

			/* do some basic bounds checking on the return code */
			if(process_state<-1 || process_state>2)
				process_state=STATE_UNKNOWN;

			/* check the output from the command */
			if(!strcmp(input_buffer,"")){
				process_state=STATE_UNKNOWN;
				strncpy(input_buffer,"NetSaint check command did not return any output",sizeof(input_buffer));
				input_buffer[sizeof(input_buffer)-1]='\x0';
			        }

			/* store output for later use */
			strncpy(netsaint_process_info,input_buffer,sizeof(netsaint_process_info));
			netsaint_process_info[sizeof(netsaint_process_info)-1]='\x0';
		        }
	        }

	netsaint_process_state=process_state;

	return process_state;
        }


/* calculates host state times */
void calculate_host_state_times(char *host_name,unsigned long *tmt, unsigned long *tup, float *ptup, unsigned long *tdown, float *ptdown, unsigned long *tunreachable, float *ptunreachable){
	time_t current_time;
	unsigned long total_program_time;
	unsigned long time_difference;
	unsigned long total_monitored_time;
	unsigned long time_up;
	unsigned long time_down;
	unsigned long time_unreachable;
	float percent_time_up;
	float percent_time_down;
	float percent_time_unreachable;
	hoststatus *temp_hoststatus;

	/* get host status info */
	temp_hoststatus=find_hoststatus(host_name);

	/* get the current time */
	time(&current_time);

	/* get total running time for NetSaint */
	total_program_time=(unsigned long)(current_time-program_start);

	if((unsigned long)temp_hoststatus->last_state_change==0L)
		time_difference=total_program_time;
	else
		time_difference=(unsigned long)(current_time-temp_hoststatus->last_state_change);

	
	time_up=temp_hoststatus->time_up;
	if(temp_hoststatus->status==HOST_UP)
		time_up+=time_difference;
	time_down=temp_hoststatus->time_down;
	if(temp_hoststatus->status==HOST_DOWN)
		time_down+=time_difference;
	time_unreachable=temp_hoststatus->time_unreachable;
	if(temp_hoststatus->status==HOST_UNREACHABLE)
		time_unreachable+=time_difference;

	total_monitored_time=time_up+time_down+time_unreachable;

	if(total_monitored_time==0L){
		percent_time_up=0.0;
		percent_time_down=0.0;
		percent_time_unreachable=0.0;
	        }
	else{
		percent_time_up=(float)(((float)time_up/(float)total_monitored_time)*100.0);
		percent_time_down=(float)(((float)time_down/(float)total_monitored_time)*100.0);
		percent_time_unreachable=(float)(((float)time_unreachable/(float)total_monitored_time)*100.0);
	        }

	*tmt=total_monitored_time;
	*tup=time_up;
	*ptup=percent_time_up;
	*tdown=time_down;
	*ptdown=percent_time_down;
	*tunreachable=time_unreachable;
	*ptunreachable=percent_time_unreachable;

	return;
        }


/* calculates service state times */
void calculate_service_state_times(char *host_name,char *service_desc,unsigned long *tmt, unsigned long *tok, float *ptok, unsigned long *twarn, float *ptwarn, unsigned long *tun, float *ptun, unsigned long *tcrit, float *ptcrit){
	time_t current_time;
	unsigned long total_program_time;
	unsigned long time_difference;
	unsigned long total_monitored_time;
	unsigned long time_ok;
	unsigned long time_warning;
	unsigned long time_unknown;
	unsigned long time_critical;
	float percent_time_ok;
	float percent_time_warning;
	float percent_time_unknown;
	float percent_time_critical;
	servicestatus *temp_svcstatus;

	/* get service status info */
	temp_svcstatus=find_servicestatus(host_name,service_desc);

	/* get the current time */
	time(&current_time);

	/* get total running time for NetSaint */
	total_program_time=(unsigned long)(current_time-program_start);

	if((unsigned long)temp_svcstatus->last_state_change==0L)
		time_difference=total_program_time;
	else
		time_difference=(unsigned long)(current_time-temp_svcstatus->last_state_change);

	
	time_ok=temp_svcstatus->time_ok;
	time_warning=temp_svcstatus->time_warning;
	time_unknown=temp_svcstatus->time_unknown;
	time_critical=temp_svcstatus->time_critical;

	if(temp_svcstatus->state_type==HARD_STATE){

		if(temp_svcstatus->status==SERVICE_OK || temp_svcstatus->status==SERVICE_RECOVERY)
			time_ok+=time_difference;
		if(temp_svcstatus->status==SERVICE_WARNING)
			time_warning+=time_difference;
		if(temp_svcstatus->status==SERVICE_UNKNOWN)
			time_unknown+=time_difference;
		if(temp_svcstatus->status==SERVICE_CRITICAL || temp_svcstatus->status==SERVICE_UNREACHABLE || temp_svcstatus->status==SERVICE_HOST_DOWN)
			time_critical+=time_difference;
	        }
	else{
		if(temp_svcstatus->last_hard_state==SERVICE_OK || temp_svcstatus->last_hard_state==SERVICE_RECOVERY)
			time_ok+=time_difference;
		if(temp_svcstatus->last_hard_state==SERVICE_WARNING)
			time_warning+=time_difference;
		if(temp_svcstatus->last_hard_state==SERVICE_UNKNOWN)
			time_unknown+=time_difference;
		if(temp_svcstatus->last_hard_state==SERVICE_CRITICAL || temp_svcstatus->last_hard_state==SERVICE_UNREACHABLE || temp_svcstatus->last_hard_state==SERVICE_HOST_DOWN)
			time_critical+=time_difference;
	        }

	total_monitored_time=time_ok+time_warning+time_unknown+time_critical;

	if(total_monitored_time==0L){
		percent_time_ok=0.0;
		percent_time_warning=0.0;
		percent_time_unknown=0.0;
		percent_time_critical=0.0;
	        }
	else{
		percent_time_ok=(float)(((float)time_ok/(float)total_monitored_time)*100.0);
		percent_time_warning=(float)(((float)time_warning/(float)total_monitored_time)*100.0);
		percent_time_unknown=(float)(((float)time_unknown/(float)total_monitored_time)*100.0);
		percent_time_critical=(float)(((float)time_critical/(float)total_monitored_time)*100.0);
	        }

	*tmt=total_monitored_time;
	*tok=time_ok;
	*twarn=time_warning;
	*tun=time_unknown;
	*tcrit=time_critical;
	*ptok=percent_time_ok;
	*ptwarn=percent_time_warning;
	*ptun=percent_time_unknown;
	*ptcrit=percent_time_critical;

	return;
        }






/**********************************************************
 *************** COMMON HTML FUNCTIONS ********************
 **********************************************************/

void display_info_table(char *title,int refresh, authdata *current_authdata){
	time_t current_time;
	char date_time[48];

	/* read status */
	read_all_status_data(main_config_file,READ_PROGRAM_STATUS);

	printf("<TABLE CLASS='infoBox' BORDER=1 CELLSPACING=0 CELLPADDING=0>\n");
	printf("<TR><TD CLASS='infoBox'>\n");
	printf("<DIV CLASS='infoBoxTitle'>%s</DIV>\n",title);

	time(&current_time);
	get_time_string(&current_time,date_time,(int)sizeof(date_time));

	printf("Last Updated: %s<BR>\n",date_time);
	if(refresh==TRUE)
		printf("Updated every %d seconds<br>\n",refresh_rate);

	printf("NetSaint Network Monitor - <A HREF='http://www.netsaint.org' TARGET='_new' CLASS='homepageURL'>www.netsaint.org</A><BR>\n");

	if(current_authdata->username!=NULL)
		printf("Logged in as <I>%s</I><BR>\n",(!strcmp(current_authdata->username,""))?"unknown":current_authdata->username);

	get_netsaint_process_info();

	if(netsaint_process_state==STATE_OK)
		printf("- NetSaint process is running<BR>");
	else
		printf("<DIV CLASS='infoBoxBadProcStatus'>- NetSaint process may not be running!<BR>Click <A HREF='%s?type=%d'>here</A> for more info.</DIV>",EXTINFO_CGI,DISPLAY_PROCESS_INFO);

	if(program_mode==ACTIVE_MODE)
		printf("- Notifications can be sent out (active mode)<BR>");
	else
		printf("<DIV CLASS='infoBoxBadProcStatus'>- Notifications cannot be sent out (standby mode)!</DIV>");

	if(execute_service_checks==TRUE)
		printf("- Service checks are actively being executed");
	else
		printf("<DIV CLASS='infoBoxBadProcStatus'>- Service checks are not being actively executed!</DIV>");

	printf("</TD></TR>\n");
	printf("</TABLE>\n");

	return;
        }



void display_nav_table(char *url,int archive){
	char date_time[48];

	if(log_rotation_method!=LOG_ROTATION_NONE){
		printf("<table border=0 cellspacing=0 cellpadding=0 CLASS='navBox'>\n");
		printf("<tr>\n");
		printf("<td align=center valign=center CLASS='navBoxItem'>\n");
		if(archive==0){
			printf("Latest Archive<br>");
			printf("<a href='%sarchive=1'><img src='%s%s' border=0 alt='Latest Archive'></a>",url,url_images_path,LEFT_ARROW_ICON);
		        }
		else{
			printf("Earlier Archive<br>");
			printf("<a href='%sarchive=%d'><img src='%s%s' border=0 alt='Earlier Archive'></a>",url,archive+1,url_images_path,LEFT_ARROW_ICON);
		        }
		printf("</td>\n");

		printf("<td width=15></td>\n");

		printf("<td align=center CLASS='navBoxDate'>\n");
		printf("<DIV CLASS='navBoxTitle'>Log File Navigation</DIV>\n");
		if(archive==0)
			get_time_string((unsigned long*)&this_scheduled_log_rotation,date_time,(int)sizeof(date_time));
		else
			get_time_string((unsigned long*)&last_scheduled_log_rotation,date_time,(int)sizeof(date_time));
		printf("%s",date_time);
		printf("<br>to<br>");
		if(archive==0)
			printf("Present..");
		else{
			get_time_string((unsigned long*)&this_scheduled_log_rotation,date_time,(int)sizeof(date_time));
			printf("%s",date_time);
		        }
		printf("</td>\n");

		if(archive!=0){
			printf("<td width=15></td>\n");

			printf("<td align=center valign=center CLASS='navBoxItem'>\n");
			if(archive==1){
				printf("Current Log<br>");
				printf("<a href='%s'><img src='%s%s' border=0 alt='Current Log'></a>",url,url_images_path,RIGHT_ARROW_ICON);
			        }
			else{
				printf("More Recent Archive<br>");
				printf("<a href='%sarchive=%d'><img src='%s%s' border=0 alt='More Recent Archive'></a>",url,archive-1,url_images_path,RIGHT_ARROW_ICON);
			        }
			printf("</td>\n");
		        }

		printf("</tr>\n");
		printf("</table>\n");
	        }

	return;
        }

