/***********************************************************************
 *
 * CONFIG.C - NetSaint Configuration CGI
 *
 * Written by Ethan Galstad (netsaint@netsaint.org)
 * Last Modified: 06-15-2000
 *
 * This CGI program will display various configuration information.
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 ***********************************************************************/

#include "../common/config.h"
#include "../common/locations.h"
#include "../common/common.h"
#include "../common/objects.h"

#include "getcgi.h"
#include "cgiutils.h"
#include "auth.h"

extern char   main_config_file[MAX_FILENAME_LENGTH];
extern char   url_html_path[MAX_FILENAME_LENGTH];
extern char   url_docs_path[MAX_FILENAME_LENGTH];
extern char   url_images_path[MAX_FILENAME_LENGTH];
extern char   url_stylesheets_path[MAX_FILENAME_LENGTH];

extern host *host_list;
extern hostgroup *hostgroup_list;
extern contactgroup *contactgroup_list;
extern command *command_list;
extern timeperiod *timeperiod_list;
extern service *service_list;
extern contact *contact_list;


#define DISPLAY_NONE             0
#define DISPLAY_HOSTS            1
#define DISPLAY_HOSTGROUPS       2
#define DISPLAY_CONTACTS         3
#define DISPLAY_CONTACTGROUPS    4
#define DISPLAY_SERVICES         5
#define DISPLAY_TIMEPERIODS      6
#define DISPLAY_COMMANDS         7

void document_header(int);
void document_footer(void);
int process_cgivars(void);

void display_hosts(void);
void display_hostgroups(void);
void display_contacts(void);
void display_contactgroups(void);
void display_services(void);
void display_timeperiods(void);
void display_commands(void);

void unauthorized_message(void);


authdata current_authdata;

int display_type=DISPLAY_NONE;

int embedded=FALSE;


int main(void){
	int result=OK;

	/* get the arguments passed in the URL */
	process_cgivars();

	/* reset internal variables */
	reset_cgi_vars();

	/* read the CGI configuration file */
	result=read_cgi_config_file(DEFAULT_CGI_CONFIG_FILE);
	if(result==ERROR){
		document_header(FALSE);
		printf("<P><DIV class='errorMessage'>Error: Could not open CGI configuration file '%s' for reading!</DIV></P>\n",DEFAULT_CGI_CONFIG_FILE);
		document_footer();
		return ERROR;
	        }

	document_header(TRUE);

	/* read the main configuration file */
	result=read_main_config_file(main_config_file);
	if(result==ERROR){
		printf("<P><DIV class='errorMessage'>Error: Could not open main configuration file '%s' for reading!</DIV></P>\n",main_config_file);
		document_footer();
		return ERROR;
	        }

	/* read all object configuration data */
	result=read_all_object_configuration_data(main_config_file,READ_HOSTGROUPS|READ_CONTACTGROUPS|READ_HOSTS|READ_SERVICES);
	if(result==ERROR){
		printf("<P><DIV class='errorMessage'>Error: Could not read some or all object configuration data!</DIV></P>\n");
		document_footer();
		return ERROR;
                }

	/* get authentication information */
	get_authentication_information(&current_authdata);


	/* begin top table */
	printf("<table border=0 width=100%%>\n");
	printf("<tr>\n");

	/* left column of the first row */
	printf("<td align=left valign=top width=50%%>\n");
	display_info_table("Configuration",FALSE,&current_authdata);
	printf("</td>\n");

	/* right hand column of top row */
	printf("<td align=right valign=top width=50%%>\n");
	printf("</td>\n");

	/* end of top table */
	printf("</tr>\n");
	printf("</table>\n");


	switch(display_type){
	case DISPLAY_HOSTS:
		display_hosts();
		break;
	case DISPLAY_HOSTGROUPS:
		display_hostgroups();
		break;
	case DISPLAY_CONTACTS:
		display_contacts();
		break;
	case DISPLAY_CONTACTGROUPS:
		display_contactgroups();
		break;
	case DISPLAY_SERVICES:
		display_services();
		break;
	case DISPLAY_TIMEPERIODS:
		display_timeperiods();
		break;
	case DISPLAY_COMMANDS:
		display_commands();
		break;
	default:
		break;
	        }

	document_footer();

	return OK;
        }




void document_header(int use_stylesheet){
	char date_time[48];
	time_t t;

	if(embedded==TRUE)
		return;

	time(&t);
	get_expire_time_string(&t,date_time,sizeof(date_time));

	printf("Cache-Control: no-store\n");
	printf("Pragma: no-cache\n");
	printf("Last-Modified: %s\n",date_time);
	printf("Expires: %s\n",date_time);
	printf("Content-type: text/html\r\n\r\n");

	printf("<html>\n");
	printf("<head>\n");
	printf("<META HTTP-EQUIV='Pragma' CONTENT='no-cache'>\n");
	printf("<title>\n");
	printf("Configuration\n");
	printf("</title>\n");

	if(use_stylesheet==TRUE)
		printf("<LINK REL='stylesheet' TYPE='text/css' HREF='%s%s'>\n",url_stylesheets_path,CONFIG_CSS);

	printf("</head>\n");

	printf("<body CLASS='config'>\n");

	return;
        }


void document_footer(void){

	if(embedded==TRUE)
		return;

	printf("</body>\n");
	printf("</html>\n");

	return;
        }


int process_cgivars(void){
	char **variables;
	int error=FALSE;
	int x;

	variables=getcgivars();

	for(x=0;variables[x]!=NULL;x++){

		/* do some basic length checking on the variable identifier to prevent buffer overflows */
		if(strlen(variables[x])>=MAX_INPUT_BUFFER-1){
			x++;
			continue;
		        }

		/* we found the configuration type argument */
		else if(!strcmp(variables[x],"type")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			/* what information should we display? */
			if(!strcmp(variables[x],"hosts"))
				display_type=DISPLAY_HOSTS;
			else if(!strcmp(variables[x],"hostgroups"))
				display_type=DISPLAY_HOSTGROUPS;
			else if(!strcmp(variables[x],"contacts"))
				display_type=DISPLAY_CONTACTS;
			else if(!strcmp(variables[x],"contactgroups"))
				display_type=DISPLAY_CONTACTGROUPS;
			else if(!strcmp(variables[x],"services"))
				display_type=DISPLAY_SERVICES;
			else if(!strcmp(variables[x],"timeperiods"))
				display_type=DISPLAY_TIMEPERIODS;
			else if(!strcmp(variables[x],"commands"))
				display_type=DISPLAY_COMMANDS;

			/* we found the embed option */
			else if(!strcmp(variables[x],"embedded"))
				embedded=TRUE;
		        }

		/* we recieved an invalid argument */
		else
			error=TRUE;
	
	        }

	return error;
        }



void display_hosts(void){
	host *temp_host;
	hostsmember *temp_hostsmember;
	int options=0;
	int odd=0;
	char time_string[16];
	int user_has_seen_something=FALSE;
	char *bg_class="";

	printf("<P><DIV ALIGN=CENTER CLASS='dataTitle'>Host Definitions</DIV></P>\n");

	printf("<P><DIV ALIGN=CENTER>\n");
	printf("<TABLE BORDER=0 CLASS='data'>\n");

	printf("<TR>\n");
	printf("<TH CLASS='data'>Host Name</TH>");
	printf("<TH CLASS='data'>Alias/Description</TH>");
	printf("<TH CLASS='data'>Address</TH>");
	printf("<TH CLASS='data'>Parent Hosts</TH>");
	printf("<TH CLASS='data'>Max. Check Attempts</TH>");
	printf("<TH CLASS='data'>Notification Interval</TH>");
	printf("<TH CLASS='data'>Notification Options</TH>");
	printf("<TH CLASS='data'>Notification Period</TH>");
	printf("<TH CLASS='data'>Host Check Command</TH>");
	printf("<TH CLASS='data'>Event Handler</TH>");
	printf("</TR>\n");

	/* check all the hosts... */
	for(temp_host=host_list;temp_host!=NULL;temp_host=temp_host->next){

		/* check to see if user is authorized to view this host information... */
		if(is_authorized_for_host(temp_host,&current_authdata)==FALSE && is_authorized_for_configuration_information(&current_authdata)==FALSE)
			continue;

		user_has_seen_something=TRUE;

		if(odd){
			odd=0;
			bg_class="dataOdd";
		        }
		else{
			odd=1;
			bg_class="dataEven";
		        }

		printf("<TR CLASS='%s'>\n",bg_class);

		printf("<TD CLASS='%s'><a name='%s'>%s</a></TD>\n",bg_class,url_encode(temp_host->name),temp_host->name);
		printf("<TD CLASS='%s'>%s</TD>\n",bg_class,temp_host->alias);
		printf("<TD CLASS='%s'>%s</TD>\n",bg_class,temp_host->address);

		printf("<TD CLASS='%s'>",bg_class);
		for(temp_hostsmember=temp_host->parent_hosts;temp_hostsmember!=NULL;temp_hostsmember=temp_hostsmember->next){

			if(temp_hostsmember!=temp_host->parent_hosts)
				printf(", ");

			printf("<a href='%s?type=hosts#%s'>%s</a>\n",CONFIG_CGI,url_encode(temp_hostsmember->host_name),temp_hostsmember->host_name);
		        }
		if(temp_host->parent_hosts==NULL)
			printf("&nbsp;");
		printf("</TD>\n");

		printf("<TD CLASS='%s'>%d</TD>\n",bg_class,temp_host->max_attempts);

		get_interval_time_string(temp_host->notification_interval,time_string,sizeof(time_string));
		printf("<TD CLASS='%s'>%s</TD>\n",bg_class,(temp_host->notification_interval==0)?"<i>No Re-notification</I>":time_string);

		printf("<TD CLASS='%s'>",bg_class);
		options=0;
		if(temp_host->notify_on_down){
			options=1;
			printf("<A HREF='%sstatuslevels.html#HOST DOWN'>Down</A>",url_docs_path);
		        }
		if(temp_host->notify_on_unreachable){
			printf("%s<A HREF='%sstatuslevels.html#HOST UNREACHABLE'>Unreachable</A>",(options)?", ":"",url_docs_path);
			options=1;
		        }
		if(temp_host->notify_on_recovery){
			printf("%s<A HREF='%sstatuslevels.html#RECOVERY'>Recovery</A>",(options)?", ":"",url_docs_path);
			options=1;
		        }
		if(options==0)
			printf("None");
		printf("</TD>\n");

		printf("<TD CLASS='%s'>",bg_class);
		if(temp_host->notification_period==NULL)
			printf("&nbsp;");
		else
			printf("<a href='%s?type=timeperiods#%s'>%s</a>",CONFIG_CGI,url_encode(temp_host->notification_period),temp_host->notification_period);
		printf("</TD>\n");
		printf("<TD CLASS='%s'>",bg_class);
		if(temp_host->host_check_command==NULL)
			printf("&nbsp;");
		else
			printf("<a href='%s?type=commands#%s'>%s</a></TD>\n",CONFIG_CGI,url_encode(temp_host->host_check_command),temp_host->host_check_command);
		printf("</TD>\n");
		printf("<TD CLASS='%s'>",bg_class);
		if(temp_host->event_handler==NULL)
			printf("&nbsp");
		else
			printf("<a href='%s?type=commands#%s'>%s</a></TD>\n",CONFIG_CGI,url_encode(temp_host->event_handler),temp_host->event_handler);
		printf("</TD>\n");

		printf("</TR>\n");
	        }

	printf("</TABLE>\n");
	printf("</DIV>\n");
	printf("</P>\n");

	/* display some helpful info if the user wasn't authorized for any hosts... */
	if(user_has_seen_something==FALSE && host_list!=NULL)
		unauthorized_message();

	return;
        }



void display_hostgroups(void){
	hostgroup *temp_hostgroup;
	contactgroupsmember *temp_contactgroupsmember;
	hostgroupmember *temp_hostgroupmember;
	int odd=0;
	char *bg_class="";

	/* see if user is authorized to view hostgroup information... */
	if(is_authorized_for_configuration_information(&current_authdata)==FALSE){
		unauthorized_message();
		return;
	        }

	printf("<P><DIV ALIGN=CENTER CLASS='dataTitle'>Host Groups</DIV></P>\n");

	printf("<P>\n");
	printf("<DIV ALIGN=CENTER>\n");

	printf("<TABLE BORDER=0 CLASS='data'>\n");
	printf("<TR>\n");
	printf("<TH CLASS='data'>Group Name</TH>");
	printf("<TH CLASS='data'>Description</TH>");
	printf("<TH CLASS='data'>Contact Groups</TH>");
	printf("<TH CLASS='data'>Hosts</TH>");
	printf("</TR>\n");

	/* check all the hostgroups... */
	for(temp_hostgroup=hostgroup_list;temp_hostgroup!=NULL;temp_hostgroup=temp_hostgroup->next){

		if(odd){
			odd=0;
			bg_class="dataOdd";
		        }
		else{
			odd=1;
			bg_class="dataEven";
		        }

		printf("<TR CLASS='%s'>\n",bg_class);

		printf("<TD CLASS='%s'>%s</TD>",bg_class,temp_hostgroup->group_name);

		printf("<TD CLASS='%s'>%s</TD>\n",bg_class,temp_hostgroup->alias);
		printf("<TD CLASS='%s'>",bg_class);

		/* find all the contact groups for this hostgroup... */
		for(temp_contactgroupsmember=temp_hostgroup->contact_groups;temp_contactgroupsmember!=NULL;temp_contactgroupsmember=temp_contactgroupsmember->next){

			if(temp_contactgroupsmember!=temp_hostgroup->contact_groups)
				printf(", ");

			printf("<A HREF='%s?type=contactgroups#%s'>%s</A>\n",CONFIG_CGI,url_encode(temp_contactgroupsmember->group_name),temp_contactgroupsmember->group_name);
		        }
		printf("</TD>\n");

		printf("<TD CLASS='%s'>",bg_class);

		/* find all the hosts that are members of this hostgroup... */
		for(temp_hostgroupmember=temp_hostgroup->members;temp_hostgroupmember!=NULL;temp_hostgroupmember=temp_hostgroupmember->next){

			if(temp_hostgroupmember!=temp_hostgroup->members)
				printf(", ");
			printf("<A HREF='%s?type=hosts#%s'>%s</A>\n",CONFIG_CGI,url_encode(temp_hostgroupmember->host_name),temp_hostgroupmember->host_name);
		        }
		printf("</TD>\n");

		printf("</TR>\n");
	        }

	printf("</TABLE>\n");
	printf("</DIV>\n");
	printf("</P>\n");

	return;
        }



void display_contacts(void){
	contact *temp_contact;
	commandsmember *temp_commandsmember;
	int odd=0;
	int options;
	char *bg_class="";

	/* see if user is authorized to view contact information... */
	if(is_authorized_for_configuration_information(&current_authdata)==FALSE){
		unauthorized_message();
		return;
	        }

	/* read in contact definitions... */
	read_all_object_configuration_data(main_config_file,READ_CONTACTS);

	printf("<P><DIV ALIGN=CENTER CLASS='dataTitle'>Contacts</DIV></P>\n");

	printf("<P>\n");
	printf("<DIV ALIGN=CENTER>\n");

	printf("<TABLE CLASS='data'>\n");
    
	printf("<TR>\n");
	printf("<TH CLASS='data'>Contact Name</TH>");
	printf("<TH CLASS='data'>Alias</TH>");
	printf("<TH CLASS='data'>Email Address</TH>");
	printf("<TH CLASS='data'>Pager Address/Number</TH>");
	printf("<TH CLASS='data'>Service Notification Options</TH>");
	printf("<TH CLASS='data'>Host Notification Options</TH>");
	printf("<TH CLASS='data'>Service Notification Period</TH>");
	printf("<TH CLASS='data'>Host Notification Period</TH>");
	printf("<TH CLASS='data'>Service Notification Commands</TH>");
	printf("<TH CLASS='data'>Host Notification Commands</TH>");
	printf("</TR>\n");
	
	/* check all contacts... */
	for(temp_contact=contact_list;temp_contact!=NULL;temp_contact=temp_contact->next){

		if(odd){
			odd=0;
			bg_class="dataOdd";
		        }
		else{
			odd=1;
			bg_class="dataEven";
		        }

		printf("<TR CLASS='%s'>\n",bg_class);

		printf("<TD CLASS='%s'><A NAME='%s'>%s</a></TD>\n",bg_class,url_encode(temp_contact->name),temp_contact->name);
		printf("<TD CLASS='%s'>%s</TD>\n",bg_class,temp_contact->alias);
		printf("<TD CLASS='%s'><A HREF='mailto:%s'>%s</A></TD>\n",bg_class,(temp_contact->email==NULL)?"&nbsp;":temp_contact->email,(temp_contact->email==NULL)?"&nbsp;":temp_contact->email);
		printf("<TD CLASS='%s'>%s</TD>\n",bg_class,(temp_contact->pager==NULL)?"&nbsp;":temp_contact->pager);

		printf("<TD CLASS='%s'>",bg_class);
		options=0;
		if(temp_contact->notify_on_service_unknown){
			options=1;
			printf("<A HREF='%sstatuslevels.html#UNKNOWN'>Unknown</A>",url_docs_path);
		        }
		if(temp_contact->notify_on_service_warning){
			printf("%s<A HREF='%sstatuslevels.html#WARNING'>Warning</A>",(options)?", ":"",url_docs_path);
			options=1;
		        }
		if(temp_contact->notify_on_service_critical){
			printf("%s<A HREF='%sstatuslevels.html#CRITICAL'>Critical</A>",(options)?", ":"",url_docs_path);
			options=1;
		        }
		if(temp_contact->notify_on_service_recovery){
			printf("%s<A HREF='%sstatuslevels.html#RECOVERY'>Recovery</A>",(options)?", ":"",url_docs_path);
			options=1;
		        }
		if(!options)
			printf("None");
		printf("</TD>\n");

		printf("<TD CLASS='%s'>",bg_class);
		options=0;
		if(temp_contact->notify_on_host_down){
			options=1;
			printf("<A HREF='%sstatuslevels.html#HOST DOWN'>Down</A>",url_docs_path);
		        }
		if(temp_contact->notify_on_host_unreachable){
			printf("%s<A HREF='%sstatuslevels.html#HOST UNREACHABLE'>Unreachable</A>",(options)?", ":"",url_docs_path);
			options=1;
		        }
		if(temp_contact->notify_on_host_recovery){
			printf("%s<A HREF='%sstatuslevels.html#RECOVERY'>Recovery</A>",(options)?", ":"",url_docs_path);
			options=1;
		        }
		if(!options)
			printf("None");
		printf("</TD>\n");

		printf("<TD CLASS='%s'><A HREF='%s?type=timeperiods#%s'>%s</A></TD>\n",bg_class,CONFIG_CGI,url_encode(temp_contact->service_notification_period),temp_contact->service_notification_period);
		printf("<TD CLASS='%s'><A HREF='%s?type=timeperiods#%s'>%s</A></TD>\n",bg_class,CONFIG_CGI,url_encode(temp_contact->host_notification_period),temp_contact->host_notification_period);

		printf("<TD CLASS='%s'>",bg_class);
		for(temp_commandsmember=temp_contact->service_notification_commands;temp_commandsmember!=NULL;temp_commandsmember=temp_commandsmember->next){

			if(temp_commandsmember!=temp_contact->service_notification_commands)
				printf(", ");

			printf("<A HREF='%s?type=commmands#%s'>%s</A>",CONFIG_CGI,url_encode(temp_commandsmember->command),temp_commandsmember->command);
		        }
		printf("</TD>\n");

		printf("<TD CLASS='%s'>",bg_class);
		for(temp_commandsmember=temp_contact->host_notification_commands;temp_commandsmember!=NULL;temp_commandsmember=temp_commandsmember->next){

			if(temp_commandsmember!=temp_contact->host_notification_commands)
				printf(", ");

			printf("<A HREF='%s?type=commmands#%s'>%s</A>",CONFIG_CGI,url_encode(temp_commandsmember->command),temp_commandsmember->command);
		        }
		printf("</TD>\n");

		printf("</TR>\n");
	        }

	printf("</TABLE>\n");
	printf("</DIV>\n");
	printf("</P>\n");

	return;
        }



void display_contactgroups(void){
	contactgroup *temp_contactgroup;
	contactgroupmember *temp_contactgroupmember;
	int odd=0;
	char *bg_class="";

	/* see if user is authorized to view contactgroup information... */
	if(is_authorized_for_configuration_information(&current_authdata)==FALSE){
		unauthorized_message();
		return;
	        }


	printf("<P><DIV ALIGN=CENTER CLASS='dataTitle'>Contact Groups</DIV></P>\n");

	printf("<P>\n");
	printf("<DIV ALIGN=CENTER>\n");

	printf("<TABLE BORDER=0 CELLSPACING=3 CELLPADDING=0>\n");

	printf("<TR>\n");
	printf("<TH CLASS='data'>Group Name</TH>\n");
	printf("<TH CLASS='data'>Description</TH>\n");
	printf("<TH CLASS='data'>Members</TH>\n");
	printf("</TR>\n");

	/* check all the contact groups... */
	for(temp_contactgroup=contactgroup_list;temp_contactgroup!=NULL;temp_contactgroup=temp_contactgroup->next){

		if(odd){
			odd=0;
			bg_class="dataOdd";
		        }
		else{
			odd=1;
			bg_class="dataEven";
		        }

		printf("<TR CLASS='%s'>\n",bg_class);

		printf("<TD CLASS='%s'><A NAME='%s'></A>%s</TD>\n",bg_class,url_encode(temp_contactgroup->group_name),temp_contactgroup->group_name);
		printf("<TD CLASS='%s'>%s</TD>\n",bg_class,temp_contactgroup->alias);

		/* find all the contact who are members of this contact group... */
		printf("<TD CLASS='%s'>",bg_class);
		for(temp_contactgroupmember=temp_contactgroup->members;temp_contactgroupmember!=NULL;temp_contactgroupmember=temp_contactgroupmember->next){
			
			if(temp_contactgroupmember!=temp_contactgroup->members)
				printf(", ");

			printf("<A HREF='%s?type=contacts#%s'\">%s</A>\n",CONFIG_CGI,url_encode(temp_contactgroupmember->contact_name),temp_contactgroupmember->contact_name);
		        }
		printf("</TD>\n");

		printf("</TR>\n");
	        }

	printf("</TABLE>\n");
	printf("</DIV>\n");
	printf("</P>\n");

	return;
        }



void display_services(void){
	service *temp_service;
	contactgroupsmember *temp_contactgroupsmember;
	char command_line[MAX_INPUT_BUFFER];
	char *command_name="";
	int options;
	int odd=0;
	char time_string[16];
	int user_has_seen_something=FALSE;
	char *bg_class;


	/* read in service definitions... */
	read_all_object_configuration_data(main_config_file,READ_SERVICES);

	printf("<P><DIV ALIGN=CENTER CLASS='dataTitle'>Services</DIV></P>\n");

	printf("<P>\n");
	printf("<DIV ALIGN=CENTER>\n");

	printf("<TABLE BORDER=0 CLASS='data'>\n");
	printf("<TR>\n");
	printf("<TH CLASS='data'>Host</TH>\n");
	printf("<TH CLASS='data'>Service</TH>\n");
	printf("<TH CLASS='data'>Volatile?</TH>\n");
	printf("<TH CLASS='data'>Check Period</TH>\n");
	printf("<TH CLASS='data'>Normal Check Interval</TH>\n");
	printf("<TH CLASS='data'>Retry Check Interal</TH>\n");
	printf("<TH CLASS='data'>Max. Check Attempts</TH>\n");
	printf("<TH CLASS='data'>Parallelize?</TH>\n");
	printf("<TH CLASS='data'>Contact Groups</TH>\n");
	printf("<TH CLASS='data'>Notification Interval</TH>\n");
	printf("<TH CLASS='data'>Notification Options</TH>\n");
	printf("<TH CLASS='data'>Notification Period</TH>\n");
	printf("<TH CLASS='data'>Event Handler</TH>\n");
	printf("<TH CLASS='data'>Check Command</TH>\n");
	printf("</TR>\n");

	/* check all the services... */
	for(temp_service=service_list;temp_service!=NULL;temp_service=temp_service->next){

		/* check to see if user is authorized to view this service information... */
		if(is_authorized_for_service(temp_service,&current_authdata)==FALSE && is_authorized_for_configuration_information(&current_authdata)==FALSE)
			continue;

		user_has_seen_something=TRUE;

		if(odd){
			odd=0;
			bg_class="dataOdd";
		        }
		else{
			odd=1;
			bg_class="dataEven";
		        }

		printf("<TR CLASS='%s'>\n",bg_class);

		printf("<TD CLASS='%s'><A NAME='%s;",bg_class,url_encode(temp_service->host_name));
		printf("%s'></A>",url_encode(temp_service->description));
		printf("<A HREF='%s?type=hosts#%s'>%s</A></TD>\n",CONFIG_CGI,url_encode(temp_service->host_name),temp_service->host_name);
		
		printf("<TD CLASS='%s'>%s</TD>\n",bg_class,temp_service->description);

		printf("<TD CLASS='%s'>%s</TD>\n",bg_class,(temp_service->is_volatile==TRUE)?"Yes":"No");

		printf("<TD CLASS='%s'>",bg_class);
		if(temp_service->check_period==NULL)
			printf("&nbsp;");
		else
			printf("<A HREF='%s?type=timeperiods#%s'>%s</A>",CONFIG_CGI,url_encode(temp_service->check_period),temp_service->check_period);
		printf("</TD>\n");

		get_interval_time_string(temp_service->check_interval,time_string,sizeof(time_string));
		printf("<TD CLASS='%s'>%s</TD>\n",bg_class,time_string);
		get_interval_time_string(temp_service->retry_interval,time_string,sizeof(time_string));
		printf("<TD CLASS='%s'>%s</TD>\n",bg_class,time_string);

		printf("<TD CLASS='%s'>%d</TD>\n",bg_class,temp_service->max_attempts);
		printf("<TD CLASS='%s'>%s</TD>\n",bg_class,(temp_service->parallelize==TRUE)?"Yes":"No");

		printf("<TD CLASS='%s'>",bg_class);
		for(temp_contactgroupsmember=temp_service->contact_groups;temp_contactgroupsmember!=NULL;temp_contactgroupsmember=temp_contactgroupsmember->next){

			if(temp_contactgroupsmember!=temp_service->contact_groups)
				printf(", ");

		        printf("<A HREF='%s?type=contactgroups#%s'>%s</A>",CONFIG_CGI,url_encode(temp_contactgroupsmember->group_name),temp_contactgroupsmember->group_name);
		        }
		if(temp_service->contact_groups==NULL)
			printf("&nbsp;");
		printf("</TD>\n");

		get_interval_time_string(temp_service->notification_interval,time_string,sizeof(time_string));
		printf("<TD CLASS='%s'>%s</TD>\n",bg_class,(temp_service->notification_interval==0)?"<i>No Re-notification</i>":time_string);

		printf("<TD CLASS='%s'>",bg_class);
		options=0;
		if(temp_service->notify_on_unknown){
			options=1;
			printf("<A HREF='%sstatuslevels.html#UNKNOWN'>Unknown</A>",url_docs_path);
		        }
		if(temp_service->notify_on_warning){
			printf("%s<A HREF='%sstatuslevels.html#CRITICAL'>Warning</A>",(options)?", ":"",url_docs_path);
			options=1;
		        }
		if(temp_service->notify_on_critical){
			printf("%s<A HREF='%sstatuslevels.html#CRITICAL'>Critical</A>",(options)?", ":"",url_docs_path);
			options=1;
		        }
		if(temp_service->notify_on_recovery){
			printf("%s<A HREF='%sstatuslevels.html#RECOVERY'>Recovery</A>",(options)?", ":"",url_docs_path);
			options=1;
		        }
		if(!options)
			printf("None");
		printf("</TD>\n");


		printf("<TD CLASS='%s'>",bg_class);
		if(temp_service->notification_period==NULL)
			printf("&nbsp;");
		else
			printf("<A HREF='%s?type=timeperiods#%s'>%s</A>",CONFIG_CGI,url_encode(temp_service->notification_period),temp_service->notification_period);
		printf("</TD>\n");

		printf("<TD CLASS='%s'>",bg_class);
		if(temp_service->event_handler==NULL)
			printf("&nbsp;");
		else
			printf("<A HREF='%s?type=commands#%s'>%s</A>",CONFIG_CGI,url_encode(temp_service->event_handler),temp_service->event_handler);
		printf("</TD>\n");

		strncpy(command_line,temp_service->service_check_command,sizeof(command_line));
		command_line[sizeof(command_line)-1]='\x0';
		command_name=strtok(command_line,"!");
		printf("<TD CLASS='%s'><A HREF='%s?type=commands#%s'>%s</A></TD>\n",bg_class,CONFIG_CGI,url_encode(command_name),temp_service->service_check_command);

		printf("</TR>\n");
	        }

	printf("</TABLE>\n");
	printf("</DIV>\n");
	printf("</P>\n");

	/* display some helpful info if the user wasn't authorized for any services... */
	if(user_has_seen_something==FALSE && service_list!=NULL)
		unauthorized_message();

	return;
        }



void display_timeperiods(void){
	timerange *temp_timerange;
	timeperiod *temp_timeperiod;
	int odd=0;
	int day=0;
	char *bg_class="";
	char timestring[10];
	int hours=0;
	int minutes=0;
	int seconds=0;

	/* see if user is authorized to view time period information... */
	if(is_authorized_for_configuration_information(&current_authdata)==FALSE){
		unauthorized_message();
		return;
	        }

	/* read in time period definitions... */
	read_all_object_configuration_data(main_config_file,READ_TIMEPERIODS);

	printf("<P><DIV ALIGN=CENTER CLASS='dataTitle'>Time Periods</DIV></P>\n");

	printf("<P>\n");
	printf("<DIV ALIGN=CENTER>\n");

	printf("<TABLE BORDER=0 CLASS='data'>\n");
	printf("<TR>\n");
	printf("<TH CLASS='data'>Name</TH>\n");
	printf("<TH CLASS='data'>Alias/Description</TH>\n");
	printf("<TH CLASS='data'>Sunday Time Ranges</TH>\n");
	printf("<TH CLASS='data'>Monday Time Ranges</TH>\n");
	printf("<TH CLASS='data'>Tuesday Time Ranges</TH>\n");
	printf("<TH CLASS='data'>Wednesday Time Ranges</TH>\n");
	printf("<TH CLASS='data'>Thursday Time Ranges</TH>\n");
	printf("<TH CLASS='data'>Friday Time Ranges</TH>\n");
	printf("<TH CLASS='data'>Saturday Time Ranges</TH>\n");
	printf("</TR>\n");

	/* check all the time periods... */
	for(temp_timeperiod=timeperiod_list;temp_timeperiod!=NULL;temp_timeperiod=temp_timeperiod->next){

		if(odd){
			odd=0;
			bg_class="dataOdd";
		        }
		else{
			odd=1;
			bg_class="dataEven";
		        }

		printf("<TR CLASS='%s'>\n",bg_class);

		printf("<TD CLASS='%s'><A NAME='%s'>%s</A></TD>\n",bg_class,url_encode(temp_timeperiod->name),temp_timeperiod->name);
		printf("<TD CLASS='%s'>%s</TD>\n",bg_class,temp_timeperiod->alias);

		for(day=0;day<7;day++){

			printf("<TD CLASS='%s'>",bg_class);

			for(temp_timerange=temp_timeperiod->days[day];temp_timerange!=NULL;temp_timerange=temp_timerange->next){

				if(temp_timerange!=temp_timeperiod->days[day])
					printf(", ");

				hours=temp_timerange->range_start/3600;
				minutes=(temp_timerange->range_start-(hours*3600))/60;
				seconds=temp_timerange->range_start-(hours*3600)-(minutes*60);
				snprintf(timestring,sizeof(timestring)-1,"%02d:%02d:%02d",hours,minutes,seconds);
				timestring[sizeof(timestring)-1]='\x0';
				printf("%s - ",timestring);

				hours=temp_timerange->range_end/3600;
				minutes=(temp_timerange->range_end-(hours*3600))/60;
				seconds=temp_timerange->range_end-(hours*3600)-(minutes*60);
				snprintf(timestring,sizeof(timestring)-1,"%02d:%02d:%02d",hours,minutes,seconds);
				timestring[sizeof(timestring)-1]='\x0';
				printf("%s",timestring);
			        }
			if(temp_timeperiod->days[day]==NULL)
				printf("&nbsp;");

			printf("</TD>\n");
		        }

		printf("</TR>\n");
	        }

	printf("</TABLE>\n");
	printf("</DIV>\n");
	printf("</P>\n");

	return;
        }



void display_commands(void){
	command *temp_command;
	int odd=0;
	char *bg_class="";

	/* see if user is authorized to view command information... */
	if(is_authorized_for_configuration_information(&current_authdata)==FALSE){
		unauthorized_message();
		return;
	        }

	/* read in command definitions... */
	read_all_object_configuration_data(main_config_file,READ_COMMANDS);

	printf("<P><DIV ALIGN=CENTER CLASS='dataTitle'>Commands</DIV></P>\n");

	printf("<P><DIV ALIGN=CENTER>\n");
	printf("<TABLE BORDER=0 CLASS='data'>\n");
	printf("<TR><TH CLASS='data'>Command Name</TH><TH CLASS='data'>Command Line</TH></TR>\n");

	/* check all commands */
	for(temp_command=command_list;temp_command!=NULL;temp_command=temp_command->next){

		if(odd){
			odd=0;
			bg_class="dataEven";
		        }
		else{
			odd=1;
			bg_class="dataOdd";
		        }

		printf("<TR CLASS='%s'>\n",bg_class);

		printf("<TD CLASS='%s'><A NAME='%s'></A>%s</TD>\n",bg_class,url_encode(temp_command->name),temp_command->name);
		printf("<TD CLASS='%s'>%s</TD>\n",bg_class,temp_command->command_line);

		printf("</TR>\n");
	        }

	printf("</TABLE>\n");
	printf("</DIV></P>\n");

	return;
        }



void unauthorized_message(void){

	printf("<P><DIV CLASS='errorMessage'>It appears as though you do not have permission to view the configuration information you requested...</DIV></P>\n");
	printf("<P><DIV CLASS='errorDescription'>If you believe this is an error, check the HTTP server authentication requirements for accessing this CGI<br>");
	printf("and check the authorization options in your CGI configuration file.</DIV></P>\n");

	return;
	}


