/**************************************************************************
 *
 * EXTINFO.C -  NetSaint Extended Information CGI
 *
 * Copyright (c) 1999-2000 Ethan Galstad (netsaint@netsaint.org)
 * Last Modified: 08-20-2000
 *
 * License:
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *************************************************************************/

#include "../common/config.h"
#include "../common/locations.h"
#include "../common/common.h"
#include "../common/objects.h"
#include "../common/comments.h"
#include "../common/statusdata.h"

#include "cgiutils.h"
#include "getcgi.h"
#include "auth.h"
#include "edata.h"

extern char             netsaint_check_command[MAX_INPUT_BUFFER];
extern char             netsaint_process_info[MAX_INPUT_BUFFER];
extern int              netsaint_process_state;
extern int              refresh_rate;

extern time_t		program_start;
extern int		program_mode;
extern time_t           last_mode_change;
extern int              daemon_mode;
extern time_t           last_command_check;
extern time_t           last_log_rotation;
extern int              execute_service_checks;
extern int              accept_passive_service_checks;
extern int              event_handlers_enabled;
extern int              obsess_over_services;

extern char main_config_file[MAX_FILENAME_LENGTH];
extern char url_html_path[MAX_FILENAME_LENGTH];
extern char url_stylesheets_path[MAX_FILENAME_LENGTH];
extern char url_docs_path[MAX_FILENAME_LENGTH];
extern char url_images_path[MAX_FILENAME_LENGTH];
extern char url_logo_images_path[MAX_FILENAME_LENGTH];
extern char log_file[MAX_FILENAME_LENGTH];

extern comment           *comment_list;


#define MAX_MESSAGE_BUFFER		4096

#define HEALTH_WARNING_PERCENTAGE       85
#define HEALTH_CRITICAL_PERCENTAGE      75

void document_header(int);
void document_footer(void);
int process_cgivars(void);

authdata current_authdata;

char *host_name="";
char *service_desc="";

int display_type=DISPLAY_PROCESS_INFO;

void show_process_info(void);
void show_host_info(void);
void show_service_info(void);
void show_network_health(void);

void display_comments(int);


extern hoststatus *hoststatus_list;
extern servicestatus *servicestatus_list;

int embedded=FALSE;
int display_header=TRUE;



int main(void){
	int result=OK;
	char temp_buffer[MAX_INPUT_BUFFER];
	hostextinfo *temp_hostextinfo=NULL;
	host *temp_host=NULL;
	

	/* get the arguments passed in the URL */
	process_cgivars();

	/* reset internal variables */
	reset_cgi_vars();

	/* read the CGI configuration file */
	result=read_cgi_config_file(DEFAULT_CGI_CONFIG_FILE);
	if(result==ERROR){
		document_header(FALSE);
		printf("<P><DIV class='errorMessage'>Error: Could not open CGI configuration file '%s' for reading!</DIV></P>\n",DEFAULT_CGI_CONFIG_FILE);
		document_footer();
		return ERROR;
	        }

	document_header(TRUE);

	/* read the main configuration file */
	result=read_main_config_file(main_config_file);
	if(result==ERROR){
		printf("<P><DIV class='errorMessage'>Error: Could not open main configuration file '%s' for reading!</DIV></P>\n",main_config_file);
		document_footer();
		return ERROR;
	        }

	/* read all object configuration data */
	result=read_all_object_configuration_data(main_config_file,READ_HOSTGROUPS|READ_CONTACTGROUPS|READ_HOSTS|READ_SERVICES);
	if(result==ERROR){
		printf("<P><DIV class='errorMessage'>Error: Could not read some or all object configuration data!</DIV></P>\n");
		document_footer();
		return ERROR;
                }

	/* read all status data */
	result=read_all_status_data(main_config_file,READ_PROGRAM_STATUS|READ_HOST_STATUS|READ_SERVICE_STATUS);
	if(result==ERROR){
		printf("<P><DIV class='errorMessage'>Error: Could not read host and service status information!</DIV></P>\n");
		document_footer();
		free_memory();
		return ERROR;
                }

	/* read in extended host information */
	read_extended_object_config_data(DEFAULT_CGI_CONFIG_FILE);

	/* get authentication information */
	get_authentication_information(&current_authdata);


	if(display_header==TRUE){

		/* begin top table */
		printf("<table border=0 width=100%%>\n");
		printf("<tr>\n");

		/* left column of the first row */
		printf("<td align=left valign=top width=33%%>\n");

		if(display_type==DISPLAY_HOST_INFO)
			snprintf(temp_buffer,sizeof(temp_buffer)-1,"Information For Host <I>%s</I>",host_name);
		else if(display_type==DISPLAY_SERVICE_INFO)
			snprintf(temp_buffer,sizeof(temp_buffer)-1,"Information For Service <I>%s</I> On Host <I>%s</I>",service_desc,host_name);
		else if(display_type==DISPLAY_NETWORK_HEALTH)
			snprintf(temp_buffer,sizeof(temp_buffer)-1,"Network Health");
		else
			snprintf(temp_buffer,sizeof(temp_buffer)-1,"NetSaint Process Information");
		temp_buffer[sizeof(temp_buffer)-1]='\x0';
		display_info_table(temp_buffer,TRUE,&current_authdata);

		/* find the host */
		if(display_type==DISPLAY_HOST_INFO || display_type==DISPLAY_SERVICE_INFO)
			temp_host=find_host(host_name,NULL);

		if((display_type==DISPLAY_HOST_INFO || display_type==DISPLAY_SERVICE_INFO) && temp_host!=NULL){
			printf("<TABLE BORDER=1 CELLPADDING=0 CELLSPACING=0 CLASS='linkBox'>\n");
			printf("<TR><TD CLASS='linkBox'>\n");
			if(display_type==DISPLAY_SERVICE_INFO)
				printf("<A HREF='%s?type=%d&host=%s'>View Information For This Host</A><br>\n",EXTINFO_CGI,DISPLAY_HOST_INFO,url_encode(host_name));
			printf("<A HREF='%s?host=%s'>View Status Detail For This Host</A><BR>\n",STATUS_CGI,url_encode(host_name));
			if(display_type==DISPLAY_HOST_INFO){
				printf("<A HREF='%s?host=%s'>View History For This Host</A><BR>\n",HISTORY_CGI,url_encode(host_name));
#ifdef USE_TRENDS
				printf("<A HREF='%s?host=%s'>View Trends For This Host</A><BR>\n",TRENDS_CGI,url_encode(host_name));
#endif
				printf("<A HREF='%s?host=%s'>View Notifications This Host</A>\n",NOTIFICATIONS_CGI,url_encode(host_name));
		                }
			else{
				printf("<A HREF='%s?host=%s&",HISTORY_CGI,url_encode(host_name));
				printf("service=%s'>View History For This Service</A><BR>\n",url_encode(service_desc));
#ifdef USE_TRENDS
				printf("<A HREF='%s?host=%s&",TRENDS_CGI,url_encode(host_name));
				printf("service=%s'>View Trends For This Service</A><BR>\n",url_encode(service_desc));
#endif
				printf("<A HREF='%s?host=%s&",NOTIFICATIONS_CGI,url_encode(host_name));
				printf("service=%s'>View Notifications This Service</A>\n",url_encode(service_desc));
		                }
			printf("</TD></TR>\n");
			printf("</TABLE>\n");
	                }

		printf("</td>\n");

		/* middle column of top row */
		printf("<td align=center valign=center width=33%%>\n");

		if((display_type==DISPLAY_HOST_INFO || display_type==DISPLAY_SERVICE_INFO) && temp_host!=NULL){

			if(display_type==DISPLAY_SERVICE_INFO)
				printf("<DIV CLASS='dataTitle'>%s</DIV>on<BR>\n",service_desc);
			printf("<DIV CLASS='dataTitle'>%s</DIV>\n",temp_host->alias);
			printf("<DIV CLASS='data'>%s</DIV>\n",temp_host->address);

			temp_hostextinfo=find_hostextinfo(host_name);
			if(temp_hostextinfo!=NULL){
				if(temp_hostextinfo->icon_image!=NULL)
					printf("<img src='%s%s' border=0 alt='%s'><BR CLEAR=ALL>",url_logo_images_path,temp_hostextinfo->icon_image,(temp_hostextinfo->icon_image_alt==NULL)?"":temp_hostextinfo->icon_image_alt);
				if(temp_hostextinfo->icon_image_alt!=NULL)
					printf("<font size=-1><i>( %s )</i><font>\n",temp_hostextinfo->icon_image_alt);
		                }
 	                }

		printf("</td>\n");

		/* right column of top row */
		printf("<td align=right valign=bottom width=33%%>\n");

		if(display_type==DISPLAY_HOST_INFO){
			if(temp_hostextinfo!=NULL && temp_hostextinfo->notes_url!=NULL && strcmp(temp_hostextinfo->notes_url,"")){
				printf("<A HREF='%s'><img src='%s%s' border=0 alt='View Additional Notes For This Host'></A>\n",temp_hostextinfo->notes_url,url_images_path,NOTES_ICON);
				printf("<BR CLEAR=ALL><FONT SIZE=-1><I>There Are Additional<BR>Notes For This Host...</I></FONT>\n");
		                }
	                }

		printf("</td>\n");

		/* end of top table */
		printf("</tr>\n");
		printf("</table>\n");

	        }

	printf("<BR>\n");

	if(display_type==DISPLAY_HOST_INFO)
		show_host_info();
	else if(display_type==DISPLAY_SERVICE_INFO)
		show_service_info();
	else if(display_type==DISPLAY_NETWORK_HEALTH)
		show_network_health();
	else
		show_process_info();

	document_footer();

	/* free all allocated memory */
	free_memory();
	free_extended_data();
	free_comment_data();

	return OK;
        }



void document_header(int use_stylesheet){
	char date_time[48];
	time_t current_time;
	time_t expire_time;

	printf("Cache-Control: no-store\n");
	printf("Pragma: no-cache\n");
	printf("Refresh: %d\n",refresh_rate);

	time(&current_time);
	get_expire_time_string(&current_time,date_time,(int)sizeof(date_time));
	printf("Last-Modified: %s\n",date_time);

	expire_time=(time_t)0L;
	get_expire_time_string(&expire_time,date_time,(int)sizeof(date_time));
	printf("Expires: %s\n",date_time);

	printf("Content-type: text/html\r\n\r\n");

	if(embedded==TRUE)
		return;

	printf("<html>\n");
	printf("<head>\n");
	printf("<title>\n");
	printf("Extended Information\n");
	printf("</title>\n");

	if(use_stylesheet==TRUE)
		printf("<LINK REL='stylesheet' TYPE='text/css' HREF='%s%s'>",url_stylesheets_path,EXTINFO_CSS);

	printf("</head>\n");

	printf("<body CLASS='extinfo'>\n");

	return;
        }


void document_footer(void){

	if(embedded==TRUE)
		return;

	printf("</body>\n");
	printf("</html>\n");

	return;
        }


int process_cgivars(void){
	char **variables;
	int error=FALSE;
	int temp_type;
	int x;

	variables=getcgivars();

	for(x=0;variables[x]!=NULL;x++){

		/* do some basic length checking on the variable identifier to prevent buffer overflows */
		if(strlen(variables[x])>=MAX_INPUT_BUFFER-1){
			x++;
			continue;
		        }

		/* we found the display type */
		else if(!strcmp(variables[x],"type")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
				}
			temp_type=atoi(variables[x]);
			if(temp_type==DISPLAY_HOST_INFO)
				display_type=DISPLAY_HOST_INFO;
			else if(temp_type==DISPLAY_SERVICE_INFO)
				display_type=DISPLAY_SERVICE_INFO;
			else if(temp_type==DISPLAY_NETWORK_HEALTH)
				display_type=DISPLAY_NETWORK_HEALTH;
			else
				display_type=DISPLAY_PROCESS_INFO;
			}

		/* we found the host name */
		else if(!strcmp(variables[x],"host")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			host_name=(char *)malloc(strlen(variables[x])+1);
			if(host_name==NULL)
				host_name="";
			else
				strcpy(host_name,variables[x]);
			}

		/* we found the service name */
		else if(!strcmp(variables[x],"service")){
			x++;
			if(variables[x]==NULL){
				error=TRUE;
				break;
			        }

			service_desc=(char *)malloc(strlen(variables[x])+1);
			if(service_desc==NULL)
				service_desc="";
			else
				strcpy(service_desc,variables[x]);
			}

		/* we found the embed option */
		else if(!strcmp(variables[x],"embedded"))
			embedded=TRUE;

		/* we found the noheader option */
		else if(!strcmp(variables[x],"noheader"))
			display_header=FALSE;
	        }

	return error;
        }



void show_process_info(void){
	char state_string[MAX_INPUT_BUFFER];
	char *state_class="";
	char date_time[48];
	time_t current_time;
	unsigned long run_time;
	char run_time_string[24];
	int days=0;
	int hours=0;
	int minutes=0;
	int seconds=0;

	/* make sure the user has rights to view system information */
	if(is_authorized_for_system_information(&current_authdata)==FALSE){

		printf("<P><DIV CLASS='errorMessage'>It appears as though you do not have permission to view process information...</DIV></P>\n");
		printf("<P><DIV CLASS='errorDescription'>If you believe this is an error, check the HTTP server authentication requirements for accessing this CGI<br>");
		printf("and check the authorization options in your CGI configuration file.</DIV></P>\n");

		return;
	        }

	printf("<P>\n");
	printf("<DIV ALIGN=CENTER>\n");

	printf("<TABLE BORDER=0 CELLPADDING=20>\n");
	printf("<TR><TD>\n");

	printf("<DIV CLASS='dataTitle'>Program Information</DIV>\n");

	printf("<TABLE BORDER=0 CLASS='data'>\n");

	printf("<TR><TH CLASS='data'>Variable</TH><TH CLASS='data'>Value</TH></TR>\n");

	/* program start time */
	get_time_string(&program_start,date_time,(int)sizeof(date_time));
	printf("<TR CLASS='dataOdd'><TD CLASS='dataOdd'>Program Start Time</TD><TD CLASS='dataOdd'>%s</TD></TR>\n",date_time);

	/* total running time */
	time(&current_time);
	run_time=(unsigned long)(current_time-program_start);
	get_time_breakdown(run_time,&days,&hours,&minutes,&seconds);
	sprintf(run_time_string,"%d days %d hrs %d min %d sec",days,hours,minutes,seconds);
	printf("<TR CLASS='dataEven'><TD CLASS='dataEven'>Total Running Time</TD><TD CLASS='dataEven'>%s</TD></TR>\n",run_time_string);

	/* current program mode */
	printf("<TR CLASS='dataOdd'><TD CLASS='dataOdd'>Current Program Mode</TD><TD><DIV CLASS='mode%s'>&nbsp;&nbsp;%s&nbsp;&nbsp;</DIV></TD></TR>\n",(program_mode==ACTIVE_MODE)?"ACTIVE":"STANDBY",(program_mode==ACTIVE_MODE)?"ACTIVE":"STANDBY");

	/* last mode change*/
	get_time_string(&last_mode_change,date_time,(int)sizeof(date_time));
	printf("<TR CLASS='dataEven'><TD CLASS='dataEven'>Last Program Mode Change</TD><TD CLASS='dataEven'>%s</TD></TR>\n",date_time);

	/* service check execution */
	printf("<TR CLASS='dataOdd'><TD CLASS='dataOdd'>Service Checks Being Executed?</TD><TD CLASS='dataOdd'><DIV CLASS='checks%s'>&nbsp;&nbsp;%s&nbsp;&nbsp;</DIV></TD></TR>\n",(execute_service_checks==TRUE)?"ENABLED":"DISABLED",(execute_service_checks==TRUE)?"YES":"NO");

	/* passive service check acceptance */
	printf("<TR CLASS='dataEven'><TD CLASS='dataEven'>Passive Service Checks Being Accepted?</TD><TD CLASS='dataEven'><DIV CLASS='checks%s'>&nbsp;&nbsp;%s&nbsp;&nbsp;</DIV></TD></TR>\n",(accept_passive_service_checks==TRUE)?"ENABLED":"DISABLED",(accept_passive_service_checks==TRUE)?"YES":"NO");

	/* event handlers enabled */
	printf("<TR CLASS='dataOdd'><TD CLASS='dataOdd'>Event Handlers Enabled?</TD><TD CLASS='dataOdd'><DIV CLASS='eventhandlers%s'>&nbsp;&nbsp;%s&nbsp;&nbsp;</DIV></TD></TR>\n",(event_handlers_enabled==TRUE)?"ENABLED":"DISABLED",(event_handlers_enabled==TRUE)?"YES":"NO");

	/* obsessing over services */
	printf("<TR CLASS='dataEven'><TD CLASS='dataEven'>Obsessing Over Services?</TD><TD CLASS='dataEven'>%s</TD></TR>\n",(obsess_over_services==TRUE)?"Yes":"No");

	/* daemon mode */
	printf("<TR CLASS='dataOdd'><TD CLASS='dataOdd'>Running As A Daemon?</TD><TD CLASS='dataOdd'>%s</TD></TR>\n",(daemon_mode==TRUE)?"Yes":"No");

	/* last external check */
	get_time_string(&last_command_check,date_time,(int)sizeof(date_time));
	printf("<TR CLASS='dataEven'><TD CLASS='dataEven'>Last External Command Check</TD><TD CLASS='dataEven'>%s</TD></TR>\n",(last_command_check==(time_t)0)?"N/A":date_time);

	/* last log file rotation */
	get_time_string(&last_log_rotation,date_time,(int)sizeof(date_time));
	printf("<TR CLASS='dataOdd'><TD CLASS='dataOdd'>Last Log File Rotation</TD><TD CLASS='dataOdd'>%s</TD></TR>\n",(last_log_rotation==(time_t)0)?"N/A":date_time);

	printf("</TABLE>\n");

	printf("</TD><TD>\n");

	printf("<DIV ALIGN=CENTER>\n");
	printf("<DIV CLASS='commandTitle'>Process Commands</DIV>\n");

	printf("<TABLE BORDER=1 CELLPADDING=0 CELLSPACING=0 CLASS='command'>\n");
	printf("<TR><TD>\n");

	if(netsaint_process_state==STATE_OK){
		printf("<TABLE BORDER=0 CELLPADDING=0 CELLSPACING=0 CLASS='command'>\n");

		printf("<TR CLASS='command'><TD><img src='%s%s' border=0 ALT='Shutdown the NetSaint Process'></td><td CLASS='command'><a href='%s?cmd_typ=%d'>Shutdown the NetSaint process</a></td></tr>\n",url_images_path,STOP_ICON,COMMAND_CGI,CMD_SHUTDOWN_PROCESS);
		printf("<TR CLASS='command'><TD><img src='%s%s' border=0 ALT='Restart the NetSaint Process'></td><td CLASS='command'><a href='%s?cmd_typ=%d'>Restart the NetSaint process</a></td></tr>\n",url_images_path,RESTART_ICON,COMMAND_CGI,CMD_RESTART_PROCESS);

		if(program_mode==ACTIVE_MODE)
			printf("<TR CLASS='command'><TD><img src='%s%s' border=0 ALT='Switch To Standby Mode'></td><td CLASS='command'><a href='%s?cmd_typ=%d'>Switch to standby mode</a></td></tr>\n",url_images_path,STANDBY_ICON,COMMAND_CGI,CMD_ENTER_STANDBY_MODE);
		else
			printf("<TR CLASS='command'><TD><img src='%s%s' border=0 ALT='Switch To Active Mode'></td><td CLASS='command'><a href='%s?cmd_typ=%d'>Switch to active mode</a></td></tr>\n",url_images_path,ACTIVE_ICON,COMMAND_CGI,CMD_ENTER_ACTIVE_MODE);

		if(execute_service_checks==TRUE)
			printf("<TR CLASS='command'><TD><img src='%s%s' border=0 ALT='Stop Executing Service Checks'></td><td CLASS='command'><a href='%s?cmd_typ=%d'>Stop executing service checks</a></td></tr>\n",url_images_path,DISABLED_ICON,COMMAND_CGI,CMD_STOP_EXECUTING_SVC_CHECKS);
		else
			printf("<TR CLASS='command'><TD><img src='%s%s' border=0 ALT='Start Executing Service Checks'></td><td CLASS='command'><a href='%s?cmd_typ=%d'>Start executing service checks</a></td></tr>\n",url_images_path,ENABLED_ICON,COMMAND_CGI,CMD_START_EXECUTING_SVC_CHECKS);

		if(accept_passive_service_checks==TRUE)
			printf("<TR CLASS='command'><TD><img src='%s%s' border=0 ALT='Stop Accepting Passive Service Checks'></td><td CLASS='command'><a href='%s?cmd_typ=%d'>Stop accepting passive service checks</a></td></tr>\n",url_images_path,DISABLED_ICON,COMMAND_CGI,CMD_STOP_ACCEPTING_PASSIVE_SVC_CHECKS);
		else
			printf("<TR CLASS='command'><TD><img src='%s%s' border=0 ALT='Start Accepting Passive Service Checks'></td><td CLASS='command'><a href='%s?cmd_typ=%d'>Start accepting passive service checks</a></td></tr>\n",url_images_path,ENABLED_ICON,COMMAND_CGI,CMD_START_ACCEPTING_PASSIVE_SVC_CHECKS);
		if(event_handlers_enabled==TRUE)
			printf("<TR CLASS='command'><TD><img src='%s%s' border=0 ALT='Disable Event Handlers'></td><td CLASS='command'><a href='%s?cmd_typ=%d'>Disable event handlers</a></td></tr>\n",url_images_path,DISABLED_ICON,COMMAND_CGI,CMD_DISABLE_EVENT_HANDLERS);
		else
			printf("<TR CLASS='command'><TD><img src='%s%s' border=0 ALT='Enable Event Handlers'></td><td CLASS='command'><a href='%s?cmd_typ=%d'>Enable event handlers</a></td></tr>\n",url_images_path,ENABLED_ICON,COMMAND_CGI,CMD_ENABLE_EVENT_HANDLERS);
		if(obsess_over_services==TRUE)
			printf("<TR CLASS='command'><TD><img src='%s%s' border=0 ALT='Stop Obsessing Over Services'></td><td CLASS='command'><a href='%s?cmd_typ=%d'>Stop obsessing over services</a></td></tr>\n",url_images_path,DISABLED_ICON,COMMAND_CGI,CMD_STOP_OBSESSING_OVER_SVC_CHECKS);
		else
			printf("<TR CLASS='command'><TD><img src='%s%s' border=0 ALT='Start Obsessing Over Services'></td><td CLASS='command'><a href='%s?cmd_typ=%d'>Start obsessing over services</a></td></tr>\n",url_images_path,ENABLED_ICON,COMMAND_CGI,CMD_START_OBSESSING_OVER_SVC_CHECKS);

		printf("</TABLE>\n");
	        }
	else{
		printf("<DIV ALIGN=CENTER CLASS='infoMessage'>It appears as though NetSaint is not running, so commands are temporarily unavailable...\n");
		if(!strcmp(netsaint_check_command,"")){
			printf("<BR><BR>\n");
			printf("Hint: It looks as though you have not defined a command for checking the process state by supplying a value for the <b>netsaint_check_command</b> option in the CGI configuration file.<BR>\n");
			printf("Read the documentation for more information on checking the status of the NetSaint process in the CGIs.\n");
		        }
		printf("</DIV>\n");
	        }

	printf("</TD></TR>\n");
	printf("</TABLE>\n");

	printf("</TD></TR></TABLE>\n");


	printf("<P>");
	printf("<DIV ALIGN=CENTER>\n");

	printf("<DIV CLASS='dataTitle'>Process State Information</DIV>\n");

	printf("<TABLE BORDER=0 CLASS='data'>\n");

	printf("<TR><TH CLASS='data'>Variable</TH><TH CLASS='data'>Value</TH></TR>\n");

	if(netsaint_process_state==STATE_OK){
		strcpy(state_string,"OK");
		state_class="processOK";
		}
	else if(netsaint_process_state==STATE_WARNING){
		strcpy(state_string,"WARNING");
		state_class="processWARNING";
		}
	else if(netsaint_process_state==STATE_CRITICAL){
		strcpy(state_string,"CRITICAL");
		state_class="processCRITICAL";
		}
	else{
		strcpy(state_string,"UNKNOWN");
		state_class="processUNKNOWN";
		}


	/* process state */
	printf("<TR CLASS='dataOdd'><TD CLASS='dataOdd'>Process Status</TD><TD><DIV CLASS='%s'>&nbsp;&nbsp;%s&nbsp;&nbsp;</DIV></TD></TR>\n",state_class,state_string);

	/* process check command */
	printf("<TR CLASS='dataEven'><TD CLASS='dataEven'>Process Check Command</TD><TD CLASS='dataEven'>%s</TD></TR>\n",netsaint_check_command);

	/* process check command result */
	printf("<TR CLASS='dataOdd'><TD CLASS='dataOdd'>Check Command Output</TD><TD CLASS='dataOdd'>%s</TD></TR>\n",netsaint_process_info);

	printf("</TABLE>\n");

	printf("</DIV>\n");
	printf("</P>\n");



	return;
	}


void show_host_info(void){
	hoststatus *temp_hoststatus;
	host *temp_host;
	char date_time[48];
	char state_string[MAX_INPUT_BUFFER];
	char *bg_class="";
	unsigned long total_monitored_time;
	unsigned long time_up;
	unsigned long time_down;
	unsigned long time_unreachable;
	float percent_time_up;
	float percent_time_down;
	float percent_time_unreachable;
	char time_up_string[48];
	char time_down_string[48];
	char time_unreachable_string[48];
	char total_time_string[48];
	int days;
	int hours;
	int minutes;
	int seconds;


	/* get host info */
	temp_host=find_host(host_name,NULL);

	/* make sure the user has rights to view host information */
	if(is_authorized_for_host(temp_host,&current_authdata)==FALSE){

		printf("<P><DIV CLASS='errorMessage'>It appears as though you do not have permission to view information for this host...</DIV></P>\n");
		printf("<P><DIV CLASS='errorDescription'>If you believe this is an error, check the HTTP server authentication requirements for accessing this CGI<br>");
		printf("and check the authorization options in your CGI configuration file.</DIV></P>\n");

		return;
	        }

	/* get host status info */
	temp_hoststatus=find_hoststatus(host_name);

	/* make sure host information exists */
	if(temp_host==NULL){
		printf("<P><DIV CLASS='errorMessage'>Error: Host Not Found!</DIV></P>>");
		return;
		}
	if(temp_hoststatus==NULL){
		printf("<P><DIV CLASS='errorMessage'>Error: Host Status Information Not Found!</DIV></P");
		return;
		}


	printf("<DIV ALIGN=CENTER>\n");
	printf("<TABLE BORDER=0 CELLPADDING=0 WIDTH=100%%>\n");
	printf("<TR>\n");

	printf("<TD ALIGN=CENTER VALIGN=CENTER CLASS='stateInfoPanel'>\n");
	
	printf("<DIV CLASS='dataTitle'>Host State Information</DIV>\n");

	if(temp_hoststatus->last_check==0L)
		printf("<P><DIV ALIGN=CENTER>This host has not yet been checked, so status information is not available.</DIV></P>\n");

	else{

		if(temp_hoststatus->status==HOST_UP){
			strcpy(state_string,"UP");
			bg_class="hostUP";
		        }
		else if(temp_hoststatus->status==HOST_DOWN){
			strcpy(state_string,"DOWN");
			bg_class="hostDOWN";
		        }
		else if(temp_hoststatus->status==HOST_UNREACHABLE){
			strcpy(state_string,"UNREACHABLE");
			bg_class="hostUNREACHABLE";
		        }

		printf("<TABLE BORDER=0 CLASS='data'>\n");
		printf("<TR><TH CLASS='data'>Variable</TH><TH CLASS='data'>Value</TH></TR>\n");

		printf("<TR CLASS='dataEven'><TD CLASS='dataEven'>Host Status</td><td CLASS='dataEven'><DIV CLASS='%s'>&nbsp;&nbsp;%s&nbsp;&nbsp;%s&nbsp;&nbsp;</DIV></td></tr>\n",bg_class,state_string,(temp_hoststatus->problem_has_been_acknowledged==TRUE)?"(Has been acknowledged)":"");

		printf("<TR CLASS='dataOdd'><TD CLASS='dataOdd'>Status Information</td><td CLASS='dataOdd'>%s</td></tr>\n",temp_hoststatus->information);

		get_time_string(&temp_hoststatus->last_check,date_time,(int)sizeof(date_time));
		printf("<TR CLASS='dataEven'><TD CLASS='dataEven'>Last Status Check</td><td CLASS='dataEven'>%s</td></tr>\n",date_time);

		printf("<TR CLASS='dataOdd'><TD CLASS='dataOdd'>Host Checks Enabled?</TD><TD CLASS='dataOdd'><DIV CLASS='checks%s'>&nbsp;&nbsp;%s&nbsp;&nbsp;</DIV></TD></TR>\n",(temp_hoststatus->checks_enabled==TRUE)?"ENABLED":"DISABLED",(temp_hoststatus->checks_enabled==TRUE)?"YES":"NO");

		get_time_string(&temp_hoststatus->last_state_change,date_time,(int)sizeof(date_time));
		printf("<TR CLASS='dataEven'><TD CLASS='dataEven'>Last State Change</td><td CLASS='dataEven'>%s</td></tr>\n",(temp_hoststatus->last_state_change==(time_t)0)?"N/A":date_time);

		get_time_string(&temp_hoststatus->last_notification,date_time,(int)sizeof(date_time));
		printf("<TR CLASS='dataOdd'><TD CLASS='dataOdd'>Last Host Notification</td><td CLASS='dataOdd'>%s</td></tr>\n",(temp_hoststatus->last_notification==(time_t)0)?"N/A":date_time);

		printf("<TR CLASS='dataEven'><TD CLASS='dataEven'>Current Notification Number</td><td CLASS='dataEven'>&nbsp;&nbsp;%d&nbsp;&nbsp;</td></tr>\n",temp_hoststatus->current_notification_number);

		printf("<TR CLASS='dataOdd'><TD CLASS='dataOdd'>Host Notifications Enabled?</td><td CLASS='dataOdd'><DIV CLASS='notifications%s'>&nbsp;&nbsp;%s&nbsp;&nbsp;</DIV></td></tr>\n",(temp_hoststatus->notifications_enabled)?"ENABLED":"DISABLED",(temp_hoststatus->notifications_enabled)?"YES":"NO");

		printf("<TR CLASS='dataEven'><TD CLASS='dataEven'>Event Handler Enabled?</td><td CLASS='dataEven'><DIV CLASS='eventhandlers%s'>&nbsp;&nbsp;%s&nbsp;&nbsp;</DIV></td></tr>\n",(temp_hoststatus->event_handler_enabled)?"ENABLED":"DISABLED",(temp_hoststatus->event_handler_enabled)?"YES":"NO");

		get_time_string(&temp_hoststatus->last_update,date_time,(int)sizeof(date_time));
		printf("<TR CLASS='dataOdd'><TD CLASS='dataOdd'>Last Update</td><td CLASS='dataOdd'>%s</td></tr>\n",(temp_hoststatus->last_update==(time_t)0)?"N/A":date_time);

		printf("</TABLE>\n");
	        }

	printf("</TD>\n");

	printf("<TD ALIGN=CENTER>\n");
	printf("<TABLE BORDER=0 CELLPADDING=20><TR>\n");

	printf("<TD ALIGN=CENTER VALIGN=TOP CLASS='stateStatisticsPanel'>\n");

	printf("<DIV CLASS='dataTitle'>Host State Statistics</DIV>\n");

	if(temp_hoststatus->last_check==0L)
		printf("<P><DIV ALIGN=CENTER>This host has not yet been checked, so state statistics are not available.</DIV></P>\n");

	else{

		/* calculate host state times */
		calculate_host_state_times(temp_host->name,&total_monitored_time,&time_up,&percent_time_up,&time_down,&percent_time_down,&time_unreachable,&percent_time_unreachable);

		get_time_breakdown(total_monitored_time,&days,&hours,&minutes,&seconds);
		snprintf(total_time_string,sizeof(total_time_string)-1,"%dd %dh %dm %ds",days,hours,minutes,seconds);
		get_time_breakdown(time_up,&days,&hours,&minutes,&seconds);
		snprintf(time_up_string,sizeof(time_up_string)-1,"%dd %dh %dm %ds",days,hours,minutes,seconds);
		get_time_breakdown(time_down,&days,&hours,&minutes,&seconds);
		snprintf(time_down_string,sizeof(time_down_string)-1,"%dd %dh %dm %ds",days,hours,minutes,seconds);
		get_time_breakdown(time_unreachable,&days,&hours,&minutes,&seconds);
		snprintf(time_unreachable_string,sizeof(time_unreachable_string)-1,"%dd %dh %dm %ds",days,hours,minutes,seconds);

		printf("<TABLE BORDER=0 CLASS='data'>\n");
		printf("<TR><TH CLASS='data'>State</TH><TH CLASS='data'>Time</TH><TH CLASS='data'>%% Time</TH></TR>\n");

		printf("<tr CLASS='dataEven'><td CLASS='hostUP'>UP</td><td CLASS='dataEven'>%s</td><td CLASS='dataEven'>%2.1f%%</td></tr>\n",time_up_string,percent_time_up);
		printf("<tr CLASS='dataOdd'><td CLASS='hostDOWN'>DOWN</td><td CLASS='dataOdd'>%s</td><td CLASS='dataOdd'>%2.1f%%</td></tr>\n",time_down_string,percent_time_down);
		printf("<tr CLASS='dataEven'><td CLASS='hostUNREACHABLE'>UNREACHABLE</td><td CLASS='dataEven'>%s</td><td CLASS='dataEven'>%2.1f%%</td></tr>\n",time_unreachable_string,percent_time_unreachable);

		printf("<tr><td colspan=3></td></tr>\n");
		printf("<tr CLASS='dataOdd'><td CLASS='dataOdd'>All States</td><td CLASS='dataOdd'>%s</td><td CLASS='dataOdd'>100.0%%</td></tr>\n",total_time_string);
		printf("</table>\n");
		}

	printf("</TD>\n");

	printf("</TR><TR>\n");

	printf("<TD ALIGN=CENTER VALIGN=TOP CLASS='commandPanel'>\n");

	printf("<DIV CLASS='commandTitle'>Host Commands</DIV>\n");

	printf("<TABLE BORDER=1 CELLPADDING=0 CELLSPACING=0 CLASS='command'><TR><TD>\n");

	if(netsaint_process_state==STATE_OK){

		printf("<TABLE BORDER=0 CELLSPACING=0 CELLPADDING=0 CLASS='command'>\n");

		if(temp_hoststatus->checks_enabled==TRUE)
			printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Disable Checks Of This Host'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s'>Disable checks of this host</a></td></tr>\n",url_images_path,DISABLED_ICON,COMMAND_CGI,CMD_DISABLE_HOST_CHECK,url_encode(host_name));
		else
			printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Enable Checks Of This Host'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s'>Enable checks of this host</a></td></tr>\n",url_images_path,ENABLED_ICON,COMMAND_CGI,CMD_ENABLE_HOST_CHECK,url_encode(host_name));
		if(temp_hoststatus->status!=HOST_UP && temp_hoststatus->last_notification!=(time_t)0){
			if(temp_hoststatus->problem_has_been_acknowledged==FALSE)
				printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Acknowledge This Host Problem'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s'>Acknowledge this host problem</a></td></tr>\n",url_images_path,ACKNOWLEDGEMENT_ICON,COMMAND_CGI,CMD_ACKNOWLEDGE_HOST_PROBLEM,url_encode(host_name));
			else
				printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Remove Problem Acknowledgement'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s'>Remove problem acknowledgement</a></td></tr>\n",url_images_path,REMOVE_ACKNOWLEDGEMENT_ICON,COMMAND_CGI,CMD_REMOVE_HOST_ACKNOWLEDGEMENT,url_encode(host_name));
		        }
		if(temp_hoststatus->notifications_enabled==TRUE)
			printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Disable Notifications For This Host'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s'>Disable notifications for this host</a></td></tr>\n",url_images_path,NOTIFICATIONS_DISABLED_ICON,COMMAND_CGI,CMD_DISABLE_HOST_NOTIFICATIONS,url_encode(host_name));
		else
			printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Enable Notifications For This Host'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s'>Enable notifications for this host</a></td></tr>\n",url_images_path,NOTIFICATION_ICON,COMMAND_CGI,CMD_ENABLE_HOST_NOTIFICATIONS,url_encode(host_name));
		if(temp_hoststatus->status!=HOST_UP)
			printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Delay Next Host Notification'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s'>Delay next host notification</a></td></tr>\n",url_images_path,DELAY_ICON,COMMAND_CGI,CMD_DELAY_HOST_NOTIFICATION,url_encode(host_name));
		printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Disable Notifications For All Services On This Host'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s'>Disable notifications for all services on this host</a></td></tr>\n",url_images_path,NOTIFICATIONS_DISABLED_ICON,COMMAND_CGI,CMD_DISABLE_HOST_SVC_NOTIFICATIONS,url_encode(host_name));
		printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Enable Notifications For All Services On This Host'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s'>Enable notifications for all services on this host</a></td></tr>\n",url_images_path,NOTIFICATION_ICON,COMMAND_CGI,CMD_ENABLE_HOST_SVC_NOTIFICATIONS,url_encode(host_name));
		printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Schedule An Immediate Check Of All Services On This Host'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s'>Schedule an immediate check of all services on this host</a></td></tr>\n",url_images_path,DELAY_ICON,COMMAND_CGI,CMD_IMMEDIATE_HOST_SVC_CHECKS,url_encode(host_name));
		printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Disable All Services On This Host'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s'>Disable all services on this host</a></td></tr>\n",url_images_path,DISABLED_ICON,COMMAND_CGI,CMD_DISABLE_HOST_SVC_CHECKS,url_encode(host_name));
		printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Enable All Services On This Host'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s'>Enable all services on this host</a></td></tr>\n",url_images_path,ENABLED_ICON,COMMAND_CGI,CMD_ENABLE_HOST_SVC_CHECKS,url_encode(host_name));
		if(temp_hoststatus->event_handler_enabled==TRUE)
			printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Disable Event Handler For This Host'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s'>Disable event handler for this host</a></td></tr>\n",url_images_path,DISABLED_ICON,COMMAND_CGI,CMD_DISABLE_HOST_EVENT_HANDLER,url_encode(host_name));
		else
			printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Enable Event Handler For This Host'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s'>Enable event handler for this host</a></td></tr>\n",url_images_path,ENABLED_ICON,COMMAND_CGI,CMD_ENABLE_HOST_EVENT_HANDLER,url_encode(host_name));

		printf("</TABLE>\n");
	        }
	else{
		printf("<DIV ALIGN=CENTER CLASS='infoMessage'>It appears as though NetSaint is not running, so commands are temporarily unavailable...<br>\n");
		printf("Click <a href='%s?type=%d'>here</a> to view NetSaint process information</DIV>\n",EXTINFO_CGI,DISPLAY_PROCESS_INFO);
	        }
	printf("</TD></TR></TABLE>\n");

	printf("</TD>\n");

	printf("</TR>\n");
	printf("</TABLE></TR>\n");

	printf("<TR>\n");

	printf("<TD COLSPAN=2 ALIGN=CENTER VALIGN=TOP CLASS='commentPanel'>\n");

	/* display comments */
	display_comments(HOST_COMMENT);

	printf("</TD>\n");

	printf("</TR>\n");
	printf("</TABLE>\n");
	printf("</DIV>\n");

	return;
	}


void show_service_info(void){
	service *temp_service;
	char date_time[48];
	servicestatus *temp_svcstatus;
	char state_string[MAX_INPUT_BUFFER];
	char *bg_class="";
	float percent_time_ok;
	float percent_time_warning;
	float percent_time_unknown;
	float percent_time_critical;
	char time_ok_string[48];
	char time_warning_string[48];
	char time_unknown_string[48];
	char time_critical_string[48];
	char total_time_string[48];
	unsigned long time_ok;
	unsigned long time_warning;
	unsigned long time_unknown;
	unsigned long time_critical;
	unsigned long total_monitored_time;
	int days;
	int hours;
	int minutes;
	int seconds;

	/* find the service */
	temp_service=find_service(host_name,service_desc,NULL);

	/* make sure the user has rights to view service information */
	if(is_authorized_for_service(temp_service,&current_authdata)==FALSE){

		printf("<P><DIV CLASS='errorMessage'>It appears as though you do not have permission to view information for this service...</DIV></P>\n");
		printf("<P><DIV CLASS='errorDescription'>If you believe this is an error, check the HTTP server authentication requirements for accessing this CGI<br>");
		printf("and check the authorization options in your CGI configuration file.</DIV></P>\n");

		return;
	        }

	/* get service status info */
	temp_svcstatus=find_servicestatus(host_name,service_desc);

	/* make sure service information exists */
	if(temp_service==NULL){
		printf("<P><DIV CLASS='errorMessage'>Error: Service Not Found!</DIV></P>");
		return;
		}
	if(temp_svcstatus==NULL){
		printf("<P><DIV CLASS='errorMessage'>Error: Service Status Not Found!</DIV></P>");
		return;
		}


	printf("<DIV ALIGN=CENTER>\n");
	printf("<TABLE BORDER=0 CELLPADDING=0 WIDTH=100%%>\n");
	printf("<TR>\n");

	printf("<TD ALIGN=CENTER VALIGN=CENTER CLASS='stateInfoPanel'>\n");
	
	printf("<DIV CLASS='dataTitle'>Service State Information</DIV>\n");

	if(temp_svcstatus->last_check==0L)
		printf("<P><DIV ALIGN=CENTER>This service has not yet been checked, so its current status information and state statistics are not available.</DIV></P>\n");

	else{

		printf("<TABLE BORDER=0 CLASS='data'>\n");
		printf("<tr><TH CLASS='data'>Variable</th><TH CLASS='data'>Value</th></tr>\n");

		if(temp_svcstatus->status==SERVICE_OK || temp_svcstatus->status==SERVICE_RECOVERY){
			strcpy(state_string,"OK");
			bg_class="serviceOK";
			}
		else if(temp_svcstatus->status==SERVICE_WARNING){
			strcpy(state_string,"WARNING");
			bg_class="serviceWARNING";
			}
		else if(temp_svcstatus->status==SERVICE_CRITICAL || temp_svcstatus->status==SERVICE_UNREACHABLE || temp_svcstatus->status==SERVICE_HOST_DOWN){
			strcpy(state_string,"CRITICAL");
			bg_class="serviceCRITICAL";
			}
		else{
			strcpy(state_string,"UNKNOWN");
			bg_class="serviceUNKNOWN";
			}
		printf("<tr CLASS='dataEven'><td CLASS='dataEven'>Current Status</td><td CLASS='dataEven'><DIV CLASS='%s'>&nbsp;&nbsp;%s&nbsp;&nbsp;%s&nbsp;&nbsp;</DIV></td></tr>\n",bg_class,state_string,(temp_svcstatus->problem_has_been_acknowledged==TRUE)?"(Has been acknowledged)":"");

		printf("<tr CLASS='dataOdd'><td CLASS='dataOdd'>Status Information</td><td CLASS='dataOdd'>%s</td></tr>\n",temp_svcstatus->information);

		printf("<tr CLASS='dataEven'><td CLASS='dataEven'>Current Attempt</td><td CLASS='dataEven'>%d/%d</td></tr>\n",temp_svcstatus->current_attempt,temp_svcstatus->max_attempts);

		printf("<tr CLASS='dataOdd'><td CLASS='dataOdd'>State Type</td><td CLASS='dataOdd'>%s</td></tr>\n",(temp_svcstatus->state_type==HARD_STATE)?"HARD":"SOFT");

		printf("<tr CLASS='dataEven'><td CLASS='dataEven'>Last Check Type</td><td CLASS='dataEven'>%s</td></tr>\n",(temp_svcstatus->check_type==SERVICE_CHECK_ACTIVE)?"ACTIVE":"PASSIVE");

		get_time_string(&temp_svcstatus->last_check,date_time,(int)sizeof(date_time));
		printf("<tr CLASS='dataOdd'><TD CLASS='dataOdd'>Last Check Time</td><td CLASS='dataOdd'>%s</td></tr>\n",date_time);

		get_time_string(&temp_svcstatus->next_check,date_time,(int)sizeof(date_time));
		printf("<tr CLASS='dataEven'><td CLASS='dataEven'>Next Scheduled Check</td><td CLASS='dataEven'>%s</td></tr>\n",(temp_svcstatus->checks_enabled && temp_svcstatus->next_check!=(time_t)0)?date_time:"N/A");

		printf("<tr CLASS='dataOdd'><TD CLASS='dataOdd'>Latency</td><td CLASS='dataOdd'>%lu seconds</td></tr>\n",temp_svcstatus->latency);

		printf("<tr CLASS='dataEven'><td CLASS='dataEven'>Check Duration</td><td CLASS='dataEven'>%lu seconds</td></tr>\n",temp_svcstatus->execution_time);

		printf("<tr CLASS='dataOdd'><TD CLASS='dataOdd'>Service Checks Enabled?</td><td CLASS='dataOdd'><DIV CLASS='checks%s'>&nbsp;&nbsp;%s&nbsp;&nbsp;</DIV></td></tr>\n",(temp_svcstatus->checks_enabled)?"ENABLED":"DISABLED",(temp_svcstatus->checks_enabled)?"YES":"NO");

		printf("<tr CLASS='dataEven'><TD CLASS='dataEven'>Passive Checks Being Accepted?</td><td CLASS='dataEven'><DIV CLASS='checks%s'>&nbsp;&nbsp;%s&nbsp;&nbsp;</DIV></td></tr>\n",(temp_svcstatus->accept_passive_service_checks==TRUE)?"ENABLED":"DISABLED",(temp_svcstatus->accept_passive_service_checks)?"YES":"NO");

		get_time_string(&temp_svcstatus->last_state_change,date_time,(int)sizeof(date_time));
		printf("<tr CLASS='dataOdd'><td CLASS='dataOdd'>Last State Change</td><td CLASS='dataOdd'>%s</td></tr>\n",(temp_svcstatus->last_state_change==(time_t)0)?"N/A":date_time);

		get_time_string(&temp_svcstatus->last_notification,date_time,(int)sizeof(date_time));
		printf("<tr CLASS='dataEven'><td CLASS='dataEven'>Last Service Notification</td><td CLASS='dataEven'>%s</td></tr>\n",(temp_svcstatus->last_notification==(time_t)0)?"N/A":date_time);

		get_time_string(&temp_svcstatus->last_notification,date_time,(int)sizeof(date_time));
		printf("<tr CLASS='dataOdd'><td CLASS='dataOdd'>Current Notification Number</td><td CLASS='dataOdd'>%d</td></tr>\n",temp_svcstatus->current_notification_number);

		printf("<tr CLASS='dataEven'><td CLASS='dataEven'>Service Notifications Enabled?</td><td CLASS='dataEven'><DIV CLASS='notifications%s'>&nbsp;&nbsp;%s&nbsp;&nbsp;</DIV></td></tr>\n",(temp_svcstatus->notifications_enabled)?"ENABLED":"DISABLED",(temp_svcstatus->notifications_enabled)?"YES":"NO");

		printf("<TR CLASS='dataOdd'><TD CLASS='dataOdd'>Event Handler Enabled?</td><td CLASS='dataOdd'><DIV CLASS='eventhandlers%s'>&nbsp;&nbsp;%s&nbsp;&nbsp;</DIV></td></tr>\n",(temp_svcstatus->event_handler_enabled)?"ENABLED":"DISABLED",(temp_svcstatus->event_handler_enabled)?"YES":"NO");

		get_time_string(&temp_svcstatus->last_update,date_time,(int)sizeof(date_time));
		printf("<tr CLASS='dataEven'><td CLASS='dataEven'>Last Update</td><td CLASS='dataEven'>%s</td></tr>\n",(temp_svcstatus->last_update==(time_t)0)?"N/A":date_time);

		printf("</table>\n");
                }
	

	printf("</TD>\n");

	printf("<TD ALIGN=CENTER>\n");
	printf("<TABLE BORDER=0 CELLPADDING=20 ><TR>\n");

	printf("<TD ALIGN=CENTER VALIGN=TOP CLASS='stateStatisticsPanel'>\n");

	printf("<DIV CLASS='dataTitle'>Service State Statistics</DIV>\n");

	if(temp_svcstatus->last_check==0L)
		printf("<P><DIV ALIGN=CENTER>This service has not yet been checked, so state statistics are not available.</DIV></P>\n");

	else{

		/* calculate service state times */
		calculate_service_state_times(temp_service->host_name,temp_service->description,&total_monitored_time,&time_ok,&percent_time_ok,&time_warning,&percent_time_warning,&time_unknown,&percent_time_unknown,&time_critical,&percent_time_critical);

		get_time_breakdown(time_ok,&days,&hours,&minutes,&seconds);
		snprintf(time_ok_string,sizeof(time_ok_string)-1,"%dd %dh %dm %ds",days,hours,minutes,seconds);
		get_time_breakdown(time_warning,&days,&hours,&minutes,&seconds);
		snprintf(time_warning_string,sizeof(time_warning_string)-1,"%dd %dh %dm %ds",days,hours,minutes,seconds);
		get_time_breakdown(time_unknown,&days,&hours,&minutes,&seconds);
		snprintf(time_unknown_string,sizeof(time_unknown_string)-1,"%dd %dh %dm %ds",days,hours,minutes,seconds);
		get_time_breakdown(time_critical,&days,&hours,&minutes,&seconds);
		snprintf(time_critical_string,sizeof(time_critical_string)-1,"%dd %dh %dm %ds",days,hours,minutes,seconds);

		get_time_breakdown(total_monitored_time,&days,&hours,&minutes,&seconds);
		snprintf(total_time_string,sizeof(total_time_string)-1,"%dd %dh %dm %ds",days,hours,minutes,seconds);

		printf("<TABLE BORDER=0 CLASS='data'>\n");
		printf("<TR><TH CLASS='data'>State</TH><TH CLASS='data'>Time</TH><TH CLASS='data'>%% Time</TH></TR>\n");

		printf("<tr CLASS='dataEven'><td CLASS='serviceOK'>OK</td><td CLASS='dataEven'>%s</td><td CLASS='dataEven'>%2.1f%%</td></tr>\n",time_ok_string,percent_time_ok);
		printf("<tr CLASS='dataOdd'><td CLASS='serviceWARNING'>WARNING</td><td CLASS='dataOdd'>%s</td><td CLASS='dataOdd'>%2.1f%%</td></tr>\n",time_warning_string,percent_time_warning);
		printf("<tr CLASS='dataEven'><td CLASS='serviceUNKNOWN'>UNKNOWN</td><td CLASS='dataEven'>%s</td><td CLASS='dataEven'>%2.1f%%</td></tr>\n",time_unknown_string,percent_time_unknown);
		printf("<tr CLASS='dataOdd'><td CLASS='serviceCRITICAL'>CRITICAL</td><td CLASS='dataOdd'>%s</td><td CLASS='dataOdd'>%2.1f%%</td></tr>\n",time_critical_string,percent_time_critical);

		printf("<tr><td colspan=3></td></tr>\n");
		printf("<tr CLASS='dataEven'><td CLASS='dataEven'>All States</td><td CLASS='dataEven'>%s</td><td CLASS='dataEven'>100.0%%</td></tr>\n",total_time_string);
		printf("</table>\n");

		if(temp_svcstatus->state_type==SOFT_STATE)
			printf("<P><DIV ALIGN=CENTER>NOTE: This service is currently in a soft state, so the state statistics may be temporarily inaccurate.  When the service returns to a hard state, the statistics will be correct.</DIV></P>\n");
		}

	printf("</TD>\n");

	printf("</TR>\n");
	printf("<TR>\n");

	printf("<TD ALIGN=CENTER VALIGN=TOP CLASS='commandPanel'>\n");

	printf("<DIV CLASS='dataTitle'>Service Commands</DIV>\n");

	printf("<TABLE BORDER=1 CELLSPACING=0 CELLPADDING=0 CLASS='command'>\n");
	printf("<TR><TD>\n");

	if(netsaint_process_state==STATE_OK){
		printf("<TABLE BORDER=0 CELLSPACING=0 CELLPADDING=0 CLASS='command'>\n");

		if(temp_svcstatus->checks_enabled){
			if(temp_svcstatus->status!=SERVICE_OK && temp_svcstatus->status!=SERVICE_RECOVERY && temp_svcstatus->status!=SERVICE_PENDING && temp_svcstatus->state_type==HARD_STATE && temp_svcstatus->last_notification!=(time_t)0){
				if(temp_svcstatus->problem_has_been_acknowledged==FALSE){
					printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Acknowledge This Service Problem'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s",url_images_path,ACKNOWLEDGEMENT_ICON,COMMAND_CGI,CMD_ACKNOWLEDGE_SVC_PROBLEM,url_encode(host_name));
					printf("&service=%s'>Acknowledge this service problem</a></td></tr>\n",url_encode(service_desc));
				        }
				else{
					printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Remove Problem Acknowledgement'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s",url_images_path,REMOVE_ACKNOWLEDGEMENT_ICON,COMMAND_CGI,CMD_REMOVE_SVC_ACKNOWLEDGEMENT,url_encode(host_name));
					printf("&service=%s'>Remove problem acknowledgement</a></td></tr>\n",url_encode(service_desc));
				        }
			        }
			if(temp_svcstatus->notifications_enabled){
				printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Disable Notifications For This Service'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s",url_images_path,NOTIFICATIONS_DISABLED_ICON,COMMAND_CGI,CMD_DISABLE_SVC_NOTIFICATIONS,url_encode(host_name));
				printf("&service=%s'>Disable notifications for this service</a></td></tr>\n",url_encode(service_desc));
				if(temp_svcstatus->status!=SERVICE_OK){
					printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Delay Next Service Notification'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s",url_images_path,DELAY_ICON,COMMAND_CGI,CMD_DELAY_SVC_NOTIFICATION,url_encode(host_name));
					printf("&service=%s'>Delay next service notification</a></td></tr>\n",url_encode(service_desc));
			                }
			        }
			else{
				printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Enable Notifications For This Service'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s",url_images_path,NOTIFICATION_ICON,COMMAND_CGI,CMD_ENABLE_SVC_NOTIFICATIONS,url_encode(host_name));
				printf("&service=%s'>Enable notifications for this service</a></td></tr>\n",url_encode(service_desc));
			        }
			printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Disable Checks Of This Service'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s",url_images_path,DISABLED_ICON,COMMAND_CGI,CMD_DISABLE_SVC_CHECK,url_encode(host_name));
			printf("&service=%s'>Disable checks of this service</a></td></tr>\n",url_encode(service_desc));
			printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Delay Next Service Check'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s",url_images_path,DELAY_ICON,COMMAND_CGI,CMD_DELAY_SVC_CHECK,url_encode(host_name));
			printf("&service=%s'>Delay next service check</a></td></tr>\n",url_encode(service_desc));
			printf("<tr CLASS='data'><td><img src='%s%s' border=0 ALT='Schedule an Immediate Check of this Service'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s",url_images_path,DELAY_ICON,COMMAND_CGI,CMD_IMMEDIATE_SVC_CHECK,url_encode(host_name));
			printf("&service=%s'>Schedule an immediate check of this service</a></td></tr>\n",url_encode(service_desc));
	                }
		else{
			printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Enable Checks Of This Service'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s",url_images_path,ENABLED_ICON,COMMAND_CGI,CMD_ENABLE_SVC_CHECK,url_encode(host_name));
			printf("&service=%s'>Enable checks of this service</a></td></tr>\n",url_encode(service_desc));
	                }
		if(temp_svcstatus->accept_passive_service_checks==TRUE){
			printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Stop Accepting Passive Checks For This Service'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s",url_images_path,DISABLED_ICON,COMMAND_CGI,CMD_DISABLE_PASSIVE_SVC_CHECKS,url_encode(host_name));
			printf("&service=%s'>Stop accepting passive checks for this service</a></td></tr>\n",url_encode(service_desc));
		        }
		else{
			printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Start Accepting Passive Checks For This Service'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s",url_images_path,ENABLED_ICON,COMMAND_CGI,CMD_ENABLE_PASSIVE_SVC_CHECKS,url_encode(host_name));
			printf("&service=%s'>Start accepting passive checks for this service</a></td></tr>\n",url_encode(service_desc));
		        }
		if(temp_svcstatus->event_handler_enabled==TRUE){
			printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Disable Event Handler For This Service'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s",url_images_path,DISABLED_ICON,COMMAND_CGI,CMD_DISABLE_SVC_EVENT_HANDLER,url_encode(host_name));
			printf("&service=%s'>Disable event handler for this service</a></td></tr>\n",url_encode(service_desc));
		        }
		else{
			printf("<tr CLASS='command'><td><img src='%s%s' border=0 ALT='Enable Event Handler For This Service'></td><td CLASS='command'><a href='%s?cmd_typ=%d&host=%s",url_images_path,ENABLED_ICON,COMMAND_CGI,CMD_ENABLE_SVC_EVENT_HANDLER,url_encode(host_name));
			printf("&service=%s'>Enable event handler for this service</a></td></tr>\n",url_encode(service_desc));
		        }

		printf("</table>\n");
	        }
	else{
		printf("<DIV CLASS='infoMessage'>It appears as though NetSaint is not running, so commands are temporarily unavailable...<br>\n");
		printf("Click <a href='%s?type=%d'>here</a> to view NetSaint process information</DIV>\n",EXTINFO_CGI,DISPLAY_PROCESS_INFO);
	        }

	printf("</td></tr>\n");
	printf("</table>\n");

	printf("</TD>\n");

	printf("</TR></TABLE></TD>\n");
	printf("</TR><TR>\n");

	printf("<TD COLSPAN=2 ALIGN=CENTER VALIGN=TOP CLASS='commentPanel'>\n");

	/* display comments */
	display_comments(SERVICE_COMMENT);

	printf("</TD>\n");

	printf("</TR>\n");
	printf("</TABLE>\n");
	printf("</DIV>\n");

	return;
	}


void show_network_health(void){
	hoststatus *temp_hoststatus;
	host *temp_host;
	int total_hosts=0;
	int total_pending_hosts=0;
	int total_host_health=0;
	int host_health_percentage=0;
	servicestatus *temp_servicestatus;
	service *temp_service;
	int total_services=0;
	int total_disabled_services=0;
	int total_pending_services=0;
	int total_service_health=0;
	int service_health_percentage=0;
	char host_health_image[16];
	char service_health_image[16];

	/* check host health  */
	for(temp_hoststatus=hoststatus_list;temp_hoststatus!=NULL;temp_hoststatus=temp_hoststatus->next){

		temp_host=find_host(temp_hoststatus->host_name,NULL);

		/* make sure user is authorized to view this host */
		if(is_authorized_for_host(temp_host,&current_authdata)==FALSE)
			continue;

		if(temp_hoststatus->status!=HOST_PENDING)
			total_hosts++;
		else
			total_pending_hosts++;

		if(temp_hoststatus->status==HOST_UP)
			total_host_health++;
		
	        }

	/* calculate host health percentage... */
	if(total_hosts==0)
		host_health_percentage=0;
	else
		host_health_percentage=(total_host_health*100)/total_hosts;


	/* check service health */
	for(temp_servicestatus=servicestatus_list;temp_servicestatus!=NULL;temp_servicestatus=temp_servicestatus->next){

		temp_service=find_service(temp_servicestatus->host_name,temp_servicestatus->description,NULL);

		/* make sure user is authorized to view this host */
		if(is_authorized_for_service(temp_service,&current_authdata)==FALSE)
			continue;

		if(temp_servicestatus->checks_enabled==FALSE){
			total_disabled_services++;
			continue;
		        }
		else if(temp_servicestatus->status!=SERVICE_PENDING)
			total_services++;
		else
			total_pending_services++;

		if(temp_servicestatus->status==SERVICE_OK)
			total_service_health+=2;
		else if(temp_servicestatus->status==SERVICE_WARNING || temp_servicestatus->status==SERVICE_UNKNOWN)
			total_service_health++;
	        }

	/* calculate service health percentage... */
	if(total_services==0)
		service_health_percentage=0;
	else
		service_health_percentage=(total_service_health*100)/(total_services*2);



	if(host_health_percentage<HEALTH_CRITICAL_PERCENTAGE)
		strncpy(host_health_image,THERM_CRITICAL_IMAGE,sizeof(host_health_image));
	else if(host_health_percentage<HEALTH_WARNING_PERCENTAGE)
		strncpy(host_health_image,THERM_WARNING_IMAGE,sizeof(host_health_image));
	else
		strncpy(host_health_image,THERM_OK_IMAGE,sizeof(host_health_image));
	host_health_image[sizeof(host_health_image)-1]='\x0';

	if(service_health_percentage<HEALTH_CRITICAL_PERCENTAGE)
		strncpy(service_health_image,THERM_CRITICAL_IMAGE,sizeof(service_health_image));
	else if(service_health_percentage<HEALTH_WARNING_PERCENTAGE)
		strncpy(service_health_image,THERM_WARNING_IMAGE,sizeof(service_health_image));
	else
		strncpy(service_health_image,THERM_OK_IMAGE,sizeof(service_health_image));
	service_health_image[sizeof(service_health_image)-1]='\x0';


	printf("<p>\n");
	printf("<div align=center>\n");

	printf("<table border=0 cellpadding=20>\n");
	printf("<tr>\n");
	printf("<td align=center valign=top>\n");
	
	printf("<p>\n");
	printf("<table border=1 cellspacing=0 cellpadding=0 CLASS='thermBox'>\n");
	printf("<tr>\n");
	printf("<td>\n");
	printf("<table border=0>\n");
	printf("<tr>\n");
	printf("<td><img src='%s%s' border=0></td>\n",url_images_path,THERM_SCALE_IMAGE);
	printf("<td valign=bottom><img src='%s%s' border=0 width=25 height=%d alt='%d%% Health'></td>\n",url_images_path,host_health_image,host_health_percentage,host_health_percentage);
	printf("</tr>\n");
	printf("</table>\n");
	printf("</td>\n");
	printf("</tr>\n");
	printf("</table>\n");
	printf("Host Health: <strong>%d%%</strong>\n",host_health_percentage);
	printf("</p>\n");

	printf("<p>\n");
	printf("<table border=1 cellpadding=0 cellspacing=0 CLASS='healthBox'>\n");
	printf("<tr>\n");
	printf("<td>Hosts Checked:</td><td><strong>%d</strong></td>\n",total_hosts);
	printf("</tr>\n");
	printf("<tr>\n");
	printf("<td>Hosts Pending:</td><td><strong>%d</strong></td>\n",total_pending_hosts);
	printf("</tr>\n");
	printf("</table>\n");
	printf("</p>\n");

	printf("</td>\n");
	printf("<td align=center valign=top>\n");

	printf("<p>\n");
	printf("<table border=1 cellspacing=0 cellpadding=0 CLASS='thermBox'>\n");
	printf("<tr>\n");
	printf("<td>\n");
	printf("<table border=0>\n");
	printf("<tr>\n");
	printf("<td><img src='%s%s' border=0></td>\n",url_images_path,THERM_SCALE_IMAGE);
	printf("<td valign=bottom><img src='%s%s' border=0 width=25 height=%d alt='%d%% Health'></td>\n",url_images_path,service_health_image,service_health_percentage,service_health_percentage);
	printf("</tr>\n");
	printf("</table>\n");
	printf("</td>\n");
	printf("</tr>\n");
	printf("</table>\n");
	printf("Service Health: <strong>%d%%</strong>\n",service_health_percentage);
	printf("</p>\n");

	printf("<p>\n");
	printf("<table border=1 cellpadding=0 cellspacing=0 CLASS='healthBox'>\n");
	printf("<tr>\n");
	printf("<td>Services Checked:</td><td><strong>%d</strong></td>\n",total_services);
	printf("</tr>\n");
	printf("<tr>\n");
	printf("<td>Services Disabled:</td><td><strong>%d</strong></td>\n",total_disabled_services);
	printf("</tr>\n");
	printf("<tr>\n");
	printf("<td>Services Pending:</td><td><strong>%d</strong></td>\n",total_pending_services);
	printf("</tr>\n");
	printf("</table>\n");
	printf("</p>\n");

	printf("</td>\n");
	printf("</tr>\n");
	printf("</table>\n");

	printf("</div>\n");
	printf("</p>\n");

	printf("<P><DIV ALIGN=CENTER CLASS='infoMessage'>\n");
	printf("Network health statistics are computed based upon the number of hosts and services that have been checked.<br>\n");
	printf("Host and services which have not yet been checked or are disabled are not included in the percentages.\n");
	printf("</DIV></P>\n");

	return;
        }



void display_comments(int type){
	host *temp_host=NULL;
	service *temp_service=NULL;
	int total_comments=0;
	int display_comment=FALSE;
	char *bg_class="";
	int odd=0;
	char date_time[48];
	comment *temp_comment;


	/* find the host or service */
	if(type==HOST_COMMENT){
		temp_host=find_host(host_name,NULL);
		if(temp_host==NULL)
			return;
	        }
	else{
		temp_service=find_service(host_name,service_desc,NULL);
		if(temp_service==NULL)
			return;
	        }


	printf("<A NAME=COMMENTS></A><DIV CLASS='commentTitle'>%s Comments</DIV>\n",(type==HOST_COMMENT)?"Host":"Service");
	printf("<TABLE BORDER=0>\n");

	printf("<tr><td valign=center><img src='%s%s' border=0 align=center></td><td CLASS='comment'>",url_images_path,COMMENT_ICON);
	if(type==HOST_COMMENT)
		printf("<a href='%s?cmd_typ=%d&host=%s'>",COMMAND_CGI,CMD_ADD_HOST_COMMENT,url_encode(host_name));
	else{
		printf("<a href='%s?cmd_typ=%d&host=%s&",COMMAND_CGI,CMD_ADD_SVC_COMMENT,url_encode(host_name));
		printf("service=%s'>",url_encode(service_desc));
	        }
	printf("Add a new comment</a></td></tr>\n");

	printf("<tr><td valign=center><img src='%s%s' border=0 align=center></td><td CLASS='comment'>",url_images_path,DELETE_ICON);
	if(type==HOST_COMMENT)
		printf("<a href='%s?cmd_typ=%d&host=%s'>",COMMAND_CGI,CMD_DEL_ALL_HOST_COMMENTS,url_encode(host_name));
	else{
		printf("<a href='%s?cmd_typ=%d&host=%s&",COMMAND_CGI,CMD_DEL_ALL_SVC_COMMENTS,url_encode(host_name));
		printf("service=%s'>",url_encode(service_desc));
	        }
	printf("Delete all comments</a></td></tr>\n");

	printf("</TABLE>\n");
	printf("</DIV>\n");
	printf("</P>\n");


	printf("<P>\n");
	printf("<DIV ALIGN=CENTER>\n");
	printf("<TABLE BORDER=0>\n");
	printf("<TR CLASS='comment'><TH CLASS='comment'>Entry Time</TH><TH CLASS='comment'>Author</TH><TH CLASS='comment'>Comment</TH><TH CLASS='comment'>Comment ID</TH><TH CLASS='comment'>Persistent</TH><TH CLASS='comment'>Actions</TH></TR>\n");

	/* read in all comments */
	read_comment_data(main_config_file);

	/* check all the comments to see if they apply to this host or service */
	for(temp_comment=comment_list;temp_comment!=NULL;temp_comment=temp_comment->next){

		display_comment=FALSE;

		if(type==HOST_COMMENT && temp_comment->comment_type==HOST_COMMENT && !strcmp(temp_comment->host_name,host_name))
			display_comment=TRUE;

		else if(type==SERVICE_COMMENT && temp_comment->comment_type==SERVICE_COMMENT && !strcmp(temp_comment->host_name,host_name) && !strcmp(temp_comment->service_description,service_desc))
			display_comment=TRUE;

		if(display_comment==TRUE){

			if(odd){
				odd=0;
				bg_class="commentOdd";
			        }
			else{
				odd=1;
				bg_class="commentEven";
			        }

			get_time_string(&temp_comment->entry_time,date_time,(int)sizeof(date_time));
			printf("<tr CLASS='%s'>",bg_class);
			printf("<td CLASS='%s'>%s</td><td CLASS='%s'>%s</td><td CLASS='%s'>%s</td><td CLASS='%s'>%d</td><td CLASS='%s'>%s</td>",bg_class,date_time,bg_class,temp_comment->author,bg_class,temp_comment->comment_data,bg_class,temp_comment->comment_id,bg_class,(temp_comment->persistent)?"Yes":"No");
			printf("<td><a href='%s?cmd_typ=%d&com_id=%d'><img src='%s%s' border=0 ALT='Delete This Comment'></td>",COMMAND_CGI,(type==HOST_COMMENT)?CMD_DEL_HOST_COMMENT:CMD_DEL_SVC_COMMENT,temp_comment->comment_id,url_images_path,DELETE_ICON);
			printf("</tr>\n");
			}
	        }

	printf("</TABLE>\n");

	/* see if this host or service has any comments associated with it */
	if(type==HOST_COMMENT)
		total_comments=number_of_host_comments(temp_host->name);
	else
		total_comments=number_of_service_comments(temp_service->host_name,temp_service->description);
	if(total_comments==0)
		printf("This %s has no comments associated with it",(type==HOST_COMMENT)?"host":"service");


	return;
        }

