/* $Id: check_radius.c,v 1.10.2.1 2001/01/17 22:38:45 karldebisschop Exp $ */

#include "config.h"
#include "common.h"
#include "utils.h"
#include <radiusclient.h>

#define PROGNAME "check_radius"

int process_arguments(int, char **);
void print_usage(void);
void print_help(void);

char *server=NULL;
int port=PW_AUTH_UDP_PORT;
char *username=NULL;
char *password=NULL;
char *expect=NULL;
char *config_file=NULL;
int retries=1;
int verbose=FALSE;

ENV *env = NULL;

int
main (int argc, char **argv)
{
	UINT4 service;
	char msg[BUFFER_LEN];
	SEND_DATA       data={0};
	int             result;
	UINT4 client_id; 

	if(process_arguments(argc,argv)==ERROR)
		usage("Could not parse arguments\n");

	if ((config_file && rc_read_config(config_file)) ||
	    rc_read_dictionary(rc_conf_str("dictionary")))
	  terminate(STATE_UNKNOWN,"Config file error");

	service = PW_AUTHENTICATE_ONLY;

	if (!(rc_avpair_add(&data.send_pairs, PW_SERVICE_TYPE, &service, 0) &&
	rc_avpair_add(&data.send_pairs, PW_USER_NAME, username, 0) &&
	rc_avpair_add(&data.send_pairs, PW_USER_PASSWORD, password, 0)))
	  terminate(STATE_UNKNOWN,"Out of Memory?");

	/* 
	 * Fill in NAS-IP-Address 
	 */ 
 
	if ((client_id = rc_own_ipaddress()) == 0) 
		return (ERROR_RC); 
 
	if (rc_avpair_add(&(data.send_pairs),PW_NAS_IP_ADDRESS,&client_id,0 )==NULL) 
		return (ERROR_RC); 

	rc_buildreq(&data,PW_ACCESS_REQUEST,server,port,timeout_interval,retries);

	result = rc_send_server(&data, msg);
	rc_avpair_free(data.send_pairs);
	if (data.receive_pairs) rc_avpair_free(data.receive_pairs);

	if (result == TIMEOUT_RC) terminate(STATE_CRITICAL,"Timeout");
	if (result == ERROR_RC) terminate(STATE_CRITICAL,"Auth Error");
	if (result == BADRESP_RC) terminate(STATE_WARNING,"Auth Failed");
	if (expect && !strstr(msg,expect)) terminate(STATE_WARNING,msg); 
	if (result == OK_RC) terminate(STATE_OK,"Auth OK");
	return(0);
}



/* process command-line arguments */
int process_arguments(int argc, char **argv)
{
	int c;

#ifdef HAVE_GETOPT_H
	int option_index = 0;
	static struct option long_options[] =
	{ 
		{"hostname",required_argument,0,'H'},
		{"port",    required_argument,0,'P'},
		{"username",required_argument,0,'u'},
		{"password",required_argument,0,'p'},
		{"filename",required_argument,0,'F'},
		{"expect",  required_argument,0,'e'},
		{"retries", required_argument,0,'r'},
		{"timeout", required_argument,0,'t'},
		{"verbose", no_argument,      0,'v'},
		{"version", no_argument,      0,'V'},
		{"help",    no_argument,      0,'h'},
		{0,0,0,0}
	};
#endif

 	if(argc<2) return ERROR;
	
	if (argc == 9) {
		config_file=argv[1];
		username=argv[2];
		password=argv[3];
		if (is_intpos(argv[4]))
			timeout_interval=atoi(argv[4]);
		else
			usage("Timeout interval must be a positive integer");
		if (is_intpos(argv[5]))
			retries=atoi(argv[5]);
		else
			usage("Number of retries must be a positive integer");
		server=argv[6];
		if (is_intpos(argv[7]))
			port=atoi(argv[7]);
		else
			usage("Server port must be a positive integer");
		expect=argv[8];
		return OK;
	}

	while (1){
#ifdef HAVE_GETOPT_H
		c = getopt_long(argc,argv,"+hVvH:P:F:u:p:t:r:e:",long_options,&option_index);
#else
		c = getopt(argc,argv,"+hVvH:P:F:u:p:t:r:e:");
#endif

		if (c==-1||c==EOF||c==1)
			break;

		switch (c)
			{
			case '?': /* print short usage statement if args not parsable */
				printf("%s: Unknown argument: %s\n\n",my_basename(argv[0]),optarg);
				print_usage();
				exit(STATE_UNKNOWN);
			case 'h': /* help */
				print_help();
				exit(OK);
			case 'V': /* version */
				print_revision(my_basename(argv[0]),"$Revision: 1.10.2.1 $");
				exit(OK);
			case 'v': /* verbose mode */
				verbose=TRUE;
				break;
			case 'H': /* hostname */
				if(is_host(optarg)==FALSE){
					printf("Invalid host name/address\n\n");
					print_usage();
					exit(STATE_UNKNOWN);
				}
				server=optarg;
				break;
			case 'P': /* port */
				if (is_intnonneg(optarg))
					port=atoi(optarg);
				else
					usage("Server port must be a positive integer");
				break;
			case 'u': /* username */
				username=optarg;
				break;
			case 'p': /* password */
				password=optarg;
				break;
			case 'F': /* configuration file */
				config_file=optarg;
				break;
			case 'e': /* expect */
				expect=optarg;
				break;
			case 'r': /* retries */
				if (is_intpos(optarg))
					retries=atoi(optarg);
				else
					usage("Number of retries must be a positive integer");
				break;
			case 't': /* timeout */
				if (is_intpos(optarg))
					timeout_interval=atoi(optarg);
				else
					usage("Timeout interval must be a positive integer");
				break;
			}
	}
	return OK;
}




void print_usage(void)
{
	printf("Usage: %s -H host [-P port] -F config_file -u username -p password -t timeout -r retries -e expect\n",PROGNAME);
}





void print_help(void)
{
	print_revision(PROGNAME,"$Revision: 1.10.2.1 $");
	printf
		("Copyright (c) 2000 Robert August Vincent II/Karl DeBisschop\n\n"
		 "This plugin tests an radius server on the specified host.\n\n");
	print_usage();
	printf
		("Options:\n"
		 " -H, --hostname=HOST\n"
		 "    Host name argument for servers using host headers (use numeric\n"
		 "    address if possible to bypass DNS lookup).\n"
		 " -P, --port=INTEGER\n"
		 "    Port number (default: %d)\n"
		 " -u, --username=STRING\n"
		 "    The user to authenticate\n"
		 " -p, --password=STRING\n"
		 "    Password for autentication (SECURITY RISK)\n"
		 " -F, --filename=STRING\n"
		 "    Configuration file\n"
		 " -e, --expect=STRING\n"
		 "    Response string to expect from the server\n"
		 " -r, --retries=INTEGER\n"
		 "    Number of times to retry a failed connection\n"
		 " -t, --timeout=INTEGER\n"
		 "    Seconds before connection times out (default: %d)\n"
		 " -v\n"
		 "    Show details for command-line debugging (do not use with netsaint server)\n"
		 " -h, --help\n"
		 "    Print detailed help screen\n"
		 " -V, --version\n"
		 "    Print version information\n",
		 port,timeout_interval);
}
