/******************************************************************************
 *
 * CHECK_TCP.C
 *
 * Program: TCP port plugin for NetSaint
 * License: GPL
 * Copyright (c) 1999 Ethan Galstad (netsaint@linuxbox.com)
 *
 * $Id: check_tcp.c,v 1.20 2000/11/03 14:52:43 karldebisschop Exp $
 *
 * Description:
 *
 * This plugin will attempt to connect to the specified port
 * on the host.  Successul connects return STATE_OK, refusals
 * and timeouts return STATE_CRITICAL, other errors return
 * STATE_UNKNOWN.
 *
 * License Information:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *****************************************************************************/

#include "config.h"
#include "common.h"
#include "netutils.h"
#include "utils.h"

#define TCP_PROTOCOL 1
#define UDP_PROTOCOL 2

int process_arguments(int, char **);
void print_usage(void);
void print_help(void);

char *PROGNAME=NULL;
char *SERVICE=NULL;
char *SEND=NULL;
char *EXPECT=NULL;
char *QUIT=NULL;
int PROTOCOL=0;
int PORT=0;

int server_port=0;
char *server_address=NULL;
char *server_send=NULL;
char *server_quit=NULL;
char **server_expect=NULL;
int server_expect_count=0;
char **warn_codes=NULL;
int warn_codes_count=0;
char **crit_codes=NULL;
int crit_codes_count=0;
int delay=0;
int warning_time=0;
int check_warning_time=FALSE;
int critical_time=0;
int check_critical_time=FALSE;
int verbose=FALSE;

int main(int argc, char **argv){
	int sd;
	int result;
	int i;
	char buffer[MAX_INPUT_BUFFER]="";
	char *status=NULL;
	char *output=NULL;
	char *ptr=NULL;

	if (strstr(argv[0],"check_udp")) {
		PROGNAME=strscpy(PROGNAME,"check_udp");
		SERVICE=strscpy(SERVICE,"UDP");
		SEND=NULL;
		EXPECT=NULL;
		QUIT=NULL;
		PROTOCOL=UDP_PROTOCOL;
		PORT=0;
	}
	else if (strstr(argv[0],"check_tcp")) {
		PROGNAME=strscpy(PROGNAME,"check_tcp");
		SERVICE=strscpy(SERVICE,"TCP");
		SEND=NULL;
		EXPECT=NULL;
		QUIT=NULL;
		PROTOCOL=TCP_PROTOCOL;
		PORT=0;
	}
	else if (strstr(argv[0],"check_ftp")) {
		PROGNAME=strscpy(PROGNAME,"check_ftp");
		SERVICE=strscpy(SERVICE,"FTP");
		SEND=NULL;
		EXPECT=strscpy(EXPECT,"220");
		QUIT=strscpy(QUIT,"QUIT\n");
		PROTOCOL=TCP_PROTOCOL;
		PORT=21;
	}
	else if (strstr(argv[0],"check_smtp")) {
		PROGNAME=strscpy(PROGNAME,"check_smtp");
		SERVICE=strscpy(SERVICE,"SMTP");
		SEND=NULL;
		EXPECT=strscpy(EXPECT,"220");
		QUIT=strscpy(QUIT,"QUIT\n");
		PROTOCOL=TCP_PROTOCOL;
		PORT=25;
	}
	else if (strstr(argv[0],"check_pop")) {
		PROGNAME=strscpy(PROGNAME,"check_pop");
		SERVICE=strscpy(SERVICE,"POP");
		SEND=NULL;
		EXPECT=strscpy(EXPECT,"110");
		QUIT=strscpy(QUIT,"QUIT\n");
		PROTOCOL=TCP_PROTOCOL;
		PORT=110;
	}
	else if (strstr(argv[0],"check_imap")) {
		PROGNAME=strscpy(PROGNAME,"check_imap");
		SERVICE=strscpy(SERVICE,"IMAP");
		SEND=NULL;
		EXPECT=strscpy(EXPECT,"* OK");
		QUIT=strscpy(QUIT,"a1 LOGOUT\n");
		PROTOCOL=TCP_PROTOCOL;
		PORT=143;
	}
	else if (strstr(argv[0],"check_nntp")) {
		PROGNAME=strscpy(PROGNAME,"check_nntp");
		SERVICE=strscpy(SERVICE,"NNTP");
		SEND=NULL;
		EXPECT=NULL;
		server_expect=realloc(server_expect,++server_expect_count);
		server_expect[server_expect_count-1]=strscpy(EXPECT,"200");
		server_expect=realloc(server_expect,++server_expect_count);
		server_expect[server_expect_count-1]=strscpy(NULL,"201");
		QUIT=strscpy(QUIT,"QUIT\n");
		PROTOCOL=TCP_PROTOCOL;
		PORT=119;
	}
	else {
		usage("ERROR: Generic check_tcp called with unknown service\n");
	}

	server_address=strscpy(NULL,"127.0.0.1");
	server_port=PORT;
	server_send=SEND;
	server_quit=QUIT;

	if(process_arguments(argc,argv)==ERROR)
		usage("Could not parse arguments\n");

	/* use default expect if none listed in process_arguments() */
	if (EXPECT && server_expect_count==0) {
		server_expect=malloc(1);
		server_expect[server_expect_count-1]=EXPECT;
	}

	/* initialize alarm signal handling */
	signal(SIGALRM,socket_timeout_alarm_handler);

	/* set socket timeout */
	alarm(socket_timeout);

	/* try to connect to the host at the given port number */
	time(&start_time);
	if(PROTOCOL==UDP_PROTOCOL)
		result=my_udp_connect(server_address,server_port,&sd);
	else /* default is TCP */
		result=my_tcp_connect(server_address,server_port,&sd);

	if (result==STATE_CRITICAL)
		return STATE_CRITICAL;

	if (server_send!=NULL) { /* Something to send? */
		snprintf(buffer,MAX_INPUT_BUFFER-1,"%s\n\n",server_send);
		buffer[MAX_INPUT_BUFFER-1]=0;
		send(sd,buffer,strlen(buffer),0);
	}

	if (delay>0){
		start_time=start_time+delay;
		sleep(delay);
	}

	if (server_send || server_expect_count>0) {

		/* watch for the expect string */
		if (recv(sd,buffer,MAX_INPUT_BUFFER-1,0)>=0) {
			status = strscat(status,buffer);
		}
		strip(status);
/*  		ptr = strpbrk(status,"\r\n"); */
/*  		ptr[0]=0; */

		/* return a CRITICAL status if we couldn't read any data */
		if(status==NULL) terminate(STATE_CRITICAL,"No data received from host\n");

		if (status && verbose) printf("%s\n",status);
	 
		if (server_expect_count>0) {
			for (i=0;;i++) {
				printf("%d %d\n",i,server_expect_count);
				if (i>=server_expect_count) 
					terminate(STATE_WARNING,"Invalid response from host\n");
				if (strstr(status,server_expect[i]))
					break;
			}
		}
	}

	if(server_quit)
		send(sd,server_quit,strlen(server_quit),0);

	/* close the connection */
	close(sd);

	time(&end_time);

	if(check_critical_time==TRUE && (end_time-start_time)>critical_time)
		result=STATE_CRITICAL;
	else if(check_warning_time==TRUE && (end_time-start_time)>warning_time)
		result=STATE_WARNING;

	/* reset the alarm */
	alarm(0);

	printf
		("%s %s - %d second response time on port %d",
		 SERVICE,
		 state_text(result),
		 (int)(end_time-start_time),
		 server_port);

	if (status)
		printf(" [%s]\n",status);
	else
		printf("\n");

	return result;
}







/* process command-line arguments */
int process_arguments(int argc, char **argv)
{
  int c;

#ifdef HAVE_GETOPT_H
	int option_index = 0;
	static struct option long_options[] =
	{ 
		{"hostname"      ,required_argument,0,'H'},
		{"critical-time" ,required_argument,0,'c'},
		{"warning-time"  ,required_argument,0,'w'},
		{"critical-codes",required_argument,0,'C'},
		{"warning-codes" ,required_argument,0,'W'},
		{"timeout"       ,required_argument,0,'t'},
		{"protocol"      ,required_argument,0,'P'},
		{"port"          ,required_argument,0,'p'},
		{"send"          ,required_argument,0,'s'},
		{"expect"        ,required_argument,0,'e'},
		{"quit"          ,required_argument,0,'q'},
		{"delay"         ,required_argument,0,'d'},
		{"verbose"       ,no_argument,      0,'v'},
		{"version"       ,no_argument,      0,'V'},
		{"help"          ,no_argument,      0,'h'},
		{0,0,0,0}
	};
#endif

 	if(argc<2) usage("No arguments found\n");
	
	/* backwards compatibility */
  for (c=1;c<argc;c++) {
    if(strcmp("-to",argv[c])==0)
      strcpy(argv[c],"-t");
    else if (strcmp("-wt",argv[c])==0)
      strcpy(argv[c],"-w");
    else if (strcmp("-ct",argv[c])==0)
      strcpy(argv[c],"-c");
	}

	if (! is_option(argv[1])){
		server_address=argv[1];
		argv[1]=argv[0];
		argv=&argv[1];
		argc--;
	}

	while (1){
#ifdef HAVE_GETOPT_H
		c = getopt_long(argc,argv,"+hVvH:s:e:q:c:w:t:p:C:W:d:",long_options,&option_index);
#else
		c = getopt(argc,argv,"+hVvH:s:e:q:c:w:t:p:C:W:d:");
#endif

		if (c==-1||c==EOF||c==1) break;

		switch (c)
			{
			case '?': /* print short usage statement if args not parsable */
				printf("%s: Unknown argument: %s\n\n",my_basename(argv[0]),optarg);
				print_usage();
				exit(STATE_UNKNOWN);
			case 'h': /* help */
				print_help();
				exit(STATE_OK);
			case 'V': /* version */
				print_revision(PROGNAME,"$Revision: 1.20 $");
				exit(STATE_OK);
			case 'v': /* verbose mode */
				verbose=TRUE;
				break;
			case 'H': /* hostname */
				if(is_host(optarg)==FALSE)
					usage("Invalid host name/address\n");
				server_address=optarg;
				break;
			case 'c': /* critical */
				if(!is_intnonneg(optarg))
					usage("Critical threshold must be a nonnegative integer\n");
				critical_time=atoi(optarg);
				check_critical_time=TRUE;
				break;
			case 'w': /* warning */
				if(!is_intnonneg(optarg))
					usage("Warning threshold must be a nonnegative integer\n");
				warning_time=atoi(optarg);
				check_warning_time=TRUE;
				break;
			case 'C':
				crit_codes=realloc(crit_codes,++crit_codes_count);
				crit_codes[crit_codes_count-1]=optarg;
				break;
			case 'W':
				warn_codes=realloc(warn_codes,++warn_codes_count);
				warn_codes[warn_codes_count-1]=optarg;
				break;
			case 't': /* timeout */
				if(!is_intpos(optarg))
					usage("Timeout interval must be a positive integer\n");
				socket_timeout=atoi(optarg);
				break;
			case 'p': /* port */
				if(!is_intpos(optarg))
					usage("Server port must be a positive integer\n");
				server_port=atoi(optarg);
				break;
			case 's':
				server_send=optarg;
				break;
			case 'e':
				EXPECT=NULL;
				if (server_expect_count==0)
					server_expect = malloc(++server_expect_count);
				else
					server_expect = realloc(server_expect,++server_expect_count);
				server_expect[server_expect_count-1]=optarg;
				break;
			case 'q':
				server_quit=optarg;
				break;
			case 'd':
				if (is_intpos(optarg))
					delay=atoi(optarg);
				else
					usage("Delay must be a positive integer\n");
				break;
			}
	}

	if (server_address==NULL)
		usage("You must provide a server address\n");

	return OK;
}





void print_usage(void)
{
	printf
		("Usage: %s -H host -p port [-w warn_time] [-c crit_time] [-s send]\n"
		 "         [-e expect] [-W wait] [-t to_sec] [-v]\n",PROGNAME);
}





void print_help(void)
{
	print_revision(PROGNAME,"$Revision: 1.20 $");
	printf
		("Copyright (c) 1999 Ethan Galstad (netsaint@linuxbox.com)\n\n"
		 "This plugin tests %s connections with the specified host.\n\n",
		 SERVICE);
	print_usage();
	printf
		("Options:\n"
		 " -H, --hostname=ADDRESS\n"
		 "    Host name argument for servers using host headers (use numeric\n"
		 "    address if possible to bypass DNS lookup).\n"
		 " -p, --port=INTEGER\n"
		 "    Port number\n"
		 " -s, --sedn=STRING\n"
		 "    String to send to the server\n"
		 " -e, --expect=STRING\n"
		 "    String to expect in server response"
		 " -W, --wait=INTEGER\n"
		 "    Seconds to wait between sending string and polling for response\n"
		 " -w, --warning=INTEGER\n"
		 "    Response time to result in warning status (seconds)\n"
		 " -c, --critical=INTEGER\n"
		 "    Response time to result in critical status (seconds)\n"
		 " -t, --timeout=INTEGER\n"
		 "    Seconds before connection times out (default: %d)\n"
		 " -v"
		 "    Show details for command-line debugging (do not use with netsaint server)\n"
		 " -h, --help\n"
		 "    Print detailed help screen\n"
		 " -V, --version\n"
		 "    Print version information\n",
		 DEFAULT_SOCKET_TIMEOUT);
}
