/*
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 * nettable.c - The table of network interfaces for 'intop'.
 *
 * Luca Deri     <deri@ntop.org>
 * Rocco Carbone <rocco@ntop.org>
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*
 * ntop header file(s)
 */
#include "ntop.h"

#include "intop.h"



/*
 *
 * The table of pointers to the descriptors of network interfaces.
 *
 *  +------------+
 *  |            |--------> interface [0]
 *  +------------+
 *  |            |--------> interface [1]
 *  +------------+
 *  |            |-- // --> not yet allocated
 *  +------------+
 *  |   ......   |
 *  +------------+
 *  |            |--------> interface [n]
 *  +------------+
 *
 */
static NtopInterface * nettable [MAX_NUM_DEVICES];
static int netsize = MAX_NUM_DEVICES;                 /* total number of entries in the table */
static int netused = 0;                               /* number of used slots in the table */

/*
 * Initialization function.
 */
static void inittable (void)
{
  static int initialized = 0;
  int i;

  if (! initialized)
    {
      /*
       * Clean the table of pointers to the descriptors of network interfaces.
       */
      for (i = 0; i < netsize; i ++)
	nettable [i] = NULL;

      initialized = 1;
    }
}


#if (0)
static void netzero (NtopInterface * n)
{
  n -> name    = NULL;
  n -> flags   = INTERFACE_DOWN;

  n -> addr    = 0;
  n -> ipdot   = NULL;
  n -> fqdn    = NULL;

  n -> network . s_addr = 0;
  n -> netmask . s_addr = 0;
  n -> ifAddr  . s_addr = 0;

  n -> started  = 0;
  n -> firstpkt = 0;
  n -> lastpkt  = 0;

  n -> pcapPtr  = NULL;
  n -> snaplen  = 0;
  n -> datalink = 0;

  n -> filter   = NULL;
  n -> fd       = -1;

  /* Printing section */
  n -> hashing = 0;
  n -> ethv    = 0;
  n -> ipv     = 0;
  n -> tcpv    = 0;

  /* Packets section */
  n -> droppedPackets = 0L;
  n -> ethernetPkts   = 0L;
  n -> broadcastPkts  = 0L;
  n -> multicastPkts  = 0L;

  n -> hash_hostTraffic = NULL;
}
#endif


/*
 * Lookup for the first free position in the table.
 */
static int netposition (void)
{
  int i;

  for (i = 0; i < netsize; i ++)
    {
      /*
       * Skip allocated entries.
       */
      if (nettable [i])
	continue;

      return (i);
    }

  /*
   * No space left on device.
   */
  return (-1);
}


/*
 * Return the # of active network interfaces.
 */
int netno (void)
{
  return (netused);
}


/*
 * Allocate a new descriptor for the interface.
 */
int intopmalloc (char * interface, int snaplen, pcap_t * ph, char * filter)
{
  NtopInterface * n;

  struct in_addr inaddr;
  struct hostent * host;
  bpf_u_int32 network = {0};
  bpf_u_int32 netmask = {0};
  char ebuf [PCAP_ERRBUF_SIZE] = {0};

  int pos = netposition ();

  if (pos == -1)
    return (pos);

  /*
   * Address manipulation.
   */
  inaddr . s_addr = intop_lookupaddr (interface);
  host = gethostbyaddr ((char *) & inaddr . s_addr, sizeof (inaddr . s_addr), AF_INET);

  /*
   * Determine the interface network number and its network mask.
   */
  if (pcap_lookupnet (interface, & network, & netmask, ebuf) != -1)
    {
      network = htonl (network);
      netmask = htonl (netmask);
    }
  else  /* IP-less interfaces */
    {
      network = htonl (0);
      netmask = htonl (0xFFFFFFFF);
    }


  initDevices(interface);
  initLibpcap("", numDevices);
  n = & device [pos];

  n -> name    = strdup (interface);
  n -> flags   = INTERFACE_DOWN;

  n -> addr    = inaddr . s_addr;
  n -> ipdot   = strdup (inet_ntoa (inaddr));
  n -> fqdn    = strdup (host ? host -> h_name : "unresolved");

  n -> network . s_addr = network;
  n -> netmask . s_addr = netmask;
  n -> started  = 0;
  n -> firstpkt = 0;
  n -> lastpkt  = 0;

  n -> pcapPtr  = ph;
  n -> snaplen  = snaplen;
  n -> datalink = pcap_datalink (ph); /* type of underlying network and data-link encapsulation method */

  n -> filter   = filter ? strdup (filter) : NULL;

  n -> fd       = pcap_fileno (ph);

  /* Printing section */
  n -> fdv     = NULL;
  n -> hashing = 0;
  n -> ethv   = 0;
  n -> ipv    = 0;
  n -> tcpv   = 0;

  /* Packets section */
  n -> droppedPkts    = 0L;
  n -> ethernetPkts   = 0L;
  n -> broadcastPkts  = 0L;
  n -> multicastPkts  = 0L;

  /*
   * Keep the pointer to the table.
   */
  inittable ();

  nettable [pos] = n;

  netused ++;

  return (pos);
}


/*
 * Free the descriptor of a network interface.
 */
void intopfree (int i)
{
  inittable ();

  if (i > -1 && i < netsize && nettable [i])
    {
      free (nettable [i] -> name);

      if (nettable [i] -> ipdot)
	free (nettable [i] -> ipdot);
      if (nettable [i] -> fqdn)
	free (nettable [i] -> fqdn);

#if defined(WE_NEED_A_PROJECT_MEETING)

      ***** ASK LUCA *******

  We need a project meeting to define a common way of memory allocation

      free (nettable [i]);
#endif

      nettable [i] = NULL;
      netused --;
    }
}


/*
 * Function to lookup by values.
 */


/*
 * Lookup for 'name'.  If found return a small integer
 * (the index in the table);  -1 otherwise.
 */
int intop_lookup_interface (char * name)
{
  int i;

  if (! name)
    return (-1);

  for (i = 0; i < netsize; i ++)
    {
      /*
       * Skip empty entries.
       */
      if (! nettable [i])
	continue;

      if (! strcmp (nettable [i] -> name, name))
	return (i);
    }

  return (-1);
}


/*
 * -_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_
 *                          Lookup by index
 * -_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_
 */


/*
 * Return the name for the interface at position 'i'.
 */
char * intop_interface (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (NULL);

  return (nettable [i] ? nettable [i] -> name : NULL);
}


/*
 * Return the flags for the interface at position 'i'.
 */
int get_intop_flags (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (-1);

  return (nettable [i] ? nettable [i] -> flags : -1);
}


/*
 * Set the flags for the interface at position 'i'.
 */
int set_intop_flags (int i, int flags)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (-1);

  return (nettable [i] ? nettable [i] -> flags = flags : -1);
}



/*
 * Return the IP address in dot notation for the interface at position 'i'.
 */
char * intop_ipdot (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (NULL);

  return (nettable [i] ? nettable [i] -> ipdot : NULL);
}


/*
 * Return the IP address in 'human' notation for the interface at position 'i'.
 */
char * intop_iphuman (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (NULL);

  return (nettable [i] ? nettable [i] -> fqdn : NULL);
}


/*
 * Return the netmask for the interface at position 'i'.
 */
u_int32_t intop_netmask (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> netmask.s_addr : 0);
}


/*
 * Return the LBNL pcap pointer for the interface at position 'i'.
 */
pcap_t * intop_pcap (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (NULL);

  return (nettable [i] ? nettable [i] -> pcapPtr : NULL);
}


/*
 * Return (human-readable) data-link encapsulation type for the interface at position 'i'.
 */
char * intop_dltype (int i)
{
  char * s;

  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (NULL);

  /*
   * Empty entry.
   */
  if (! nettable [i])
    return (NULL);

  switch (nettable [i] -> datalink)
    {
    case DLT_NULL:
      s = "no link-layer encapsulation";
      break;

    case DLT_EN10MB:
      s = "Ethernet (10Mb)";
      break;

    case DLT_EN3MB:
      s = "Experimental Ethernet (3Mb)";
      break;

    case DLT_AX25:
      s = "Amateur Radio AX.25";
      break;

    case DLT_PRONET:
      s = "Proteon ProNET Token Ring";
      break;

    case DLT_CHAOS:
      s = "Chaos";
      break;

    case DLT_IEEE802:
      s = "IEEE 802 Networks";
      break;

    case DLT_ARCNET:
      s = "ARCNET";
      break;

    case DLT_SLIP:
      s = "Serial Line IP";
      break;

    case DLT_PPP:
      s = "Point-to-point Protocol";
      break;

    case DLT_FDDI:
      s = "FDDI";
      break;

    case DLT_ATM_RFC1483:
      s = "LLC/SNAP encapsulated atm";
      break;

    case DLT_RAW:
      s = "raw IP";
      break;

    case DLT_SLIP_BSDOS:
      s = "BSD/OS Serial Line IP";
      break;

    case DLT_PPP_BSDOS:
      s = "BSD/OS Point-to-point Protocol";
      break;

    default:
      s = "Unknown";
      break;
    }

  return (s);
}


/*
 * Return the filter as known by the user for the interface at position 'i'.
 */
char * get_intop_filter (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (NULL);

  return (nettable [i] ? nettable [i] -> filter : NULL);
}


/*
 * Set the filter for the interface at position 'i'.
 */
int set_intop_filter (int i, char * filter)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (-1);

  if (! nettable [i])
    return (-1);

  if (nettable [i] -> filter)
    free (nettable [i] -> filter);

  nettable [i] -> filter = filter ? strdup (filter) : NULL;

  return (0);
}


/*
 * Return the # of captured Ethernet Packets for the interface at position 'i'.
 */
TrafficCounter ethpkts (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> ethernetPkts : 0);
}


/*
 * Return the # of captured Broadcast Packets for the interface at position 'i'.
 */
TrafficCounter broadcastpkts (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> broadcastPkts : 0);
}


/*
 * Return the # of captured Multicast Packets for the interface at position 'i'.
 */
TrafficCounter multicastpkts (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> multicastPkts : 0);
}


/*
 * Return the # of Packets dropped by the application for the interface at position 'i'.
 */
TrafficCounter droppedpkts (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> droppedPkts : 0);
}


/*
 * Return the # of Packets received by the application for the interface at position 'i'.
 */
TrafficCounter receivedpkts (int i)
{
  struct pcap_stat pcapstats = {0};

  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  if (! nettable [i])
    return (0);

  if (pcap_stats (nettable [i] -> pcapPtr, & pcapstats) != 0)
    return (0);

  return ((TrafficCounter) pcapstats . ps_recv);
}


/*
 * Return the # of Packets dropped by the kernel for the interface at position 'i'.
 */
TrafficCounter kerneldroppedpkts (int i)
{
  struct pcap_stat pcapstats = {0};

  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  if (! nettable [i])
    return (0);

  if (pcap_stats (nettable [i] -> pcapPtr, & pcapstats) != 0)
    return (0);

  return ((TrafficCounter) pcapstats . ps_drop);
}


/*
 * Return the # of Ethetnet Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter ethbytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> ethernetBytes : 0);
}


/*
 * Return the # of IP Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter ipbytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> ipBytes : 0);
}


/*
 * Return the # of TCP Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter tcpbytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> tcpBytes : 0);
}


/*
 * Return the # of UDP Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter udpbytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> udpBytes : 0);
}

/*
 * Return the # of Other IP Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter otheripbytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> otherIpBytes : 0);
}


/*
 * Return the # of ICMP Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter icmpbytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> icmpBytes : 0);
}


/*
 * Return the # of NetBIOS Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter netbiosbytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> netbiosBytes : 0);
}


/*
 * Return the # of (r)ARP Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter arpbytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> arpRarpBytes : 0);
}


/*
 * Return the # of AppleTalk Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter appletalkbytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> atalkBytes : 0);
}


/*
 * Return the # of IPX Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter ipxbytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> ipxBytes : 0);
}


/*
 * Return the # of DECNet Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter decnetbytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> decnetBytes : 0);
}


/*
 * Return the # of DLC Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter dlcbytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> dlcBytes : 0);
}


/*
 * Return the # of OSPF Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter ospfbytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> ospfBytes : 0);
}


/*
 * Return the # of EGP Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter egpbytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> egpBytes : 0);
}


/*
 * Return the # of IGMP Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter igmpbytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> igmpBytes : 0);
}


/*
 * Return the # of OSI Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter osibytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> osiBytes : 0);
}


/*
 * Return the # of QNX Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter qnxbytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> qnxBytes : 0);
}


/*
 * Return the # of Other Bytes received by the application for the interface at position 'i'.
 */
TrafficCounter otherbytes (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (0);

  return (nettable [i] ? nettable [i] -> otherBytes : 0);
}


#if 0
static int hostsno (int current)
{
  int hostsno = 0;
  int i;
  HostTraffic * ht = NULL;

  if (current == -1)
    return (0);

  for (i = 1; i <  device [current] .actualHashSize; i ++)
    {
      /*
       * Skip unused entries.
       */
      if (! (ht = device [current] . hash_hostTraffic [i]))
	continue;

      hostsno ++;
    }
  return (hostsno);
}
#endif


/*
 * Return the pointer to the descriptor for the interface at position 'i'.
 */
NtopInterface * nethandler (int i)
{
  /*
   * Sanity check.
   */
  if (i < 0 || i >= netsize)
    return (NULL);

  return (nettable [i]);
}


/*
 * Return the pointer to the next interface descriptor.
 *
 *
 * This function also updates (and shoud not!!!!!)
 * the # of valid hosts in hash_hostTraffic
 * to make happy the function in uptime.c
 */
NtopInterface * netnext (void)
{
  static int last = 0;
  int i;
#if (1)
  int j;
  int hostsno = 0;
#endif

  last = last % netsize;

  for (i = last; i < netsize; i ++)
    if (nettable [i])
      {
	last = i + 1;

#if (1)
	for (j = 1; j <  nettable [i] -> actualHashSize; j ++)
	  {
	    /*
	     * Skip unused entries.
	     */
	    if (! nettable [i] -> hash_hostTraffic [j])
	      continue;

	    hostsno ++;
	  }
	nettable [i] -> hostsno = hostsno;
#endif
	return (nettable [i]);
      }

  last = netsize;
  return (NULL);
}
