/*
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 * sniff.c - enable packet capture on the given network interface.
 *
 * Luca Deri     <deri@ntop.org>
 * Rocco Carbone <rocco@ntop.org>
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*
 * ntop header file(s)
 */
#include "ntop.h"

#include "intop.h"


/*
 * Start collecting and processing packets.
 */
int intop_sniff (int argc, char * argv [])
{
  /*
   * Notice the command name.
   */
  char * commandname = argv [0];

  int c;
#define USAGE(xxx) \
  printf ("Usage: %s [-h] [-q] [-i interface]\n", xxx)
  char * optstring = "hqi:";

  /*
   * Reserve here space for the local variables.
   */
  int quiet = 0;
  char * interface = NULL;
  int current;
  int comargc = 0;
  char ** comargv = NULL;
  char * command = NULL;
  int rc = 0;
  /*
   * BPF filter and program.
   */
  int roomsize;
  char * fltr;
  struct bpf_program bpf_program = {0};
  /*
   * LBNL pcap handler.
   */
  pcap_t * ph = NULL;
  bpf_u_int32 netmask = {0};

  /*
   * Unique thread identifier.
   */
  int tid;

  optind = 0;
  optarg = NULL;

  /*
   * Parse command line options to the application via standard system calls.
   */
  while ((c = getopt (argc, argv, optstring)) != -1)
    {
      switch (c)
	{
	case 'h':
	  USAGE (commandname);
	  return (0);

	default:
	  USAGE (commandname);
	  return (-1);

	case 'q':
	  quiet = ! quiet;
	  break;

	case 'i':
	  interface = optarg;
	  break;
	}
    }


  /*
   * Safe to play with the 'active' interface (if any)
   * in case no specific one was chosen by the user.
   */
  if (! interface && ! (interface = intop_interface (active)))
    {
      printf ("No interface is currently enabled for packet sniffing.\n");
      return (-1);
    }

  /*
   * Lookup for the given name in the table of enabled interfaces.
   */
  current = intop_lookup_interface (interface);

  /*
   * Depending on the state of the network interface do...
   *
   */
  switch (get_intop_flags (current))
    {
    case -1:               /* never seen before */
    case FLAG_INTERFACE_DOWN:
      /*                    argv[0]      -i    interface      \0 */
      command = malloc (strlen ("open") + 4 + strlen (interface) + 1);
      sprintf (command, "open -i %s", interface);
      comargv = buildargv (command);
      comargc = 3;

      rc = intop_open (comargc, comargv);

      freeargv (comargv);
      free (command);

      if (rc == -1)
	return (-1);

      /*
       * I need to set current to the correct value now.
       */
      current = intop_lookup_interface (interface);

      break;

    case FLAG_INTERFACE_READY:  /* we can go! */
      break;

    case FLAG_INTERFACE_ENABLED:
      /*
       * Avoid to enable packet capturing several times for the same interface.
       */
      printf ("%s: packet capturing is already active for %s.\n", commandname, interface);
      return (-1);
    }

#if defined(CFG_MULTITHREADED)

  /*
   * Build a filter from all remaining command line arguments.
   */
  roomsize = 0;
  fltr = NULL;
  while (optind < argc)
    {
      roomsize += (strlen (argv [optind]) + 1 + 1);
      if (fltr)
	{
	  strcat (fltr, " ");
	  fltr = realloc (fltr, roomsize);
	  strcat (fltr, argv [optind ++]);
	}
      else
	{
	  fltr = malloc (roomsize);
	  strcpy (fltr, argv [optind ++]);
	}
    }

  /*
   * Compile the optional 'filter' into a BPF program.
   * By default optimization on the resulting code is required.
   */
  if (fltr)
    {
      set_intop_filter (current, fltr);
      free (fltr);
    }


  if ((fltr = get_intop_filter (current)))
    {
      ph = intop_pcap (current);
      netmask = intop_netmask (current);

      if (pcap_compile (ph, & bpf_program, fltr, 1, netmask) == -1)
	{
	  printf ("%s: cannot compile the filter [%s] (reason: %s)\n",
		  commandname, fltr, pcap_geterr (ph));
	  set_intop_filter(current, NULL);
	  return (-1);
	}

      /*
       * And apply the fltr to the handler.
       */
      if (pcap_setfilter (ph, & bpf_program) == -1)
	{
	  printf ("%s: cannot set the filter [%s] (reason: %s)\n",
		  commandname, fltr, pcap_geterr (ph));
	  set_intop_filter (current, NULL);
	  return (-1);
	}
    }

  if (! quiet)
    {
      if (fltr)
	printf ("starting sniffer on %s using %s...\n", interface, fltr);
      else
	printf ("starting sniffer on %s (with no filter enabled)...\n", interface);
      fflush (stdout);
    }

  /*
   * Set the time the interface was enabled for sniffing.
   */
  nethandler (current) -> started = NOW;

  tid = createThread (& myGlobals.device [current] . pcapDispatchThreadId,
		      pcapDispatch, (char *) current);

  /*
   * Change the status of the interface to ENABLED.
   */
  if (! tid)
    set_intop_flags (active, FLAG_INTERFACE_ENABLED);

  return (tid);

#else

  printf ("Sorry: intop does not works (yet) withoout threads.  I will try to fix it asap.\n");
  printf ("Sorry: for the incovenience.\n\n");
  fflush (stdout);
  return (-1);

#endif /* CFG_MULTITHREADED */
}
