#
# table.pl - show ntop's Trafich hash.
#
# Copyright (C) 2003 Rocco Carbone <rocco@ntop.org>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; see the file COPYING. If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
#

#
# the %traffic hash keeps all information of interest
# for the purpose of reporting statistics access
#
# They include:
#     hwaddress   => unique ethernet address (MAC address)
#     vendor      => unique MAC address vendor name
#     ipaddress   => its IP address in dot notation
#     hostname    => its, locally resolved, hostname
#     domain      => its, locally resolved, DNS domain name
#     nbtname     => its NetBIOS name
#     nbtdomain   => its NetBIOS domain
#     first       => time the host was first seen
#     last        => time the host was last seen

#
# useful variables
#
$__progname__   = "table.pl";
$__version__    = "0.0.1";
$__updated__    = "2003 Mar 10";
$__boottime__   = time();
$__bootdate__   = localtime($__boottime__);

# Hey Netscape, this is for you
my $htmltitle   = "NTOP Statistics -- Traffic Hash";
my $bodyname    = "Traffic Report";
my $title       = "Traffic Summary";
my $description = "Shows the %traffic hash";

#
# useful variables
#

my %colorinfo = (
		 yellow => {
			    colors => {
				       bg    => '#ffffff',
				       forte => '#f6e97d'
				      },
			    rowcolors => [
					  '"#ffffff"',
					  '"#fbf5c6"'
					 ],
			    images => {
				       left   => '"/pep/y_left.gif"',
				       right  => '"/pep/y_right.gif"',
				       bottom => '"/pep/bottom_y_200.gif"'
				      }
			   },
		 blue => {
			  colors => {
				     bg    => '#ffffff',
				     forte => '#7397ff'
				    },
			  rowcolors => [
					'#ffffff', '#d8e4f1'
				       ],
			  images => {
				     left   => '"/pep/b_left.gif"',
				     right  => '"/pep/b_right.gif"',
				     bottom => '"/pep/bottom_b_200.gif"'
				    }
			 },
		 red => {
			 colors => {
				    bg    => '#ffffff',
				    forte => '#d4303d'
				   },
			 rowcolors => [
				      '#ffffff', '#ffdfdf'
				     ],
			 images => {
				    left   => '"/pep/r_left.gif"',
				    right  => '"/pep/r_right.gif"',
				    bottom => '"/pep/bottom_r_200.gif"'
				   }
			},
		 green => {
			   colors => {
				      bg    => '#ffffff',
				      forte => '#90dc98'
				     },
			   rowcolors => [
					'#ffffff', '#ddf1d8'
					],
			   images => {
				      left   => '"/pep/g_left.gif"',
				      right  => '"/pep/g_right.gif"',
				      bottom => '"/pep/bottom_g_200.gif"'
				     }
			  }
	       );

#
# build various arrays from the %traffic hash
#
my @vals = values %traffic;

# sub arrays sorted by various kind of values
my @sorted_by_hwaddress = (sort { $a->{hwaddress} cmp $b->{hwaddress} } @vals)[0..9];
my @sorted_by_vendor = (sort known_vendors @vals)[0..29];

# packets
my @sorted_by_pktsent = (sort { $b->{pktsent} <=> $a->{pktsent} } @vals)[0..9];
my @sorted_by_pktrcvd = (sort { $b->{pktreceived} <=> $a->{pktreceived} } @vals)[0..9];

# bytes
my @sorted_by_bytessent = (sort { $b->{bytessent} <=> $a->{bytessent} } @vals)[0..9];
my @sorted_by_bytesrcvd = (sort { $b->{bytesreceived} <=> $a->{bytesreceived} } @vals)[0..9];

my @sorted_by_lastseen = reverse((sort { $a->{lastseen} <=> $b->{lastseen} } @vals)[0..14]);
my @sorted_by_firstseen = (sort { $a->{firstseen} <=> $b->{firstseen} } @vals)[0..39];
my @sorted_by_firstseen_rev = (reverse (sort { $a->{lastseen} <=> $b->{lastseen} } @vals))[0..24];

my %hosts = (
	     vals     => \@sorted_by_hwaddress,
	     fields   => [
			  "hwaddress", '30%', "left",
			  "hostname",  '30%', "left",
			  "ipaddress", '30%', "right"
			 ],
	     color    => $colorinfo{blue},
	     title    => "Top " . scalar @sorted_by_hwaddress . " Hosts viewed by ntop",
	     subtitle => "Alphabetically sorted by Hardware Address"
	    );

my %vendors = (
	       vals     => \@sorted_by_vendor,
	       fields   => [
			    "hwaddress", '30%', "left",
			    "vendor",    '30%', "center",
			    "ipaddress", '30%', "right"
			   ],
	       color    => $colorinfo{blue},
	       title    => "Top " . scalar @sorted_by_vendor . " Hosts viewed by ntop",
	       subtitle => "Alphabetically sorted by Vendor Name"
	      );

my %pktsent = (
	       vals     => \@sorted_by_pktsent,
	       fields   => [
			    "hwaddress", '30%', "left",
			    "hostname",  '30%', "left",
			    "pktsent",   '30%', "right"
			   ],
	       color    => $colorinfo{red},
	       title    => "Top " . scalar @sorted_by_pktsent . " Hosts viewed by ntop",
	       subtitle => "Numerically sorted by the # of packet sent"
	     );

my %pktrcvd = (
	       vals     => \@sorted_by_pktrcvd,
	       fields   => [
			    "hwaddress",   '30%', "left",
			    "hostname",    '30%', "left",
			    "pktreceived", '30%', "right"
			   ],
	       color    => $colorinfo{red},
	       title    => "Top " . scalar @sorted_by_pktrcvd . " Hosts viewed by ntop",
	       subtitle => "Numerically sorted by the # of packet received"
	     );

my %bytessent = (
		 vals     => \@sorted_by_bytessent,
		 fields   => [
			      "hwaddress", '30%', "left",
			      "hostname",  '30%', "left",
			      "bytessent", '30%', "right"
			     ],
		 color    => $colorinfo{green},
		 title    => "Top " . scalar @sorted_by_bytessent . " Hosts viewed by ntop",
		 subtitle => "Numerically sorted by the # of bytes sent"
	     );

my %bytesrcvd = (
		 vals     => \@sorted_by_bytesrcvd,
		 fields   => [
			      "hwaddress",     '30%', "left",
			      "hostname",      '30%', "left",
			      "bytesreceived", '30%', "right"
			     ],
		 color    => $colorinfo{green},
		 title    => "Top " . scalar @sorted_by_bytesrcvd . " Hosts viewed by ntop",
		 subtitle => "Numerically sorted by the # of bytes received"
	     );

my %last  = (
	     vals     => \@sorted_by_lastseen,
	     fields   => [
			  "hwaddress", '30%', "left",
			  "hostname",  '30%', "left",
			  "lastseen",  '30%', "center"
			 ],
	     color    => $colorinfo{green},
	     title    => "Top " . scalar @sorted_by_lastseen . " Hosts viewed by ntop",
	     subtitle => "Alphabetically sorted by Last Seen"
	    );

my %first = (
	     vals     => \@sorted_by_firstseen,
	     fields   => [
			  "hwaddress", '30%', "left",
			  "hostname",  '30%', "left",
			  "firstseen", '30%', "center"
			 ],
	     color    => $colorinfo{yellow},
	     title    => "Top " . scalar @sorted_by_firstseen . " Hosts viewed by ntop",
	     subtitle => "Alphabetically sorted by First Seen"
	    );

my %first_rev = (
		 vals     => \@sorted_by_firstseen_rev,
		 fields   => [
			      "hwaddress", '30%', "left",
			      "hostname",  '30%', "left",
			      "firstseen", '30%', "center"
			     ],
		 color    => $colorinfo{red},
		 title    => "Top " . scalar @sorted_by_firstseen_rev . " Hosts viewed by ntop",
		 subtitle => "Alphabetically sorted by First Seen (reverse)"
		);

my @views = (\%hosts, \%vendors,
	     \%pktsent, \%pktrcvd,
	     \%bytessent, \%bytesrcvd,
	     \%last, \%first, \%first_rev);


select STDOUT;

#
# print HTML header and title
#
html_header($htmltitle);

#
# start printing HTML body
#
html_start_body($bodyname);

#
# print report's title with its brief description
#
print "<center><font size=\"6\"><b>$description</b></font></table>\n";

#
# print the matrix
#
print_matrix(\@views, 2, 400);

#
# finish HTML output
#
html_stop_body();


#
# print HTML header and title
#
sub html_header {
  my $titlepage = shift;

  #
  # immediately start printing the magic line which tells to the world
  # we are playing with an HTML document
  #
  # print "Content-type: text/html\n\n";

  print "Content-type: text/html\n";
  print "Cache-Control: no-cache\n";
  print "Expires: 0\n\n";


  print
    "<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML 2.0//EN\">\n",
    "<html>\n",
    "<!-- ntop statistic page -->\n",
    "<!-- Rocco Carbone <rocco\@ntop.org> -->\n",
    "<head>\n",
    "<link rel=stylesheet href=/style.css type=\"text/css\">\n",
    "<meta http-equiv=Pragma content=no-cache>\n",
    "<meta http-equiv=Cache-Control content=no-cache>\n",
    "<meta http-equiv=\"Content-Type\" content=\"text/html; charset=ISO-8859-1\">",
    "<meta name=\"robots\" content=\"ntop statistics\">\n",
    "<meta name=\"keywords\" content=\"ntop\">\n",
    "</head>\n",
    "<title>$titlepage</title>\n";
}


#
# start printing HTML body
#
sub html_start_body {
  my $title = shift;

  print "<body bgcolor=#ffffff text=#000000 link=#1111cc vlink=#cc0000 alink=#888888>\n";
  print "<center>\n";
  print "<table border=0 cellpadding=1 cellspacing=2>\n";
  print "<tr>\n";
  print "<td align=center>\n";
  print "<a href=\"/plugins/pep?available.pl\">all available scripts</a>\n";
  print "</td>\n";
  print "<td align=center>\n";
  print "<a href=\"/plugins/pep?table.pl\">reload</a>\n";
  print "</td>\n";
  print "<td width=15></td>\n";
  print "<td align=center>\n";
  print "</td>\n";
  print "</tr>\n";
  print "</table>\n";
}


#
# finish printing HTML body (aka footer)
#
sub html_stop_body {
  print "<p>\n";
  print "Generated by $__progname__ version $__version__\n";
  print "</p>\n";
  print "<table border=0 cellpadding=1 cellspacing=2>\n";
  print "<tr>\n";
  print "<td align=left>\n";
  print "<font size=2>\n";
  print "Questions or comments: <a href=\"mailto:rocco\@ntop.org\"</a><br>\n";
  print "&copy; 2000-2003 by Rocco Carbone - Updated $__updated__\n";
  print "</font>\n";
  print "</td>\n";
  print "</tr>\n";
  print "</table>\n";
  print "</center>\n";
  print "</body>\n";
  print "</html>\n";
}


#
# Table header
#
sub table_header {

  my ($color, $cols, $title, $subtitle) = @_;

  $subtitle =~ s/ /\&nbsp;/g;      # replace blanks for html rendering

  print
    "<tr>
      <td bgcolor=$color->{colors}{forte} valign=top>
       <div align=left><img height=\"30\" src=$color->{images}{left} width=\"12\"></div>
      </td>
      <td align=center bgcolor=$color->{colors}{forte} ";
  if ($cols > 2) {
    my $colspan = $cols - 2;
    print "colspan=\"$colspan\"";
  }
  print ">
       <div align=center><font face=\"Arial, sans-serif\" size=-1>
        <b>$title<br></b>$subtitle</font>
       </div>
      </td>
      <td bgcolor=$color->{colors}{forte} valign=top>
        <div align=right><img height=30 src=$color->{images}{right} width=\"12\"></div>
      </td>
     </tr>";
}


#
# a table row
#
# $a points to an array where:
#  the first element is always the "info"
#  the second element is the "width" of the field
#  the third element is the "align" of the field
#
sub table_row {

  my ($bg, $i, $a) = @_;

  my $face = "arial";

  print
    "<tr bgcolor=$bg>
      <td height=12pt width=\"6%\" align=center>
       <font face=$face><b>&nbsp;$i.</b></font>
      </td>";

  foreach $e (@$a) {

    $e->[0] =~ s/ /\&nbsp;/g;      # replace blanks for html rendering

    print
      "<td height=12pt width=$e->[1] align=$e->[2]>
        <font face=$face size=-1>&nbsp;$e->[0]</font>
       </td>";
  }
    print "</tr>";
}


#
# Table footer
#
sub table_footer {

  my ($cols, $color) = @_;

  print
    "<tr>
      <td colspan=$cols><img valign=top height=\"9\" hspace=\"0\" src=$color width=100%></td>
     </tr>";
}


#
# How to print a Table
#
# $a is a reference to the array of the references to the hashes with the values
# $fields is a reference to an array of the keys to print
# $size is the size of the table in pixel
# $color is a reference to the hash with the specification of the colors and images to use
# $sorted is ...
#
sub table_print {

  my ($a, $fields, $size, $color, $title, $subtitle) = @_;

  if (! ref $a) {
    return;
  }

  print "<table bgcolor=$color->{colors}{bg} border=\"0\" cellpadding=\"0\" cellspacing=\"0\" width=\"$size\">";

  $cols = 1 + scalar @$fields / 3;

  #
  # this is the header of the table that is in turn another table
  # (1 row and 3 cols) with two small images at the left and right size
  # and a title and a subtitle in the middle (centered)
  #
  table_header($color, $cols, $title, $subtitle);

  my $i = 0;
  my $k = scalar @{$color->{rowcolors}};

  # an empty row
  # table_row($color->{rowcolors}[$i % $k], 0, (), $cols);

  # and now print a table with all the information
  foreach my $e (@$a) {

    # @rows is the array to the values to print in a row
    my @rows = ();

    my $kk = scalar @$fields;

    for ($jj = 0; $jj < $kk; $jj += 3) {

      my $info;
      #
      # special case to manage dates
      #
      if ($fields->[$jj + 0] eq "firstseen" || $fields->[$jj + 0] eq "lastseen") {
	$info = seen($e->{$fields->[$jj]});
      }
      else {
	$info = $e->{$fields->[$jj]};
      }

      my @specs = ($info, $fields->[$jj + 1], $fields->[$jj + 2]);

      @rows = (@rows, \@specs);
    }

    table_row($color->{rowcolors}[$i % $k], ++ $i, \@rows, $cols);
  }

  table_footer($cols, $color->{images}{bottom});

  print "</table>";
}


#
# How to print a Matrix
#
# $v is the reference to the array of all the tables to display
# $cols is the # of columns
# $size is the size of the table in pixel
#
sub print_matrix {

  my ($v, $cols, $size) = @_;

  #
  # what should be displayed
  #
  my @views = @$v;

  my ($i, $j);

  # how many rows in the matrix of tables?
  my $rows = int(($#views + 1) / $cols);
  if (($#views + 1) % $cols) {
    $rows++;
  }

  print "<table border=\"0\" cellpadding=\"6\" cellspacing=\"0\" width=\"100%\">";

  for ($i = 0; $i < $rows; $i ++) {
    print "<tr>";
    for ($j = 0; $j < $cols; $j ++) {
      print "<td valign=top>";
      table_print($views[$i * $cols + $j]->{vals},
		  $views[$i * $cols + $j]->{fields},
		  $size, $views[$i * $cols + $j]->{color},
		  $views[$i * $cols + $j]->{title},
		  $views[$i * $cols + $j]->{subtitle});
      print "</td>";
    }
    print "</tr>";
  }
  print "</table>";
}


#
# compare two entries based on the concept of "resolved vendor name"
#
sub known_vendors {
  my $v1 = $a->{vendor};
  my $v2 = $b->{vendor};

  # reverse sort due to unresolved vendor names
  $b->{vendor} cmp $a->{vendor}
}


#
# compare two entries based on the "aged" concept
#
sub aged {
  ($a->{lastseen} - $a->{firstseen}) <=> ($b->{lastseen} - $b->{firstseen})
}

#
# DaysInSecs
#
sub days {
  my ($t1, $t2) = @_;

  return int(($t2 - $t1) / 86400);
}


#
# HoursInSecs
#
sub hours {
  my ($t1, $t2) = @_;

  return int(($t2 - $t1 - (days($t1, $t2) * 86400)) / 3600);
}


#
# MinutesInSecs
#
sub mins {
  my ($t1, $t2) = @_;

  return int(($t2 - $t1 - (days($t1, $t2) * 86400) - (hours($t1, $t2) * 3600)) / 60);
}


sub prettydelta {
  my ($t1, $t2) = @_;

  my $d = days($t1, $t2);
  my $h = hours($t1, $t2);
  my $m = mins($t1, $t2);
  my $s = ($t2 - $t1) % 60;

  if ($d) {
    return sprintf("%3d %02d:%02d:%02d", $d, $h, $m, $s);
  }
  elsif ($h) {
    return sprintf("%2d:%02d:%02d", $h, $m, $s);
  }
  else {
    return sprintf("%02d:%02d", $m, $s);
  }
}


sub seen {

  my ($seen) = @_;

  my ($sec, $min, $hour, $day, $month, $year, $wday) = (localtime($seen)) [0, 1, 2, 3, 4, 5, 6];

  $year += 1900;
  $month = sprintf "%02d", $month + 1;
  $hour  = sprintf "%02d", $hour;
  $min   = sprintf "%02d", $min;
  $sec   = sprintf "%02d", $sec;

  return $year . "/" . $month . "/" . $day . "&nbsp;" . $hour . ":" . $min . ":" . $sec;
}

1;
