/*
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 * filter.c - Read/Write the BPF filter associated to a network interface.
 *
 * Luca Deri     <deri@ntop.org>
 * Rocco Carbone <rocco@ntop.org>
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*
 * ntop header file(s)
 */
#include "ntop.h"

#include "intop.h"


/*
 * Get/Set the BPF filter associated to a network interface.
 */
int intop_set_filter (int argc, char * argv [])
{
  /*
   * Notice the command name.
   */
  char * commandname = argv [0];

  int c;
#define USAGE(xxx) \
  printf ("Usage: %s [-h] [-i interface] [expression]\n", xxx)
  char * optstring = "hi:";

  /*
   * Reserve here space for the local variables.
   */
  char * interface = NULL;
  int current;
  /*
   * LBNL pcap handler.
   */
  pcap_t * ph = NULL;
  bpf_u_int32 netmask = {0};

  /*
   * BPF filter and program.
   */
  int roomsize;
  char * fltr;
  struct bpf_program bpf_program = {0};

  optind = 0;
  optarg = NULL;

  /*
   * Parse command line options to the application via standard system calls.
   */
  while ((c = getopt (argc, argv, optstring)) != -1)
    {
      switch (c)
	{
	case 'h':
	  USAGE (commandname);
	  return (0);

	default:
	  USAGE (commandname);
	  return (-1);

	case 'i':
	  interface = optarg;
	  break;
	}
    }

  /*
   * Safe to play with the 'active' interface (if any)
   * in case no specific one was chosen by the user.
   */
  if (! interface && ! (interface = intop_interface (active)))
    {
      printf ("No interface is currently enabled for packet sniffing.\n");
      return (-1);
    }

  /*
   * Lookup for the given name in the table of enabled interfaces.
   */
  current = intop_lookup_interface (interface);
  if (current == -1)
    {
      printf ("%s: unknown interface %s.\n", commandname, interface);
      return (-1);
    }

  /*
   * Avoid to modify parameters for interfaces not yet enabled via LBNL.
   */
  if (get_intop_flags (current) == FLAG_INTERFACE_DOWN)
    {
      printf ("%s: unknown interface %s.\n", commandname, interface);
      return (-1);
    }


  /*
   * Build a filter from all remaining command line arguments.
   */
  roomsize = 0;
  fltr = NULL;
  while (optind < argc)
    {
      roomsize += (strlen (argv [optind]) + 1 + 1);
      if (fltr)
	{
	  strcat (fltr, " ");
	  fltr = realloc (fltr, roomsize);
	  strcat (fltr, argv [optind ++]);
	}
      else
	{
	  fltr = malloc (roomsize);
	  strcpy (fltr, argv [optind ++]);
	}
    }


  /*
   * Compile the mandatory 'filter' into a BPF program.
   * By default optimization on the resulting code is required.
   */
  if (fltr)
    {
      ph = intop_pcap (current);
      netmask = intop_netmask (current);

      if (pcap_compile (ph, & bpf_program, fltr, 1, netmask) == -1)
	{
	  printf ("%s: cannot compile the filter [%s]  (reason: %s)\n",
		  commandname, fltr, pcap_geterr (ph));
	  return (-1);
	}

      /*
       * And apply the filter to the handler.
       */
      if (pcap_setfilter (ph, & bpf_program) == -1)
	{
	  printf ("%s: cannot set the filter [%s] (reason: %s)\n",
		  commandname, fltr, pcap_geterr (ph));
	  return (-1);
	}

      set_intop_filter (current, fltr);
      free (fltr);
    }
  else
    {
      fltr = get_intop_filter (current);
      if (fltr)
	printf ("current value for filtering packets is set to: [%s].\n", fltr);
      else
	printf ("no filter has been currently set for interface %s.\n", interface);
    }

  return (0);
}
