/*
 *  Copyright (C) 2001-2002 Luca Deri <deri@ntop.org>
 *
 * 	                    http://www.ntop.org/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "ntop.h"

/* 
   This file includes code courtesy of:
   
   Dale E Reed, Jr. <daler@iea-software.com> 
   Christian Hammers <ch@westend.com>
*/

/* Forward */
static void sendNetFlow(int sendPartialPak);

/* *********************************** */

static int isNetFlowEnabled(u_int actualDeviceId) {
  char key[64], value[64];

  switch(myGlobals.device[actualDeviceId].exportNetFlow) {
  case NETFLOW_EXPORT_DISABLED:
    return(0);
    break;
  case NETFLOW_EXPORT_ENABLED:
    return(1);
    break;
  default: /* NETFLOW_EXPORT_UNKNOWN */ 
    value[0] = '\0';
    sprintf(key, "%s.exportNetFlow", myGlobals.device[actualDeviceId].name);

    if(!fetchPrefsValue(key, value, sizeof(value))) {
      storePrefsValue(key, "No");
      myGlobals.device[actualDeviceId].exportNetFlow = NETFLOW_EXPORT_DISABLED;
      return(0);
    } else {
      if(!strcmp(value, "No")) {
	myGlobals.device[actualDeviceId].exportNetFlow = NETFLOW_EXPORT_DISABLED;
	return(0);
      } else {
	myGlobals.device[actualDeviceId].exportNetFlow = NETFLOW_EXPORT_ENABLED;
	return(1);
      }
    }
    break;
  }
}

/* *********************************** */

static int isNetFlowExportEnabled() {
  if((myGlobals.netFlowOutSocket > 0)
     && (myGlobals.netFlowDest.sin_addr.s_addr != 0)) {
    return(1);
  } else
    return(0);
}

/* *********************************** */

void termNetFlowExporter() {
  if(myGlobals.netFlowOutSocket != 0) {
    sendNetFlow(TRUE);
    traceEvent(TRACE_INFO, "Cisco NetFlow exporter terminated.");
#ifndef WIN32
    close(myGlobals.netFlowOutSocket);
#else
    closesocket(myGlobals.netFlowOutSocket);
#endif
  }
}

/* *********************************** */

static void initFlowHeader(struct flow_ver5_hdr *flowHeader, int numCount) {
  flowHeader->version        = htons(5);
  flowHeader->count          = htons(numCount);
  flowHeader->sysUptime      = htonl((myGlobals.actTime-myGlobals.initialSniffTime)*1000);
  flowHeader->unix_secs      = htonl(myGlobals.actTime);
  flowHeader->unix_nsecs     = htonl(0);
  flowHeader->flow_sequence  = htonl(myGlobals.globalFlowSequence);
  flowHeader->engine_type    = 0;
  flowHeader->engine_id      = 0;

  myGlobals.globalFlowSequence += numCount;
}

/* *********************************** */

static void sendNetFlow(int sendPartialPak) {

  if(!isNetFlowExportEnabled()) return; /* NetFlow has not been enabled yet */

  if((myGlobals.globalFlowPktCount >= V5FLOWS_PER_PAK)
     || ((myGlobals.globalFlowPktCount > 0) && (sendPartialPak == TRUE))) {    
    /* Buffer is full, or forceSend flag is set - Send it and reset */
    int rc;

    initFlowHeader(&myGlobals.theRecord.flowHeader, myGlobals.globalFlowPktCount);
    
    rc = sendto(myGlobals.netFlowOutSocket, 
		(void*)&myGlobals.theRecord, 
		/* Fix below courtesy of Rami AlZaid <rami@alzaid.com> */
		(myGlobals.globalFlowPktCount*sizeof(struct flow_ver5_rec)
		 +sizeof(struct flow_ver5_hdr)),
		0, (struct sockaddr *)&myGlobals.netFlowDest,
		sizeof(myGlobals.netFlowDest));

    if(rc <= 0)
      traceEvent(TRACE_ERROR, "Netflow send failure (sock=%d)(errno=%d)", 
		 myGlobals.netFlowOutSocket, errno);
    else if(myGlobals.netFlowDebug) {
      int i;
      
      traceEvent(TRACE_INFO, "Sent NetFlow packet.");

      for(i=0; i<myGlobals.globalFlowPktCount; i++) {
	struct in_addr a, b;
	char buf[32], buf1[32];
	
	a.s_addr = ntohl(myGlobals.theRecord.flowRecord[i].srcaddr);
	b.s_addr = ntohl(myGlobals.theRecord.flowRecord[i].dstaddr);

	traceEvent(TRACE_INFO, "%2d) %s:%d <-> %s:%d %u/%u (proto=%d)",		   
		     i+1, _intoa(a, buf, sizeof(buf)),
		     ntohs(myGlobals.theRecord.flowRecord[i].srcport),
		     _intoa(b, buf1, sizeof(buf1)),
		     ntohs(myGlobals.theRecord.flowRecord[i].dstport),
		     ntohl(myGlobals.theRecord.flowRecord[i].dPkts),
		     ntohl(myGlobals.theRecord.flowRecord[i].dOctets),
		     myGlobals.theRecord.flowRecord[i].prot);
      }

      traceEvent(TRACE_INFO, "Exported %d NetFlow's...", myGlobals.globalFlowPktCount);
    }

    myGlobals.numNetFlowsPktsSent += myGlobals.globalFlowPktCount;
    memset(&myGlobals.theRecord, 0, sizeof(myGlobals.theRecord));
    myGlobals.globalFlowPktCount = 0;
  }
}

/* *********************************** */

/* 
   First/Last calculation time fixed by
   Wies-Software <wies@wiessoft.de>
*/

void sendTCPSessionFlow(IPSession *theSession, int actualDeviceId) {
  u_int32_t srcAddr, dstAddr;

  if(!isNetFlowExportEnabled() || (!isNetFlowEnabled(actualDeviceId)))
     return; /* NetFlow has not been enabled yet */
    
  if((myGlobals.device[actualDeviceId].
      hash_hostTraffic[checkSessionIdx(theSession->initiatorIdx)] == NULL)
     || (myGlobals.device[actualDeviceId].
	 hash_hostTraffic[checkSessionIdx(theSession->remotePeerIdx)] == NULL)) {
    traceEvent(TRACE_INFO, "Failed sanity check in sendTCPSessionFlow!");
    return; /* Sanity check */
  }

  srcAddr = htonl(myGlobals.device[actualDeviceId].hash_hostTraffic[checkSessionIdx(theSession->initiatorIdx)]
		  ->hostIpAddress.s_addr);
  dstAddr = htonl(myGlobals.device[actualDeviceId].hash_hostTraffic[checkSessionIdx(theSession->remotePeerIdx)]
		  ->hostIpAddress.s_addr);

  if((srcAddr != 0) 
     && (dstAddr != 0)
     && (theSession->bytesSent.value > 0)) {
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].srcaddr  = srcAddr;
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dstaddr  = dstAddr;
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].input    = htons(actualDeviceId);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].output   = htons(255 /* unknown device */);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dPkts    = htonl(theSession->pktSent);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dOctets  = htonl(theSession->bytesSent.value);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].First    = htonl((theSession->firstSeen-myGlobals.initialSniffTime)*1000);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].Last     = htonl((theSession->lastSeen-myGlobals.initialSniffTime)*1000);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].srcport  = htons(theSession->sport);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dstport  = htons(theSession->dport);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].prot     = 6 /* TCP */;

    myGlobals.globalFlowPktCount++; sendNetFlow(FALSE);

    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].srcaddr = dstAddr;
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dstaddr = srcAddr;
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].input   = htons(actualDeviceId);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].output  = htons(255 /* unknown device */);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dPkts   = htonl(theSession->pktRcvd);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dOctets = htonl(theSession->bytesRcvd.value);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].First   = htonl((theSession->firstSeen-myGlobals.initialSniffTime)*1000);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].Last    = htonl((theSession->lastSeen-myGlobals.initialSniffTime)*1000);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].srcport = htons(theSession->dport);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dstport = htons(theSession->sport);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].prot    = 6 /* TCP */;

    myGlobals.globalFlowPktCount++; sendNetFlow(FALSE);
  }
}

/* *********************************** */

void sendICMPflow(HostTraffic *srcHost,
		  HostTraffic *dstHost,
		  u_int length, u_int actualDeviceId) {
  u_int32_t srcAddr, dstAddr;

  if(!isNetFlowExportEnabled()) return; /* NetFlow has not been enabled yet */

  if((!isNetFlowEnabled(actualDeviceId))
     || (dstHost->hostIpAddress.s_addr == myGlobals.netFlowDest.sin_addr.s_addr)) {
    /*
      Flows generated by NetFlow collector host
      MUST be ignored (avoid race conditions)
    */
    return;
  }

  srcAddr = htonl(srcHost->hostIpAddress.s_addr);
  dstAddr = htonl(dstHost->hostIpAddress.s_addr);

  if((srcAddr != 0)
     && (dstAddr != 0)
     && (length > 0)) {
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].srcaddr = srcAddr;
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dstaddr = dstAddr;
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].input   = htons(actualDeviceId);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].output  = htons(255 /* unknown device */);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dPkts   = htonl(1);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dOctets = htonl(length);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].First   = htonl((myGlobals.actTime-myGlobals.initialSniffTime)*1000);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].Last    = myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].First;
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].prot    = 1 /* ICMP */;
    myGlobals.globalFlowPktCount++; sendNetFlow(FALSE);
  }
}

/* *********************************** */

void sendUDPflow(HostTraffic *srcHost,
		 HostTraffic *dstHost,
		 u_int sport, u_int dport,
		 u_int length,
		 u_int actualDeviceId) {
  u_int32_t srcAddr, dstAddr;

  if(!isNetFlowExportEnabled() || (!isNetFlowEnabled(actualDeviceId)))
     return; /* NetFlow has not been enabled yet */

  if((dstHost->hostIpAddress.s_addr == myGlobals.netFlowDest.sin_addr.s_addr)
     && (myGlobals.netFlowDest.sin_port == dport)) {
    /*
      Flows generated by UDP packets due to NetFlow
      MUST be ignored
    */
    return;
  }

  srcAddr = htonl(srcHost->hostIpAddress.s_addr);
  dstAddr = htonl(dstHost->hostIpAddress.s_addr);

  if(srcAddr && dstAddr && length) {    
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].srcaddr = srcAddr;
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dstaddr = dstAddr;
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].input   = htons(actualDeviceId);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].output  = htons(255 /* unknown device */);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dPkts   = htonl(1);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dOctets = htonl(length);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].First   = htonl((myGlobals.actTime-myGlobals.initialSniffTime)*1000);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].Last    = myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].First;
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].srcport = htons(sport);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dstport = htons(dport);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].prot    = 17 /* UDP */;
    
    myGlobals.globalFlowPktCount++; sendNetFlow(FALSE);
  }
}

/* *********************************** */

void sendOTHERflow(HostTraffic *srcHost,
		   HostTraffic *dstHost,
		   u_int8_t proto,
		   u_int length,
		   u_int actualDeviceId) {
  u_int32_t srcAddr, dstAddr;

  if((!isNetFlowExportEnabled()) || (!isNetFlowEnabled(actualDeviceId)))
    return; /* NetFlow has not been enabled yet */

  if(dstHost->hostIpAddress.s_addr == myGlobals.netFlowDest.sin_addr.s_addr) {
    /*
      Flows generated by NetFlow collector host
      MUST be ignored (avoid race conditions)
    */
    return;
  }

  srcAddr = htonl(srcHost->hostIpAddress.s_addr);
  dstAddr = htonl(dstHost->hostIpAddress.s_addr);

  if(srcAddr && dstAddr && length) {    
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].srcaddr = srcAddr;
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dstaddr = dstAddr;
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].input   = htons(actualDeviceId);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].output  = htons(255 /* unknown device */);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dPkts   = htonl(1);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].dOctets = htonl(length);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].First   = htonl((myGlobals.actTime-myGlobals.initialSniffTime)*1000);
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].Last    = myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].First;
    myGlobals.theRecord.flowRecord[myGlobals.globalFlowPktCount].prot    = proto;
    
    myGlobals.globalFlowPktCount++; sendNetFlow(FALSE);
  }
}
