/*
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 * info.c - Display detailed about the actual state of a network interface.
 *
 * Luca Deri     <deri@ntop.org>
 * Rocco Carbone <rocco@ntop.org>
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*
 * ntop header file(s)
 */
#include "ntop.h"

#include "intop.h"

/*
 * Print current network interface information
 * in a format readable for humans.
 */
int intop_info (int argc, char * argv [])
{
  /*
   * Notice the command name.
   */
  char * commandname = argv [0];

  int c;
#define USAGE(xxx) \
  printf ("Usage: %s [-h]\n\
           [-i interface]\n", xxx)
  char * optstring = "hi:";

  /*
   * Reserve here space for the local variables.
   */
  char * interface = NULL;
  int current;
  NtopInterface * n;
  time_t now = NOW;

  optind = 0;
  optarg = NULL;

  /*
   * Parse command line options to the application via standard system calls.
   */
  while ((c = getopt (argc, argv, optstring)) != -1)
    {
      switch (c)
	{
	case 'h':
	  USAGE (commandname);
	  return (0);

	default:
	  USAGE (commandname);
	  return (-1);

	case 'i':
	  interface = optarg;
	  break;
	}
    }

  if (optind < argc)
    {
      printf ("\nWrong option(s): \" ");
      while (optind < argc)
	printf ("%s ", argv [optind ++]);
      printf ("\"\n");
      USAGE (commandname);
      printf ("\n");
      return (-1);
    }

  /*
   * Safe to play with the 'active' interface (if any)
   * in case no specific one was chosen by the user.
   */
  if (! interface && ! (interface = intop_interface (active)))
    {
      printf ("No interface is currently enabled for packet sniffing.\n");
      return (-1);
    }

  /*
   * Lookup for the given name in the table of enabled interfaces.
   */
  current = intop_lookup_interface (interface);
  if (current == -1)
    {
      printf ("%s: unknown interface %s.\n", commandname, interface);
      return (-1);
    }

  /*
   * General information about the interface.
   */

  n = nethandler (current);

  {
    char * fqdn = intop_iphuman (current);

    if (strchr (fqdn, '.'))
      printf ("Network interface: %s [%s] %s [%s] set to %s mode\n",
	      interface, intop_dltype (current),
	      fqdn, intop_ipdot (current),
	      get_intop_flags (current) == INTERFACE_READY ? "non promiscuous" : "promiscuous");
    else
      printf ("Network interface: %s [%s] %s.%s [%s] set to %s mode\n",
	      interface, intop_dltype (current),
	      fqdn, myGlobals.domainName, intop_ipdot (current),
	      get_intop_flags (current) == INTERFACE_READY ? "non promiscuous" : "promiscuous");
  }


  printf ("Sampling since    :  [%s] %s", formatSeconds (now - n -> started), ctime (& n -> started));
  printf ("\n");

  printf ("Packets:\n");
  printf ("  Received           : %s\n", cformatPkts (receivedpkts (current)));
  printf ("  Dropped by program : %s\n", cformatPkts (droppedpkts (current)));
  printf ("  Dropped by kernel  : %s\n", cformatPkts (kerneldroppedpkts (current)));

  printf ("  Ethernet           : %s\n", cformatPkts (ethpkts (current)));
  printf ("    Unicast          : %s\n",
	  cformatPkts (ethpkts (current) - broadcastpkts (current) - multicastpkts (current)));
  printf ("    Broadcast        : %s\n", cformatPkts (broadcastpkts (current)));
  printf ("    Multicast        : %s\n", cformatPkts (multicastpkts (current)));


  printf ("\n");

  printf ("Bytes:\n");
  printf ("  Total              : %s\n", cformatBytes (ethbytes (current)));
  printf ("   IP                : %s\n", cformatBytes (ipbytes (current)));
  printf ("   TCP               : %s\n", cformatBytes (tcpbytes (current)));
  printf ("   UDP               : %s\n", cformatBytes (udpbytes (current)));
  printf ("   Other IP          : %s\n", cformatBytes (otheripbytes (current)));
  printf ("     ICMP            : %s\n", cformatBytes (icmpbytes (current)));
  printf ("     NetBIOS         : %s\n", cformatBytes (netbiosbytes (current)));
  printf ("     r(ARP)          : %s\n", cformatBytes (arpbytes (current)));
  printf ("     AppleTalk       : %s\n", cformatBytes (appletalkbytes (current)));
  printf ("     IPX             : %s\n", cformatBytes (ipxbytes (current)));
  printf ("     DecNET          : %s\n", cformatBytes (decnetbytes (current)));
  printf ("     DLC             : %s\n", cformatBytes (dlcbytes (current)));
  printf ("     OSPF            : %s\n", cformatBytes (ospfbytes (current)));
  printf ("     EGP             : %s\n", cformatBytes (egpbytes (current)));
  printf ("     IGMP            : %s\n", cformatBytes (igmpbytes (current)));
  printf ("     OSI             : %s\n", cformatBytes (osibytes (current)));
  printf ("     QNX             : %s\n", cformatBytes (qnxbytes (current)));
  printf ("     Other           : %s\n", cformatBytes (otherbytes (current)));

  printf ("\n");

  fflush (stdout);

  return (0);
}
