/*
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 * lsdev.c - list all network interfaces suitable for being used with intop.
 *
 * Luca Deri     <deri@ntop.org>
 * Rocco Carbone <rocco@ntop.org>
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*
 * ntop header file(s)
 */
#include "ntop.h"

#include "intop.h"


/*
 * Show all available network interfaces attached to the system.
 * To be included in the list the interface must be configured up
 * and it should be accessible via the LBNL pcap library.
 */
int intop_lsdev (int argc, char * argv [])
{
  /*
   * Notice the command name.
   */
  char * commandname = argv [0];

  int c;
#define USAGE(xxx) \
  printf ("Usage: %s [-h]\n", xxx)
  char * optstring = "h";

  /*
   * Reserve here space for the local variables.
   */
  int sockfd;
  int found = 0;      /* # of configured interfaces */
  int avail = 0;      /* # of available interfaces */
  int i;

  /*
   * Array of structures (memory buffer for SIOCGIFCONF)
   */
  struct ifreq room [MAX_NUM_DEVICES];
  struct ifconf ifc;
  struct ifreq * first;
  struct ifreq * last;
  struct ifreq * current;
  struct ifreq flags;                             /* flags for the interface */

  /*
   * LBNL pcap variables.
   */
  char ebuf [PCAP_ERRBUF_SIZE] = {0};
  pcap_t * ph;

  optind = 0;
  optarg = NULL;
  memset(&room, 0, sizeof(room));

  /*
   * Parse command line options to the application via standard system calls.
   */
  while ((c = getopt (argc, argv, optstring)) != -1)
    {
      switch (c)
	{
	case 'h':
	  USAGE (commandname);
	  return (0);

	default:
	  USAGE (commandname);
	  return (-1);
	}
    }

  if (optind < argc)
    {
      printf ("\nWrong option(s): \" ");
      while (optind < argc)
	printf ("%s ", argv [optind ++]);
      printf ("\"\n");
      USAGE (commandname);
      printf ("\n");
      return (-1);
    }

  printf ("Warning: this function could not work on systems where struct ifreq has the sa_len field.\n");
  printf ("Warning: I will try to fix it asap.  Sorry for the incovenience.\n\n");

  /*
   * Open a socket to query for network configuration parameters.
   */
  sockfd = socket (AF_INET, SOCK_DGRAM, 0);
  if (sockfd == -1)
    {
      printf ("%s: cannot open socket: errno = %d\n", commandname, errno);
      return (-1);
    }

  /*
   * Set output memory variables.
   */
  ifc . ifc_len = sizeof (room);
  ifc . ifc_buf = (caddr_t) room;

  /*
   * Retrieve interface list for this system.
   */
  if (ioctl (sockfd, SIOCGIFCONF, & ifc) == -1 ||
      ifc . ifc_len < sizeof (struct ifreq))
    {
      printf ("%s: cannot retrieve network interface list: errno = %d\n", commandname, errno);
      close (sockfd);
      return (-1);
    }

  /*
   * Set pointers to the returned buffer.
   */
  current = first = (struct ifreq *) ifc . ifc_req;
  last = (struct ifreq *) (first + ifc . ifc_len);

  /*
   * This is the # of configured interfaces at this time on this system.
   */
  found = (last - first) / sizeof (struct ifreq);

  if (found)
    {
      printf ("%d interface(s) were found on this system.\n", found);
      printf ("The list of those suitable for using with %s is:\n", progName);
    }

  for (i = 0; i < found; i ++, current ++)
    {
      /*
       * Skip dummy and virtual interfaces
       */
      if (! strncmp (current -> ifr_name, "dummy", 5) ||
	  strchr (current -> ifr_name, ':'))
	continue;

      /*
       * Some systems return multiple entries if the interface has multi addresses.
       * I have to manage this condition here.
       */

      memset (& flags, 0, sizeof (flags));
      strncpy (flags . ifr_name, current -> ifr_name, sizeof (flags . ifr_name));

      /*
       * Get the interface flags.
       */
      if (ioctl (sockfd, SIOCGIFFLAGS, & flags) == -1)
	{
	  if (errno == ENXIO)
	    continue;

	  printf ("cannot get flags for interface %.*s: errno = %d\n",
		  (int) sizeof (flags . ifr_name), flags . ifr_name, errno);
	  continue;
	}

      /*
       * Show, and count, only the enabled interfaces.
       */
      if (! (flags . ifr_flags & IFF_UP))
	continue;

      /*
       * Be sure the LBNL library can play with it.
       */
      ph = pcap_open_live (flags . ifr_name, 68, 1, 1000, ebuf);
      if (! ph)
	continue;

      pcap_close (ph);

      avail ++;
      printf ("%d. %.*s\n", avail, (int) sizeof (flags . ifr_name), flags . ifr_name);
    }

  close (sockfd);

  if (! avail)
    {
      printf ("no suitable interface found.\n");
      return (-1);
    }

  return (0);
}
