/*
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 * open.c - open a network interface to look at packets on the network.
 *
 * Luca Deri     <deri@ntop.org>
 * Rocco Carbone <rocco@ntop.org>
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*
 * ntop header file(s)
 */
#include "ntop.h"

#include "intop.h"


/*
 * Obtain a LBNL packet capture descriptor
 * to look at packets on the network.
 */
int intop_open (int argc, char * argv [])
{
  /*
   * Notice the command name.
   */
  char * commandname = argv [0];

  int c;
#define USAGE(xxx) \
  printf ("Usage: %s [-h] [-q] [-i interface] [-s snaplen] [ expression ]\n", xxx)
  char * optstring = "hqi:s:";

  /*
   * Reserve here space for the local variables.
   */
  int quiet = 0;
  char * interface = NULL;
  int current;
  int snaplen = DEFAULT_SNAPLEN;
  char ebuf [PCAP_ERRBUF_SIZE] = {0};
  /*
   * LBNL pcap handler.
   */
  pcap_t * ph = NULL;

  /*
   * BPF filter.
   */
  int roomsize;
  char * fltr;

  optind = 0;
  optarg = NULL;

  /*
   * Parse command line options to the application via standard system calls.
   */
  while ((c = getopt (argc, argv, optstring)) != -1)
    {
      switch (c)
	{
	case 'h':
	  USAGE (commandname);
	  return (0);

	default:
	  USAGE (commandname);
	  return (-1);

	case 'q':
	  quiet = ! quiet;
	  break;

	case 'i':
	  interface = optarg;
	  break;

	case 's':
	  snaplen = atoi (optarg);
	  break;
	}
    }

  /*
   * Check for valid names here.
   */
  if (! interface)
    {
      printf ("No interface selected, please specify one with -i\n");
      printf ("The list of all suitable interfaces can be obtained via the 'lsdev' command.\n");
      fflush (stdout);
      return (-1);
    }

  /*
   * Lookup for the given name in the table of enabled interfaces
   * (to avoid multiple open on the same network interface).
   */
  current = intop_lookup_interface (interface);

  if (current != -1)
    {
      printf ("%s: interface %s already enabled for packet capturing.\n", commandname, interface);
      fflush (stdout);
      return (-1);
    }

  /*
   * Time to initialize LBNL pcap library for the specified
   * interface in promiscuous mode.
   * The read timeout is set to 1 sec by default.
   */
  ph = pcap_open_live (interface, snaplen, 1, 1000, ebuf);
  if (! ph)
    {
      printf ("%s: cannot initialize the LBNL pcap package (reason: %s)\n",
	      commandname, ebuf);
      fflush (stdout);
      return (-1);
    }

  /*
   * Build a filter from all remaining command line arguments.
   */
  roomsize = 0;
  fltr = NULL;
  while (optind < argc)
    {
      roomsize += (strlen (argv [optind]) + 1 + 1);
      if (fltr)
	{
	  strcat (fltr, " ");
	  fltr = realloc (fltr, roomsize);
	  strcat (fltr, argv [optind ++]);
	}
      else
	{
	  fltr = malloc (roomsize);
	  strcpy (fltr, argv [optind ++]);
	}
    }

  /*
   * Get a new descriptor and save current parameters
   * to the table of interfaces managed by this program.
   * Also play with the interface stack.
   */
  previous = active;
  active = intopmalloc (interface, snaplen, ph, fltr);

  if (fltr)
    free (fltr);

  /*
   * And update the status of the interface to READY.
   */
  set_intop_flags (active, INTERFACE_READY);

  /*
   * Update user prompt to include the active interface.
   */
  userprompt (interface);

  /*
   * Additional initialization only to make happy the ntop engine.
   */
  myGlobals . device [myGlobals . numDevices] . pcapPtr = ph;
  myGlobals . device [myGlobals . numDevices] . name = strdup (interface);

  resetStats ();
  initCounters ();

  return (0);
}
