/*
 *  Copyright (C) 1998-2002 Luca Deri <deri@ntop.org>
 *                      
 *  			    http://www.ntop.org/
 *  					
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "ntop.h"
#include "globals-report.h"

/* #define DEBUG */

typedef struct nfsEntries {
  HostTraffic* host;
  Counter bytesSent, bytesRcvd;
  Counter lastBytesSent, lastBytesRcvd;
  float sentThpt, rcvdThpt;
} NfsEntries;

static int nfsColumnSort = 0;
static NfsEntries *nfsEntries[SHORTHASHNAMESIZE];
static time_t nextNfsUpdate;
 
/* ********************* */

static NfsEntries* findNfsEntry(struct in_addr addr) {
  unsigned long key = addr.s_addr % SHORTHASHNAMESIZE;
  unsigned int numSearches = 0;

  while((nfsEntries[key] != NULL) 
	&& (numSearches++ < SHORTHASHNAMESIZE)
	&& (nfsEntries[key]->host->hostIpAddress.s_addr == addr.s_addr))
    break;

  if(nfsEntries[key] == NULL) {
    HostTraffic* host = findHostByNumIP(intoa(addr), myGlobals.actualReportDeviceId);

    if(host == NULL)
      return(NULL); /* This shouldn't happen */

    nfsEntries[key] = (NfsEntries*)malloc(sizeof(NfsEntries));
    memset(nfsEntries[key], 0, sizeof(NfsEntries));
    nfsEntries[key]->host = host;
    return(nfsEntries[key]);
  } else if(nfsEntries[key]->host->hostIpAddress.s_addr == addr.s_addr)
    return(nfsEntries[key]);
  else
    return(NULL);
}

/* ********************* */

static void updateNfsThpt(void) {
  unsigned long now, secsDiff;
  int i;

  now = time(NULL);
  secsDiff = now-nextNfsUpdate+THROUGHPUT_REFRESH_TIME;

#ifdef DEBUG
  traceEvent(TRACE_INFO, "Updating NFS thpt...\n");
#endif

  for(i=0; i<SHORTHASHNAMESIZE; i++)
    if(nfsEntries[i] != NULL) {
      float diff;

      diff = (float)(nfsEntries[i]->bytesSent - nfsEntries[i]->lastBytesSent);
      nfsEntries[i]->sentThpt = diff/secsDiff;
#ifdef DEBUG
      traceEvent(TRACE_INFO, "%s) %.2f/", 
		 nfsEntries[i]->host->hostSymIpAddress, 
		 nfsEntries[i]->sentThpt);
#endif

      diff = (float)(nfsEntries[i]->bytesRcvd-nfsEntries[i]->lastBytesRcvd);
      nfsEntries[i]->rcvdThpt = diff/secsDiff;
#ifdef DEBUG
      traceEvent(TRACE_INFO, "%.2f\n", nfsEntries[i]->rcvdThpt);
#endif

      nfsEntries[i]->lastBytesSent = nfsEntries[i]->bytesSent;
      nfsEntries[i]->lastBytesRcvd = nfsEntries[i]->bytesRcvd;
    }

  nextNfsUpdate = now+THROUGHPUT_REFRESH_TIME;
}

/* ********************* */

static void handleNFSPacket(u_char *_deviceId,
			    const struct pcap_pkthdr *h, const u_char *p) {
  struct ip ip;
  struct udphdr nfsPktHdr;
  u_int hlen, length;
#ifdef DEBUG
  u_short sport, dport;
#endif
  NfsEntries *srcHost, *dstHost;
  int deviceId, actualDeviceId;

#ifdef WIN32
  deviceId = 0;
#else
  deviceId = *_deviceId; /* Fix courtesy of Berthold Gunreben <bg@suse.de> */
#endif

  actualDeviceId = getActualInterface(deviceId);

  length = h->len-myGlobals.headerSize[myGlobals.device[deviceId].datalink];

  memcpy(&ip, (p+myGlobals.headerSize[myGlobals.device[deviceId].datalink]), sizeof(struct ip));
  hlen = (u_int)ip.ip_hl * 4;
  memcpy(&nfsPktHdr, (p+myGlobals.headerSize[myGlobals.device[deviceId].datalink]+hlen), 
	 sizeof(struct udphdr));  

#ifdef DEBUG
  sport = ntohs(nfsPktHdr.uh_sport);
  dport = ntohs(nfsPktHdr.uh_dport);

  traceEvent(TRACE_INFO, "NFS %s:%d ", intoa(ip.ip_src), sport);
  traceEvent(TRACE_INFO, "-> %s:%d [len=%d]\n", intoa(ip.ip_dst), dport, length);
#endif

  ip.ip_src.s_addr = ntohl(ip.ip_src.s_addr); /* Big/little endian */
  srcHost = findNfsEntry(ip.ip_src);
  if(srcHost != NULL) srcHost->bytesSent += length;

  ip.ip_dst.s_addr = ntohl(ip.ip_dst.s_addr); /* Big/little endian */
  dstHost = findNfsEntry(ip.ip_dst);
  if(dstHost != NULL) dstHost->bytesRcvd += length;
  
  if(time(NULL) > nextNfsUpdate)
    updateNfsThpt();
}

/* ******************************* */

static int sortNFShosts(const void *_a, const void *_b) {
  NfsEntries **a = (NfsEntries **)_a;
  NfsEntries **b = (NfsEntries **)_b;
  int rc;
  
  if((a == NULL) && (b != NULL)) {
    traceEvent(TRACE_WARNING, "WARNING (1)\n");
    return(1);
  } else if((a != NULL) && (b == NULL)) {
    traceEvent(TRACE_WARNING, "WARNING (2)\n");
    return(-1);
  } else if((a == NULL) && (b == NULL)) {
    traceEvent(TRACE_WARNING, "WARNING (3)\n");
    return(0);
  }

  switch(nfsColumnSort) {
  default:
  case 1:
#ifdef MULTITHREADED
    accessMutex(&myGlobals.addressResolutionMutex, "sortNFShosts");
#endif 
    rc = strcasecmp((*a)->host->hostSymIpAddress, (*b)->host->hostSymIpAddress);
#ifdef MULTITHREADED
    releaseMutex(&myGlobals.addressResolutionMutex);
#endif 
    return(rc);
    break;

  case 2:
    if((*a)->bytesSent < (*b)->bytesSent)
      return(1);
    else if ((*a)->bytesSent > (*b)->bytesSent)
      return(-1);
    else
      return(0);
    break;

  case 3:
    if((*a)->sentThpt < (*b)->sentThpt)
      return(1);
    else if ((*a)->sentThpt > (*b)->sentThpt)
      return(-1);
    else
      return(0);
    break;

  case 4:
    if((*a)->bytesRcvd < (*b)->bytesRcvd)
      return(1);
    else if ((*a)->bytesRcvd > (*b)->bytesRcvd)
      return(-1);
    else
      return(0);
    break;

  case 5:
    if((*a)->rcvdThpt < (*b)->rcvdThpt)
      return(1);
    else if ((*a)->rcvdThpt > (*b)->rcvdThpt)
      return(-1);
    else
      return(0);
    break;

  }  
}

/* ****************************** */

static void handleNfsWatchHTTPrequest(char* url) {
  static short everUpdated = 0;
  char tmpStr[2*BUF_SIZE];
  int numEntries = 0, i, revertOrder=0;
  char *pluginName = "<A HREF=/plugins/nfsWatch";
  char *sign[16];
  NfsEntries *tmpNfsEntries[SHORTHASHNAMESIZE];
  float maxSentThpt=-1, maxRcvdThpt=-1;

  if(!everUpdated) {
    updateNfsThpt();
    everUpdated = 1;
  }

  for(i=0; i<16; i++) sign[i] = "";

  if(url[0] == '\0') {
    nfsColumnSort = 0;
  } else {
    if(url[0] == '-') {
      revertOrder = 1;
      nfsColumnSort = atoi(&url[1]);
    } else {
      nfsColumnSort = atoi(url);
    }
  }

  if(!revertOrder) sign[nfsColumnSort] = "-";

  for(i=0; i<SHORTHASHNAMESIZE; i++) {
    if(nfsEntries[i] != NULL) {
      tmpNfsEntries[numEntries++] = nfsEntries[i];
      if(nfsEntries[i]->sentThpt > maxSentThpt) maxSentThpt = nfsEntries[i]->sentThpt;
      if(nfsEntries[i]->sentThpt > maxRcvdThpt) maxRcvdThpt = nfsEntries[i]->rcvdThpt;
    }
  }

  sendHTTPHeader(HTTP_TYPE_HTML, 0);
  printHTMLheader("Welcome to nfsWatch", 0);

  if(numEntries > 0) {
    sendString("<CENTER>\n");
    sendString("<TABLE BORDER><TR "TR_ON">");

    if(snprintf(tmpStr, sizeof(tmpStr), "<TH "TH_BG">%s?%s1>Host</A></TH>"
	    "<TH "TH_BG">%s?%s2>Data&nbsp;Sent</A></TH>"
	    "<TH "TH_BG" COLSPAN=2>%s?%s3>Actual&nbsp;Sent&nbsp;Thpt</A></TH>"
	    "<TH "TH_BG">%s?%s4>Data&nbsp;Rcvd</A></TH>"
	    "<TH "TH_BG" COLSPAN=2>%s?%s5>Actual&nbsp;Rcvd&nbsp;Thpt</A></TH>"
	    "</TR>\n",
	    pluginName, sign[1],
	    pluginName, sign[2],
	    pluginName, sign[3],
	    pluginName, sign[4],
		pluginName, sign[5]) < 0) BufferTooShort();

    sendString(tmpStr);
 
    quicksort(tmpNfsEntries, numEntries, sizeof(NfsEntries*), sortNFShosts);

    for(i=0; i<numEntries; i++) {
      NfsEntries *theEntry;
      char bar[512];

      if(revertOrder)
	theEntry = tmpNfsEntries[numEntries-i-1];
      else
	theEntry = tmpNfsEntries[i];
      if(snprintf(tmpStr, sizeof(tmpStr), "<TR "TR_ON" %s>%s"
	      "<TD "TD_BG" ALIGN=RIGHT>%s</TD>"
	      "<TD "TD_BG" ALIGN=RIGHT>%s</TD>",
	      getRowColor(), makeHostLink(theEntry->host, 1, 1, 0),
	      formatBytes(theEntry->bytesSent, 1),
	      formatThroughput(theEntry->sentThpt)) < 0) 
	BufferTooShort();
      sendString(tmpStr);
      printBar(bar, sizeof(bar), (unsigned short)((theEntry->sentThpt*100)/maxSentThpt), 100, 1);

      if(snprintf(tmpStr, sizeof(tmpStr), "<TD "TD_BG" ALIGN=RIGHT>%s</TD>"
	      "<TD "TD_BG" ALIGN=RIGHT>%s</TD>\n",
	      formatBytes(theEntry->bytesRcvd, 1),
	      formatThroughput(theEntry->rcvdThpt)) < 0) 
	BufferTooShort();
      sendString(tmpStr);

      printBar(bar, sizeof(bar), (unsigned short)((theEntry->rcvdThpt*100)/maxRcvdThpt), 100, 1);

      sendString("</TR>\n");
    }

    sendString("</TABLE></CENTER><p>\n");
  } else {
    printNoDataYet();
  }

  printHTMLtrailer();  
}

/* ****************************** */

static void termNfsFunct(void) {
  traceEvent(TRACE_INFO, "Thanks for using nfsWatchPlugin...\n"); 
  traceEvent(TRACE_INFO, "Done.\n"); 
  fflush(stdout);
}

/* ****************************** */

static PluginInfo nfsPluginInfo[] = {
  { "nfsWatchPlugin",
    "This plugin handles NFS traffic",
    "1.0",           /* version */
    "<A HREF=http://luca.ntop.org/>L.Deri</A>", 
    "nfsWatch",      /* http://<host>:<port>/plugins/nfsWatch */
    0, /* Not Active */
    NULL, /* no special startup after init */
    termNfsFunct,    /* TermFunc   */
    handleNFSPacket, /* PluginFunc */
    handleNfsWatchHTTPrequest,
    "port 2049"  /* BPF filter: filter all the nfs (nfsd = port 2049) packets */
  }
};

/* Plugin entry fctn */
#ifdef STATIC_PLUGIN
PluginInfo* nfsPluginEntryFctn(void) {
#else
PluginInfo* PluginEntryFctn(void) {
#endif
  
  traceEvent(TRACE_INFO, "Welcome to %s. (C) 1999 by Luca Deri.\n",  
	     nfsPluginInfo->pluginName);

  memset(nfsEntries, 0, sizeof(NfsEntries*));
  nextNfsUpdate = time(NULL)+THROUGHPUT_REFRESH_TIME;

  return(nfsPluginInfo);
}
