/*************************************************
* Base Classes Header File                       *
* (C) 1999-2002 The OpenCL Project               *
*************************************************/

#ifndef OPENCL_BASE_H__
#define OPENCL_BASE_H__

#include <string>
#include <opencl/config.h>
#include <opencl/exceptn.h>
#include <opencl/symkey.h>

namespace OpenCL {

/*************************************************
* Algorithm Base Class                           *
*************************************************/
class Algorithm
   {
   public:
      std::string name() const { return alg_name; }
      virtual void clear() throw() = 0;
      Algorithm(const std::string& n) : alg_name(n) {}
      virtual ~Algorithm() {}
   private:
      const std::string alg_name;
   };

/*************************************************
* Symmetric Algorithm Base Class                 *
*************************************************/
class SymmetricAlgorithm : public Algorithm
   {
   public:
      virtual void set_key(const byte[], u32bit) throw(Invalid_Key_Length) = 0;
      void set_key(const SymmetricKey&) throw(Invalid_Key_Length);
      bool valid_keylength(u32bit) const;
      u32bit default_keylength() const { return max; }
      SymmetricAlgorithm(const std::string&, u32bit, u32bit, u32bit);
      virtual ~SymmetricAlgorithm() {}
   private:
      const u32bit max, min, mod;
   };

/*************************************************
* Block Cipher Base Class                        *
*************************************************/
class BlockCipher : public SymmetricAlgorithm
   {
   public:
      void encrypt(const byte in[], byte out[]) const { enc(in, out); }
      void decrypt(const byte in[], byte out[]) const { dec(in, out); }
      void encrypt(byte block[]) const { enc(block, block); }
      void decrypt(byte block[]) const { dec(block, block); }
      u32bit block_size() const { return block_sz; }
      virtual BlockCipher* clone() const = 0;
      BlockCipher(const std::string&, u32bit, u32bit, u32bit = 0, u32bit = 1);
      virtual ~BlockCipher() {}
   private:
      virtual void enc(const byte[], byte[]) const = 0;
      virtual void dec(const byte[], byte[]) const = 0;
      const u32bit block_sz;
   };

/*************************************************
* Stream Cipher Base Class                       *
*************************************************/
class StreamCipher : public SymmetricAlgorithm
   {
   public:
      void encrypt(const byte i[], byte o[], u32bit len) { cipher(i, o, len); }
      void decrypt(const byte i[], byte o[], u32bit len) { cipher(i, o, len); }
      void encrypt(byte in[], u32bit len) { cipher(in, in, len); }
      void decrypt(byte in[], u32bit len) { cipher(in, in, len); }
      virtual StreamCipher* clone() const = 0;
      StreamCipher(const std::string&, u32bit, u32bit = 0, u32bit = 1);
      virtual ~StreamCipher() {}
   private:
      virtual void cipher(const byte[], byte[], u32bit) = 0;
   };

/*************************************************
* Random Access Stream Cipher Base Class         *
*************************************************/
class RandomAccessStreamCipher : public StreamCipher
   {
   public:
      virtual void seek(u32bit) = 0;
      virtual RandomAccessStreamCipher* clone() const = 0;
      RandomAccessStreamCipher(const std::string&, u32bit,
                               u32bit = 0, u32bit = 1);
      virtual ~RandomAccessStreamCipher() {}
   };

/*************************************************
* Buffered Computation Base Class                *
*************************************************/
class BufferedComputation : public Algorithm
   {
   public:
      void process(const byte[], u32bit, byte[]);
      void process(const std::string&, byte[]);
      void update(const byte[], u32bit);
      void update(const std::string&);
      virtual void final(byte[]) = 0;
      u32bit out_length() const { return out_len; }
      BufferedComputation(const std::string name, u32bit n) :
         Algorithm(name), out_len(n) {}
      virtual ~BufferedComputation() {}
   private:
      virtual void add_data(const byte[], u32bit) = 0;
      const u32bit out_len;
   };

/*************************************************
* Hash Function Base Class                       *
*************************************************/
class HashFunction : public BufferedComputation
   {
   public:
      virtual HashFunction* clone() const = 0;
      HashFunction(const std::string&, u32bit);
      virtual ~HashFunction() {}
   };

/*************************************************
* MAC Base Class                                 *
*************************************************/
class MessageAuthCode : public BufferedComputation, public SymmetricAlgorithm
   {
   public:
      virtual MessageAuthCode* clone() const = 0;
      MessageAuthCode(const std::string&, u32bit, u32bit,
                      u32bit = 0, u32bit = 1);
      virtual ~MessageAuthCode() {}
   };

/*************************************************
* Random Number Generator Base Class             *
*************************************************/
class RandomNumberGenerator : public Algorithm
   {
   public:
      virtual byte random() = 0;
      virtual void randomize(byte[], u32bit);
      virtual u32bit random_u32bit();
      virtual void add_entropy(const byte[], u32bit) throw() = 0;
      virtual void add_entropy(class EntropySource&, bool) = 0;
      RandomNumberGenerator(const std::string& n) : Algorithm(n) {}
      virtual ~RandomNumberGenerator() {}
   };

/*************************************************
* Entropy Source Base Class                      *
*************************************************/
class EntropySource
   {
   public:
      virtual u32bit fast_poll(byte[], u32bit) = 0;
      virtual u32bit slow_poll(byte[], u32bit) = 0;
      virtual ~EntropySource() {}
   };

}

#endif
