/*************************************************
* BigInt Encoding/Decoding Source File           *
* (C) 1999-2002 The OpenCL Project               *
*************************************************/

#include <cctype>
#include <opencl/bigint.h>
#include <opencl/encoder.h>
#include <opencl/numthry.h>

namespace OpenCL {

u32bit encoded_size(const BigInt& n, Encoding code)
   {
   static const double LOG_2_BASE_10 = 0.30102999566;
   if(code == Binary)
      return ((n.bits() + 7) / 8);
   else if(code == Hexadecimal)
      return ((n.bits() + 3) / 4);
   else if(code == Decimal)
      return (u32bit)((n.bits() * LOG_2_BASE_10) + 1);
   else throw Invalid_Argument("Cannot encode BigInt with unknown base");
   }

void encode(byte output[], const BigInt& n, Encoding code)
   {
   if(code == Binary)
      n.binary_encode(output);
   else if(code == Hexadecimal)
      {
      SecureVector<byte> binary = n.binary_encode();
      for(u32bit j = 0; j != binary.size(); j++)
         Hex::encode(binary + j, output + 2*j);
      }
   else if(code == Decimal)
      {
      static const BigInt TEN = 10;
      BigInt copy = n;
      copy.set_sign(Positive);
      u32bit output_size = encoded_size(n, Decimal);
      for(u32bit j = 0; j != output_size; j++)
         {
         BigInt remainder;
         divide(copy, TEN, copy, remainder);
         output[output_size - 1 - j] = remainder.at(0) + '0';
         if(copy.is_zero()) break;
         }
      }
   else
      throw Invalid_Argument("Cannot encode BigInt with unknown base");
   }

SecureVector<byte> encode(const BigInt& n, Encoding code)
   {
   SecureVector<byte> output(encoded_size(n, code));
   encode(output, n, code);
   return output;
   }

BigInt decode(const SecureVector<byte>& buf, Encoding code)
   {
   return decode(buf, buf.size(), code);
   }

BigInt decode(const byte buf[], u32bit length, Encoding code)
   {
   BigInt r;
   if(code == Binary)
      r.binary_decode(buf, length);
   else if(code == Hexadecimal)
      {
      if(length % 2)
         throw Format_Error("BigInt hex decode: input must be in full bytes");
      SecureVector<byte> binary(length / 2);
      for(u32bit j = 0; j != binary.size(); j++)
         {
         if(!std::isxdigit(buf[2*j]) || !std::isxdigit(buf[2*j+1]))
            throw Invalid_Argument("BigInt: Invalid hexadecimal string");
         Hex::decode(buf + 2*j, binary[j]);
         }
      r.binary_decode(binary);
      }
   else if(code == Decimal)
      {
      for(u32bit j = 0; j != length; j++)
         {
         if(!std::isdigit(buf[j]))
            throw Invalid_Argument("BigInt: Invalid decimal string");
         r = 10 * r + (buf[j] - '0');
         }
      }
   else
      throw Invalid_Argument("Cannot decode BigInt with unknown base");
   return r;
   }

}
