/*************************************************
* Global RNG Source File                         *
* (C) 1999-2002 The OpenCL Project               *
*************************************************/

#include <opencl/rng.h>
#include <opencl/mutex.h>

namespace OpenCL {

namespace {

RandomNumberGenerator* global_rng = 0;
Mutex* global_rng_lock = 0;

}

namespace Global_RNG {

/*************************************************
* Remove entropy from the global RNG             *
*************************************************/
void randomize(byte output[], u32bit size)
   {
   if(!global_rng)
      throw Exception("Global_RNG::randomize: The global RNG is not set");
   global_rng_lock->lock();
   global_rng->randomize(output, size);
   global_rng_lock->unlock();
   }

/*************************************************
* Remove entropy from the global RNG             *
*************************************************/
byte random()
   {
   byte ret;
   randomize(&ret, 1);
   return ret;
   }

/*************************************************
* Add entropy to the global RNG                  *
*************************************************/
void add_entropy(const byte entropy[], u32bit size)
   {
   if(!global_rng) return;
   global_rng_lock->lock();
   global_rng->add_entropy(entropy, size);
   global_rng_lock->unlock();
   }

/*************************************************
* Add entropy to the global RNG                  *
*************************************************/
void add_entropy(EntropySource& src, bool slow_poll)
   {
   if(!global_rng) return;
   global_rng_lock->lock();
   global_rng->add_entropy(src, slow_poll);
   global_rng_lock->unlock();
   }

}

/*************************************************
* Set the global RNG                             *
*************************************************/
void set_global_rng(RandomNumberGenerator* rng)
   {
   if(!global_rng_lock) global_rng_lock = get_mutex();

   SecureBuffer<byte, 128> state;
   if(global_rng)
      Global_RNG::randomize(state, state.size());

   global_rng_lock->lock();
   if(global_rng)
      delete global_rng;
   global_rng = rng;
   global_rng_lock->unlock();

   Global_RNG::add_entropy(state, state.size());
   }

}
