/*************************************************
* BigInt Header File                             *
* (C) 1999-2002 The OpenCL Project               *
*************************************************/

#ifndef OPENCL_BIGINT_H__
#define OPENCL_BIGINT_H__

#include <opencl/opencl.h>

namespace OpenCL {

enum Encoding { Decimal = 10, Hexadecimal = 16, Binary = 256 };
enum Sign { Negative = 0, Positive = 1 };

class BigInt
   {
   public:
      struct DivideByZero : public Exception
         { DivideByZero() : Exception("A BigInt tried to divide by zero") {} };

      enum NumberType { Random, Prime, SafePrime };

      BigInt& operator+=(const BigInt& n) { return (*this = *this + n); }
      BigInt& operator-=(const BigInt& n) { return (*this = *this - n); }
      BigInt& operator*=(const BigInt& n) { return (*this = *this * n); }
      BigInt& operator/=(const BigInt& n) { return (*this = *this / n); }
      BigInt& operator%=(const BigInt& n) { return (*this = *this % n); }
      u32bit  operator%=(u32bit n)        { *this = *this % n; return reg[0]; }
      BigInt& operator<<=(u32bit n)       { return (*this = *this << n); }
      BigInt& operator>>=(u32bit n)       { return (*this = *this >> n); }

      friend BigInt operator+(const BigInt&, const BigInt&);
      friend BigInt operator-(const BigInt&, const BigInt&);
      friend BigInt operator*(const BigInt&, const BigInt&);
      friend BigInt operator/(const BigInt&, const BigInt&);
      friend BigInt operator%(const BigInt&, const BigInt&);
      friend u32bit operator%(const BigInt&, u32bit);
      friend BigInt operator<<(const BigInt&, u32bit);
      friend BigInt operator>>(const BigInt&, u32bit);

      BigInt operator-() const;
      bool operator !() const { return (!is_nonzero()); }

      BigInt& operator++();
      BigInt& operator--();
      BigInt  operator++(int) { BigInt tmp = *this; ++*this; return tmp; }
      BigInt  operator--(int) { BigInt tmp = *this; --*this; return tmp; }

      void add(u32bit);
      void sub(u32bit);

      s32bit cmp(const BigInt&, bool = true) const;
      bool is_even() const { return (get_bit(0) == 0); }
      bool is_odd()  const { return (get_bit(0) == 1); }
      bool is_nonzero() const { return (!is_zero()); }
      bool is_zero() const;

      void set_bit(u32bit);
      void clear_bit(u32bit);
      bool get_bit(u32bit n) const;
      u32bit at(u32bit) const;

      bool is_negative() const { return (sign() == Negative); }
      bool is_positive() const { return (sign() == Positive); }
      Sign sign() const { return (signedness); }
      Sign reverse_sign() const;
      void flip_sign();
      void set_sign(Sign);
      BigInt abs() const;

      u32bit word_size() const { return (8 * sizeof(u32bit)); }
      u32bit size() const { return reg.size(); }
      u32bit sig_words() const;
      u32bit bytes() const;
      u32bit bits() const;

      void shrink() { reg.resize(sig_words()); }

      const u32bit* data() const { return reg.ptr(); }
      SecureVector<u32bit>& get_reg() { return reg; }
      u32bit& operator[](u32bit index) { return reg[index]; }
      SecureVector<byte> binary_encode() const;
      void binary_encode(byte[]) const;
      void binary_decode(const SecureVector<byte>&);
      void binary_decode(const byte[], u32bit);

      static const BigInt& zero() { static const BigInt x = 0; return x; }
      static const BigInt& one()  { static const BigInt x = 1; return x; }

      void swap(BigInt&);

      friend void positive_divide(BigInt&, BigInt&, BigInt&, BigInt&);

      BigInt(u64bit = 0);
      BigInt(const BigInt&);
      BigInt(const std::string&);
      BigInt(const SecureVector<byte>&, Encoding = Binary);
      BigInt(const byte[], u32bit, Encoding = Binary);
      BigInt(NumberType, u32bit);
      BigInt(Sign, u32bit);
      BigInt(const u32bit*, u32bit, Sign = Positive);
   private:
      void grow_to(u32bit n) const { reg.grow_to(n); }
      Sign signedness;
      SecureVector<u32bit> reg;
   };

/*************************************************
* Comparison Functions                           *
*************************************************/
inline bool operator==(const BigInt& a, const BigInt& b)
   { return (a.cmp(b) == 0); }
inline bool operator!=(const BigInt& a, const BigInt& b)
   { return (a.cmp(b) != 0); }
inline bool operator<=(const BigInt& a, const BigInt& b)
   { return (a.cmp(b) <= 0); }
inline bool operator>=(const BigInt& a, const BigInt& b)
   { return (a.cmp(b) >= 0); }
inline bool operator<(const BigInt& a, const BigInt& b)
   { return (a.cmp(b) < 0); }
inline bool operator>(const BigInt& a, const BigInt& b)
   { return (a.cmp(b) > 0); }

/*************************************************
* Miscellaneous Functions                        *
*************************************************/
u32bit encoded_size(const BigInt&, Encoding = Binary);
SecureVector<byte> encode(const BigInt&, Encoding = Binary);
void encode(byte[], const BigInt&, Encoding = Binary);
BigInt decode(const byte[], u32bit, Encoding = Binary);
BigInt decode(const SecureVector<byte>&, Encoding = Binary);

std::ostream& operator<<(std::ostream&, const BigInt&);
std::istream& operator>>(std::istream&, BigInt&);

}

namespace std {

inline void swap(OpenCL::BigInt& a, OpenCL::BigInt& b) { a.swap(b); }

}

#endif
