/*************************************************
* Memory Locking Allocator Source File           *
* (C) 1999-2002 The OpenCL Project               *
*************************************************/

#include <opencl/lock_mem.h>
#include <opencl/util.h>
#include <cstdlib>
#include <sys/types.h>
#include <sys/mman.h>

namespace OpenCL {

/*************************************************
* Constructor                                    *
*************************************************/
MemoryLocking_Allocator::MemoryLocking_Allocator(u32bit blocks, u32bit size)
   : ManagedAllocator(true, check_size(size))
   {
   lock = get_mutex();

   size = check_size(size);
   allocated.resize(blocks);
   for(u32bit j = 0; j != allocated.size(); j++)
      {
      allocated[j].buf = std::malloc(size);
      if(!allocated[j].buf)
         throw Exception("MemoryLocking_Allocator: Ran out of memory");

      allocated[j].length = size;
      allocated[j].in_use = false;
      if(mlock(allocated[j].buf, allocated[j].length))
         throw Exception("MemoryLocking_Allocator: Locking failed");
      }
   }

/*************************************************
* Destructor                                     *
*************************************************/
MemoryLocking_Allocator::~MemoryLocking_Allocator()
   {
   for(u32bit j = 0; j != allocated.size(); j++)
      {
      if(allocated[j].in_use)
         throw Exception("~MemoryLocking_Allocator: buffer still in use");
      munlock(allocated[j].buf, allocated[j].length);
      std::free(allocated[j].buf);
      }
   release_mutex(lock);
   }

/*************************************************
* Check the buffer size                          *
*************************************************/
u32bit MemoryLocking_Allocator::check_size(u32bit size) const
   {
   if(size >= 2048 && power_of_2(size))
      return size;
   else
      return 8192;
   }

/*************************************************
* Allocation                                     *
*************************************************/
void* MemoryLocking_Allocator::alloc_block(u32bit n) const
   {
   void* out = 0;

   lock->lock();

   for(u32bit j = 0; j != allocated.size(); j++)
      if(!allocated[j].in_use && allocated[j].length == n)
         {
         allocated[j].in_use = true;
         out = allocated[j].buf;
         break;
         }

   lock->unlock();

   if(out) return out;

   out = std::malloc(n);
   if(out)
      mlock(out, n);

   return out;
   }

/*************************************************
* Deallocation                                   *
*************************************************/
void MemoryLocking_Allocator::dealloc_block(void* ptr, u32bit n) const
   {
   if(ptr == 0) return;

   lock->lock();

   for(u32bit j = 0; j != allocated.size(); j++)
      if(allocated[j].buf == ptr)
         {
         if(!allocated[j].in_use || allocated[j].length != n)
            throw Exception("MemoryLocking_Allocator: Internal failure");

         allocated[j].in_use = false;
         lock->unlock();
         return;
         }

   lock->unlock();

   munlock(ptr, n);
   std::free(ptr);
   }

}
