/*************************************************
* Memory Mapping Allocator Source File           *
* (C) 1999-2002 The OpenCL Project               *
*************************************************/

#ifndef _XOPEN_SOURCE
  #define _XOPEN_SOURCE 500
#endif
#ifndef _XOPEN_SOURCE_EXTENDED
  #define _XOPEN_SOURCE_EXTENDED 1
#endif

#include <opencl/mmap_mem.h>
#include <opencl/exceptn.h>
#include <opencl/util.h>
#include <cstring>
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>

namespace OpenCL {

/*************************************************
* Constructor                                    *
*************************************************/
MemoryMapping_Allocator::MemoryMapping_Allocator(const std::string& dir)
   : ManagedAllocator(true, getpagesize())
   {
   path = temp_dir(dir);
   path += "/opencl_XXXXXX";
   lock = get_mutex();

   cache.resize(CACHE_SIZE);
   for(u32bit j = 0; j != cache.size(); j++)
      {
      cache[j].length = getpagesize();
      cache[j].buf = mmap_block(cache[j].length);
      cache[j].fd = allocated[cache[j].buf].first;
      allocated.erase(allocated.find(cache[j].buf));
      }
   }

/*************************************************
* Destructor                                     *
*************************************************/
MemoryMapping_Allocator::~MemoryMapping_Allocator()
   {
   for(u32bit j = 0; j != cache.size(); j++)
      munmap_block(cache[j].buf, cache[j].length, cache[j].fd);
   }

/*************************************************
* Return true if the program is suid/sgid        *
*************************************************/
bool MemoryMapping_Allocator::is_suid() const
   {
   if(getuid() != geteuid()) return true;
   if(getgid() != getegid()) return true;
   return false;
   }

/*************************************************
* Return the name of a good temporary directory  *
*************************************************/
std::string MemoryMapping_Allocator::temp_dir(const std::string& dir) const
   {
   if(!is_suid())
      {
      if(getenv("TMPDIR")) return std::string(getenv("TMPDIR"));
      if(getenv("TMP")) return std::string(getenv("TMP"));
      }
   if(dir != "") return dir;
   return "/tmp";
   }

/*************************************************
* Guess the page size of the system              *
*************************************************/
u32bit MemoryMapping_Allocator::getpagesize() const
   {
#if defined(_SC_PAGE_SIZE)
   u32bit pagesize = sysconf(_SC_PAGE_SIZE);
   if(pagesize >= 512 && power_of_2(pagesize))
      return pagesize;
#endif
   return 8192;
   }

/*************************************************
* Allocation                                     *
*************************************************/
void* MemoryMapping_Allocator::alloc_block(u32bit n) const
   {
   lock->lock();
   for(u32bit j = 0; j != cache.size(); j++)
      {
      if(cache[j].length == n && cache[j].buf != 0)
         {
         void* ptr = cache[j].buf;
         allocated[ptr] = std::pair<int, u32bit>(cache[j].fd, cache[j].length);
         cache[j].buf = 0;
         cache[j].length = 0;
         cache[j].fd = 0;
         lock->unlock();
         return ptr;
         }
      }
   lock->unlock();

   return mmap_block(n);
   }

/*************************************************
* Deallocation                                   *
*************************************************/
void MemoryMapping_Allocator::dealloc_block(void* ptr, u32bit n) const
   {
   if(ptr == 0) return;

   lock->lock();

   if(allocated.find(ptr) == allocated.end())
      throw Exception("MemoryMapping_Allocator: pointer not found");
   if(allocated[ptr].second != n)
      throw Exception("MemoryMapping_Allocator: bad length for memory block");

   for(u32bit j = 0; j != cache.size(); j++)
      if(cache[j].buf == 0)
         {
         cache[j].buf = ptr;
         cache[j].length = n;
         cache[j].fd = allocated[ptr].first;
         lock->unlock();
         return;
         }
   lock->unlock();

   munmap_block(ptr, n, allocated[ptr].first);
   }

/*************************************************
* Memory Map a File into Memory                  *
*************************************************/
void* MemoryMapping_Allocator::mmap_block(u32bit n) const
   {
   char* filepath = new char[path.length() + 1];
   std::strcpy(filepath, path.c_str());

   mode_t old_umask = umask(077);
   int file = mkstemp(filepath);
   if(file == -1)
      throw Exception("MemoryMapping_Allocator: Could not create file");

   if(unlink(filepath))
      throw Exception("MemoryMapping_Allocator: Could not unlink file");
   delete[] filepath;
   umask(old_umask);

   lseek(file, n, SEEK_SET);
   if(write(file, "\0", 1) != 1)
      throw Exception("MemoryMapping_Allocator: Could not write to file");
   void* ptr = mmap(0, n, PROT_READ | PROT_WRITE, MAP_SHARED, file, 0);
   if(ptr == (void*)MAP_FAILED)
      throw Exception("MemoryMapping_Allocator: Could not map file");

   lock->lock();
   allocated[ptr] = std::make_pair<int, u32bit>(file, n);
   lock->unlock();

   return ptr;
   }

/*************************************************
* Remove a Memory Map                            *
*************************************************/
void MemoryMapping_Allocator::munmap_block(void* ptr, u32bit n,
                                           u32bit fd) const
   {
   static const u32bit OVERWRITE_PASSES = 7;
   static const byte PATTERNS[12] = { 0x00, 0xFF, 0xAA, 0x55, 0x11, 0xEE,
                                      0xDE, 0xAD, 0x77, 0x88, 0x33, 0xCC };

   if(ptr == 0) return;

   for(u32bit j = 0; j != OVERWRITE_PASSES; j++)
      {
      std::memset(ptr, PATTERNS[j % 12], n);
      if(msync(ptr, n, MS_SYNC))
         throw Exception("MemoryMapping_Allocator: Sync operation failed");
      }
   std::memset(ptr, 0, n);
   if(msync(ptr, n, MS_SYNC))
      throw Exception("MemoryMapping_Allocator: Sync operation failed");

   if(munmap(ptr, n))
      throw Exception("MemoryMapping_Allocator: Could not unmap file");

   if(close(fd))
      throw Exception("MemoryMapping_Allocator: Could not close file");
   }

}
