/*************************************************
* FTW EntropySource Source File                  *
* (C) 1999-2002 The OpenCL Project               *
*************************************************/

#define __USE_XOPEN_EXTENDED
#include <opencl/es_ftw.h>
#include <opencl/util.h>
#include <fstream>
#include <cstring>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>

namespace OpenCL {

/*************************************************
* Fast Poll                                      *
*************************************************/
u32bit FTW_EntropySource::fast_poll(byte out[], u32bit length)
   {
   if(position == BUFFERSIZE) gather();
   u32bit copied = std::min(length, BUFFERSIZE - position);
   copy_mem(out, buffer + position, copied);
   position += copied;
   return copied;
   }

/*************************************************
* Slow Poll                                      *
*************************************************/
u32bit FTW_EntropySource::slow_poll(byte out[], u32bit length)
   {
   gather();
   u32bit got = length;
   while(length)
      {
      u32bit copied = std::min(length, BUFFERSIZE - position);
      copy_mem(out, buffer + position, copied);
      position += copied;
      out += copied;
      length -= copied;
      gather();
      }
   return got;
   }

/*************************************************
* Gather Entropy                                 *
*************************************************/
void FTW_EntropySource::gather()
   {
   gather_from_dir(path);
   position = 0;
   }

/*************************************************
* Gather Entropy From Directory Tree             *
*************************************************/
void FTW_EntropySource::gather_from_dir(const std::string& dirname)
   {
   if(dirname == "") return;

   DIR* dir = opendir(dirname.c_str());
   if(dir == 0) return;

   dirent* entry = readdir(dir);
   while(entry)
      {
      if((std::strcmp(entry->d_name, ".") == 0) ||
         (std::strcmp(entry->d_name, "..") == 0))
         { entry = readdir(dir); continue; }

      std::string filename = dirname + '/' + entry->d_name;

      struct stat stat_buf;
      if(lstat(filename.c_str(), &stat_buf) == -1)
         { entry = readdir(dir); continue; }

      if(S_ISREG(stat_buf.st_mode))
         gather_from_file(filename);
      if(S_ISDIR(stat_buf.st_mode))
         gather_from_dir(filename);
      entry = readdir(dir);
      }
   closedir(dir);
   }

/*************************************************
* Gather Entropy From A File                     *
*************************************************/
void FTW_EntropySource::gather_from_file(const std::string& filename)
   {
   int fd = open(filename.c_str(), O_RDONLY);
   if(fd == -1) return;

   SecureBuffer<byte, BUFFERSIZE> read_buf;
   while(true)
      {
      ssize_t got = read(fd, read_buf, BUFFERSIZE);
      if(got <= 0) return;
      xor_buf(buffer, read_buf, got);
      }
   }

}
