/*************************************************
* Unix EntropySource Source File                 *
* (C) 1999-2002 The OpenCL Project               *
*************************************************/

#define __USE_XOPEN_EXTENDED
#include <opencl/es_unix.h>
#include <opencl/util.h>
#include <stdio.h>

namespace OpenCL {

namespace {

bool Source_Cmp(const Source& a, const Source& b)
   { return (a.priority < b.priority); }

}

/*************************************************
* Unix_EntropySource Constructor                 *
*************************************************/
Unix_EntropySource::Unix_EntropySource(bool add_defaults)
   {
   if(add_defaults)
      add_default_sources();
   position = 0;
   }

/*************************************************
* Unix_EntropySource Constructor                 *
*************************************************/
Unix_EntropySource::Unix_EntropySource(const Source srcs[], u32bit count,
                                       bool add_defaults)
   {
   add_sources(srcs, count);
   if(add_defaults)
      add_default_sources();
   position = 0;
   }

/*************************************************
* Add the default sources                        *
*************************************************/
void Unix_EntropySource::add_default_sources()
   {
   u32bit count = 0;
   while(true)
      {
      if(DEFAULT_SOURCES[count].name == "") break;
      count++;
      }
   add_sources(DEFAULT_SOURCES, count);
   }

/*************************************************
* Fast Poll                                      *
*************************************************/
u32bit Unix_EntropySource::fast_poll(byte out[], u32bit length)
   {
   static const u32bit MAX_PRIORITY = 2;
   return gather(out, length, MAX_PRIORITY);
   }

/*************************************************
* Slow Poll                                      *
*************************************************/
u32bit Unix_EntropySource::slow_poll(byte out[], u32bit length)
   {
   static const u32bit MAX_PRIORITY = 4;
   return gather(out, length, MAX_PRIORITY);
   }

/*************************************************
* Gather Entropy From Several Sources            *
*************************************************/
u32bit Unix_EntropySource::gather(byte out[], u32bit length, u32bit prio)
   {
   u32bit got = 0;
   for(u32bit j = 0; j != sources.size(); j++)
      {
      if(sources[j].priority > prio && got >= length) break;
      if(sources[j].priority > prio + 2) break;
      if(sources[j].working)
         got += gather_entropy(sources[j]);
      }
   u32bit copied = std::min(BUFFERSIZE - position, length);
   copy_mem(out, buffer + position, copied);
   position = (position + copied) % BUFFERSIZE;
   return copied;
   }

/*************************************************
* Gather Entropy from a Source                   *
*************************************************/
u32bit Unix_EntropySource::gather_entropy(Source& src)
   {
   static const u32bit MIN_OUTPUT = 10;

   std::string command = src.name + " " + src.args + " 2>/dev/null";

   SecureBuffer<byte, BUFFERSIZE> output;
   u32bit total_output = 0;

   timestamp();

   FILE* pipe = popen(command.c_str(), "r");
   if(!pipe)
      throw Exception("Unix_EntropySource: popen failed");
   while(!ferror(pipe) && !feof(pipe))
      {
      u32bit got = fread(output, 1, BUFFERSIZE, pipe);
      total_output += got;
      xor_buf(buffer, output, got);
      }

   if(pclose(pipe) == 127*256)   src.working = false;
   if(total_output < MIN_OUTPUT) src.working = false;

   timestamp();

   return total_output;
   }

/*************************************************
* Add a timestamp value to the buffer            *
*************************************************/
void Unix_EntropySource::timestamp()
   {
   u64bit time = system_clock();
   xor_buf(buffer, (byte*)&time, 8);
   }

/*************************************************
* Add sources to the list                        *
*************************************************/
void Unix_EntropySource::add_sources(const Source srcs[], u32bit count)
   {
   sources.insert(sources.end(), srcs, srcs + count);
   std::sort(sources.begin(), sources.end(), Source_Cmp);
   }

}
