/*************************************************
* Allocator Factory Source File                  *
* (C) 1999-2002 The OpenCL Project               *
*************************************************/

#include <opencl/allocate.h>
#include <opencl/secalloc.h>
#include <opencl/defalloc.h>
#include <opencl/exceptn.h>
#include <map>

namespace OpenCL {

namespace {

class AllocatorFactory
   {
   public:
      SecureAllocator* get(const std::string& type)
         {
         if(alloc.find(type) == alloc.end()) return 0;
         return alloc[type];
         }
      void add(const std::string& type, SecureAllocator* allocator)
         { alloc[type] = allocator; }
      AllocatorFactory() { alloc["default"] = new Default_Allocator; }
      ~AllocatorFactory()
         {
         std::map<std::string, SecureAllocator*>::iterator iter;
         for(iter = alloc.begin(); iter != alloc.end(); iter++)
            delete iter->second;
         }
   private:
      std::map<std::string, SecureAllocator*> alloc;
   };

AllocatorFactory factory;
std::string default_type = "default";

/*************************************************
* Try to get an allocator of the specified type  *
*************************************************/
SecureAllocator* try_alloc(const std::string& type)
   {
   SecureAllocator* alloc = factory.get(type);
   if(alloc) return alloc->clone();
   return 0;
   }

}

/*************************************************
* Get an allocator                               *
*************************************************/
SecureAllocator* get_allocator(const std::string& type)
   {
   SecureAllocator* alloc;

   alloc = try_alloc(type);
   if(alloc) return alloc;

   alloc = try_alloc(default_type);
   if(alloc) return alloc;

   alloc = try_alloc("default");
   if(alloc) return alloc;

   throw Exception("Couldn't find an allocator to use in get_allocator");
   }

/*************************************************
* Release an allocator                           *
*************************************************/
void release_allocator(SecureAllocator* alloc)
   {
   if(!alloc->single_copy())
      delete alloc;
   }

/*************************************************
* Set the default allocator type                 *
*************************************************/
void set_default_allocator(const std::string& type)
   {
   default_type = type;
   }

/*************************************************
* Add new allocator type                         *
*************************************************/
bool add_allocator_type(const std::string& type, SecureAllocator* alloc)
   {
   if(type == "" || factory.get(type))
      return false;
   factory.add(type, alloc);
   return true;
   }

/*************************************************
* Initalize the memory subsystem                 *
*************************************************/
void initalize_memory_subsystem()
   {
   factory.add("default", new Default_Allocator);
   }

}
