/*************************************************
* Base Classes Source File                       *
* (C) 1999-2002 The OpenCL Project               *
*************************************************/

#include <opencl/base.h>

namespace OpenCL {

/*************************************************
* Symmetric Algorithm Constructor                *
*************************************************/
SymmetricAlgorithm::SymmetricAlgorithm(const std::string& name, u32bit key_min,
                                       u32bit key_max, u32bit key_mod) :
   Algorithm(name),
   max(key_max ? key_max : key_min), min(key_min), mod(key_mod) {}

/*************************************************
* Query if the keylength is valid                *
*************************************************/
bool SymmetricAlgorithm::valid_keylength(u32bit length) const
   {
   return ((length >= min) && (length <= max) && (length % mod == 0));
   }

/*************************************************
* Set the key                                    *
*************************************************/
void SymmetricAlgorithm::set_key(const SymmetricKey& key)
   throw(Invalid_Key_Length)
   {
   set_key(key, key.length());
   }

/*************************************************
* Block Cipher Constructor                       *
*************************************************/
BlockCipher::BlockCipher(const std::string& name, u32bit block,
                         u32bit key_min, u32bit key_max, u32bit key_mod) :
   SymmetricAlgorithm(name, key_min, key_max, key_mod), block_sz(block) {}

/*************************************************
* Stream Cipher Constructor                      *
*************************************************/
StreamCipher::StreamCipher(const std::string& name, u32bit key_min,
                           u32bit key_max, u32bit key_mod) :
   SymmetricAlgorithm(name, key_min, key_max, key_mod) {}

/*************************************************
* Random Access Stream Cipher Constructor        *
*************************************************/
RandomAccessStreamCipher::RandomAccessStreamCipher(const std::string& name,
                                                   u32bit key_min,
                                                   u32bit key_max,
                                                   u32bit key_mod) :
   StreamCipher(name, key_min, key_max, key_mod) {}

/*************************************************
* Hash Function Constructor                      *
*************************************************/
HashFunction::HashFunction(const std::string& name, u32bit hlen) :
   BufferedComputation(name, hlen) {}

/*************************************************
* MAC Constructor                                *
*************************************************/
MessageAuthCode::MessageAuthCode(const std::string& name, u32bit mlen,
                                 u32bit key_min, u32bit key_max,
                                 u32bit key_mod) :
   BufferedComputation(name, mlen),
   SymmetricAlgorithm(name, key_min, key_max, key_mod) {}

/*************************************************
* Hashing/MACing                                 *
*************************************************/
void BufferedComputation::update(const byte in[], u32bit n)
   {
   add_data(in, n);
   }

void BufferedComputation::update(const std::string& str)
   {
   update((const byte*)str.c_str(), str.size());
   }

void BufferedComputation::process(const byte in[], u32bit len, byte out[])
   {
   update(in, len);
   final(out);
   }

void BufferedComputation::process(const std::string& in, byte out[])
   {
   update(in);
   final(out);
   }

/*************************************************
* Randomize an array of bytes                    *
*************************************************/
void RandomNumberGenerator::randomize(byte out[], u32bit len)
   {
   for(u32bit j = 0; j != len; j++)
      out[j] ^= random();
   }

/*************************************************
* Return a random u32bit                         *
*************************************************/
u32bit RandomNumberGenerator::random_u32bit()
   {
   return ((random() << 24) | (random() << 16) | (random() << 8) | (random()));
   }

}
