/*************************************************
* BigInt Operators Source File                   *
* (C) 1999-2002 The OpenCL Project               *
*************************************************/

#include <opencl/bigint.h>
#include <opencl/numthry.h>
#include <opencl/mp_core.h>

namespace OpenCL {

/*************************************************
* Addition Operator                              *
*************************************************/
BigInt operator+(const BigInt& x, const BigInt& y)
   {
   if((x.sign() == y.sign()))
      {
      BigInt z(x.sign(), std::max(x.size(), y.size()) + 1);
      bigint_add(x.data(), x.size(), y.data(), y.size(), z.get_reg());
      return z;
      }
   else if(x.is_positive())
      return (x - y.abs());
   else
      return (y - x.abs());
   }

/*************************************************
* Subtraction Operator                           *
*************************************************/
BigInt operator-(const BigInt& x, const BigInt& y)
   {
   s32bit relative_size = bigint_cmp(x.data(), x.size(), y.data(), y.size());
   if(relative_size == 0 && (x.sign() == y.sign())) return 0;
   if(relative_size == 0 && (x.sign() != y.sign())) return (x << 1);

   BigInt z(Positive, std::max(x.size(), y.size()) + 1);

   if(relative_size == -1)
      {
      if(x.sign() == y.sign())
         bigint_sub(y.data(), y.size(), x.data(), x.sig_words(), z.get_reg());
      else
         bigint_add(x.data(), x.size(), y.data(), y.size(), z.get_reg());
      z.set_sign(y.reverse_sign());
      }
   if(relative_size == 1)
      {
      if(x.sign() == y.sign())
         bigint_sub(x.data(), x.size(), y.data(), y.sig_words(), z.get_reg());
      else
         bigint_add(x.data(), x.size(), y.data(), y.size(), z.get_reg());
      z.set_sign(x.sign());
      }
   return z;
   }

/*************************************************
* Multiplication Operator                        *
*************************************************/
BigInt operator*(const BigInt& x, const BigInt& y)
   {
   Sign sign = Positive;
   if(x.sign() != y.sign())
      sign = Negative;

   BigInt z(sign, x.sig_words() + y.sig_words() + 1);
   bigint_mul(x.data(), x.sig_words(), y.data(), y.sig_words(), z.get_reg());
   return z;
   }

/*************************************************
* Division Operator                              *
*************************************************/
BigInt operator/(const BigInt& x, const BigInt& y)
   {
   BigInt q, r;
   divide(x, y, q, r);
   return q;
   }

/*************************************************
* Modulo Operator                                *
*************************************************/
BigInt operator%(const BigInt& n, const BigInt& mod)
   {
   if(mod.is_zero()) throw BigInt::DivideByZero();
   if(mod.is_negative())
      throw Invalid_Argument("BigInt: negative modulus not legal");

   BigInt q, r;
   divide(n, mod, q, r);
   return r;
   }

/*************************************************
* Modulo Operator                                *
*************************************************/
u32bit operator%(const BigInt& n, u32bit mod)
   {
   if(mod == 0) throw BigInt::DivideByZero();

   BigInt q, r;
   divide(n, mod, q, r);
   return r.reg[0];
   }

/*************************************************
* Left Shift Operator                            *
*************************************************/
BigInt operator<<(const BigInt& x, u32bit shift)
   {
   if(shift == 0) return x;
   const u32bit shift_words = shift / 32, shift_bits  = shift % 32;

   BigInt y(x.sign(), x.sig_words() + shift_words + (shift_bits ? 1 : 0));
   bigint_shl(y.get_reg(), x.data(), x.sig_words(), shift_words, shift_bits);
   return y;
   }

/*************************************************
* Right Shift Operator                           *
*************************************************/
BigInt operator>>(const BigInt& x, u32bit shift)
   {
   if(shift == 0) return x;
   const u32bit shift_words = shift / 32, shift_bits  = shift % 32;
   if(x.size() <= shift_words) return 0;

   SecureVector<u32bit> y(x.size() - shift_words);
   bigint_shr(y, x.data(), x.size(), shift_words, shift_bits);
   return BigInt(y, y.size(), x.sign());
   }

/*************************************************
* Prefix Increment Operator                      *
*************************************************/
BigInt& BigInt::operator++()
   {
   if(is_negative()) sub(1);
   else              add(1);
   return (*this);
   }

/*************************************************
* Prefix Decrement Operator                      *
*************************************************/
BigInt& BigInt::operator--()
   {
   if(is_negative()) add(1);
   else              sub(1);
   return (*this);
   }

/*************************************************
* Comparison Function                            *
*************************************************/
s32bit BigInt::cmp(const BigInt& n, bool check_signs) const
   {
   if(check_signs)
      {
      if(n.is_positive() && this->is_negative()) return -1;
      if(n.is_negative() && this->is_positive()) return 1;
      if(n.is_negative() && this->is_negative())
         return (-bigint_cmp(data(), size(), n.data(), n.size()));
      }
   return bigint_cmp(data(), size(), n.data(), n.size());
   }

}
