/*************************************************
* Division Algorithm Source File                 *
* (C) 1999-2002 The OpenCL Project               *
*************************************************/

#include <opencl/numthry.h>
#include <opencl/mp_core.h>

namespace OpenCL {

/*************************************************
* Solve x = q * y + r                            *
*************************************************/
void divide(const BigInt& x_arg, const BigInt& y_arg, BigInt& q, BigInt& r)
   {
   if(y_arg.is_zero()) throw BigInt::DivideByZero();

   BigInt x = x_arg, y = y_arg;
   x.set_sign(Positive);
   y.set_sign(Positive);

   positive_divide(x, y, q, r);

   if(x_arg.sign() == Negative)
      {
      q.flip_sign();
      if(r.is_nonzero()) { --q; r = y_arg.abs() - r; }
      }
   if(y_arg.sign() == Negative)
      q.flip_sign();
   }

/*************************************************
* Solve x = q * y + r                            *
*************************************************/
void positive_divide_nonmodify(const BigInt& x_arg, const BigInt& y_arg,
                               BigInt& q, BigInt& r)
   {
   BigInt x = x_arg, y = y_arg;
   positive_divide(x, y, q, r);
   }

/*************************************************
* Solve x = q * y + r                            *
*************************************************/
void positive_divide(BigInt& x, BigInt& y, BigInt& q, BigInt& r)
   {
   if(y.is_zero()) throw BigInt::DivideByZero();
   if(x.is_negative() || y.is_negative())
      throw Invalid_Argument("Arguments to positive_divide must be positive");

   s32bit compare = x.cmp(y);
   if(compare == -1) { q = BigInt::zero(); r = x; return; }
   if(compare ==  0) { q = BigInt::one(); r = BigInt::zero(); return; }

   u32bit shifts = 0;
   while(y.at(y.sig_words() - 1) < 0x80000000)
      { x <<= 1; y <<= 1; shifts++; }

   x.shrink();
   y.shrink();

   u32bit n = x.size() - 1, t = y.size() - 1;
   q.reg.create(n - t + 1);
   if(n <= t)
      {
      while(x > y) { x -= y; q.add(1); }
      r = x >> shifts;
      return;
      }

   r = y << (32 * (n-t));
   while(x >= r) { x -= r; q[n-t]++; }

   for(u32bit j = n; j != t; j--)
      {
      // I have never seen this code path taken, which makes me nervous
      if(x.at(j) == y.at(t))
         {
         printf("off_chance = 1\n");
         q[j-t-1] = 0xFFFFFFFF;
         }
      else
         q[j-t-1] = (((u64bit)x.at(j) << 32) | x.at(j-1)) / y.at(t);

      while(bigint_divcore(q.at(j-t-1), y.at(t), y.at(t-1),
                           x.at(j), x.at(j-1), x.at(j-2)))
         q[j-t-1]--;

      x -= (q.at(j-t-1) * y) << (32 * (j-t-1));
      if(x.is_negative())
         {
         x += y << (32 * (j-t-1));
         q[j-t-1]--;
         }
      }
   r = x >> shifts;
   }

}
