/*************************************************
* Filter Source File                             *
* (C) 1999-2002 The OpenCL Project               *
*************************************************/

#include <opencl/filter.h>
#include <opencl/secqueue.h>

namespace OpenCL {

/*************************************************
* Filter Constructor                             *
*************************************************/
Filter::Filter(u32bit count)
   {
   set_port_count(count);
   }

/*************************************************
* Set/Reset next                                 *
*************************************************/
void Filter::set_port_count(u32bit n)
   {
   next.clear();
   next.resize(n);
   port_num = 0;
   filter_owns = 0;
   }

/*************************************************
* Send data to all ports                         *
*************************************************/
void Filter::send(const byte input[], u32bit length)
   {
   bool nothing_attached = true;
   for(u32bit j = 0; j != total_ports(); j++)
      if(next[j])
         {
         if(write_queue.size())
            next[j]->write(write_queue, write_queue.size());
         next[j]->write(input, length);
         nothing_attached = false;
         }
   if(nothing_attached)
      write_queue.append(input, length);
   else if(write_queue.has_items())
      write_queue.destroy();
   }

/*************************************************
* Start a new message                            *
*************************************************/
void Filter::new_msg()
   {
   start_msg();
   for(u32bit j = 0; j != total_ports(); j++)
      if(next[j])
         next[j]->new_msg();
   }

/*************************************************
* End the current message                        *
*************************************************/
void Filter::finish_msg()
   {
   end_msg();
   for(u32bit j = 0; j != total_ports(); j++)
      if(next[j])
         next[j]->finish_msg();
   }

/*************************************************
* Attach a filter to the current port            *
*************************************************/
void Filter::attach(Filter* new_filter)
   {
   if(new_filter)
      {
      Filter* last = this;
      while(last->get_next())
         last = last->get_next();
      last->next[last->current_port()] = new_filter;
      }
   }

/*************************************************
* Set the active port on a filter                *
*************************************************/
void Filter::set_port(u32bit new_port)
   {
   if(new_port >= total_ports())
      throw Invalid_Argument("Filter: Invalid port number");
   port_num = new_port;
   }

/*************************************************
* Return the next Filter in the logical chain    *
*************************************************/
Filter* Filter::get_next() const
   {
   if(next.size() >= port_num)
      return next[port_num];
   return 0;
   }

/*************************************************
* Destroy this Filter and it's children          *
*************************************************/
void destroy(Filter*& to_kill)
   {
   if(!to_kill) return;
   for(u32bit j = 0; j != to_kill->total_ports(); j++)
      destroy(to_kill->next[j]);
   delete to_kill;
   to_kill = 0;
   }

}
