/*************************************************
* BigInt Algorithms Source File                  *
* (C) 1999-2002 The OpenCL Project               *
*************************************************/

#include <opencl/mp_core.h>
#include <cstring>

extern "C" {

/*************************************************
* Addition                                       *
*************************************************/
void bigint_add(const u32bit* x, u32bit x_size,
                const u32bit* y, u32bit y_size, u32bit* z)
   {
   if(x_size < y_size) { bigint_add(y, y_size, x, x_size, z); return; }

   u32bit carry = 0;
   for(u32bit j = 0; j != y_size; j++)
      {
      u64bit sum = ((u64bit)x[j] + y[j]) + carry;
      z[j] = (sum & 0xFFFFFFFF);
      carry = (sum >> 32) & 0xFFFFFFFF;
      }
   for(u32bit j = y_size; j != x_size; j++)
      {
      u64bit sum = (u64bit)x[j] + carry;
      z[j] = (sum & 0xFFFFFFFF);
      carry = (sum >> 32) & 0xFFFFFFFF;
      }
   z[x_size] = carry;
   }

/*************************************************
* Subtraction                                    *
*************************************************/
void bigint_sub(const u32bit* x, u32bit x_size,
                const u32bit* y, u32bit y_size, u32bit* z)
   {
   static const u64bit OFFSET = 0x100000000;
   u32bit borrow = 0;
   for(u32bit j = 0; j != y_size; j++)
      {
      u64bit temp = ((OFFSET + x[j]) - y[j]) - borrow;
      borrow = ((temp < OFFSET) ? 1 : 0);
      z[j] = (u32bit)(temp - OFFSET);
      }
   for(u32bit j = y_size; j != x_size; j++)
      {
      u64bit temp = (OFFSET + x[j]) - borrow;
      borrow = ((temp < OFFSET) ? 1 : 0);
      z[j] = (u32bit)(temp - OFFSET);
      }
   }

/*************************************************
* Multiplication                                 *
*************************************************/
void bigint_mul(const u32bit* x, u32bit x_size,
                const u32bit* y, u32bit y_size, u32bit* z)
   {
   for(u32bit j = 0; j != x_size; j++)
      {
      u32bit carry = 0;
      for(u32bit k = 0; k != y_size; k++)
         {
         u64bit product = (u64bit)z[j+k] + (u64bit)x[j] * y[k] + carry;
         z[j+k] = (product & 0xFFFFFFFF);
         carry = (product >> 32) & 0xFFFFFFFF;
         }
      z[j+y_size] = carry;
      }
   }

/*************************************************
* Shift Left                                     *
*************************************************/
void bigint_shl(u32bit* y, const u32bit* x, u32bit x_size,
                u32bit word_shift, u32bit bit_shift)
   {
   for(u32bit j = 0; j != x_size; j++)
      y[j + word_shift] = x[j];
   if(bit_shift)
      {
      u32bit carry = 0;
      for(u32bit j = word_shift; j != x_size + word_shift + 1; j++)
         {
         u32bit temp = y[j];
         y[j] = (temp << bit_shift) | carry;
         carry = (temp >> (32 - bit_shift));
         }
      }
   }

/*************************************************
* Shift Right                                    *
*************************************************/
void bigint_shr(u32bit* y, const u32bit* x, u32bit x_size,
                u32bit word_shift, u32bit bit_shift)
   {
   if(x_size < word_shift) return;

   for(u32bit j = 0; j != x_size - word_shift; j++)
      y[j] = x[j + word_shift];
   if(bit_shift)
      {
      u32bit carry = 0;
      for(u32bit j = x_size - word_shift; j > 0; j--)
         {
         u32bit temp = y[j-1];
         y[j-1] = (temp >> bit_shift) | carry;
         carry = (temp << (32 - bit_shift));
         }
      }
   }

/*************************************************
* Division Core                                  *
*************************************************/
bool bigint_divcore(u32bit q, u32bit y_1, u32bit y_2,
                    u32bit x1, u32bit x2, u32bit x3)
   {
   u32bit last = 0;
   u64bit product = 0;

   product = (u64bit)q * y_2;
   y_2 = (product & 0xFFFFFFFF);

   product = (u64bit)q * y_1 + ((product >> 32) & 0xFFFFFFFF);
   y_1 = (product & 0xFFFFFFFF);
   last = (product >> 32) & 0xFFFFFFFF;

   if(last > x1) return true;
   if(last < x1) return false;
   if(y_1 > x2)  return true;
   if(y_1 < x2)  return false;
   if(y_2 > x3)  return true;
   if(y_2 < x3)  return false;
   return false;
   }

/*************************************************
* Compare                                        *
*************************************************/
s32bit bigint_cmp(const u32bit* x, u32bit x_size,
                  const u32bit* y, u32bit y_size)
   {
   if(x_size < y_size) { return (-bigint_cmp(y, y_size, x, x_size)); }

   while(x_size > y_size)
      {
      if(x[x_size-1])
         return 1;
      x_size--;
      }
   for(u32bit j = x_size; j > 0; j--)
      {
      if(x[j-1] > y[j-1]) return 1;
      if(x[j-1] < y[j-1]) return -1;
      }
   return 0;
   }

}
