/*************************************************
* Pipe Source File                               *
* (C) 1999-2002 The OpenCL Project               *
*************************************************/

#include <opencl/pipe.h>
#include <opencl/secqueue.h>

namespace OpenCL {

/*************************************************
* Pipe Constructor                               *
*************************************************/
Pipe::Pipe(Filter* f1, Filter* f2, Filter* f3, Filter* f4)
   {
   init();
   append(f1);
   append(f2);
   append(f3);
   append(f4);
   }

/*************************************************
* Pipe Constructor                               *
*************************************************/
Pipe::Pipe(Filter* filter_array[], u32bit count)
   {
   init();
   for(u32bit j = 0; j != count; j++)
      append(filter_array[j]);
   }

/*************************************************
* Initialize the Pipe                            *
*************************************************/
void Pipe::init()
   {
   pipe = 0;
   default_read = 0;
   locked = false;
   }

/*************************************************
* Destroy the Pipe                               *
*************************************************/
void Pipe::destruct(Filter* to_kill)
   {
   if(!to_kill) return;
   if(dynamic_cast<SecureQueue*>(to_kill)) return;
   for(u32bit j = 0; j != to_kill->total_ports(); j++)
      destruct(to_kill->next[j]);
   delete to_kill;
   to_kill = 0;
   }

/*************************************************
* Set the default read message                   *
*************************************************/
void Pipe::set_default_msg(u32bit msg)
   {
   if(msg >= messages.size())
      throw Invalid_Argument("Pipe: this default message number is too high");
   default_read = msg;
   }

/*************************************************
* Start a new message                            *
*************************************************/
void Pipe::start_msg()
   {
   if(locked) return;
   find_endpoints(pipe);
   pipe->new_msg();
   locked = true;
   }

/*************************************************
* End the current message                        *
*************************************************/
void Pipe::end_msg()
   {
   if(!locked) return;
   pipe->finish_msg();
   locked = false;
   }

/*************************************************
* Find the endpoints of the Pipe                 *
*************************************************/
void Pipe::find_endpoints(Filter* f)
   {
   for(u32bit j = 0; j != f->total_ports(); j++)
      if(f->next[j] && !dynamic_cast<SecureQueue*>(f->next[j]))
         find_endpoints(f->next[j]);
      else
         {
         SecureQueue* q = new SecureQueue;
         f->next[j] = q;
         messages.push_back(q);
         }
   }

/*************************************************
* Append a Filter to the Pipe                    *
*************************************************/
void Pipe::append(Filter* filter)
   {
   if(locked)
      throw Exception("Cannot attach to a Pipe while it is locked");
   if(!filter) return;
   if(!pipe) pipe = filter;
   else      pipe->attach(filter);
   }

/*************************************************
* Prepend a Filter to the Pipe                   *
*************************************************/
void Pipe::prepend(Filter* filter)
   {
   if(locked)
      throw Exception("Cannot prepend to a Pipe while it is locked");
   if(!filter) return;
   if(pipe) filter->attach(pipe);
   pipe = filter;
   }

/*************************************************
* Pop a Filter off the Pipe                      *
*************************************************/
void Pipe::pop()
   {
   if(locked)
      throw Exception("Cannot pop off a Pipe while it is locked");
   if(!pipe) return;
   if(pipe->total_ports() > 1)
      throw Exception("Cannot pop off a Filter with multiple ports");
   Filter* f = pipe;
   u32bit owns = f->owns();
   pipe = pipe->next[0];
   delete f;

   while(owns--)
      {
      f = pipe;
      pipe = pipe->next[0];
      delete f;
      }
   }

}
