/*************************************************
* Pipe Reading/Writing Source File               *
* (C) 1999-2002 The OpenCL Project               *
*************************************************/

#include <opencl/pipe.h>
#include <opencl/secqueue.h>

namespace OpenCL {

/*************************************************
* Write into a Pipe                              *
*************************************************/
void Pipe::write(const byte input[], u32bit length)
   {
   if(!locked)
      throw Exception("Cannot write to a Pipe while it is unlocked");
   pipe->write(input, length);
   }

/*************************************************
* Write a string into a Pipe                     *
*************************************************/
void Pipe::write(const std::string& str)
   {
   write((const byte*)str.c_str(), str.size());
   }

/*************************************************
* Read some data from the pipe                   *
*************************************************/
u32bit Pipe::read(byte output[], u32bit length)
   {
   return read(default_read, output, length);
   }

/*************************************************
* Read some data from the pipe                   *
*************************************************/
u32bit Pipe::read(u32bit msg, byte output[], u32bit length)
   {
   if(msg >= messages.size())
      throw Exception("Pipe::read: msg num " + to_string(msg) + " too high");
   return messages[msg]->read(output, length);
   }

/*************************************************
* Read a byte from the pipe                      *
*************************************************/
u32bit Pipe::read(u32bit msg, byte& out)
   {
   return read(msg, &out, 1);
   }

/*************************************************
* Return all data in the pipe                    *
*************************************************/
SecureVector<byte> Pipe::read_all()
   {
   SecureVector<byte> buffer(remaining());
   read(buffer, buffer.size());
   return buffer;
   }

/*************************************************
* Return all data in the pipe                    *
*************************************************/
SecureVector<byte> Pipe::read_all(u32bit msg)
   {
   SecureVector<byte> buffer(remaining(msg));
   read(msg, buffer, buffer.size());
   return buffer;
   }

/*************************************************
* Return all data in the pipe as a string        *
*************************************************/
std::string Pipe::read_all_as_string()
   {
   static const u32bit BUFFERSIZE = DEFAULT_BUFFERSIZE;
   SecureBuffer<byte, BUFFERSIZE> buffer;
   std::string str;
   while(remaining())
      {
      u32bit got = read(buffer, BUFFERSIZE);
      str.append((char*)buffer.ptr(), got);
      }
   return str;
   }

/*************************************************
* Return all data in the pipe as a string        *
*************************************************/
std::string Pipe::read_all_as_string(u32bit msg)
   {
   static const u32bit BUFFERSIZE = DEFAULT_BUFFERSIZE;
   SecureBuffer<byte, BUFFERSIZE> buffer;
   std::string str;
   while(remaining(msg))
      {
      u32bit got = read(msg, buffer, BUFFERSIZE);
      str.append((char*)buffer.ptr(), got);
      }
   return str;
   }

/*************************************************
* Find out how many bytes are ready to read      *
*************************************************/
u32bit Pipe::remaining(u32bit msg) const
   {
   if(msg >= messages.size())
      throw Exception("Pipe::remaining: msg num " + to_string(msg) +
                      " too high");
   return messages[msg]->size();
   }

/*************************************************
* Peek at some data in the pipe                  *
*************************************************/
u32bit Pipe::peek(u32bit msg, byte output[], u32bit length)
   {
   if(msg >= messages.size())
      throw Exception("Pipe::peek: msg num " + to_string(msg) + " too high");
   return messages[msg]->peek(output, length);
   }

/*************************************************
* Peek at a byte in the pipe                     *
*************************************************/
u32bit Pipe::peek(u32bit msg, byte& out)
   {
   return peek(msg, &out, 1);
   }

}
