//
// OpenVRML
//
// Copyright (C) 2000  Braden N. McDaniel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//

header "post_include_hpp" {
# include <memory>
# include "field.h"
# include "nodeptr.h"
# include "fieldvalueptr.h"
# define ANTLR_LBRACE {
# define ANTLR_RBRACE }

namespace OpenVRML {
    class Scope;
    class ScriptNode;
    class NodeType;
    class Doc2;
}

namespace {
    class Vrml97Scanner : public antlr::TokenStream {
    public:
        static const int EOF_;
        //
        // The following identifiers for constants must match those in the file
        // Vrml97TokenTypes.txt.
        //
        static const int PERIOD;
        static const int LBRACKET;
        static const int RBRACKET;
        static const int LBRACE;
        static const int RBRACE;
        static const int ID;
        static const int INTEGER;
        static const int HEX_INTEGER;
        static const int REAL;
        static const int STRING;
        static const int KEYWORD_DEF;
        static const int KEYWORD_EVENTIN;
        static const int KEYWORD_EVENTOUT;
        static const int KEYWORD_EXPOSEDFIELD;
        static const int KEYWORD_EXTERNPROTO;
        static const int KEYWORD_FALSE;
        static const int KEYWORD_FIELD;
        static const int KEYWORD_IS;
        static const int KEYWORD_NULL;
        static const int KEYWORD_PROTO;
        static const int KEYWORD_ROUTE;
        static const int KEYWORD_TO;
        static const int KEYWORD_TRUE;
        static const int KEYWORD_USE;
        static const int FIELDTYPE_SFBOOL;
        static const int FIELDTYPE_SFCOLOR;
        static const int FIELDTYPE_SFFLOAT;
        static const int FIELDTYPE_SFIMAGE;
        static const int FIELDTYPE_SFINT32;
        static const int FIELDTYPE_SFNODE;
        static const int FIELDTYPE_SFROTATION;
        static const int FIELDTYPE_SFSTRING;
        static const int FIELDTYPE_SFTIME;
        static const int FIELDTYPE_SFVEC2F;
        static const int FIELDTYPE_SFVEC3F;
        static const int FIELDTYPE_MFCOLOR;
        static const int FIELDTYPE_MFFLOAT;
        static const int FIELDTYPE_MFINT32;
        static const int FIELDTYPE_MFNODE;
        static const int FIELDTYPE_MFROTATION;
        static const int FIELDTYPE_MFSTRING;
        static const int FIELDTYPE_MFTIME;
        static const int FIELDTYPE_MFVEC2F;
        static const int FIELDTYPE_MFVEC3F;

        Vrml97Scanner(std::istream &);

        virtual antlr::RefToken nextToken();

        size_t line() const;
        size_t col() const;

    private:
        void _getNextChar();
        void _identifyKeyword(antlr::Token &);
        void _identifyFieldType(antlr::Token &);
        void _identifyTerminalSymbol(antlr::Token &);

        std::istream & _istm;
        size_t      _line;
        size_t      _col;
        int         _c;
        int         _prevChar;
        int         _prevTokenType;
        bool        _readTooMuch;
        bool        _expectingFieldType;
    };
}

namespace OpenVRML ANTLR_LBRACE
}

header "pre_include_cpp" {
# include "Vrml97Parser.hpp"
ANTLR_RBRACE
}

header "post_include_cpp" {
# include <assert.h>
# include <ctype.h>
# include <iostream>
# include <antlr/CommonToken.hpp>
# include "scope.h"
# include "script.h"
# include "private.h"

namespace {

bool isValidIdFirstChar(char);
bool isValidIdRestChars(char);
bool isWhitespaceChar(char);
bool isNewlineChar(char);
bool isHexDigit(char);

const int Vrml97Scanner::EOF_                   (antlr::Token::EOF_TYPE);

//
// The values and names for these constants *must* match those expressed in the
// file Vrml97TokenTypes.txt.
//
const int Vrml97Scanner::PERIOD                 (4);
const int Vrml97Scanner::LBRACKET               (5);
const int Vrml97Scanner::RBRACKET               (6);
const int Vrml97Scanner::LBRACE                 (7);
const int Vrml97Scanner::RBRACE                 (8);
const int Vrml97Scanner::ID                     (9);
const int Vrml97Scanner::INTEGER                (10);
const int Vrml97Scanner::HEX_INTEGER            (11);
const int Vrml97Scanner::REAL                   (12);
const int Vrml97Scanner::STRING                 (13);
const int Vrml97Scanner::KEYWORD_DEF            (14);
const int Vrml97Scanner::KEYWORD_EVENTIN        (15);
const int Vrml97Scanner::KEYWORD_EVENTOUT       (16);
const int Vrml97Scanner::KEYWORD_EXPOSEDFIELD   (17);
const int Vrml97Scanner::KEYWORD_EXTERNPROTO    (18);
const int Vrml97Scanner::KEYWORD_FALSE          (19);
const int Vrml97Scanner::KEYWORD_FIELD          (20);
const int Vrml97Scanner::KEYWORD_IS             (21);
const int Vrml97Scanner::KEYWORD_NULL           (22);
const int Vrml97Scanner::KEYWORD_PROTO          (23);
const int Vrml97Scanner::KEYWORD_ROUTE          (24);
const int Vrml97Scanner::KEYWORD_TO             (25);
const int Vrml97Scanner::KEYWORD_TRUE           (26);
const int Vrml97Scanner::KEYWORD_USE            (27);
const int Vrml97Scanner::FIELDTYPE_SFBOOL       (28);
const int Vrml97Scanner::FIELDTYPE_SFCOLOR      (29);
const int Vrml97Scanner::FIELDTYPE_SFFLOAT      (30);
const int Vrml97Scanner::FIELDTYPE_SFIMAGE      (31);
const int Vrml97Scanner::FIELDTYPE_SFINT32      (32);
const int Vrml97Scanner::FIELDTYPE_SFNODE       (33);
const int Vrml97Scanner::FIELDTYPE_SFROTATION   (34);
const int Vrml97Scanner::FIELDTYPE_SFSTRING     (35);
const int Vrml97Scanner::FIELDTYPE_SFTIME       (36);
const int Vrml97Scanner::FIELDTYPE_SFVEC2F      (37);
const int Vrml97Scanner::FIELDTYPE_SFVEC3F      (38);
const int Vrml97Scanner::FIELDTYPE_MFCOLOR      (39);
const int Vrml97Scanner::FIELDTYPE_MFFLOAT      (40);
const int Vrml97Scanner::FIELDTYPE_MFINT32      (41);
const int Vrml97Scanner::FIELDTYPE_MFNODE       (42);
const int Vrml97Scanner::FIELDTYPE_MFROTATION   (43);
const int Vrml97Scanner::FIELDTYPE_MFSTRING     (44);
const int Vrml97Scanner::FIELDTYPE_MFTIME       (45);
const int Vrml97Scanner::FIELDTYPE_MFVEC2F      (46);
const int Vrml97Scanner::FIELDTYPE_MFVEC3F      (47);

Vrml97Scanner::Vrml97Scanner(std::istream & istm)
  : _istm(istm), _line(1), _col(0), _c(' '), _prevChar('\0'), _prevTokenType(0),
    _readTooMuch(false), _expectingFieldType(false)
{}

antlr::RefToken Vrml97Scanner::nextToken()
{
    using std::string;
    using antlr::RefToken;
    using antlr::CommonToken;

    RefToken    token(new CommonToken);
    string      tokenString;

    if (_readTooMuch) {
        _readTooMuch = false;
    } else {
        _getNextChar();
    }

    while (isWhitespaceChar(_c) || (_c == '#')) {
        if (_c == '#') {
            while (!(isNewlineChar(_c) || _c == EOF)) { _getNextChar(); }
        } else {
            _getNextChar();
        }
    }

    if (_c == EOF) {
        token->setType(EOF_);
    } else if (isValidIdFirstChar(_c)) {
        //
        // in an identifier or a keyword
        //
        token->setType(ID);

        while (isValidIdRestChars(_c)) {
            tokenString += _c;
            _getNextChar();
        }

        _readTooMuch = true;

        token->setText(tokenString);

        if (_expectingFieldType) {
            _identifyFieldType(*token);
            _expectingFieldType = false;
        }
        _identifyKeyword(*token);

    } else if ((_c == '.') || (_c == '+') || (_c == '-') || isdigit(_c)) {
        //
        // probably in a number
        //

        if ((_c == '+') || (_c == '-')) {
            tokenString += _c;
            _getNextChar();
            _readTooMuch = true;
        }

        if (isdigit(_c)) {
            //
            // definitely in a number
            //
            token->setType(INTEGER);

            tokenString += _c;

            _getNextChar();

            if ((_prevChar == '0') && ((_c == 'x') || (_c == 'X'))) {
                //
                // in an integer expressed in hexadecimal
                //
                token->setType(HEX_INTEGER);

                tokenString += _c;
                _getNextChar();
                while (isHexDigit(_c)) {
                    tokenString += _c;
                    _getNextChar();
                }
            } else {
                while (isdigit(_c)) {
                    tokenString += _c;
                    _getNextChar();
                }
            }

            if (_c == '.') {
                //
                // in a floating-point number
                //
                token->setType(REAL);

                tokenString += _c;
                _getNextChar();

                while (isdigit(_c)) {
                    tokenString += _c;
                    _getNextChar();
                }

                if ((_c == 'E') || (_c == 'e')) {
                    //
                    // in an exponent
                    //
                    tokenString += _c;
                    _getNextChar();

                    if ((_c == '+') || (_c == '-') || isdigit(_c)) {
                        //
                        // exponent may be signed
                        //
                        tokenString += _c;
                        _getNextChar();

                        while (isdigit(_c)) {
                            tokenString += _c;
                            _getNextChar();
                        }
                    }
                }
            } else if ((_c == 'E') || (_c == 'e')) {
                //
                // in an exponent
                //
                token->setType(REAL);

                tokenString += _c;
                _getNextChar();

                if ((_c == '+') || (_c == '-') || isdigit(_c)) {
                    //
                    // exponent may be signed
                    //
                    tokenString += _c;
                    _getNextChar();

                    while (isdigit(_c)) {
                        tokenString += _c;
                        _getNextChar();
                    }
                }
            }

            _readTooMuch = true;

        } else if (_c == '.') {
            //
            // in a floating-point number or a lone full-stop (as in a ROUTE)
            //

            tokenString += _c;
            _getNextChar();

            if (isdigit(_c)) {

                token->setType(REAL);

                while (isdigit(_c)) {
                    tokenString += _c;
                    _getNextChar();
                }

                if ((_c == 'E') || (_c == 'e')) {
                    //
                    // in an exponent
                    //
                    tokenString += _c;
                    _getNextChar();

                    if ((_c == '+') || (_c == '-') || isdigit(_c)) {
                        //
                        // exponent may be signed
                        //
                        tokenString += _c;
                        _getNextChar();

                        while (isdigit(_c)) {
                            tokenString += _c;
                            _getNextChar();
                        }
                    }
                }
            } else {
                token->setType(PERIOD);
            }

            _readTooMuch = true;
        }

        token->setText(tokenString);

    } else if (_c == '"') {
        //
        // in a string
        //
        token->setType(STRING);

        tokenString += _c;
        _getNextChar();

        char prevChar('\0');
        while ((_c != '"') || (prevChar == '\\')) {
            tokenString += _c;
            prevChar = _c;
            _getNextChar();
        }
        tokenString += _c; // add the closing quote

        token->setText(tokenString);

    } else {
        //
        // terminal symbol or unidentified character
        //
        tokenString += _c;

        token->setText(tokenString);

        _identifyTerminalSymbol(*token);
    }

    token->setLine(_line);
    _prevTokenType = token->getType();

    return token;
}

size_t Vrml97Scanner::line() const
{
    return _line;
}

size_t Vrml97Scanner::col() const
{
    return _col;
}

void Vrml97Scanner::_getNextChar() {
    _prevChar = _c;
    _c = _istm.get();
    ++_col; // Increment the column count;

    //
    // Increment the line count (and reset the column count to zero) if the
    // current character is a newline character EXCEPT if the current character
    // is a linefeed AND the previous character is a carriage return.
    //
    if (isNewlineChar(_c)) {
        if (!((_c == 0x0a) && (_prevChar == 0x0d))) {
            ++_line;
            _col = 0;
        }
    }
}

void Vrml97Scanner::_identifyKeyword(antlr::Token & token)
{
    std::string const tokenText(token.getText());

    if      (tokenText == "DEF")            { token.setType(KEYWORD_DEF); }
    else if (tokenText == "eventIn")        { _expectingFieldType = true;
                                              token.setType(KEYWORD_EVENTIN); }
    else if (tokenText == "eventOut")       { _expectingFieldType = true;
                                              token.setType(KEYWORD_EVENTOUT); }
    else if (tokenText == "exposedField")   { _expectingFieldType = true;
                                              token.setType(KEYWORD_EXPOSEDFIELD); }
    else if (tokenText == "EXTERNPROTO")    { token.setType(KEYWORD_EXTERNPROTO); }
    else if (tokenText == "FALSE")          { token.setType(KEYWORD_FALSE); }
    else if (tokenText == "field")          { _expectingFieldType = true;
                                              token.setType(KEYWORD_FIELD); }
    else if (tokenText == "IS")             { token.setType(KEYWORD_IS); }
    else if (tokenText == "NULL")           { token.setType(KEYWORD_NULL); }
    else if (tokenText == "PROTO")          { token.setType(KEYWORD_PROTO); }
    else if (tokenText == "ROUTE")          { token.setType(KEYWORD_ROUTE); }
    else if (tokenText == "TO")             { token.setType(KEYWORD_TO); }
    else if (tokenText == "TRUE")           { token.setType(KEYWORD_TRUE); }
    else if (tokenText == "USE")            { token.setType(KEYWORD_USE); }
}

void Vrml97Scanner::_identifyFieldType(antlr::Token & token)
{
    assert(_expectingFieldType);

    std::string const tokenText(token.getText());

    if      (tokenText == "SFBool")     { token.setType(FIELDTYPE_SFBOOL); }
    else if (tokenText == "SFColor")    { token.setType(FIELDTYPE_SFCOLOR); }
    else if (tokenText == "SFFloat")    { token.setType(FIELDTYPE_SFFLOAT); }
    else if (tokenText == "SFImage")    { token.setType(FIELDTYPE_SFIMAGE); }
    else if (tokenText == "SFInt32")    { token.setType(FIELDTYPE_SFINT32); }
    else if (tokenText == "SFNode")     { token.setType(FIELDTYPE_SFNODE); }
    else if (tokenText == "SFRotation") { token.setType(FIELDTYPE_SFROTATION); }
    else if (tokenText == "SFString")   { token.setType(FIELDTYPE_SFSTRING); }
    else if (tokenText == "SFTime")     { token.setType(FIELDTYPE_SFTIME); }
    else if (tokenText == "SFVec2f")    { token.setType(FIELDTYPE_SFVEC2F); }
    else if (tokenText == "SFVec3f")    { token.setType(FIELDTYPE_SFVEC3F); }
    else if (tokenText == "MFColor")    { token.setType(FIELDTYPE_MFCOLOR); }
    else if (tokenText == "MFFloat")    { token.setType(FIELDTYPE_MFFLOAT); }
    else if (tokenText == "MFInt32")    { token.setType(FIELDTYPE_MFINT32); }
    else if (tokenText == "MFNode")     { token.setType(FIELDTYPE_MFNODE); }
    else if (tokenText == "MFRotation") { token.setType(FIELDTYPE_MFROTATION); }
    else if (tokenText == "MFString")   { token.setType(FIELDTYPE_MFSTRING); }
    else if (tokenText == "MFTime")     { token.setType(FIELDTYPE_MFTIME); }
    else if (tokenText == "MFVec2f")    { token.setType(FIELDTYPE_MFVEC2F); }
    else if (tokenText == "MFVec3f")    { token.setType(FIELDTYPE_MFVEC3F); }
}

void Vrml97Scanner::_identifyTerminalSymbol(antlr::Token & token)
{
    std::string const tokenText(token.getText());

    if      (tokenText == "[")  { token.setType(LBRACKET); }
    else if (tokenText == "]")  { token.setType(RBRACKET); }
    else if (tokenText == "{")  { token.setType(LBRACE); }
    else if (tokenText == "}")  { token.setType(RBRACE); }
}

bool isValidIdFirstChar(char c) {
    if (((c >= 0x30) && (c <= 0x39))
        || (c == 0x2b)
        || (c == 0x2d)
        || !isValidIdRestChars(c)) {

        return false;
    }

    return true;
}

bool isValidIdRestChars(char c) {
    if (   (c <= 0x20)
        || (c == 0x22)
        || (c == 0x23)
        || (c == 0x27)
        || (c == 0x2c)
        || (c == 0x2e)
        || (c == 0x5b)
        || (c == 0x5c)
        || (c == 0x5d)
        || (c == 0x7b)
        || (c == 0x7d)
        || (c == 0x7f)) {

        return false;
    }

    return true;
}

bool isWhitespaceChar(char c) {
    if (   (c == 0x0d)      // carriage return
        || (c == 0x0a)      // linefeed
        || (c == 0x20)      // space
        || (c == 0x09)      // tab
        || (c == 0x2c)) {   // comma

        return true;
    }

    return false;
}

bool isNewlineChar(char c) {
    return ((c == 0x0a) || (c == 0x0d));
}

bool isHexDigit(char c) {
    if (          isdigit(c)
        || (c == 'A') || (c == 'a')
        || (c == 'B') || (c == 'b')
        || (c == 'C') || (c == 'c')
        || (c == 'D') || (c == 'd')
        || (c == 'E') || (c == 'e')
        || (c == 'F') || (c == 'f')) {

        return true;
    }

    return false;
}

struct InterfaceIdEquals_ : std::unary_function<OpenVRML::NodeInterface, bool> {
    explicit InterfaceIdEquals_(const std::string & interfaceId):
        interfaceId(&interfaceId)
    {}

    bool operator()(const OpenVRML::NodeInterface & interface) const
    {
        return interface.id == *this->interfaceId;
    }

private:
    const std::string * interfaceId;
};

} // namespace

namespace OpenVRML ANTLR_LBRACE

}

options {
    language="Cpp";
}

class Vrml97Parser extends Parser;
options {
    k=1;
    importVocab=Vrml97;
}

{
public:
    Vrml97Parser(antlr::TokenStream & lexer, const std::string & uri):
        antlr::LLkParser(lexer, 1),
        uri(uri)
    {
        setTokenNames(_tokenNames);
    }

private:
    const std::string uri;
}

vrmlScene[OpenVRML::Browser & browser,
          OpenVRML::MFNode & mfNode]
{
    const ScopePtr scope(new Vrml97RootScope(browser, this->uri));
}
    :   (statement[browser, mfNode, scope])*
    ;

statement[OpenVRML::Browser & browser,
          OpenVRML::MFNode & mfNode,
          const OpenVRML::ScopePtr & scope]
    {
        OpenVRML::NodePtr node;
        OpenVRML::NodeTypePtr nodeType;
    }
    : node=nodeStatement[browser, scope] {
            //
            // If we are unable to parse a node, node will be null.
            //
            if (node) {
                mfNode.addNode(node);
            }
        }
    | protoStatement[browser, scope]
    | routeStatement[*scope]
    ;

nodeStatement[OpenVRML::Browser & browser,
              const OpenVRML::ScopePtr & scope]
returns [OpenVRML::NodePtr n]
options { defaultErrorHandler=false; }
    :   KEYWORD_DEF id0:ID n=node[browser, scope, id0->getText()]
    |   KEYWORD_USE id1:ID
        {
            assert(scope);
            n.reset(scope->findNode(id1->getText()));
            if (!n) {
                throw antlr::SemanticException("Node \"" + id1->getText()
                                    + "\" has not been defined in this scope.",
                                    this->uri, LT(0)->getLine());
            }
        }
    |   n=node[browser, scope, std::string()]
    ;
    exception
    catch [antlr::RecognitionException & ex] {
        reportError(ex);
    }

protoStatement[OpenVRML::Browser & browser,
               const OpenVRML::ScopePtr & scope]
    //
    // XXX What if the node type already exists in the scope? Probably need to
    // XXX handle an exception here.
    //
    : externproto[browser, scope]
    | proto[browser, scope]
    ;

proto[OpenVRML::Browser & browser,
      const OpenVRML::ScopePtr & scope]
{
    NodeClassPtr nodeClass;
    ScopePtr protoScope;
}
    : KEYWORD_PROTO id:ID {
            nodeClass.reset(new ProtoNodeClass(browser));
            protoScope.reset(new Scope(id->getText(), scope));
        }
        LBRACKET (
            {
                // XXX Hmm... We give each interface declaration its own scope
                // XXX here. This is wasteful; Vrml97RootScope is expensive and
                // XXX we only *really* need this for SFNode and MFNode
                // XXX fields/exposedFields.
                ScopePtr interfaceDeclScope(new Vrml97RootScope(browser,
                                                                this->uri));
            }
            protoInterfaceDeclaration[interfaceDeclScope,
                                      static_cast<ProtoNodeClass &>(*nodeClass)]
        )* RBRACKET LBRACE protoBody[protoScope,
                                     static_cast<ProtoNodeClass &>(*nodeClass)]
        RBRACE
        {
            //
            // Add the new NodeClass (prototype definition) to the Browser's
            // NodeClassMap.
            //
            // First, construct the id for the node implementation.
            //
            std::string implId = scope->id;
            while (protoScope) {
                implId += '#' + protoScope->id;
                protoScope = protoScope->parent;
            }
            const Browser::NodeClassMap::value_type value(implId, nodeClass);
            browser.nodeClassMap.insert(value);

            //
            // PROTO's implicitly introduce a new node type as well...
            //
            const NodeTypePtr nodeType =
                    nodeClass->createType(id->getText(),
                                          static_cast<ProtoNodeClass &>(*nodeClass)
                                              .protoNodeType.getInterfaces());
            assert(nodeType);
            assert(scope);
            scope->addNodeType(nodeType);
        }
    ;

protoInterfaceDeclaration[const OpenVRML::ScopePtr & scope,
                          OpenVRML::ProtoNodeClass & proto]
{
    using OpenVRML::NodeInterface;
    using antlr::SemanticException;

    NodeInterface::Type it(NodeInterface::invalidType);
    OpenVRML::FieldValue::Type ft(FieldValue::invalidType);
    OpenVRML::FieldValuePtr fv;
}
    : it=eventInterfaceType ft=fieldType id0:ID {
            try {
                switch (it) {
                case NodeInterface::eventIn:
                    proto.addEventIn(ft, id0->getText());
                    break;

                case NodeInterface::eventOut:
                    proto.addEventOut(ft, id0->getText());
                    break;

                default:
                    assert(false);
                }
            } catch (std::invalid_argument & ex) {
                throw SemanticException(ex.what(), this->uri, LT(0)->getLine());
            }
        }
    | it=fieldInterfaceType ft=fieldType id1:ID
        fv=fieldValue[proto.browser, scope, ft] {
            assert(fv);
            try {
                switch (it) {
                case NodeInterface::field:
                    proto.addField(id1->getText(), fv);
                    break;

                case NodeInterface::exposedField:
                    proto.addExposedField(id1->getText(), fv);
                    break;

                default:
                    assert(false);
                }
            } catch (std::invalid_argument & ex) {
                throw SemanticException(ex.what(), this->uri, LT(0)->getLine());
            }
        }
    ;

eventInterfaceType returns [OpenVRML::NodeInterface::Type it = OpenVRML::NodeInterface::invalidType]
    :   KEYWORD_EVENTIN  { it = OpenVRML::NodeInterface::eventIn; }
    |   KEYWORD_EVENTOUT { it = OpenVRML::NodeInterface::eventOut; }
    ;

fieldInterfaceType returns [OpenVRML::NodeInterface::Type it = OpenVRML::NodeInterface::invalidType]
    :   KEYWORD_FIELD        { it = OpenVRML::NodeInterface::field; }
    |   KEYWORD_EXPOSEDFIELD { it = OpenVRML::NodeInterface::exposedField; }
    ;

protoBody[const OpenVRML::ScopePtr & scope,
          OpenVRML::ProtoNodeClass & proto]
{
    OpenVRML::NodePtr n;
}
    : (protoStatement[proto.browser, scope])*
        n=protoNodeStatement[proto, scope] { assert(n); proto.addRootNode(n); }
        (protoBodyStatement[proto, scope])*
    ;

protoBodyStatement[OpenVRML::ProtoNodeClass & proto,
                   const OpenVRML::ScopePtr & scope]
{
    OpenVRML::NodePtr n;
}
    : n=protoNodeStatement[proto, scope] { assert(n); proto.addRootNode(n); }
    | protoStatement[proto.browser, scope]
    | routeStatement[*scope]
    ;

protoNodeStatement[OpenVRML::ProtoNodeClass & proto,
                   const OpenVRML::ScopePtr & scope]
returns [OpenVRML::NodePtr n]
options { defaultErrorHandler=false; }
{
    using antlr::SemanticException;
}
    : KEYWORD_DEF id0:ID n=protoNode[proto, scope, id0->getText()]
    | KEYWORD_USE id1:ID {
            n.reset(scope->findNode(id1->getText()));
            if (!n) {
                throw SemanticException("Node \"" + id1->getText()
                                    + "\" has not been defined in this scope.",
                                    this->uri, LT(0)->getLine());
            }
        }
    | n=protoNode[proto, scope, std::string()]
    ;

externproto[OpenVRML::Browser & browser, const OpenVRML::ScopePtr & scope]
{
    OpenVRML::NodeInterfaceSet interfaces;
    OpenVRML::MFString urlList;
    OpenVRML::NodeTypePtr nodeType;
}
    : KEYWORD_EXTERNPROTO id:ID LBRACKET
        (externInterfaceDeclaration[interfaces])* RBRACKET
        urlList=externprotoUrlList {
            for (size_t i = 0; i < urlList.getLength(); ++i) {
            	Browser::NodeClassMap::const_iterator pos =
                        browser.nodeClassMap.find(urlList.getElement(i));
                if (pos != browser.nodeClassMap.end()) {
                    nodeType = pos->second->createType(id->getText(),
                                                       interfaces);
                    break;
                }
            }
            //
            // If we weren't able to create a NodeType, that means that we
            // don't already have a NodeClass for the node. Currently we only
            // support referring to existing NodeClasses with EXTERNPROTO;
            // adding new NodeClasses via EXTERNPROTO is not supported. In
            // practice, this means that the ordinary way of using EXTERNPROTOs
            // in VRML worlds will fail.
            //
            if (nodeType) { scope->addNodeType(nodeType); }
        }
    ;

externInterfaceDeclaration[OpenVRML::NodeInterfaceSet & interfaces]
        {
            using OpenVRML::NodeInterface;
            using OpenVRML::FieldValue;
            using antlr::SemanticException;
            NodeInterface::Type it(NodeInterface::invalidType);
            FieldValue::Type ft(FieldValue::invalidType);
        }
    :   it=interfaceType ft=fieldType id:ID {
            const NodeInterface interface(it, ft, id->getText());
            try {
                interfaces.add(interface);
            } catch (std::invalid_argument & ex) {
            	throw SemanticException(ex.what(), this->uri, LT(0)->getLine());
            }
        }
    ;

interfaceType
returns [OpenVRML::NodeInterface::Type it = OpenVRML::NodeInterface::invalidType]
    : it=eventInterfaceType
    | it=fieldInterfaceType
    ;

externprotoUrlList returns [OpenVRML::MFString urlList]
        {
            using std::string;
            using std::vector;
            using OpenVRML::MFString;

            string s;
        }
    :   s=stringValue { urlList = MFString(1, &s); }
    |   LBRACKET { vector<string> stringVector; }
        ( s=stringValue { stringVector.push_back(s); } )* RBRACKET
        { urlList = MFString(stringVector.size(), &stringVector[0]); }
    ;

routeStatement[const OpenVRML::Scope & scope]
    :   KEYWORD_ROUTE fromNodeId:ID PERIOD fromInterfaceId:ID
        KEYWORD_TO toNodeId:ID PERIOD toInterfaceId:ID
        {
            using OpenVRML::FieldValue;
            using OpenVRML::Node;
            using OpenVRML::NodePtr;
            using antlr::SemanticException;

            Node * const fromNode = scope.findNode(fromNodeId->getText());
            if (!fromNode) {
                throw SemanticException("Node \"" + fromNodeId->getText()
                                    + "\" has not been defined in this scope.",
                                    this->uri, LT(0)->getLine());
            }

            Node * const toNode = scope.findNode(toNodeId->getText());
            if (!toNode) {
                throw SemanticException("Node \"" + toNodeId->getText()
                        + "\" has not been defined in this scope.",
                        this->uri, LT(0)->getLine());
            }

            try {
                fromNode->addRoute(fromInterfaceId->getText(),
       	                           NodePtr(toNode), toInterfaceId->getText());
            } catch (std::runtime_error & ex) {
                throw SemanticException(ex.what(), this->uri, LT(0)->getLine());
            }
        }
    ;

node[OpenVRML::Browser & browser,
     const OpenVRML::ScopePtr & scope,
     const std::string & nodeId]
returns [OpenVRML::NodePtr n]
options { defaultErrorHandler = false; }
{
    using OpenVRML::NodeTypePtr;
    using OpenVRML::NodePtr;
    using OpenVRML::ScriptNode;
    using antlr::SemanticException;

    NodeTypePtr nodeType;
}
    : { !LT(1)->getText().compare("Script") }? scriptId:ID {
            n.reset(new ScriptNode(browser.scriptNodeClass, scope));
            if (!nodeId.empty()) { n->setId(nodeId); }

            ScriptNode * const scriptNode = n->toScript();
            assert(scriptNode);
        } LBRACE (
            nodeBodyElement[scope, *n]
            | scriptInterfaceDeclaration[scope, *scriptNode]
        )* RBRACE

    | nodeTypeId:ID {
            nodeType = scope->findType(nodeTypeId->getText());
            if (!nodeType) {
                throw SemanticException("Unknown node type \""
                                        + nodeTypeId->getText() + "\".",
                                        this->uri, LT(0)->getLine());
            }

            n = NodePtr(nodeType->createNode(scope));

            if (!nodeId.empty()) { n->setId(nodeId); }
        } LBRACE (nodeBodyElement[scope, *n])* RBRACE
    ;

nodeBodyElement[const OpenVRML::ScopePtr & scope,
                OpenVRML::Node & node]
{
    using OpenVRML::FieldValue;
    using antlr::SemanticException;
    FieldValue::Type ft(FieldValue::invalidType);
    FieldValuePtr fv;
}
    : id:ID {
            ft = node.nodeType.hasField(id->getText());
            if (ft == FieldValue::invalidType) {
                ft = node.nodeType.hasExposedField(id->getText());
                if (ft == FieldValue::invalidType) {
                    throw SemanticException(node.nodeType.id
                                    + " node has no field or exposedField \""
                                    + id->getText() + "\"",
                                    this->uri, LT(0)->getLine());
                }
            }
        }
        fv=fieldValue[node.nodeType.nodeClass.browser, scope, ft] {
            assert(fv);
            node.setField(id->getText(), *fv);
        }
    |   routeStatement[*scope]
    |   protoStatement[node.nodeType.nodeClass.browser, scope]
    ;

scriptInterfaceDeclaration[const OpenVRML::ScopePtr & scope,
                           OpenVRML::ScriptNode & node]
    {
        using OpenVRML::NodeInterface;
        using OpenVRML::FieldValue;
        using antlr::SemanticException;
        NodeInterface::Type it(NodeInterface::invalidType);
        FieldValue::Type ft(FieldValue::invalidType);
    }
    : it=eventInterfaceType ft=fieldType id:ID {
            const NodeInterfaceSet::const_iterator pos =
                    node.nodeType.getInterfaces().findInterface(id->getText());
            if (pos != node.nodeType.getInterfaces().end()) {
                throw SemanticException("Interface \"" + id->getText()
                                    + "\" already declared for Script node.",
                                    this->uri, LT(0)->getLine());
            }
            switch (it) {
            case NodeInterface::eventIn:
                node.addEventIn(ft, id->getText());
                break;
            case NodeInterface::eventOut:
                node.addEventOut(ft, id->getText());
                break;
            default:
                assert(false);
            }
        }
    | scriptFieldInterfaceDeclaration[scope, node]
    ;

scriptFieldInterfaceDeclaration[const OpenVRML::ScopePtr & scope,
                                OpenVRML::ScriptNode & node]
{
    using std::find_if;
    using OpenVRML::FieldValue;
    using antlr::SemanticException;

    FieldValue::Type ft = FieldValue::invalidType;
    FieldValuePtr fv;
}
    : KEYWORD_FIELD ft=fieldType id:ID
        fv=fieldValue[node.nodeType.nodeClass.browser, scope, ft] {
            assert(fv);
            const NodeInterfaceSet & interfaces = node.nodeType.getInterfaces();
            const NodeInterfaceSet::const_iterator pos =
                    interfaces.findInterface(id->getText());
            if (pos != interfaces.end()) {
                throw SemanticException("Interface \"" + id->getText()
                                    + "\" already declared for Script node.",
                                    this->uri, LT(0)->getLine());
            }
            node.addField(id->getText(), fv);
        }
    ;

protoNode[OpenVRML::ProtoNodeClass & proto,
          const OpenVRML::ScopePtr & scope,
          const std::string & nodeId]
returns [OpenVRML::NodePtr n]
options { defaultErrorHandler=false; }
{
    using OpenVRML::NodeTypePtr;
    using OpenVRML::NodePtr;
    using OpenVRML::ScriptNode;
    using antlr::SemanticException;

    NodeTypePtr nodeType;
}
    : { !LT(1)->getText().compare("Script") }? scriptId:ID {
            n.reset(new ScriptNode(proto.browser.scriptNodeClass, scope));
            if (!nodeId.empty()) { n->setId(nodeId); }

            ScriptNode * const scriptNode = n->toScript();
            assert(scriptNode);
        }
        LBRACE (
            protoNodeBodyElement[proto, scope, *n]
            | protoScriptInterfaceDeclaration[proto, scope, *scriptNode]
        )* RBRACE

    | nodeTypeId:ID {
            nodeType = scope->findType(nodeTypeId->getText());
            if (!nodeType) {
                throw SemanticException("Unknown node type \""
                                        + nodeTypeId->getText() + "\".",
                                        this->uri, LT(0)->getLine());
            }
            n = nodeType->createNode(scope);
            if (!nodeId.empty()) { n->setId(nodeId); }
        }
        LBRACE (protoNodeBodyElement[proto, scope, *n])* RBRACE
    ;

protoNodeBodyElement[OpenVRML::ProtoNodeClass & proto,
                     const OpenVRML::ScopePtr & scope,
                     OpenVRML::Node & node]
{
    using OpenVRML::FieldValue;
    using antlr::SemanticException;

    FieldValue::Type ft(FieldValue::invalidType);
    FieldValuePtr fv;
}
    : { node.nodeType.hasField(LT(1)->getText())
        || node.nodeType.hasExposedField(LT(1)->getText()) }? id:ID {
            ft = node.nodeType.hasField(id->getText());
            if (ft == FieldValue::invalidType) {
                ft = node.nodeType.hasExposedField(id->getText());
                if (ft == FieldValue::invalidType) {
                    throw SemanticException(node.nodeType.id
                                    + " node has no field or exposedField \""
                                    + id->getText() + "\".",
                                    this->uri, LT(0)->getLine());
                }
            }
        } (
            (
                fv=protoFieldValue[proto, scope, ft] {
                    assert(fv);
                    node.setField(id->getText(), *fv);
                }
            )
            | isStatement[proto, node, id->getText()]
        )
    | eventId:ID isStatement[proto, node, eventId->getText()]
    | routeStatement[*scope]
    | protoStatement[proto.browser, scope]
    ;

isStatement[OpenVRML::ProtoNodeClass & proto, OpenVRML::Node & node,
            std::string const & nodeInterfaceId]
{
    using antlr::SemanticException;
}
    : KEYWORD_IS id:ID {
            try {
                proto.addIS(node, nodeInterfaceId, id->getText());
            } catch (std::bad_alloc & ex) {
                throw;
            } catch (std::runtime_error & ex) {
                //
                // ex should be UnsupportedInterface, NodeInterfaceTypeMismatch,
                // or FieldValueTypeMismatch.
                //
                throw SemanticException(ex.what(), this->uri, LT(0)->getLine());
            }
        }
    ;

protoScriptInterfaceDeclaration[OpenVRML::ProtoNodeClass & proto,
                                const OpenVRML::ScopePtr & scope,
                                OpenVRML::ScriptNode & node]
{
    using OpenVRML::NodeInterface;
    using OpenVRML::FieldValue;
    using antlr::SemanticException;
    NodeInterface::Type it(NodeInterface::invalidType);
    FieldValue::Type ft(FieldValue::invalidType);
}
    : it=eventInterfaceType ft=fieldType id:ID {
            const NodeInterfaceSet::const_iterator pos =
                    node.nodeType.getInterfaces().findInterface(id->getText());
            if (pos != node.nodeType.getInterfaces().end()) {
                throw SemanticException("Interface \"" + id->getText()
                                    + "\" already declared for Script node.",
                                    this->uri, LT(0)->getLine());
            }
            switch (it) {
            case NodeInterface::eventIn:
                node.addEventIn(ft, id->getText());
                break;
            case NodeInterface::eventOut:
                node.addEventOut(ft, id->getText());
                break;
            default:
                assert(false);
            }
        } (isStatement[proto, node, id->getText()])?
    |   protoScriptFieldInterfaceDeclaration[proto, scope, node]
    ;

protoScriptFieldInterfaceDeclaration[OpenVRML::ProtoNodeClass & proto,
                                     const OpenVRML::ScopePtr & scope,
                                     OpenVRML::ScriptNode & node]
{
    using std::find_if;
    using OpenVRML::FieldValue;
    using antlr::SemanticException;

    FieldValue::Type ft(FieldValue::invalidType);
    FieldValuePtr fv;
}
    : KEYWORD_FIELD ft=fieldType id:ID {
            //
            // We need to check if the fieldId is an exact match for any
            // existing interface for the Script node; so, we don't use
            // NodeInterfaceSet::findInterface.
            //
            const NodeInterfaceSet & interfaces = node.nodeType.getInterfaces();
            const NodeInterfaceSet::const_iterator pos =
                    find_if(interfaces.begin(), interfaces.end(),
                            InterfaceIdEquals_(id->getText()));
            if (pos != interfaces.end()) {
                throw SemanticException("Interface \"" + id->getText()
                                    + "\" already declared for Script node.",
                                    this->uri, LT(0)->getLine());
            }
        } (
            (
                fv=protoFieldValue[proto, scope, ft] {
                    assert(fv);
                    node.addField(id->getText(), fv);
                }
            )
            | (
                KEYWORD_IS protoFieldId:ID {
                    //
                    // First, get the field value from the ProtoNodeClass'
                    // fieldValueMap.
                    //
                    ProtoNodeClass::DefaultValueMap::const_iterator pos =
                            proto.defaultValueMap.find(protoFieldId->getText());
                    if (pos == proto.defaultValueMap.end()) {
                        throw SemanticException("Proto has no field \""
                                                + protoFieldId->getText()
                                                + "\".", this->uri,
                                                LT(0)->getLine());
                    }

                    //
                    // Now, add the field to the Script node with the default
                    // value.
                    //
                    node.addField(id->getText(), pos->second);

                    //
                    // Finally, add the IS mapping.
                    //
                    try {
                        proto.addIS(node, id->getText(),
                                    protoFieldId->getText());
                    } catch (std::bad_alloc & ex) {
                        throw;
                    } catch (std::runtime_error & ex) {
                        //
                        // ex should be UnsupportedInterface,
                        // NodeInterfaceTypeMismatch, or FieldValueTypeMismatch.
                        //
                        throw SemanticException(ex.what(), this->uri,
                                                LT(0)->getLine());
                    }
                }
            )
        )
    ;

fieldType
returns [OpenVRML::FieldValue::Type ft = OpenVRML::FieldValue::invalidType]
{
    using OpenVRML::FieldValue;
}
    : FIELDTYPE_MFCOLOR     { ft = FieldValue::mfcolor; }
    | FIELDTYPE_MFFLOAT     { ft = FieldValue::mffloat; }
    | FIELDTYPE_MFINT32     { ft = FieldValue::mfint32; }
    | FIELDTYPE_MFNODE      { ft = FieldValue::mfnode; }
    | FIELDTYPE_MFROTATION  { ft = FieldValue::mfrotation; }
    | FIELDTYPE_MFSTRING    { ft = FieldValue::mfstring; }
    | FIELDTYPE_MFTIME      { ft = FieldValue::mftime; }
    | FIELDTYPE_MFVEC2F     { ft = FieldValue::mfvec2f; }
    | FIELDTYPE_MFVEC3F     { ft = FieldValue::mfvec3f; }
    | FIELDTYPE_SFBOOL      { ft = FieldValue::sfbool; }
    | FIELDTYPE_SFCOLOR     { ft = FieldValue::sfcolor; }
    | FIELDTYPE_SFFLOAT     { ft = FieldValue::sffloat; }
    | FIELDTYPE_SFIMAGE     { ft = FieldValue::sfimage; }
    | FIELDTYPE_SFINT32     { ft = FieldValue::sfint32; }
    | FIELDTYPE_SFNODE      { ft = FieldValue::sfnode; }
    | FIELDTYPE_SFROTATION  { ft = FieldValue::sfrotation; }
    | FIELDTYPE_SFSTRING    { ft = FieldValue::sfstring; }
    | FIELDTYPE_SFTIME      { ft = FieldValue::sftime; }
    | FIELDTYPE_SFVEC2F     { ft = FieldValue::sfvec2f; }
    | FIELDTYPE_SFVEC3F     { ft = FieldValue::sfvec3f; }
    ;

fieldValue[OpenVRML::Browser & browser,
           const OpenVRML::ScopePtr & scope,
           OpenVRML::FieldValue::Type ft]
returns [OpenVRML::FieldValuePtr fv]
options { defaultErrorHandler=false; }
{
    using OpenVRML::FieldValue;
}
    : { (ft == FieldValue::sfnode) || (ft == FieldValue::mfnode) }?
        fv=nodeFieldValue[browser, scope, ft]
    | fv=nonNodeFieldValue[ft]
    ;

protoFieldValue[OpenVRML::ProtoNodeClass & proto,
                const OpenVRML::ScopePtr & scope,
                OpenVRML::FieldValue::Type ft]
returns [OpenVRML::FieldValuePtr fv]
{
    using OpenVRML::FieldValue;
}
    : { (ft == FieldValue::sfnode) || (ft == FieldValue::mfnode) }?
        fv=protoNodeFieldValue[proto, scope, ft] { assert(fv); }
    | fv=nonNodeFieldValue[ft] { assert(fv); }
    ;

nonNodeFieldValue[OpenVRML::FieldValue::Type ft]
returns [OpenVRML::FieldValuePtr fv = OpenVRML::FieldValuePtr(0)]
options { defaultErrorHandler=false; }
{
    using OpenVRML::FieldValue;
}
    : { ft == FieldValue::sfbool }? fv=sfBoolValue
    | { ft == FieldValue::sfcolor }? fv=sfColorValue
    | { ft == FieldValue::sffloat }? fv=sfFloatValue
    | { ft == FieldValue::sfimage }? fv=sfImageValue
    | { ft == FieldValue::sfint32 }? fv=sfInt32Value
    | { ft == FieldValue::sfrotation }? fv=sfRotationValue
    | { ft == FieldValue::sfstring }? fv=sfStringValue
    | { ft == FieldValue::sftime }? fv=sfTimeValue
    | { ft == FieldValue::sfvec2f }? fv=sfVec2fValue
    | { ft == FieldValue::sfvec3f }? fv=sfVec3fValue
    | { ft == FieldValue::mfcolor }? fv=mfColorValue
    | { ft == FieldValue::mffloat }? fv=mfFloatValue
    | { ft == FieldValue::mfint32 }? fv=mfInt32Value
    | { ft == FieldValue::mfrotation }? fv=mfRotationValue
    | { ft == FieldValue::mfstring }? fv=mfStringValue
    | { ft == FieldValue::mftime }? fv=mfTimeValue
    | { ft == FieldValue::mfvec2f }? fv=mfVec2fValue
    | fv=mfVec3fValue
    ;

nodeFieldValue[OpenVRML::Browser & browser,
               const OpenVRML::ScopePtr & scope,
               OpenVRML::FieldValue::Type ft]
returns [OpenVRML::FieldValuePtr fv]
options { defaultErrorHandler=false; }
{
    using OpenVRML::FieldValue;
}
    : { ft == FieldValue::sfnode }? fv=sfNodeValue[browser, scope]
    | fv=mfNodeValue[browser, scope]
    ;

protoNodeFieldValue[OpenVRML::ProtoNodeClass & proto,
                    const OpenVRML::ScopePtr & scope,
                    OpenVRML::FieldValue::Type ft]
returns [OpenVRML::FieldValuePtr fv]
options { defaultErrorHandler=false; }
    : { ft == OpenVRML::FieldValue::sfnode }?
        fv=protoSfNodeValue[proto, scope]
    | fv=protoMfNodeValue[proto, scope]
    ;

sfBoolValue returns [OpenVRML::FieldValuePtr sbv]
options { defaultErrorHandler=false; }
{
    bool val(false);
}
    : val=boolValue { sbv.reset(new SFBool(val)); }
    ;

boolValue returns [bool val = false]
options { defaultErrorHandler=false; }
    : KEYWORD_TRUE { val = true; }
    | KEYWORD_FALSE { val = false; }
    ;

sfColorValue returns [OpenVRML::FieldValuePtr scv]
options { defaultErrorHandler=false; }
{
    float c[3];
}
    : colorValue[c] { scv.reset(new SFColor(c[0], c[1], c[2])); }
    ;

mfColorValue
returns [OpenVRML::FieldValuePtr mcv = OpenVRML::FieldValuePtr(new MFColor)]
options { defaultErrorHandler=false; }
{
    float c[3];
    MFColor & mfcolor = static_cast<MFColor &>(*mcv);
}
    : colorValue[c] {
    	    mfcolor.setLength(1);
            mfcolor.setElement(0, c);
        }
    | LBRACKET (colorValue[c] {
            mfcolor.setLength(mfcolor.getLength() + 1);
            mfcolor.setElement(mfcolor.getLength() - 1, c);
        })* RBRACKET
    ;

colorValue[float c[3]]
options { defaultErrorHandler=false; }
{
    float r(0.0f), g(0.0f), b(0.0f);
}
    : r=colorComponent g=colorComponent b=colorComponent { c[0] = r;
                                                           c[1] = g;
                                                           c[2] = b; }
    ;

//
// Issue a warning if a color component is not in [0, 1].
//
colorComponent returns [float val = 0.0f]
options { defaultErrorHandler=false; }
    : val=floatValue {
            if (val < 0.0 || val > 1.0) {
                this->reportWarning("Color component values must be from 0 to 1.");
            }
        }
    ;

sfFloatValue returns [OpenVRML::FieldValuePtr sfv]
options { defaultErrorHandler=false; }
{
    float f(0.0f);
}
    : f=floatValue { sfv.reset(new SFFloat(f)); }
    ;

mfFloatValue returns [OpenVRML::FieldValuePtr mfv]
options { defaultErrorHandler=false; }
{
    float f(0.0f);
}
    : f=floatValue { mfv.reset(new MFFloat(1, &f)); }
    | LBRACKET { std::vector<float> floatVector; }
        (f=floatValue { floatVector.push_back(f); })* RBRACKET
        { mfv.reset(new MFFloat(floatVector.size(), &floatVector[0])); }
    ;

floatValue returns [float val = 0.0f]
options { defaultErrorHandler=false; }
    : f0:REAL     { val = atof(f0->getText().c_str()); }
    | f1:INTEGER  { val = atof(f1->getText().c_str()); }
    ;

sfImageValue returns [OpenVRML::FieldValuePtr siv]
options { defaultErrorHandler=false; }
{
    unsigned long w(0L), h(0L), com(0L), pixel(0L);
}
    : w=intValue h=intValue com=intValue
        { std::vector<unsigned char> pixelVector; }
        (
            pixel=intValue
            {
                // need to confirm the cross-platform-ness of this, it
                // looks kind of ugly but might in fact be ok. basically,
                // we read the value as an integer, then strip off the
                // bytes one by one.
		for (int i = com - 1; i >= 0; i--) {
                    pixelVector.push_back(static_cast<unsigned char>(pixel >> (8 * i) & 0xff));
                }
            }
        )*
        {
            // if somebody gives us a really, really, really big
            // pixeltexture, then we will crash. in the age of dos
            // attacks, we have to assume that someone will feed us a
	    // too-big texture to see if we barf. good behavior
	    // would be to detect outsized w/h and bail. casting away
	    // the compiler warning is not helpful. there are other
            // bigger bugs to fry, so I guess it's ok for now.
            //
            if (pixelVector.size() != (w * h * com)) {
                throw antlr::SemanticException("Wrong number of pixel values for SFImage.",
                                               this->uri, LT(0)->getLine());
            }
            siv.reset(new SFImage(w, h, com, &pixelVector[0]));
        }
    ;

sfInt32Value returns [OpenVRML::FieldValuePtr siv]
options { defaultErrorHandler=false; }
{
    long i(0L);
}
    : i=intValue { siv.reset(new SFInt32(i)); }
    ;

mfInt32Value returns [OpenVRML::FieldValuePtr miv]
options { defaultErrorHandler=false; }
{
    long i(0L);
}
    : i=intValue { miv.reset(new MFInt32(1, &i)); }
    | LBRACKET { std::vector<long> longVector; }
        (i=intValue { longVector.push_back(i); })* RBRACKET
        { miv.reset(new MFInt32(longVector.size(), &longVector[0])); }
    ;

intValue returns [long val = 0]
options { defaultErrorHandler=false; }
    :   i0:INTEGER      { val = atol(i0->getText().c_str()); }
    |  	i1:HEX_INTEGER  { val = strtol(i1->getText().c_str(), 0, 16); }
    ;

sfNodeValue[OpenVRML::Browser & browser,
            const OpenVRML::ScopePtr & scope]
returns [OpenVRML::FieldValuePtr snv]
{
    OpenVRML::NodePtr n;
}
    : n=nodeStatement[browser, scope] { snv.reset(new SFNode(n)); }
    | KEYWORD_NULL                    { snv.reset(new SFNode); }
    ;

protoSfNodeValue[OpenVRML::ProtoNodeClass & proto,
                 const OpenVRML::ScopePtr & scope]
returns [OpenVRML::FieldValuePtr snv]
{
    OpenVRML::NodePtr n;
}
    : n=protoNodeStatement[proto, scope] { snv.reset(new SFNode(n)); }
    | KEYWORD_NULL                       { snv.reset(new SFNode); }
    ;

mfNodeValue[OpenVRML::Browser & browser,
            const OpenVRML::ScopePtr & scope]
returns [OpenVRML::FieldValuePtr mnv]
    { OpenVRML::NodePtr n; }
    : n=nodeStatement[browser, scope] {
            if (n) { mnv.reset(new MFNode(1, &n)); }
        }
    | LBRACKET { mnv.reset(new MFNode); } (
            n=nodeStatement[browser, scope] {
                if (n) { static_cast<MFNode &>(*mnv).addNode(n); }
            }
        )* RBRACKET
    ;

protoMfNodeValue[OpenVRML::ProtoNodeClass & proto,
                 const OpenVRML::ScopePtr & scope]
returns [OpenVRML::FieldValuePtr mnv]
    { OpenVRML::NodePtr n; }
    : n=protoNodeStatement[proto, scope] { mnv.reset(new MFNode(1, &n)); }
    | LBRACKET { mnv.reset(new MFNode); } (
            n=protoNodeStatement[proto, scope] {
                static_cast<MFNode &>(*mnv).addNode(n);
            }
        )* RBRACKET
    ;

sfRotationValue returns [OpenVRML::FieldValuePtr srv]
    { float r[4]; }
    : rotationValue[r] { srv.reset(new SFRotation(r)); }
    ;

mfRotationValue
returns [OpenVRML::FieldValuePtr mrv = OpenVRML::FieldValuePtr(new MFRotation)]
{
    float r[4];
    MFRotation & mfrotation = static_cast<MFRotation &>(*mrv);
}
    : rotationValue[r] {
            mfrotation.setLength(1);
            mfrotation.setElement(0, r);
        }
    | LBRACKET (rotationValue[r] {
            mfrotation.setLength(mfrotation.getLength() + 1);
            mfrotation.setElement(mfrotation.getLength() - 1, r);
        })* RBRACKET
    ;

//
// Issue a warning here if the vector isn't normalized.
//
rotationValue[float r[4]]
options { defaultErrorHandler=false; }
    {
        using OpenVRML_::length;
        using OpenVRML_::fpequal;
        using OpenVRML_::normalize;
        float x(0.0f), y(0.0f), z(0.0f), rot(0.0f);
    }
    : x=floatValue y=floatValue z=floatValue rot=floatValue {
            r[0] = x;
            r[1] = y;
            r[2] = z;
            r[3] = rot;

            const float axisLength = length(r);
            if (!fpequal(axisLength, 1.0)) {
                this->reportWarning("The axis component of a rotation must be a normalized vector.");
                if (fpequal(axisLength, 0.0)) {
                    r[2] = 1.0;
                } else {
                    normalize(r);
                }
            }
        }
    ;

sfStringValue returns [OpenVRML::FieldValuePtr ssv]
options { defaultErrorHandler=false; }
    { std::string s; }
    : s=stringValue { ssv.reset(new SFString(s)); }
    ;

mfStringValue returns [OpenVRML::FieldValuePtr msv]
options { defaultErrorHandler=false; }
    { std::string s; }
    : s=stringValue { msv.reset(new MFString(1, &s)); }
    | LBRACKET { std::vector<std::string> stringVector; }
        (s=stringValue { stringVector.push_back(s); })* RBRACKET
        { msv.reset(new MFString(stringVector.size(), &stringVector[0])); }
    ;

stringValue returns [std::string str]
options { defaultErrorHandler=false; }
    :   s:STRING
        {
            //
            // Why doesn't this work?
            //
            // str = std::string(s->getText().begin() + 1, s->getText().end() - 1);

            std::string temp(s->getText());
            str = std::string(temp.begin() + 1, temp.end() - 1);
        }
    ;

sfTimeValue returns [OpenVRML::FieldValuePtr stv]
options { defaultErrorHandler=false; }
    { double t(0.0); }
    : t=doubleValue { stv.reset(new SFTime(t)); }
    ;

mfTimeValue returns [OpenVRML::FieldValuePtr mtv]
options { defaultErrorHandler=false; }
    { double t(0.0); }
    : t=doubleValue { mtv.reset(new MFTime(1, &t)); }
    | LBRACKET { std::vector<double> doubleVector; }
        (t=doubleValue { doubleVector.push_back(t); })* RBRACKET
        { mtv.reset(new MFTime(doubleVector.size(), &doubleVector[0])); }
    ;

doubleValue returns [double val = 0.0]
options { defaultErrorHandler=false; }
    :   d0:REAL     { val = atof(d0->getText().c_str()); }
    |   d1:INTEGER  { val = atof(d1->getText().c_str()); }
    ;

sfVec2fValue returns [OpenVRML::FieldValuePtr svv]
options { defaultErrorHandler=false; }
    { float v[2]; }
    : vec2fValue[v] { svv.reset(new SFVec2f(v[0], v[1])); }
    ;

mfVec2fValue
returns [OpenVRML::FieldValuePtr mvv = OpenVRML::FieldValuePtr(new MFVec2f)]
options { defaultErrorHandler=false; }
{
    float v[2];
    MFVec2f & mfvec2f = static_cast<MFVec2f &>(*mvv);
}
    : vec2fValue[v] {
            mfvec2f.setLength(1);
            mfvec2f.setElement(0, v);
        }
    | LBRACKET (vec2fValue[v] {
            mfvec2f.setLength(mfvec2f.getLength() + 1);
            mfvec2f.setElement(mfvec2f.getLength() - 1, v);
        })* RBRACKET
    ;

vec2fValue[float v[2]]
options { defaultErrorHandler=false; }
{
    float x(0.0f), y(0.0f);
}
    : x=floatValue y=floatValue { v[0] = x;
                                  v[1] = y; }
    ;

sfVec3fValue returns [OpenVRML::FieldValuePtr svv]
options { defaultErrorHandler=false; }
{
    float v[3];
}
    : vec3fValue[v] { svv.reset(new SFVec3f(v[0], v[1], v[2])); }
    ;

mfVec3fValue
returns [OpenVRML::FieldValuePtr mvv = OpenVRML::FieldValuePtr(new MFVec3f)]
options { defaultErrorHandler=false; }
{
    float v[3];
    MFVec3f & mfvec3f = static_cast<MFVec3f &>(*mvv);
}
    : vec3fValue[v] {
            mfvec3f.setLength(1);
            mfvec3f.setElement(0, v);
        }
    | LBRACKET (vec3fValue[v] {
            mfvec3f.setLength(mfvec3f.getLength() + 1);
            mfvec3f.setElement(mfvec3f.getLength() - 1, v);
        })* RBRACKET
    ;

vec3fValue[float v[3]]
options { defaultErrorHandler=false; }
{
    float x(0.0f), y(0.0f), z(0.0f);
}
    : x=floatValue y=floatValue z=floatValue { v[0] = x;
                                               v[1] = y;
                                               v[2] = z; }
    ;
