/*
 * Pan - A Newsreader for X
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <string.h>
#include <stdlib.h>

#include <libgnomeui/libgnomeui.h>

#include <pan/base/debug.h>
#include <pan/base/group.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/serverlist.h>

#include <pan/globals.h>
#include <pan/group-ui.h>
#include <pan/util.h>

static void
group_empty_cb (GnomeDialog    * dialog,
		int              button_number,
		gpointer         groups_gp)
{
	GPtrArray * groups = (GPtrArray*) groups_gp;

	g_return_if_fail (groups!=NULL);
	g_return_if_fail (groups->len>0);

	/* process the buttonpresses */
	switch (button_number)
	{
		case 0: /* 'yes' */
		{
			GtkWidget * w = gtk_object_get_data (GTK_OBJECT(dialog), "reset");
			guint i;
			gboolean clear_counts = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(w));
			for (i=0; i<groups->len; ++i) {
				Group * group = GROUP(g_ptr_array_index(groups,i));
				group_empty (group, clear_counts);
			}
			break;
		}
		case 1: /* 'no' */
			break;

		default: /* cancel */
			break;
	};

	/* cleanup */
	g_ptr_array_free (groups, TRUE);
	gtk_widget_destroy (GTK_WIDGET(dialog));
}

void
group_empty_dialog (const GPtrArray * groups)
{
	GString * str = g_string_new (NULL);
	GtkWidget * dialog = NULL;
	GtkWidget * w = NULL;

	/* sanity check */
	g_return_if_fail (groups!=NULL);
	g_return_if_fail (groups->len>0);
	g_return_if_fail (group_is_valid(GROUP(g_ptr_array_index(groups,0))));

	pan_lock();

	/**
	***  Create dialog
	**/

	/* titlebar */
	if (groups->len > 1)
		g_string_sprintf (str, _("Empty %u groups"), groups->len);
	else {
		Group * g = GROUP(g_ptr_array_index(groups,0));
		g_string_sprintf (str, _("Empty group \"%s\""), group_get_readable_name(g));
	}

	dialog = gnome_dialog_new (str->str,
				   GNOME_STOCK_BUTTON_YES,
				   GNOME_STOCK_BUTTON_NO, NULL);

	/* body */
	if (groups->len > 1) {
		g_string_sprintf (str,
			_("Remove all messages from these %u groups?"), groups->len);
	}
	else {
		Group * group = GROUP(g_ptr_array_index(groups,0));
		g_string_sprintf (str,
			_("Remove all messages from \"%s\"?"),
			group_get_readable_name(group));
	}
	w = gtk_label_new (str->str);
	gtk_container_add (GTK_CONTAINER(GNOME_DIALOG(dialog)->vbox), w);

	/* "clear count?" check button */
	w = gtk_check_button_new_with_label (groups->len>1
		? _("Reset \"new message\" counters")
		: _("Reset \"new message\" counter"));
	gtk_container_add (GTK_CONTAINER(GNOME_DIALOG(dialog)->vbox), w);
	gtk_object_set_data (GTK_OBJECT(dialog), "reset", w);

	/* listen for clicks */
	gtk_signal_connect (GTK_OBJECT(dialog), "clicked",
			    GTK_SIGNAL_FUNC (group_empty_cb), pan_g_ptr_array_dup(groups));

        gnome_dialog_set_parent (GNOME_DIALOG(dialog), GTK_WINDOW(Pan.window));
	gtk_widget_show_all (dialog);
	pan_unlock();

	g_string_free (str, TRUE);
}

/****
*****
*****
****/

static void
new_folder_cb (gchar* string, gpointer data)
{
	Group * folder = NULL;
	Server * server = serverlist_get_named_server (INTERNAL_SERVER_NAME);
	gchar * name;
	debug_enter ("new_folder_cb");

	/* user hit cancel */
	if (string == NULL)
		return;

	/* make sure the user entered something */
	name = g_strdup (string);
	g_strstrip (name);
	if (!is_nonempty_string (name)) {
		pan_error_dialog (_("Invalid folder name: \"%s\""), string);
		g_free (name);
		return;
	}

	/* make sure there's not already a folder with that name */
	g_strdown (name);
	folder = serverlist_get_named_folder (name);
	if (folder != NULL) {
		pan_error_dialog (_("Folder \"%s\" already exists."), string);
		g_free (name);
		return;
	}

	/* make sure it's not in the reserved namespace */
	if (!strncmp(name, "pan.", 4)) {
		pan_error_dialog (_("Please don't begin folders with \"pan.\"; it confuses me terribly."));
		g_free (name);
		return;
	}

	/* create new folder */
	folder = group_new (server, name);
	group_set_is_folder (folder, TRUE);
	group_set_dirty (folder);
	server_add_groups (server, &folder, 1, NULL, NULL);

	/* cleanup */
	g_free (name);
	debug_exit ("new_folder_cb");
}

void
group_new_folder_dialog (void)
{
	gchar * prompt = g_strdup (_("New Folder Name:"));
	gnome_request_dialog (FALSE, prompt, NULL, 128,
			      new_folder_cb,
			      NULL, GTK_WINDOW(Pan.window));
	g_free (prompt);
}
