/*
 * Pan - A Newsreader for X
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <string.h>

#include <libgnomeui/libgnomeui.h>

#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>

#include <pan/identities/identity.h>
#include <pan/identities/identity-edit-ui.h>

#include <pan/util.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

typedef struct
{
	GtkWidget * dialog;

	GtkWidget * name_entry;
	GtkWidget * default_news_tb;
	GtkWidget * default_mail_tb;

	GtkWidget * author_real_entry;
	GtkWidget * author_addr_entry;

	GtkWidget * reply_to_entry;
	GtkWidget * organization_entry;
	GtkWidget * signature_entry;
	GtkWidget * attribution_entry;
	GtkWidget * msg_id_fqdn_entry;

	GtkWidget * custom_headers_text;

	gchar     * identity_name;
}
IdentityEditDialog;

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

extern GtkTooltips * ttips;

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PRIVATE
***********/

/*****
******
******  IDENTITY INFO
******
*****/

static GtkWidget*
create_identity_tab (IdentityEditDialog *d)
{
	GtkWidget * t;
	GtkWidget * f;
	GtkWidget * w;
	GtkWidget * table;
	gint        row;

	t = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER(t), GNOME_PAD_SMALL);

	f = gtk_frame_new (_("Profile Information"));
	gtk_container_add (GTK_CONTAINER(t), f);

	table = gtk_table_new (2, 3, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER(f), table);

	row = 0;
	w = gtk_label_new (_("Profile Name"));
	gtk_misc_set_alignment (GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, row, row+1, 
		GTK_FILL, GTK_FILL, 0, 0);

	d->name_entry = w = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE (table), w, 1, 2, row, row+1, 
		GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);

	row++;
	d->default_news_tb = w = gtk_check_button_new_with_label (
		_("Make this the default profile for news"));
	gtk_table_attach (GTK_TABLE (table), w, 0, 2, row, row+1, 
		GTK_FILL, GTK_FILL, 0, 0);

	row++;
	d->default_mail_tb = w = gtk_check_button_new_with_label (
		_("Make this the default profile for mail"));
	gtk_table_attach (GTK_TABLE (table), w, 0, 2, row, row+1, 
		GTK_FILL, GTK_FILL, 0, 0);


	f = gtk_frame_new (_("Required Information"));
	gtk_container_add (GTK_CONTAINER(t), f);

	table = gtk_table_new (2, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER(f), table);

	row = 0;
	w = gtk_label_new (_("Full User Name:"));
	gtk_misc_set_alignment (GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, row, row+1, 
		GTK_FILL, GTK_FILL, 0, 0);

	w = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE (table), w, 1, 2, row, row+1,
		GTK_FILL|GTK_EXPAND, GTK_FILL, 4, 0);
	d->author_real_entry = w;

	row++;
	w = gtk_label_new (_("Email Address:"));
	gtk_misc_set_alignment (GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, row, row+1,
		GTK_FILL, GTK_FILL, 4, 0);

	d->author_addr_entry = w = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(table), w, 1, 2, row, row+1,
		GTK_FILL|GTK_EXPAND, GTK_FILL, 4, 0);


	f = gtk_frame_new (_("Optional Information"));
	gtk_container_add (GTK_CONTAINER(t), f);

	table = gtk_table_new (4, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER(f), table);

	row = 0;
	w = gtk_label_new (_("Reply To:"));
	gtk_misc_set_alignment (GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, row, row+1,
		GTK_FILL, GTK_FILL, 0, 0);

	d->reply_to_entry = w = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE (table), w, 1, 2, row, row+1,
		GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);

	row++;
	w = gtk_label_new (_("Message ID FQDN:"));
	gtk_misc_set_alignment (GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, row, row+1,
		GTK_FILL, GTK_FILL, 0, 0);

	d->msg_id_fqdn_entry = w = gtk_entry_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, 
		"The domain name to use for MessageID generation", "");
	gtk_table_attach (GTK_TABLE (table), w, 1, 2, row, row+1,
		GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);

	row++;
	w = gtk_label_new (_("Organization:"));
	gtk_misc_set_alignment (GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, row, row+1,
		GTK_FILL, GTK_FILL, 0, 0);

        d->organization_entry = w = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE (table), w, 1, 2, row, row+1,
		GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);

	row++;
	w = gtk_label_new (_("Attribution:"));
	gtk_misc_set_alignment (GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, row, row+1,
		GTK_FILL, GTK_FILL, 0, 0);

        d->attribution_entry = w = gtk_entry_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, 
		"%i for Message-ID\n"
		"%a for Author and Address\n"
		"%n for Author name\n%d for Date", "");
	gtk_table_attach (GTK_TABLE (table), w, 1, 2, row, row+1,
		GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);

	row++;
	w = gtk_label_new (_("Signature File:"));
	gtk_misc_set_alignment (GTK_MISC (w), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, row, row+1,
		GTK_FILL, GTK_FILL, 0, 0);

        d->signature_entry = w = gnome_file_entry_new ("Profile", 
		_("Signature File:"));
	gtk_table_attach (GTK_TABLE (table), w, 1, 2, row, row+1,
		GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);

	return t;
}

static void
populate_identity_tab (IdentityEditDialog * d,
                       const Identity     * id,
                       gboolean             default_news,
                       gboolean             default_mail)
{
	gboolean    b;
	GtkWidget * w;

	b = id!=NULL;

	/* Identity name */
	w = d->name_entry;
	gtk_entry_set_text (GTK_ENTRY (w), b ? id->name : "");

	/* default news */
	w = d->default_news_tb;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), default_news);

	/* default mail */
	w = d->default_mail_tb;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), default_mail);

	/* author name */
	w = d->author_real_entry;
	gtk_entry_set_text (GTK_ENTRY (w), 
		b && is_nonempty_string (id->author_real) ? 
		id->author_real : "");

	/* author address */
	w = d->author_addr_entry;
	gtk_entry_set_text (GTK_ENTRY (w), 
		b && is_nonempty_string (id->author_addr) ?
		id->author_addr : "");

	/* reply to */
	w = d->reply_to_entry;
	gtk_entry_set_text (GTK_ENTRY (w), 
		 b && is_nonempty_string (id->reply_to) ?
		id->reply_to : "");

	/* msg id fqdn */
	w = d->msg_id_fqdn_entry;
	gtk_entry_set_text (GTK_ENTRY (w), 
		 b && is_nonempty_string (id->msg_id_fqdn) ?
		id->msg_id_fqdn : "");

	/* organization */
	w = d->organization_entry;
	gtk_entry_set_text (GTK_ENTRY (w), 
		b && is_nonempty_string (id->organization) ?
		id->organization : "");

	/* attribution */
	w = d->attribution_entry;
	gtk_entry_set_text (GTK_ENTRY (w), 
		b && is_nonempty_string (id->attribution) ?
		id->attribution : _("On %d, %n wrote:"));

	/* signature file */
	w = d->signature_entry;
	w = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (w));
	gtk_entry_set_text (GTK_ENTRY (w), 
		b && is_nonempty_string (id->signature) ?
		id->signature : "");
}

static void
identity_from_identity_tab (IdentityEditDialog * d, 
                            Identity           * id,
                            gboolean           * default_news,
                            gboolean           * default_mail)
{
	gchar     * pch;
	GtkWidget * w;

	g_return_if_fail (d!=NULL);
	g_return_if_fail (id!=NULL);
	g_return_if_fail (default_news!=NULL);
	g_return_if_fail (default_mail!=NULL);

	/* Identity name */
	w = d->name_entry;
	pch = gtk_editable_get_chars (GTK_EDITABLE (w), 0, -1);
	replace_gstr (&id->name, pch);
	g_strstrip (id->name);

	/* default news */
	w = d->default_news_tb;
	*default_news = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));

	/* default mail */
	w = d->default_mail_tb;
	*default_mail = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));

	/* author name */
	w = d->author_real_entry;
	pch = gtk_editable_get_chars (GTK_EDITABLE (w), 0, -1);
	replace_gstr (&id->author_real, pch);
	g_strstrip (id->author_real);

	/* author address */
	w = d->author_addr_entry;
	pch = gtk_editable_get_chars (GTK_EDITABLE (w), 0, -1);
	replace_gstr (&id->author_addr, pch);
	g_strstrip (id->author_addr);

	/* reply to */
	w = d->reply_to_entry;
	pch = gtk_editable_get_chars (GTK_EDITABLE (w), 0, -1);
	replace_gstr (&id->reply_to, pch);
	g_strstrip (id->reply_to);

	/* msg id fqdn */
	w = d->msg_id_fqdn_entry;
	pch = gtk_editable_get_chars (GTK_EDITABLE (w), 0, -1);
	replace_gstr (&id->msg_id_fqdn, pch);
	g_strstrip (id->msg_id_fqdn);

	/* organization */
	w = d->organization_entry;
	pch = gtk_editable_get_chars (GTK_EDITABLE (w), 0, -1);
	replace_gstr (&id->organization, pch);
	g_strstrip (id->organization);
	
	/* attribution */
	w = d->attribution_entry;
	pch = gtk_editable_get_chars (GTK_EDITABLE (w), 0, -1);
	replace_gstr (&id->attribution, pch);
	g_strstrip (id->attribution);

	/* signature file */
	w = d->signature_entry;
	w = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (w));
	pch = gtk_editable_get_chars (GTK_EDITABLE (w), 0, -1);
	replace_gstr (&id->signature, pch);
	g_strstrip (id->signature);
}

/*****
******
******  CUSTOM HEADERS
******
*****/

static GtkWidget*
create_headers_tab (IdentityEditDialog *d)
{
	GtkWidget * frame;
	GtkWidget * w;
	GtkWidget * scroll;

	g_return_val_if_fail (d!=NULL, NULL);

	w = gtk_text_new (NULL, NULL);
	gtk_text_set_word_wrap (GTK_TEXT (w), FALSE);
	gtk_text_set_line_wrap (GTK_TEXT (w), FALSE);
	gtk_text_set_editable (GTK_TEXT (w), TRUE);
	d->custom_headers_text = w;

	scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_container_set_border_width (GTK_CONTAINER(scroll), GNOME_PAD_SMALL);	gtk_container_add (GTK_CONTAINER(scroll), w);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
	                                GTK_POLICY_AUTOMATIC,
	                                GTK_POLICY_AUTOMATIC);
	
	frame = gtk_frame_new (_("Custom Headers"));
	gtk_container_set_border_width (GTK_CONTAINER(frame), GNOME_PAD);
	gtk_container_add (GTK_CONTAINER(frame), scroll);

	return frame;
}

static void
populate_headers_tab (IdentityEditDialog *d, const Identity *id)
{
	g_return_if_fail (d!=NULL);

	if (id!=NULL && id->custom_headers)
	{
		GString * headers = g_string_new (NULL);
		guint     i;
		gint      pos = 0;

		for (i=0; i<id->custom_headers->len; i++)
		{
			Header * h = g_ptr_array_index (id->custom_headers, i);

			g_string_sprintfa (headers, "%s: %s\n", 
				h->name, h->value);
		}

		gtk_editable_delete_text (GTK_EDITABLE (d->custom_headers_text),
		                          0, -1);
		gtk_editable_insert_text (GTK_EDITABLE (d->custom_headers_text),
		                       headers->str, headers->len, &pos);

		g_string_free (headers, TRUE);
	}
	
}

static void
identity_from_headers_tab (IdentityEditDialog *d, Identity *id)
{
	gchar  * text;
	gint     i;
	gchar ** lines;

	g_return_if_fail (d!=NULL);
	g_return_if_fail (id!=NULL);
	g_return_if_fail (id->custom_headers!=NULL);

	/* Clear all current custom headers */
	for (i=0; i<id->custom_headers->len; i++)
	{
		Header * h = g_ptr_array_index (id->custom_headers, i);

		g_free (h->name);
		g_free (h->value);
		g_free (h);
		g_ptr_array_remove (id->custom_headers, h);
	}

	text = gtk_editable_get_chars (GTK_EDITABLE (d->custom_headers_text),
	                               0, -1);
	lines = g_strsplit (text, "\n", -1);
	for (i=0; lines!=NULL && lines[i]!=NULL; i++)
	{
		gchar * delimit;

		g_strstrip (lines[i]);
		delimit = strchr (lines[i], ':');

		if (delimit != NULL)
		{
			Header * h = g_new0 (Header, 1);
			gchar  * n = g_strndup (lines[i], delimit - lines[i]);
			gchar  * v = g_strdup (delimit + 1);

			g_strstrip (n);
			g_strstrip (v);

			h->name = n;
			h->value = v;

			g_ptr_array_add (id->custom_headers, h);
		}
	}
	g_strfreev (lines);
}

/*****
******
******  
******
*****/

static gint
dialog_close_cb (GnomeDialog * dialog, gpointer data)
{
	IdentityEditDialog * d = (IdentityEditDialog*) data;

	g_free (d->identity_name);
	g_free (d);
	return FALSE;
}
 
GtkWidget*
identity_edit_dialog_new (const Identity * id,
                          gboolean         default_news,
                          gboolean         default_mail)
{
	GtkWidget *w, *n, *d_v;
	IdentityEditDialog * d;

	d = g_new0 (IdentityEditDialog, 1);

	/* dialog */

	w = d->dialog = gnome_dialog_new (_("Pan: Edit Profile"),
		GNOME_STOCK_BUTTON_OK,
		GNOME_STOCK_BUTTON_CANCEL, NULL);
	d_v = GNOME_DIALOG(w)->vbox;
	gtk_window_set_policy (GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_signal_connect (GTK_OBJECT(w), "close", GTK_SIGNAL_FUNC(dialog_close_cb), d);
	gtk_object_set_user_data (GTK_OBJECT(w), d);

	/* notebook */
	n = gtk_notebook_new ();
	w = create_identity_tab (d);
	gtk_container_add (GTK_CONTAINER(n), w);
	gtk_notebook_set_tab_label (GTK_NOTEBOOK(n), w, gtk_label_new(_("User Profile")));
	w = create_headers_tab (d);
	gtk_container_add (GTK_CONTAINER(n), w);
	gtk_notebook_set_tab_label (GTK_NOTEBOOK(n), w, gtk_label_new(_("Custom Headers")));
	gtk_box_pack_start (GTK_BOX(d_v), n, TRUE, TRUE, GNOME_PAD_SMALL);

	/* populate */
	populate_identity_tab (d, id, default_news, default_mail);
	populate_headers_tab (d, id);

	return d->dialog;
}

Identity*
identity_edit_dialog_get_identity (GtkWidget * w, 
                                   gboolean  * default_news,
                                   gboolean  * default_mail)
{
	Identity           * id = NULL;
	IdentityEditDialog * d  = 
		(IdentityEditDialog*) gtk_object_get_user_data (GTK_OBJECT(w));
	
	id = identity_new ();
	identity_from_identity_tab (d, id, default_news, default_mail);
	identity_from_headers_tab (d, id);

	/* make sure user didn't skip on any of the essential parts */
	if (!identity_is_valid (id)) {
		pan_object_unref (PAN_OBJECT(id));
		id = NULL;
	}

	return id;
}
