/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Pan - A Newsreader for X
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <glib.h>

#include <gnome-xml/parser.h>
#include <gnome-xml/xmlmemory.h>

#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/util-file.h>

#include <pan/rules/rule-xml.h>

/************
*************  UTIL
************/

static void
newline_depth (GString * appendme, gint depth)
{
	gint i;
	g_string_append_c (appendme, '\n');
	for (i=0; i<depth; ++i)
		g_string_append_c (appendme, '\t');
}

static void
pan_g_string_append_escaped (GString * gstr, gchar * escapeme)
{
	gchar * pch = pan_str_escape (escapeme);
	g_string_append (gstr, pch);
	g_free (pch);
}


/*****
******  
******    WRITING
******  
*****/

static void
write_rule (GString * gstr, const Rule * r, gint depth)
{
	gint i;

	g_return_if_fail (gstr!=NULL);
	g_return_if_fail (r!=NULL);

	/* the rule itself */
	newline_depth (gstr, depth);
	g_string_append (gstr, "<rule name=\"");
	pan_g_string_append_escaped (gstr, r->name);
	g_string_append (gstr, "\" filter_name=\"");
	pan_g_string_append_escaped (gstr, r->filter_name);
	g_string_sprintfa (gstr, "\" hits=\"%lu\" misses=\"%lu\"", 
		(gulong)r->hits, (gulong)(r->tries - r->hits));
	if (r->apply_to_incoming)
		g_string_append (gstr, " apply_to_incoming=\"t\"");

	switch (r->group_type)
	{
		case RULE_GROUP_WILDCARD:
			g_string_append (gstr, " group_type=\"w\"");
			g_string_sprintfa (gstr, " group_wildcard=\"%s\"",
				r->group_wildcard);
			break;
		case RULE_GROUP_LIST:
			if (r->group_list == NULL &&
			    r->group_list->len == 0)
				break;
			g_string_append (gstr, " group_type=\"l\"");
			g_string_append (gstr, " group_list=\"");
			for (i=0; i<r->group_list->len; i++)
			{
				if (i>0) g_string_append (gstr, ";");
				g_string_append (gstr, (gchar *)
					g_ptr_array_index(r->group_list, i));
			}
			g_string_append (gstr, "\"");
	}
	g_string_append (gstr, ">");

	/* sound */
	if ((r->action->flags & RULE_ACTION_PLAY_SOUND) != 0) {
		newline_depth (gstr, depth+1);
		g_string_append (gstr, "<sound filename=\"");
		pan_g_string_append_escaped (gstr, r->action->sound_file);
		g_string_append (gstr, "\"/>");
	}

	/* alert */
	if ((r->action->flags & RULE_ACTION_SHOW_ALERT) != 0) {
		newline_depth (gstr, depth+1);
		g_string_append (gstr, "<alert message=\"");
		pan_g_string_append_escaped (gstr, r->action->alert_message);
		g_string_append (gstr, "\"/>");
	}

	/* decode */
	if ((r->action->flags & RULE_ACTION_DECODE) != 0) {
		newline_depth (gstr, depth+1);
		g_string_append (gstr, "<decode directory=\"");
		pan_g_string_append_escaped (gstr, r->action->decode_path);
		g_string_append (gstr, "\"/>");
	}

	/* read */
	if ((r->action->flags & RULE_ACTION_MARK_READ) != 0) {
		newline_depth (gstr, depth+1);
		g_string_append (gstr, "<read/>");
	}

	/* unread */
	if ((r->action->flags & RULE_ACTION_MARK_UNREAD) != 0) {
		newline_depth (gstr, depth+1);
		g_string_append (gstr, "<unread/>");
	}

	/* flag */
	if ((r->action->flags & RULE_ACTION_TAG_FOR_RETRIEVAL) != 0) {
		newline_depth (gstr, depth+1);
		g_string_append (gstr, "<flag/>");
	}

	/* flag */
	if ((r->action->flags & RULE_ACTION_RETREIVE_BODY) != 0) {
		newline_depth (gstr, depth+1);
		g_string_append (gstr, "<fetch_body/>");
	}

	/* watch */
	if ((r->action->flags & RULE_ACTION_WATCH) != 0) {
		newline_depth (gstr, depth+1);
		g_string_append (gstr, "<watch/>");
	}

	/* ignore */
	if ((r->action->flags & RULE_ACTION_IGNORE) != 0) {
		newline_depth (gstr, depth+1);
		g_string_append (gstr, "<ignore/>");
	}

	/* discard */
	if ((r->action->flags & RULE_ACTION_DISCARD) != 0) {
		newline_depth (gstr, depth+1);
		g_string_append (gstr, "<discard/>");
	}

	newline_depth (gstr, depth);
	g_string_append (gstr, "</rule>");
}


gchar*
rule_xml_to_string  (Rule  ** rules,
                     gint     rule_qty)
{
	gint i;
	GString * str;
	gchar * retval;

	g_return_val_if_fail (rule_qty>=0, NULL);

	/* write the rules to a buf */
	str = g_string_sized_new (4096);
	for (i=0; i<rule_qty; ++i) {
		write_rule (str, rules[i], 1);
		if (i!=rule_qty)
			g_string_append (str, "\n\n");
	}

	g_string_prepend (str, "<?xml version=\"1.0\" ?>\n"
	                       "<!DOCTYPE rules SYSTEM \"rules.dtd\">\n"
	                       "<rules>\n");
	g_string_append (str, "\n</rules>");

	/* return the string */
	retval = str->str;
	g_string_free (str, FALSE);
	return retval;
}

static gboolean
rule_xml_fwrite (FILE        * fp,
                 Rule       ** rules,
                 gint          rule_qty)
{
	gchar  * pch;
	size_t   to_write, written;

	g_return_val_if_fail (fp!=NULL, 0);
	g_return_val_if_fail (rules!=NULL, 0);
	g_return_val_if_fail (rule_qty>0, 0);

	pch = rule_xml_to_string ((Rule**)rules, rule_qty);
	to_write = strlen (pch);
	written = fwrite (pch, sizeof(char), to_write, fp);
	g_free (pch);

	return to_write = written;
}

void
rule_xml_write (const gchar * filename,
                Rule       ** rules,
                gint          rule_qty)
{
	/* sanity clause */
	g_return_if_fail (is_nonempty_string(filename));
	g_return_if_fail (rule_qty>=0);

	/* write the file */

	if (rule_qty == 0) {
		unlink (filename);
	}
	else {
		FILE     * fp;
		gchar    * tmp_path = pan_make_temp (&fp);
		gboolean ok = fp ? TRUE : FALSE;

		if (ok) {
			ok = rule_xml_fwrite (fp, rules, rule_qty);
			fclose (fp);
		}
		
		if (ok) {
			ok = pan_rename (tmp_path, filename);
		}

		if (ok) {
			log_add_va (LOG_INFO, _("Wrote rules to `%s'"), 
				filename);
		}
		else {
			log_add_va (LOG_ERROR,
				_("Can't write rules to `%s': %s"),
				filename, g_strerror(errno));
		}

		g_free (tmp_path);
	}
}









/*****
******  
******  
******    READING
******  
******  
*****/

static void
get_attributes (xmlDoc             * doc,
                xmlNode            * node,
                GHashTable        ** newme_hash,
                GStringChunk      ** newme_chunk)
{
	const xmlAttr * attr = NULL;
	GHashTable * hash = g_hash_table_new (g_str_hash, g_str_equal);
	GStringChunk * strchunk = g_string_chunk_new (128);

	for (attr=node->properties; attr!=NULL; attr=attr->next) {
		const gchar * name = (const gchar*) attr->name;
		xmlChar * value = xmlNodeListGetString (doc, attr->val, FALSE);
		gchar * clear_value = pan_str_unescape ((const gchar*) value);
		if (is_nonempty_string(name) && clear_value!=NULL)
			g_hash_table_insert (hash,
			                     g_string_chunk_insert(strchunk,name),
			                     g_string_chunk_insert(strchunk,clear_value));
		g_free (clear_value);
		xmlFree (value);
	}

	*newme_hash = hash;
	*newme_chunk = strchunk;

}

static Rule*
create_rule (xmlDoc * doc, xmlNode * node)
{
	Rule * r = NULL;
	const gchar * tag = (const gchar*) node->name;

	xmlNode * child;
	const gchar * cpch;
	GHashTable * attr = NULL;
	GStringChunk * strchunk = NULL;

	if (pan_strcmp(tag,"rule")!=0)
		return NULL;

	/* init toplevel */
	r = rule_new ();
	get_attributes (doc, node, &attr, &strchunk);

	/* populate toplevel */
	cpch = (const gchar *) g_hash_table_lookup (attr, "name");
	if (cpch != NULL)
		replace_gstr (&r->name, g_strdup(cpch));

	cpch = (const gchar *) g_hash_table_lookup (attr, "filter_name");
	if (cpch != NULL)
		replace_gstr (&r->filter_name, g_strdup(cpch));

	cpch = (const gchar *) g_hash_table_lookup (attr, "hits");
	if (cpch != NULL) {
		r->hits = atoi (cpch);
		r->tries += r->hits;
	}

	cpch = (const gchar *) g_hash_table_lookup (attr, "misses");
	if (cpch != NULL)
		r->tries += atoi (cpch);

	cpch = (const gchar *) g_hash_table_lookup (attr, "apply_to_incoming");
	r->apply_to_incoming = cpch!=NULL && tolower((int)*cpch)=='t';

	cpch = (const gchar *) g_hash_table_lookup (attr, "group_type");
	if (cpch != NULL)
	{
		if (*cpch == 'w')
		{
			r->group_type = RULE_GROUP_WILDCARD;
			cpch = (const gchar *) g_hash_table_lookup (attr, "group_wildcard");
			if (cpch != NULL)
			{
				r->group_wildcard = g_strdup(cpch);
			}
		}
		else
		if (*cpch == 'l')
		{
			gchar * str;

			r->group_type = RULE_GROUP_LIST;
			cpch = (const gchar *) g_hash_table_lookup (attr, "group_list");
			r->group_list = g_ptr_array_new();
			while ((str = get_next_token_str (cpch, ';', &cpch)) != NULL)
			{
				g_ptr_array_add(r->group_list, str);
			}
		}
	}
		
	/* cleanup toplevel */
	g_hash_table_destroy (attr);
	g_string_chunk_free (strchunk);

	/* walk through children */
	for (child=node->xmlChildrenNode; child!=NULL; child=child->next)
	{
		const gchar * tag = (const gchar*) child->name;
		get_attributes (doc, child, &attr, &strchunk);

		if (!pan_strcmp (tag, "sound"))
		{
			r->action->flags |= RULE_ACTION_PLAY_SOUND;
			cpch = (const gchar *) g_hash_table_lookup (attr, "filename");
			replace_gstr (&r->action->sound_file, g_strdup(cpch));
		}
		else if (!pan_strcmp (tag, "alert"))
		{
			r->action->flags |= RULE_ACTION_SHOW_ALERT;
			cpch = (const gchar *) g_hash_table_lookup (attr, "message");
			replace_gstr (&r->action->alert_message, g_strdup(cpch));
		}
		else if (!pan_strcmp (tag, "decode"))
		{
			r->action->flags |= RULE_ACTION_DECODE;
			cpch = (const gchar *) g_hash_table_lookup (attr, "directory");
			replace_gstr (&r->action->decode_path, g_strdup(cpch));
		}
		else if (!pan_strcmp (tag, "read"))
		{
			r->action->flags |= RULE_ACTION_MARK_READ;
		}
		else if (!pan_strcmp (tag, "unread"))
		{
			r->action->flags |= RULE_ACTION_MARK_UNREAD;
		}
		else if (!pan_strcmp (tag, "flag"))
		{
			r->action->flags |= RULE_ACTION_TAG_FOR_RETRIEVAL;
		}
		else if (!pan_strcmp (tag, "fetch_body"))
		{
			r->action->flags |= RULE_ACTION_RETREIVE_BODY;
		}
		else if (!pan_strcmp (tag, "watch"))
		{
			r->action->flags |= RULE_ACTION_WATCH;
		}
		else if (!pan_strcmp (tag, "ignore"))
		{
			r->action->flags |= RULE_ACTION_IGNORE;
		}
		else if (!pan_strcmp (tag, "discard"))
		{
			r->action->flags |= RULE_ACTION_DISCARD;
		}

		/* cleanup toplevel */
		g_hash_table_destroy (attr);
		g_string_chunk_free (strchunk);
	}

	return r;
}

 
void
rule_xml_read (const gchar * filename,
               GPtrArray   * appendme_rules)
{
	xmlDocPtr doc;
	xmlNodePtr cur;

	/* sanity clause */
	g_return_if_fail (is_nonempty_string(filename));
	g_return_if_fail (appendme_rules!=NULL);
	if (!file_exists(filename)) return;

	doc = xmlParseFile (filename);
	if (doc==NULL) return;

	cur = xmlDocGetRootElement(doc);
	cur = cur->xmlChildrenNode;

	for (; cur!=NULL; cur=cur->next)
	{
		Rule * r = create_rule (doc, cur);
		if (r != NULL)
			g_ptr_array_add (appendme_rules, r);
	}

	xmlFreeDoc (doc);
}
