///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#ifndef BOUNDARY_H		// -*- c++ -*-
#define BOUNDARY_H
// A boundary is a limited are, rectangular or something else.  It might be
// used as an obstacle or as a clipping path, etc.  Borders might be drawn
// along a boundary (but not /by/ it).
#include <vector>
#include "matrix.h"

class Boundary {
public:
  virtual std::ostream& print(std::ostream&) const = 0;
  virtual std::ostream& print(std::ostream&, const Matrix& m) const = 0;

  virtual bool operator == (const Boundary& other) const = 0;
  bool operator != (const Boundary& other) {
    return !(*this == other);
  }
};

inline std::ostream& operator << (std::ostream& out, const Boundary& b) {
  return b.print(out);
}

typedef std::vector<Boundary*> BoundaryVect;

// A RectBoundary is a unit square boundary, transformed by an arbitrary
// transform matrix.
class RectBoundary : public Boundary {
public:
  RectBoundary(const Matrix& m,
	       const double width = 1, const double height = 1);
  std::ostream& print(std::ostream&) const;
  std::ostream& print(std::ostream&, const Matrix& m) const;

  bool operator == (const Boundary& other) const;
  
private:
  Matrix matrix;
  
  // Undefined ctors, avoid defaults
  RectBoundary(const RectBoundary&);
  RectBoundary();
  void operator = (const RectBoundary&);
};

// A XformedBoundary is any kind of boundary, transformed by an arbitrary
// transform matrix.
struct XformedBoundary {
  XformedBoundary(const Matrix& m, const Boundary& b)
    : bound(b), transform(m) {}
  const Boundary& bound;
  const Matrix& transform;
};
inline XformedBoundary operator * (const Matrix& m, const Boundary& b) {
  return XformedBoundary(m, b);
}
inline std::ostream& operator << (std::ostream& out, const XformedBoundary& b)
{
  return b.bound.print(out, b.transform);
}

#endif
