///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "cachedframe.h"
#include "fileerrors.h"
#include "globals.h"
#include "processman.h"
#include <gdk--.h>
#include <sigc++/signal_system.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <util/warning.h>

void Cached_Frame::clear_picture()
{
  if(picture)
    picture.release();
}

bool Cached_Frame::draw_content(View& view) 
{
  if(no_picture_generated)
    {
      no_picture_generated=false; // or it will never be generated again
      draw_content_broken(view);
      return true;
    }
  if(picture) {
    // p is the top left corner, since that is what draw_pixmap wants.
    const Gdk_Point p = view.pt2scr(get_matrix().transform(Vector(0, height)));
    if(mask)
      {
	Gdk_GC gc=view.get_gc();
	gc.set_clip_mask(mask);
	gc.set_clip_origin(p->x, p->y);
	view.get_win().draw_pixmap(gc, picture, 0, 0, p->x, p->y);
	// Actually, if the picture is just black on white, 
	// a draw_rect would do.
	gc.set_clip_mask();
	gc.set_clip_origin(0, 0);
      }
    else
      view.get_win().draw_pixmap(view.get_gc(), picture, 0, 0, p->x, p->y);
    return true;

  } else {
    if(!is_reshaping())		// generate_picture takes too long time
      {
	try
	  {
	    generate_picture(view);
	    draw_content_wait(view);
	  }
	catch(Gen_Pic_Error e)
	  {
	    switch(e.type)
	      {
	      case ASSOCIATION:
		draw_content_missing(view);
		break;
	      case GENERATION:
		draw_content_broken(view);
		break;
	      case ZEROSIZE:
		break;
	      }
	  }
      } 
    else // if reshaping
      draw_content_wait(view);
    return false;
  }
}

void Cached_Frame::end_reshape(bool revert)
{
  if(!revert)
    clear_picture();
  Pagent::end_reshape(revert);
  ready_to_draw_signal(this);
}

namespace {
  void do_clear_picture(Pagent* frame) {
    if(Cached_Frame *cf = dynamic_cast<Cached_Frame*>(frame))
      cf->clear_picture();
  }
}

Cached_Frame::Cached_Frame(Group *parent, 
			   FLength w, FLength h, const std::string& name)
  :Basic_Frame(parent, w, h, name), no_picture_generated(false),
   white_is_transparent(false),
   picture(0), mask(0), pid(-1)
{
  proc_stop_connection = process_manager.process_stopped
    .connect(SigC::slot(this, &Cached_Frame::process_stopped));
  tmp_file=process_manager.find_new_name();
  
  geometry_changed_signal.connect(slot(do_clear_picture));
}

Cached_Frame::Cached_Frame(Group* parent, const XMLNode& node)
  :Basic_Frame(parent, node), no_picture_generated(false),
   white_is_transparent(false),
   picture(0), mask(0), pid(-1)
{
  proc_stop_connection = process_manager.process_stopped
    .connect(SigC::slot(this, &Cached_Frame::process_stopped));
  tmp_file=process_manager.find_new_name();
}

Cached_Frame::~Cached_Frame()
{
  proc_stop_connection.disconnect();
  if(!process_manager.delete_file(tmp_file))
    warning << "Failed to delete " << tmp_file << std::endl;
}

void Cached_Frame::act_on_zoom_factor_change()
{
  clear_picture();
  if(pid!=-1)
    {
      process_manager.stop(pid);
      pid=-1;
    }
}

void Cached_Frame::process_stopped(pid_t pid, bool exited_normally, 
				   int exit_code)
{
  if(pid!=this->pid)
    return;
  this->pid=-1;
  debug << pid << ", exit code: " << exit_code << std::endl;
  if(!exited_normally)
    {
      warning << "Process " << this->pid << " exited abnormally" << std::endl;
      return;
    }
  clear_picture();
  GdkPixbuf *buf = gdk_pixbuf_new_from_file(tmp_file.c_str());
  GdkPixmap *topix;
  if(buf) //file may not exist
    {
      if(white_is_transparent)
	{
	  GdkBitmap *mask;
	  GdkPixbuf *buf_alpha=gdk_pixbuf_add_alpha(buf, true, 255, 255, 255);
	  gdk_pixbuf_render_pixmap_and_mask(buf_alpha, &topix, &mask, 100);
	  this->mask=Gdk_Bitmap(mask);
	  gdk_bitmap_unref(mask);
	  gdk_pixbuf_unref(buf_alpha);
	}
      else
	gdk_pixbuf_render_pixmap_and_mask(buf, &topix, 0, 0);
      gdk_pixbuf_unref(buf);
      picture = Gdk_Pixmap(topix);
      gdk_pixmap_unref(topix);
    }
  no_picture_generated=!picture;
  ready_to_draw_signal(this);
}
