// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
#include "config.h"
#include <fstream>
#include <iostream>

#include <util/warning.h>
#include <defines.h>

#include "globals.h"

#define _Str(x) #x
#define Str(x) _Str(x)

Config::Config(const std::string strFileName, const char *szPath )
  : strSETTINGS_FILE(strFileName), paper_name("A4"),
  fX(20.9903), fY(29.7039),
  reshapeboxsize(7), orientation(Papers::PORTRAIT),
  strPrintCommand("lpr"), unit(Units::PT)
{
  // Try to get the $HOME location
  char* descr = getenv("HOME");
  if (descr && !szPath)
  {
    strPath = descr;
    strPath += "/";
  }
  else
  {
    if(szPath)
      strPath = szPath;
    else
      strPath = "h:/";  // <<<<<<<<<<<<<<< DEFAULT VALUE!!! (h:/)
  }

  // default paths:
  setPath("PSInterpreter", "gs");
  setPath("PSViewer", "gv");
  setPath("FontPath", Str(FONTPATH)); 
  setPath("XSLTPath", Str(XMLPATH)); 
  
  if( !getSettings() )  
    verbose << "No $HOME/.pptout file found." << std::endl;
  // throwing an exception would be preferable
  // Isn't necessarily an error: we are unlikely to have a settings file
  // the first time we run the program.
}

Config::~Config( )
{
  if( !saveSettings() )
    warning << "Error: failed to save the settings!" << std::endl;
  // throwing an exception would be preferable
}

void Config::setOrientation( Papers::Orientation o )
{
  orientation = o;
}


Papers::Orientation Config::getOrientation(  )
{
  return orientation;
}


void Config::setPath(std::string strProgram, std::string strLocation)
{
  pathMap[strProgram] = strLocation;
  //pathTree.insert(*(new Prg(strProgram, strLocation)));
}


std::string Config::getPath(std::string strProgram)
{
  path_type::iterator iter = pathMap.find(strProgram);
  if(iter == pathMap.end() )
    return "";
  else
    return iter -> second;
}


void Config:: setPrintCmd(std::string strPrintCmd)
{
  strPrintCommand = strPrintCmd;
}

std::string Config::getPrintCmd( )
{
  return strPrintCommand;
}

bool Config::setResBoxSize( int size )
{
  if(size > 0 )
  {
    reshapeboxsize = size;
    return true;
  }
  return false;
}

int Config::getResBoxSize( )
{
  return  reshapeboxsize;
}

void Config::setUnit( Units::Unit u )
{
  unit = u;
}

Units::Unit Config::getUnit( )
{
  return unit;
}

void Config::setPaperName(std::string paper_name_)
{
  paper_name=paper_name_;
}

std::string Config::getPaperName()
{
  return paper_name;
}

void Config::setXY( float X, float Y )
{
  fX = X;
  fY = Y;
}

float Config::getX( )
{
  return fX;
}

float Config::getY( )
{
  return fY;
}

////////////////PRIVATE MEMBERS/////////////////////

bool Config::getSettings()
{
  const std::string filename = strPath + strSETTINGS_FILE;
  std::ifstream fin(filename.c_str());
  if(!fin) return false;

  //  char ch; 
  token_type type = ERR;
  headers hdr_type;
  char* szToken = new char[1];

  while(!fin.eof())
  {
    while( type != HEADER && !fin.eof() )
    { 
      delete[] szToken;
      szToken = nextToken(fin, type); 
    }

    hdr_type = getHeader(szToken);
    switch(hdr_type)
    {
    case R_BOX:
      do
      {
	delete[] szToken;
        szToken = nextToken( fin, type );
	if(type == KEY && !strncmp(szToken, "size", 4))
	{
	  delete[] szToken;
	  szToken = nextToken( fin, type );
	  if(type==VALUE && isdigit(*szToken))
	    { reshapeboxsize = atoi(szToken); }
	}
      } while(type != HEADER && !fin.eof() );
      break;

    case PAPER_SET:
      do
      {
        delete[] szToken;
        szToken = nextToken( fin, type );
	if(type != KEY) continue;
	if(!strncmp(szToken, "width", 5))
	{
	  delete[] szToken;
	  szToken = nextToken( fin, type );
	  if(type==VALUE && isdigit(*szToken))
	    { fX = atof(szToken); }
	} else if(!strncmp(szToken, "height", 6)) {
	  delete[] szToken;
	  szToken = nextToken( fin, type );
	  if(type==VALUE && isdigit(*szToken))
	    { fY = atof(szToken); }
	} else if(!strncmp(szToken, "orientation", 11)) {
	  delete[] szToken;
	  szToken = nextToken( fin, type );
	  if(type==VALUE && isdigit(*szToken))
	    { orientation = (Papers::Orientation) atoi(szToken); }
	} else if(!strncmp(szToken, "paper", 9)) {
	  delete[] szToken;
	  szToken = nextToken( fin, type );
	  if(type==VALUE && isalpha(*szToken))
	    paper_name=szToken;
	}
      } while(type != HEADER && !fin.eof() );
      break;

    case MAIN:
      do
      {
	delete[] szToken;
        szToken = nextToken( fin, type );
	if(type != KEY) continue;
	if(!strncmp(szToken, "unit", 4))
	{
	  delete[] szToken;
	  szToken = nextToken( fin, type );
	  if(type==VALUE && isdigit(*szToken))
	    { unit = (Units::Unit)atoi(szToken); }
	}
      } while(type != HEADER && !fin.eof() );
      break;

    case TOOLS:
      do
      {
	delete[] szToken;
        szToken = nextToken( fin, type );
	if(type != KEY) continue;
	if(!strncmp(szToken, "print", 5))
	{
	  delete[] szToken;
	  szToken = nextToken( fin, type );
	  if(type==VALUE && isalpha(*szToken))
	    { strPrintCommand = szToken; }
	}
      } while(type != HEADER && !fin.eof() );
      break;

    case PATHS:
      
      char* szKey;
      do
      {
	delete[] szToken;
        szToken = nextToken( fin, type );
        if(type != KEY) continue;
	szKey = szToken;
	szToken = nextToken( fin, type );
	if(type != VALUE) continue;
	pathMap[szKey] = szToken;
	delete[] szKey;
      } while(type != HEADER && !fin.eof() );
      break;

    default: type = ERR;
    }

  }

  delete[] szToken;
  fin.close();
  return true;
}


bool Config::saveSettings()
{
  std::ofstream fout( (strPath+strSETTINGS_FILE).c_str() ); //Ugly...
  if(!fout) return false;

  // this line printed a lot of binary garbage:
  //  fout << "Passepartout " << version << endl << endl;
  
  using std::endl;
  fout << "[reshape box]" <<endl
      << "size:\t" << reshapeboxsize << endl << endl;

  fout << "[paper settings]" << endl 
      << "paper:\t" << paper_name << endl
      << "width:\t" << fX << endl 
      << "height:\t" << fY << endl
      << "orientation:\t" << orientation << endl << endl;

  fout << "[main]" << endl 
      << "unit:\t" << unit << endl << endl;

  fout << "[tools]" << endl 
      << "print:\t" << strPrintCommand << endl << endl;

  fout << "[paths]" << endl;
  
  path_type::iterator iter = pathMap.begin();
  while(iter != pathMap.end() )
  {
    fout<<iter->first<<":\t"<<iter->second<< endl;
    ++iter;
  }

  fout.close();
  return true;
}


char* Config::nextToken(std::istream& fin, token_type& type )
{
  const int MAX_LENGTH = 50;
  char* szToken = new char[MAX_LENGTH+1];
  szToken[0] = EOF;
  int pos = 0;
  bool newline = false;

  char chNext = remove_spaces( fin, newline );
  //if(chNext == EOF) return string("");

  if(chNext == '[') // Header
  {
    chNext = remove_spaces( fin, newline );
    do
    {
      if(chNext == '\n')
	{ fin.unget(); break; }  // '\n' will indicate Key token later on
      szToken[pos++] = chNext;
      chNext = fin.get();
    } while( !fin.eof() && chNext != ']' && pos < MAX_LENGTH);
    type = HEADER;

  } else {
    if(newline)     //Key
    {
      while( !fin.eof() && chNext != ':' && pos < MAX_LENGTH )
	{ szToken[pos++] = chNext; chNext = fin.get(); }
      type = KEY;

    } else {        //Value (rest of the line)
      while( !fin.eof() && pos < MAX_LENGTH )
      {
        if( strchr("\n\r", chNext) ) { fin.unget(); break; }
	szToken[pos++] = chNext;
	chNext = fin.get();
      }
      type = VALUE;

    }
  }

  if(szToken[0] == EOF)
    { pos = 0; type = ERR; }// If no token could be found (eof etc)
  szToken[pos] = '\0';
  //  return string(token);
  return szToken;
}


char Config::remove_spaces(std::istream& fin, bool& eol)
{
  eol = false;
  char chNext;
  //  while( (chNext = fin.get()) )
  while( fin.get(chNext) ) // looks nicer - no compiler warning
  {
    if(chNext == '\n') eol = true;
    if( !isspace(chNext)) break;
  }

  return chNext;
}


Config::headers Config::getHeader(char *token)
{
  if(strlen(token) < 6)
  {
    if(!strncmp(token, "main", 4)) return MAIN;
    if(!strncmp(token, "tools", 5)) return TOOLS;
    if(!strncmp(token, "paths", 5)) return PATHS;
  } else {
    if(!strncmp("reshape box", token, 10)) return R_BOX;
    if(!strncmp(token, "paper settings", 14)) return PAPER_SET;
  }

  return UNDEF;
}
