// -*- c++ -*-
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
#ifndef CONFIG_H
#define CONFIG_H


#include "paper_sizes.h"
#include "units.h"
#include <string>
#include <iostream>
#include <map>


///////////////////////////////////////////////////////////////////////////////////
// Configuration class
//
// ******************************* PUBLIC METHODS *********************************
// void setOrientation (Orientation o)  Set paper orientation (portrait/landscape)
// Orientation getOrientation( )        Get current paper orientation
// int setPath(strProgram, strLocation) Sets/updates the path to a program
// string getPath(strProgram)           Gets the path to a program
// void setPrintCmd(strPrintCmd)        Change the print command line
// string getPrintCmd( )                Get current print command line
// void setResBoxSize( size )           Change the size of the reshape box
// int getResBoxSize( )                 Get the size of the reshape box
// void setUnit(Units::Unit unit)       Change the metric units used by the ruler etc
// Units::unit getUnit( )               Returns the metric units used by the ruler etc
// void setPaperName(string)
// string getPaperName()
// float setXY(fX, fY)                  Set the page size (in cm)
// float getX( )                        Get the page width
// float getY( )                        Get the page height
///////////////////////////////////////////////////////////////////////////////////


/**
 * Configuration class
 *
 * This class attempts to read settings in $HOME/.pptout
 * If that file doesn't exist it will be created using default settings.
 *
 * @author Sergej Gratchev
 * @version 0.1
 */
class Config
{
public:
   static const float fVERSION;
  const std::string strSETTINGS_FILE; //Name of the settings file



   /**
    * Constructor
    *
    * @param strFileName Name of the settings file
    * @param szPath Location of the settings file. NULL=default value ($HOME or 'h:/' if everything else fails). Don't forget the '/' on the end!
    */
  Config(const std::string strFileName=".pptout", const char *szPath=NULL);

   ~Config( );
   /**
    * Set paper orientation (portrait/landscape)
    *
    * @param o New orientation
    */
   void setOrientation( Papers::Orientation o );

   /** Get current paper orientation (portrait/landscape) */
   Papers::Orientation getOrientation( );

   /**
    * Sets/updates the path to a program
    *
    * @param strProgram Name of the program
    * @param strLocation The program's location
    */
  void setPath(std::string strProgram, std::string strLocation);

   /**
    * Gets the path to a program
    *
    * @param strProgram Name of the program
    * @return "" = No such program found
    */
  std::string getPath(std::string strProgram);

   /**
    * Change the print command line
    *
    * @param strPrintCmd New print command line.
    */
  void setPrintCmd(std::string strPrintCmd);

   /** Get current print command line */
  std::string getPrintCmd( );


   /**
    * Change the size of the reshape box
    */
   bool setResBoxSize( int size );

   /**
    * Get the size of the reshape box
    */
   int getResBoxSize( );

   /*
    * Change the typographic units used by the ruler etc
    *
    * @param unit New metric unit
    */
   void setUnit( Units::Unit u );

   /** Returns the typographic units used by the ruler etc*/
   Units::Unit getUnit( );

  void setPaperName(std::string paper_name_);
  std::string getPaperName();

   /**
    * Set the page size (in cm)
    *
    * @param fX Page width
    * @param fY Page height
    */
   void setXY( float X, float Y );

   /** Get the page width */
   float getX( );

   /** Get the page height */
   float getY( );


protected:
  std::string strPath, paper_name;
  float fX, fY;         // Page width / height
  int reshapeboxsize;

  Papers::Orientation orientation;
  float *fZoom;

  std::string strPrintCommand;
  Units::Unit unit;

  typedef std::map<std::string, std::string> path_type;
  path_type pathMap;


  /** Get settings from file */
  bool getSettings( );
  /** Save settings to file */
  bool saveSettings( );

  enum token_type { HEADER=1, KEY, VALUE, ERR };
  enum headers { UNDEF = 1, R_BOX, PAPER_SET, MAIN, TOOLS, PATHS };

  /** Returns next token */
  char *nextToken(std::istream& fin, token_type& type);
  char remove_spaces(std::istream& fin, bool& eol);

  headers getHeader( char *token );

};

#endif
