///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "globals.h"
#include "docpropsdialog.h"
#include <gtk--/box.h>
#include <gtk--/label.h>
#include <gtk--/separator.h>
#include "safeslot.h"
#include <list>
#include "paper_sizes.h"
#include "document.h"
#include "window.h"
#include "errordialog.h"
#include "fileerrors.h"
#include "config.h"
#include <util/warning.h>

Doc_Props_Dialog::Doc_Props_Dialog(Gtk::Window &parent, Document_View &view_):
  Gtk::Dialog(),
  template_button("Use template:"),
  portrait_button("Portrait"), landscape_button("Landscape"),
  single_sided_button("Single-sided"), double_sided_button("Double-sided"),
  frame("Document format"),
  file_entry("Select template"),
  view(view_),
  first_page(1, false)
{
  Gtk::HBox *template_box=manage(new Gtk::HBox),
    *sidedness_box=manage(new Gtk::HBox),
    *format_box=manage(new Gtk::HBox),
    *first_page_box=manage(new Gtk::HBox);
  Gtk::VBox *frame_box=manage(new Gtk::VBox);
    
  set_transient_for(parent);
  set_border_width(border_width);
  get_vbox()->set_spacing(5);
  set_modal(true);
  set_title("Document properties");
  set_policy(false, false, true);

  {
    std::string default_paper=config.getPaperName();
    bool default_is_known=false;
    std::list<std::string> paper_list;
    for(std::map<std::string, Papers::Size>::const_iterator i=papers.sizes.begin();
	i!=papers.sizes.end();
	i++)
      {
	paper_list.push_back(i->first);
	if(i->first==default_paper)
	  default_is_known=true;
      }
    paper_size.set_popdown_strings(paper_list);
    if(!default_is_known)
      {
	warning << "Paper type \"" << default_paper 
		<< "\" is unknown." << std::endl;
	default_paper="A4";
	config.setPaperName("A4");
      }
    paper_size.get_entry()->set_text(default_paper);
  }

  portrait_button.set_group(landscape_button.group());
  single_sided_button.set_group(double_sided_button.group());

  template_box->pack_start(template_button, false, false, 0);
  template_box->pack_start(file_entry, true, true, 0);

  format_box->pack_start(paper_size, false, false, 0);
  format_box->pack_start(portrait_button, false, false, 10);
  format_box->pack_start(landscape_button, true, true, 5);

  sidedness_box->pack_start(single_sided_button, false, false, 0);
  sidedness_box->pack_start(double_sided_button, true, true, 5);

  frame_box->pack_start(*format_box, true, false, 0);
  frame_box->pack_start(*manage(new Gtk::HSeparator()), true, false, 5);
  frame_box->pack_start(*sidedness_box, true, false, 0);
  frame.set_border_width(border_width);
  frame_box->set_border_width(border_width);
  frame.add(*frame_box);

  first_page_box->pack_start(*manage(new Gtk::Label("First page number:")), 
			    false, false, 0);
  first_page_box->pack_start(first_page, false, false, 5);

  get_vbox()->pack_start(*template_box, true, false, 0);
  get_vbox()->pack_start(frame, true, false, 0);
  get_vbox()->pack_start(*first_page_box, true, false, 0);

  template_button.set_active(true);
  portrait_button.set_active(config.getOrientation()==Papers::PORTRAIT);
  double_sided_button.set_active(true);

  paper_size.get_entry()->set_editable(false);

  Gtk::Button *ok_button = manage(new Gtk::Button(" Ok ")),
    *cancel_button = manage(new Gtk::Button(" Cancel "));
  ok_button->set_flags(GTK_CAN_DEFAULT);
  cancel_button->set_flags(GTK_CAN_DEFAULT);
  get_action_area()->pack_start(*ok_button, true, false, 0);
  get_action_area()->pack_start(*cancel_button, true, false, 0);

  ok_button->clicked.connect(safeslot(this, &Doc_Props_Dialog::ok));
  cancel_button->clicked.connect(safeslot(this, &Doc_Props_Dialog::cancel));
  template_button.clicked.connect(safeslot(this, 
					   &Doc_Props_Dialog::update_mode));
  update_mode();

  get_vbox()->show_all();
  get_action_area()->show_all();
}

void Doc_Props_Dialog::update_mode()
{
  bool active=template_button.get_active();
  frame.set_sensitive(!active);
  file_entry.set_sensitive(active);
}

void Doc_Props_Dialog::show_it(bool new_document_)
{
  Document *document=view.get_document();
  new_document=new_document_;
  if(document && !new_document)
    {
      std::string template_file=document->get_template_file();
      template_button.set_active(template_file!="");
      first_page.set(document->get_first_page_num());
      portrait_button.set_active(document->get_orientation()
				 == Papers::PORTRAIT);
      double_sided_button.set_active(document->is_doublesided());
      paper_size.get_entry()->set_text(document->get_paper_name());

      show();
      // Gtk::Entry::set_position doesn't seem to work 
      // unless the entry is shown first
      file_entry.entry.set_text(template_file);
    }
  else
    {
      template_button.set_active(true);
    }
  show();
}


void Doc_Props_Dialog::ok()
{
  Document *document=view.get_document();
  if(template_button.get_active() && file_entry.entry.get_text()=="")
    {
      Error_Dialog::view("No template file selected");
      return;
    }
  if(new_document && document)
    {
      try
	{
	  manage(new Frame_Window("", this));
	  hide();
	}
      catch(const Error::Read &e)
	{
	  Error_Dialog::view(e.message);
	}
    }
  else
    {
      try
	{
	  if(template_button.get_active()) // create from template
	    {
	      if(!document)
		document=new Document(file_entry.entry.get_text());
	      else
		document->set_template(file_entry.entry.get_text());
	      document->set_first_page_num(first_page.get());
	    }
	  else
	    {
	      Papers::Orientation orientation=
		portrait_button.get_active()
		? Papers::PORTRAIT : Papers::LANDSCAPE; 
	      if(!document)
		document=new Document();
	      document->set_template("");
	      document->set_paper_name(paper_size.get_entry()->get_text());
	      document->set_orientation(orientation);
	      document->set_doublesided(double_sided_button.get_active());
	      document->set_first_page_num(first_page.get());
	    }
	  if(new_document)
	    view.set_document(document);
	  hide();
	}
      catch(const Error::Read &e)
	{
	  Error_Dialog::view(e.message);
	}
    }
}

void Doc_Props_Dialog::cancel()
{
  hide();
}
