///
// Copyright (C) 2002, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include <strstream>
//#include <unistd.h>
#include <algorithm>
#include <stdexcept>
#include <ctime>

#include <xml++.h>

#include "document.h"
#include "fileerrors.h"
#include "page.h"
#include "globals.h"
#include "textstream.h"
#include <util/stringutil.h>
#include <util/filesys.h>
#include <util/warning.h>
#include <util/os.h>

SigC::Signal1<void, Document*> Document::changed_signal;
SigC::Signal1<void, Document*> Document::size_changed_signal;
SigC::Signal1<void, Document*> Document::filename_changed_signal;
SigC::Signal1<void, Document*> Document::selection_changed_signal;
SigC::Signal1<void, Document*> Document::streams_changed_signal;
SigC::Signal1<void, Document*> Document::deleted_signal;

namespace {			// local
  class same_name {
  public:
    same_name(const std::string& name) : name_(name) {};
    bool operator() (const Text_Stream* stream) {
      return stream->get_name() == name_;
    }
  private:
    const std::string name_;
  };
  class NameOrder {
  public:
    bool operator() (const Text_Stream* x, const Text_Stream* y) const {
      return x && y && (x->get_name() < y->get_name());
    }
  };
}

std::string Document::make_up_new_name()
{
  static unsigned int num = 1;
  while(num < UINT_MAX) {
    std::ostrstream tmp;
    tmp << "stream" << num << std::ends;
    if(find_if(text_streams.begin(), text_streams.end(), same_name(tmp.str()))
       == text_streams.end())
      return tmp.str();
    // up the number only afterwards if no match, so if the user creates a
    // stream, renames it, and creates another one, he don't loose numers.
    ++num;
  }
  // Todo:  Try again from 1, in case a stream was deleted or renamed, but
  // only once, so we don't get stuck in an eternal loop.  Don't start from 1
  // each time, to avoid confusing the user
  throw std::runtime_error("Out of automatical stream names");
}

int Document::count_selected() const {
  return selection.size();
}

const Document::Selection& Document::selected() const {
  return selection;
}

void Document::select_all(bool select)
{
  selection.clear();

  if(select)
    for(PageVec::const_iterator p = pages.begin(); p != pages.end(); ++p)
      for(Group::ChildVec::const_iterator i = (*p)->pbegin(); 
	  i != (*p)->pend(); ++i)
	selection.push_back(*i);
  
  selection_changed_signal(this);
}

void Document::deselect(Pagent* obj) 
{
  Selection::iterator i=find(selection.begin(), selection.end(), obj);
  if(i!=selection.end())
    {
      selection.erase(i);
      selection_changed_signal(this);
    }
}

void Document::select(Pagent* obj, bool deselect_old) 
{
  if(deselect_old) selection.clear();
  selection.push_back(obj);

  selection_changed_signal(this);
}

void Document::delete_selected()
{
  for(Selection::const_iterator i = selection.begin(); 
      i != selection.end();
      ++i) 
    {
      Pagent* obj = (*i)->get_parent().ungroup(*i);
      if (!obj)
	throw std::logic_error("Tried to remove pagent from bad parent.");
      delete obj;
    }
  selection.clear();

  selection_changed_signal(this);
}


Document::StreamVec Document::get_text_streams()
{
  StreamVec tmp=text_streams;
  tmp.sort(NameOrder());
  return tmp;
}

void Document::rename_text_stream(std::string old_name, std::string new_name)
{
  Text_Stream *stream=get_text_stream(old_name);
  if(!stream)
    throw Error::Text_Stream_Name();
  if(old_name==new_name)
    return; //not an error
  if(get_text_stream(new_name))
    throw Error::Text_Stream_Name();
  stream->set_name(new_name);
  streams_changed_signal(this);
}

std::string Document::add_text_stream(std::string file, std::string name, std::string transform)
{
  if(name=="") 
    name=make_up_new_name();
  for(StreamVec::const_iterator i=text_streams.begin();
      i!=text_streams.end();
      i++)
    if((*i)->get_name()==name)
      throw Error::Text_Stream_Name();
  text_streams.push_back(new Text_Stream(name, file, transform));
  streams_changed_signal(this);
  return name;
}

Text_Stream* Document::get_text_stream(std::string name)
{
  for(StreamVec::iterator i=text_streams.begin();
      i!=text_streams.end();
      i++)
    if((*i)->get_name()==name)
      return *i;
  return NULL;
}

void Document::remove_text_stream(std::string name)
{
  for(StreamVec::iterator i=text_streams.begin();
      i!=text_streams.end();
      i++)
    if((*i)->get_name()==name)
      {
	delete *i;
	text_streams.erase(i);
	streams_changed_signal(this);  
	return;
      }
}

std::string Document::to_relative(std::string filename) const
{
  return relative_path(path(this->filename), filename);
}

std::string Document::from_relative(std::string filename) const
{
  if(filename.empty() || filename[0]=='/')
    return filename;
  else
    return path(this->filename)+filename;
}

unsigned int Document::get_num_of_pages()
{
  unsigned int num=0;
  PageVec::const_iterator i=pages.begin();

  while(i++!=pages.end()) num++;

  return num;
}

void Document::delete_page(int page_num)
{
  int num_of_pages=get_num_of_pages();

  if(page_num>=first_page_num 
     && page_num<first_page_num+num_of_pages)
    {
      int j=first_page_num;
      PageVec::iterator i=pages.begin();

      while(j<page_num && j<first_page_num+num_of_pages)
	{
	  i++;
	  j++;
	}
      delete *i;
      pages.erase(i);
    }
  else throw Error::Invalid_Page_Num(page_num);
  
  // Todo: temporary fix to make sure deleted objects 
  //are not still selected:
  select_all(false); 

  changed_signal(this);
}

Page *Document::new_page(int page_num, Page *original)
{
  PageVec::iterator i=pages.begin();
  int num_of_pages=get_num_of_pages();
  Page *the_new_page;

  if(original)
    {
      XMLNode *tmp=original->save();
      the_new_page=new Page(*tmp, *this);
      delete tmp;
    }
  else
    the_new_page=new Page(*this);

  if(num_of_pages==0)
    {
      pages.push_front(the_new_page);
    }
  else if(page_num>=first_page_num 
	  && page_num<=first_page_num+num_of_pages)
    {
      int j=first_page_num;

      while(j<page_num && j<first_page_num+num_of_pages)
	{
	  i++;
	  j++;
	}
      
      pages.insert(i, the_new_page);
    }
  else 
    {
      delete the_new_page;
      the_new_page=0;
      throw Error::Invalid_Page_Num(page_num);
    }

  changed_signal(this);
  return the_new_page;
}

int Document::get_page_num_of_page(const Page *page) const {
  int j=get_first_page_num();
  for(PageVec::const_iterator i=pages.begin(); i!=pages.end(); i++, j++)
    {
      if(page==*i)
	return j;
    }
  throw Error::Invalid_Page_Num();
}

Page *Document::get_page(int page_num)
{
  int j=first_page_num;
  PageVec::iterator i=pages.begin();
  int num_of_pages=get_num_of_pages();

  if(num_of_pages==0)
    return NULL;

  while(j<page_num && j<first_page_num+num_of_pages-1)
    {
      i++;
      j++;
    }

  return *i;
}

Page *Document::get_page(std::string page_name)
{
  for(PageVec::iterator i=pages.begin();
      i!=pages.end();
      i++)
    {
      if((*i)->get_name()==page_name)
	return *i;
    }

  return 0;
}

Document::Document()
  :is_template(false), paper_name("A4"), filename(""),
   orientation(Papers::PORTRAIT), 
   the_template(0)
{
  pages.push_back(new Page(*this));
  set_doublesided(true);
  set_first_page_num(1);
}

std::list<std::string> Document::get_template_pages()
{
  std::list<std::string> tmp;
  if(the_template)
    {
      Document *t=the_template;
      for(int i=t->get_first_page_num(); 
	  i<t->get_first_page_num()+((int) t->get_num_of_pages());
	  i++)
	tmp.push_back(t->get_page(i)->get_name());
    }
  return tmp;
}

void Document::open()
{
  XMLTree tree(filename);
  xml_open(tree.root());
}

void Document::save()
{
  XMLTree tree(filename);
  if(tree.root())
    delete tree.root();
  tree.set_root(xml_save());
  if(!tree.write())
    throw Error::Write("Could not write to "+filename);
}

void Document::xml_open(XMLNode *root)
{
  std::string temp_template="";

  if(!root)
    throw Error::Read("Root node is NULL.");
  if(root->name()!="document")
    throw Error::Read("\"document\" tag missing.");

  //default values if property is not encountered:
  set_doublesided(true); 
  set_orientation(Papers::PORTRAIT);
  set_first_page_num(1);
  set_paper_name("A4");

  //read document properties:
  XMLPropertyList properties=root->properties();
  for(XMLPropertyList::iterator i=properties.begin();
      i!=properties.end();
      i++)
    {
      const std::string name=(*i)->name();
      if(name=="template")
	temp_template=(*i)->value();
      else if(name=="doublesided")
	set_doublesided((*i)->value()=="true");
      else if(name=="landscape")
	{
	  if((*i)->value()=="true")
	    set_orientation(Papers::LANDSCAPE);
	}
      else if(name=="paper_name")
	try
	  {
	    set_paper_name((*i)->value());
	  }
	catch (Error::Paper_Name e)
	  {
	    throw Error::Read("There is no paper called \""+e.name+"\".");
	  }
      else if(name=="first_page_num")
	{
	  set_first_page_num(to<int>((*i)->value()));
	}
      else
	warning << "Unknown property \"" << (*i)->name()
		<< "\" ignored in <document>." << std::endl;
    }

  if(!is_template) // templates can't have templates
    set_template(from_relative(temp_template)); 
  // the template overrides anything  explicitly stated in the document

  // read text streams and pages:
  XMLNodeList children=root->children();
  for(XMLNodeList::iterator i=children.begin();
      i!=children.end();
      i++)
    {
      std::string name=(*i)->name();
      if(name=="text_stream") 
	{
	  // if no name we can't add it because a name would be assigned for it
	  std::string name, file, transform;
	  if(const XMLProperty* n_prop = (*i)->property("name"))
	    name = n_prop->value();
	  if(const XMLProperty* f_prop = (*i)->property("file"))
	    file = f_prop->value();
	  if(const XMLProperty* x_prop = (*i)->property("transform"))
	    transform = x_prop->value();
	  
	  if(name.empty()) //permit streams without an association
	    throw Error::Read("Could not load text stream, name \""
			      + name + "\", file \"" + file + "\"");
	  
	  if(!get_text_stream(name))
	    add_text_stream(from_relative(file), name, 
			    from_relative(transform));
	  // template streams override document streams
	  // if two streams have the same name, the second will be ignored
	}
      else if(name=="page")
	{
	  Page *page=new Page(*i, *this);
	  if(!page)
	    throw Error::Read("Could not add page.");
	    pages.push_back(page);
	}
      else
	warning << "Unknown tag \"" << (*i)->name()
		<< "\" ignored in <document>." << std::endl;
    }
}

Document::Document(std::string template_file_)
  :is_template(false), filename(""), the_template(0)
{
  set_first_page_num(1);
  set_template(template_file_);
}

void Document::set_filename(std::string filename_)
{
  filename=expand_path(filename_); 
  filename_changed_signal(this);
}

void Document::set_template(std::string template_file_)
{
  template_file=template_file_;
  if(!template_file.empty())
    {
      if(the_template)
	delete the_template;
      the_template=new Document(template_file, true);
      set_doublesided(the_template->is_doublesided());
      set_orientation(the_template->get_orientation());
      set_paper_name(the_template->get_paper_name());
      StreamVec ts=the_template->get_text_streams();
      for(StreamVec::iterator i=ts.begin();
	  i!=ts.end();
	  i++)
	{
	  try
	    {
	      Text_Stream* tmp=get_text_stream((*i)->get_name());
	      if(!tmp)
		add_text_stream((*i)->get_association(), (*i)->get_name(),
				(*i)->get_transform());
	      else // override
		tmp->set_association((*i)->get_association());
	    }
	  catch (Error::Text_Stream_Name e)
	    {
	      warning << e.what() << std::endl;
	    }
	}
    }
}

Document::Document(std::string filename_, bool is_template_)
  :is_template(is_template_), the_template(0)
{
  filename=expand_path(filename_);
  open();
}

Document::~Document()
{
  deleted_signal(this);
  if(the_template)
    delete the_template;
  while(!pages.empty())
    {
      Page *page=*pages.begin();
      pages.pop_front();
      delete page;
    }
  while(!text_streams.empty())
    {
      Text_Stream *text_stream=*text_streams.begin();
      text_streams.pop_front();
      delete text_stream;
    }
}

void Document::set_doublesided(bool ds)
{
  doublesided=ds;
  changed_signal(this);
}

void Document::set_first_page_num(int num)
{
  first_page_num=num;
  changed_signal(this);
}

void Document::set_orientation(Papers::Orientation _orientation)
{
  orientation=_orientation;
  changed_signal(this);
  size_changed_signal(this);
}

void Document::set_paper_name(std::string _paper_name)
{
  if(papers.sizes.find(_paper_name)==papers.sizes.end())
    throw Error::Paper_Name(_paper_name);
  paper_name=_paper_name;
  changed_signal(this);
  size_changed_signal(this);
}

XMLNode *Document::xml_save()
{
  XMLNode *root=new XMLNode("document");
  root->add_property("paper_name", get_paper_name());

  if(the_template)
    root->add_property("template", to_relative(template_file));
  if(is_doublesided()) 
    root->add_property("doublesided", "true");
  else
    root->add_property("doublesided", "false");

  if(get_orientation()==Papers::LANDSCAPE)
    root->add_property("landscape", "true");
  else
    root->add_property("landscape", "false");

  root->add_property("first_page_num", tostr(get_first_page_num()));
  root->add_content("\n");
  for(StreamVec::iterator i=text_streams.begin(); 
      i!=text_streams.end(); 
      i++)
    {
      root->add_content("  ");
      XMLNode *tmp=root->add_child("text_stream");
      root->add_content("\n");
      tmp->add_property("name", (*i)->get_name());
      tmp->add_property("file", to_relative((*i)->get_association()));
      tmp->add_property("transform", to_relative((*i)->get_transform()));
    }
  for(PageVec::iterator i=pages.begin(); i!=pages.end(); i++)
    {
      root->add_content("  ");
      root->add_child((*i)->save());  
      root->add_content("\n");
    }
  return root;
}

void Document::print(std::ostream& out, int first_page, int last_page,
		     bool eps)
{
  using std::endl;
  
  time_t the_time=std::time(NULL);

  if(!(first_page>=first_page_num 
       && first_page<=last_page
       && last_page<=first_page_num+((int) get_num_of_pages())-1))
    throw Error::Print("Bad page interval");

  // ignore request to print multible-page EPS 
  eps=eps && first_page==last_page;

  int w = (int) (get_width()+0.5);
  int h = (int) (get_height()+0.5);
  
  if(eps) 
    {
      out << "%!PS-Adobe-3.0 EPSF-3.0" << endl
	  << "%%BoundingBox: 0 0 "<<w<<" "<<h<<endl;
    }
  else
    out << "%!PS-Adobe-3.0" << endl;
  
  out <<"%%DocumentData: Clean8Bit"<<endl
      <<"%%LanguageLevel: 2"<<endl
    // actually, we can't really know about the level of the eps files ...
      <<"%%Orientation: "
      <<(orientation==Papers::PORTRAIT?"Portrait":"Landscape")<<endl
      <<"%%Pages: "<<get_num_of_pages()<<endl
      <<"%%PageOrder: Ascend"<<endl
      <<"%%Title: "<<basename(filename)<<endl
      <<"%%CreationDate: "<<std::ctime(&the_time) // <<endl
    // ctime seems to add a newline
      <<"%%Creator: Passepartout "<<version
      <<" by Fredrik Arnerup & Rasmus Kaj"<<endl;
  if(true) // perhaps the user is into cloak and dagger stuff ...
    out << "%%For: " << os::fullname()
	<< " <" << os::username() << "@" << os::hostname() << ">"
	<< " (" << os::machine() << ", " << os::sysname()
	<< " " << os::release() << ")" << endl;
  out <<"%%EndComments"<<endl
      <<endl 
      <<"%%BeginProlog"<<endl
      <<"%%EndProlog"<<endl
      <<endl
      <<"%%BeginSetup"<<endl
      <<"%%EndSetup"<<endl
      <<endl;
  
  int page_num=first_page_num;
  for(PageVec::const_iterator i=pages.begin(); i!=pages.end(); i++)
    {
      if(page_num>=first_page && page_num<=last_page)
	{
	  out << endl << "%%Page: \""
	      <<(*i)->get_name()<<"\" "<<page_num<<endl;

	  // If we have a bounding box, we may not draw outside of it.
	  // Shouldn't really be a problem, but one never knows ...
	  if(eps)
	    out <<endl
		<<"gsave"<<endl
		<<"0 0 moveto"<<endl
		<<w<<" 0 rlineto"<<endl
		<<"0 "<<h<<" rlineto"<<endl
		<<-w<<" 0 rlineto"<<endl
		<<"closepath clip newpath"<<endl<<endl; 
	  //clip doesn't make implicit newpath

	  (*i)->print(out);

	  if(eps)
	    out<<endl<<"grestore"<<endl;
	}
      page_num++;
    }
  out << endl << "%%EOF" << endl;
}
